function Clustering(X, varargin)

% Reference:
% Evrim Acar, Rasmus Bro, Bonnie Schmidt, New Exploratory Metabonomic
% Tools, Submitted to Journal of Chemometrics, 2007.
%
%
% Clustering(X, varargin)
% This function takes as input original data, distance and clustering
% preferences and computes the model (if any selected), applies
% hierarchical clustering based on the distance and linkage method given as
% input parameters. 
%
% Inputs:   X: two-way or three-way array
%           vargin(1): flag_label 0/1
%           vargin(2): flag_color 0/1
%           vargin(3): class id
%           vargin{4}: linkage method
%           vargin{5}: distance measure
% optional: method : {'tucker', 'parafac', 'matrix'}
%           param  : {number of components, 'scores' or 'residuals'} if tucker or parafac,
%                               1  if  matrix
%                               
% Example: For three-way array, X,
%               Clustering(X,0,1,1,'complete','mahalanobis', 'tucker',{[2 2 2], 'scores'});
%               Clustering(X,0,1,2,'complete','mahalanobis', 'parafac',{2, 'scores'});
%               Clustering(X,0,1,1,'complete','mahalanobis', 'matrix', 1);
%
%          For two-way array, X
%               Clustering(X,0,1,1,'complete','mahalanobis');
%
% 
% Last modified: 05/21/2007

if isa(X,'dataset')
  Data=X.data;
else
  Data=X;
end

order=length(size(Data));

if (nargin > 3)
    flag_label=varargin{1}; 
    flag_color=varargin{2};
    class_id=varargin{3};
    method= varargin{4};
    distance=varargin{5};
    okargs = {'tucker' 'parafac' 'matrix'};
    for j=6:2:nargin-1
        pname = varargin{j};
        pval = varargin{j+1};
        k = strmatch(lower(pname), okargs);
        if isempty(k)
            error('BadParameter',...
                  'Unknown parameter name:  %s.',pname);
        elseif length(k)>1
            error('BadParameter',...
                  'Ambiguous parameter name:  %s.',pname);
        else
            switch(k)
            case 1  % TUCKER
                if ~isempty(pval)
                    if (order==3)
                        nb_comp=pval{1};
                        if (length(nb_comp)==3)
                            model=tucker(X,nb_comp);
                            if strmatch(lower(pval{2}), 'residuals')
                                Data=model.ssqresiduals{1};                                 
                            elseif strmatch(lower(pval{2}), 'scores')
                                Data=model.loads{1};
                            end
                            save tucker_model.mat model
                            D=pdist(Data,distance);                          
                         else
                            error('Component number for each mode should be entered for a TUCKER3 model');
                        end
                    else
                        error('Data : Not multiway');
                    end
                else
                    error('Component number for each mode should be entered for a TUCKER3 model');
                end
            case 2 % PARAFAC
                if ~isempty(pval)
                    if (order==3)
                        nb_comp=pval{1};
                        if (length(nb_comp)==1)
                            model=parafac(X,nb_comp);
                            if strmatch(lower(pval{2}), 'residuals')
                                Data=model.ssqresiduals{1};                                                        
                            elseif strmatch(lower(pval{2}), 'scores')
                                Data=model.loads{1};
                            end
                            save parafac_model.mat model
                            D=pdist(Data,distance);  
                        else
                            error('Component number for each mode should be the same for a PARAFAC model');
                        end
                    else
                       error('Data : Not multiway');
                    end 
                else
                    error('A component number should be entered for a PARAFAC model');
                end
            case 3 % Matrix
                if (pval==1)
                    %ignores distance and computes the frb. norm of matrix
                    %difference
                    D=pdist_threeway(Data);
                else
                    error('In order to set matrix-wise distance measure for a 3-way array, set the parameter to 1');
                end   
            end
        end
    end
else
    help Clustering.m
    error('Check input parameters');    
end

if (order==2)
    D=pdist(Data,distance);
end

%Pairwise distances between objects 
DistMatrix=squareform(D);

%Merge clusters 
Z = linkage(D,method);

%Assign which class to use in coloring
if (flag_color)
    if (class_id==1)
        chosen_class=X.class{1};
    elseif (class_id>1)
        chosen_class=X.class{1,class_id};
    end
end

%Dendrogram
initial_slidervalue=0;
if ((flag_label==1) & (flag_color==1))
    dendrogram_version1(X, Z, initial_slidervalue, 'CLASSCOLOR',chosen_class, 'LABELS', X.label{1});
elseif ((flag_label==1) & (flag_color==0))
    dendrogram_version1(X, Z, initial_slidervalue, 'skip_color', 0, 'LABELS',X.label{1});
elseif ((flag_color==1) & (flag_label==0))
    dendrogram_version1(X, Z, initial_slidervalue, 'CLASSCOLOR',chosen_class,'skip_label',0);
else
    dendrogram_version1(X, Z, initial_slidervalue, 'skip_color',0,'skip_label',0);    
end
