function [Snew] = noseproc(NoseData,baseline,transform,feature,centscal,options)

% NOSEPROC For pre-processing eletronic nose/tongue data
%
% Input
% S            Three-way IxJxK array containing the data 
%              I: number of samples
%              J: number of times measured
%              K: Number of sensors
%
% Baseline     
% manipulation Vector that defines baseline manipulation
%              [0]: None
%              [1]: Difference;    S-S0
%              [2]: Fractionel;    (S-S0)/S0
%              [3]: Relative;      S/S0
%
%              S0 is the first sensor value. Ie for sensor 3, S0 = X(:,1,3)
%              for all samples.
%
% Transform    Defines transformation of the baseline manipulated data
%              [0] : No transformations
%              [1] : Logarithm;
%              [2] : First derivative
%              [3] : Second derivative
%
% Feature      Defines the kind of features extracted from the baseline manipulated and
%              transformed data
%              [0]    : No feature selection (keep the data as is)
%              [1 v w]: Subset selection using every w'th time from time v
%              [2]    : Absolute maximum
%              [3]    : Maximal slope of adsorption part
%              [4]    : Minimal slope (negative) of desorption part
%              [5]    : Unfold data to SAMPLE x (TIME x SENSOR)
%
% Centscal     Centering and scaling options
%              [0]: None
%              [1]: Center samples to remove baseline
%              [2]: Scale sensors to equal variation
%              [3]: Do both
%
% Options      Additional settings
%              Options(1) = 1 : Show screen info 
%
% I/O [Snew] = noseproc(NoseData,baseline,transform,feature,centscal,options);
%
% Copyright, Thomas Skov & Rasmus Bro (rb@kvl.dk), 2004


% Notes for developers:
%   Calculation of derivatives can likely be improved (currently based on
%   GRADIENT.m)

% Check input
if nargin<5
  error(' FDD inputs must be given. Type HELP NOSEPROC for more information')
end
if nargin<6
  options=1;
end

if options(1)==1
  disp(' ')
  disp('     ')
  disp(' Pre-processing electronic nose/tongue data')
  disp('     ')
  disp(' ')
end


id1=find(NoseData.IncludedX{1});
id2=find(NoseData.IncludedX{2});
id3=find(NoseData.IncludedX{3});
S = squeeze(NoseData.data(id1,id2,id3));
Snew = S;

%%%%%%%%%%%%% BASELINE MANIPULATION

if length(size(S))==3
    [I,J,K]=size(S);
    if baseline == 0 % None
        if options(1)==1
            disp(' - No baseline manipulation')
        end
    elseif baseline == 1 % Difference
        for i=1:I
            for k = 1:K
                Snew(i,:,k) = S(i,:,k)-S(i,1,k);
            end
        end
        if options(1)==1
            disp(' - Subtracting S0')
        end
    elseif baseline == 2 % Fractional
        for i=1:I
            for k = 1:K
                Snew(i,:,k) = (S(i,:,k)-S(i,1,k))/S(i,1,k);
            end
        end
        if options(1)==1
            disp(' - Subtracting S0 and scaling by S0')
        end
    elseif baseline == 3 % Relative
        for i=1:I
            for k = 1:K
                Snew(i,:,k) = S(i,:,k)/S(i,1,k);
            end
        end
        if options(1)==1
            disp(' - Scaling by S0')
        end
    else
        error(' Second input BASELINE must be either 0, 1, 2 or 3. See help for more info')
    end
    
elseif length(size(S)) == 2 & size(id1,1) == 1;
    [J,K]=size(S);
    if baseline == 0 % None
        if options(1)==1
            disp(' - No baseline manipulation')
        end
    elseif baseline == 1 % Difference
        for k = 1:K
            Snew(:,k) = S(:,k)-S(1,k);
        end
        if options(1)==1
            disp(' - Subtracting S0')
        end
    
    elseif baseline == 2 % Fractional
        for k = 1:K
            Snew(:,k) = (S(:,k)-S(1,k))/S(1,k);
        end    
        if options(1)==1
            disp(' - Subtracting S0 and scaling by S0')
        end
    elseif baseline == 3 % Relative
        for k = 1:K
            Snew(:,k) = S(:,k)/S(1,k);
        end
        if options(1)==1
            disp(' - Scaling by S0')
        end
    else
        error(' Second input BASELINE must be either 0, 1, 2 or 3. See help for more info')
    end
    
elseif length(size(S)) == 2 & size(id3,1) == 1;
    [I,J]=size(S);
    if baseline == 0 % None
        if options(1)==1
            disp(' - No baseline manipulation')
        end
    elseif baseline == 1 % Difference
        for i=1:I
            Snew(i,:) = S(i,:)-S(i,1);
        end
        if options(1)==1
            disp(' - Subtracting S0')
        end
    elseif baseline == 2 % Fractional
        for i=1:I
            Snew(i,:) = (S(i,:)-S(i,1))/S(i,1);
        end
        if options(1)==1
            disp(' - Subtracting S0 and scaling by S0')
        end
    elseif baseline == 3 % Relative
        for i=1:I
            Snew(i,:) = S(i,:)/S(i,1);
        end
        if options(1)==1
            disp(' - Scaling by S0')
        end
    else
        error(' Second input BASELINE must be either 0, 1, 2 or 3. See help for more info')
    end
end

%%%%%%%%%%%%%%%% TRANSFORM

if length(size(S)) == 3
    if transform(1) == 0  % None
        if options(1)==1
            disp(' - No transformation selected')
        end
    elseif transform(1) == 1  % Logarithm
        try X(find(Snew<=0))=eps;
            X=reshape(X,I,J,K);
            for i=1:I;
                for j=1:J;
                    for k=1:K;
                        if X(i,j,k)==0
                            Snew(i,j,k)=Snew(i,j,k);
                        else
                            Snew(i,j,k)=X(i,j,k);
                        end
                    end
                end
            end
        catch
            Snew=Snew;
        end
        Snew = log(Snew);
        if options(1)==1
            disp(' - Taking logarithm of scaled data')
        end
    elseif transform(1) == 2  % First derivative
        for i=1:I
            for k=1:K
                Snew(i,:,k)=gradient(Snew(i,:,k));
            end
        end
        if options(1)==1
            disp(' - Taking first derivative of scaled data')
        end
    elseif transform(1) == 3  % Second derivative
        for i=1:I
            for k=1:K
                Snew(i,:,k)=gradient(gradient(Snew(i,:,k)));
            end
        end
        if options(1)==1
            disp(' - Taking second derivative of scaled data')
        end
    else
        error(' The third input TRANSFORM has to be 0, 1, 2, or 3. Type HELP NOSEPROC for more info')
    end
    
elseif length(size(S)) == 2 & size(id1,1) == 1;
    if transform(1) == 0  % None
        if options(1)==1
            disp(' - No transformation selected')
        end
    elseif transform(1) == 1  % Logarithm
        try X(find(Snew<=0))=eps;
            X=reshape(X,J,K);
            for j=1:J;
                for k=1:K;
                    if X(j,k)==0
                        Snew(j,k)=Snew(j,k);
                    else
                        Snew(j,k)=X(j,k);
                    end
                end
            end
        catch
            Snew=Snew;
        end
        Snew = log(Snew);
        if options(1)==1
            disp(' - Taking logarithm of scaled data')
        end
    elseif transform(1) == 2  % First derivative
        for k=1:K
            Snew(:,k)=gradient(Snew(:,k));
        end
        if options(1)==1
            disp(' - Taking first derivative of scaled data')
        end
    elseif transform(1) == 3  % Second derivative
        for k=1:K
            Snew(:,k)=gradient(gradient(Snew(:,k)));
        end
        if options(1)==1
            disp(' - Taking second derivative of scaled data')
        end
    else
        error(' The third input TRANSFORM has to be 0, 1, 2, or 3. Type HELP NOSEPROC for more info')
    end
    
elseif length(size(S)) == 2 & size(id3,1) == 1;
    if transform(1) == 0  % None
        if options(1)==1
            disp(' - No transformation selected')
        end
    elseif transform(1) == 1  % Logarithm
        try X(find(Snew<=0))=eps;
            X=reshape(X,I,J);
            for i=1:I;
                for j=1:J;
                    if X(i,j)==0
                        Snew(i,j)=Snew(i,j);
                    else
                        Snew(i,j)=X(i,j);
                    end
                end
            end
        catch
            Snew=Snew;
        end
        Snew = log(Snew);
        if options(1)==1
            disp(' - Taking logarithm of scaled data')
        end
    elseif transform(1) == 2  % First derivative
        for i=1:I
            Snew(i,:)=gradient(Snew(i,:));
        end
        if options(1)==1
            disp(' - Taking first derivative of scaled data')
        end
    elseif transform(1) == 3  % Second derivative
        for i=1:I
            Snew(i,:)=gradient(gradient(Snew(i,:)));
        end
        if options(1)==1
            disp(' - Taking second derivative of scaled data')
        end
    else
        error(' The third input TRANSFORM has to be 0, 1, 2, or 3. Type HELP NOSEPROC for more info')
    end
end

%%%%%%%%%%%%%%%% FEATURE SELECTION

if length(size(S))== 3
    if feature(1) == 0  % No feature
        if options(1)==1
            disp([' - No features selected (transformed data kept)'])
        end
    elseif feature(1) == 1  % Every w'th from v
        if length(feature)<3
            warning(' When the fourth input (feature) is set to 1, additional v and w must be given [1 v w], where w is the interval with which to select variables and v the start time')
            feature(2) = 1;
            feature(3) = 3;
        end
        if length(feature) == 3;
            v = feature(2);
            w = feature(3);      
        end;
        Snew = Snew(:,v:w:end,:);
        if options(1)==1
            disp([' - Selecting every ',num2str(feature(3)),' variable in the time mode from time ',num2str(feature(2)),' to time ',num2str(J),''])
        end
    elseif feature(1) == 2  % Absolute maximum
        for i = 1:I
            for k = 1:K
                s = squeeze(Snew(i,:,k));
                [a,b]=max(abs(s));
                Snew2(i,k) = Snew(i,b,k);
            end
        end
        Snew = Snew2;
        if options(1)==1
            disp([' - Selecting the absolute maximum (removing the time mode)'])
        end
    elseif feature(1) == 3  % Maximum slope of adsorption part
        for i = 1:I
            for k = 1:K
                s = squeeze(Snew(i,:,k));
                [a,b]=max(abs(s));
                sg = gradient(Snew(i,1:b,k));
                Snew2(i,k) = max(sg);
            end
        end
        Snew = Snew2;
        if options(1)==1
            disp([' - Selecting maximal slope of adsorption part (removing the time mode)'])
        end
    elseif feature(1) == 4  % Minimum slope of desorption part
        for i = 1:I
            for k = 1:K
                s = squeeze(Snew(i,:,k));
                [a,b]=max(abs(s));
                sg = gradient(Snew(i,b:end,k));
                Snew2(i,k) = min(sg);
            end
        end
        Snew = Snew2;
        if options(1)==1
            disp([' - Selecting minimal (absolute) slope of desorption part (removing the time mode)'])
        end
        
     elseif feature(1) == 5 %Unfold time and sensor dimension
        Snew2 = reshape(Snew,size(id1,1),(size(id2,1)*size(id3,1)));
        Snew=Snew2;
        
        if options(1) == 1
           disp([' - Unfolding the time and sensor mode'])
        end
        
     else
     error(' The fourth input FEATURE has to be 0, 1, 2, 3, 4 or 5. Type HELP NOSEPROC for more info')
  end
  
elseif length(size(S)) == 2 & size(id1,1) == 1;
    if feature(1) == 0  % No feature
        if options(1)==1
            disp([' - No features selected (transformed data kept)'])
        end
    elseif feature(1) == 1  % Every w'th from v
        if length(feature)<3
            warning(' When the fourth input (feature) is set to 1, additional v and w must be given [1 v w], where w is the interval with which to select variables and v the start time')
            feature(2) = 1;
            feature(3) = 3;
        end
        if length(feature) == 3;
            v = feature(2);
            w = feature(3);      
        end;
        Snew = Snew(v:w:end,:);
        if options(1)==1
            disp([' - Selecting every ',num2str(feature(3)),' variable in the time mode from time ',num2str(feature(2)),' to time ',num2str(J),''])
        end
    elseif feature(1) == 2  % Absolute maximum
        for k = 1:K
            s = squeeze(Snew(:,k));
            [a,b]=max(abs(s));
            Snew2(k) = Snew(b,k);
        end
        Snew = Snew2;
        if options(1)==1
            disp([' - Selecting the absolute maximum (removing the time mode)'])
        end
    elseif feature(1) == 3  % Maximum slope of adsorption part
        for k = 1:K
            s = squeeze(Snew(:,k));
            [a,b]=max(abs(s));
            sg = gradient(Snew(1:b,k));
            Snew2(k) = max(sg);
        end
        Snew = Snew2;
        if options(1)==1
            disp([' - Selecting maximal slope of adsorption part (removing the time mode)'])
        end
    elseif feature(1) == 4  % Minimum slope of desorption part
        for k = 1:K
            s = squeeze(Snew(:,k));
            [a,b]=max(abs(s));
            sg = gradient(Snew(b:end,k));
            Snew2(k) = min(sg);
        end
        Snew = Snew2;
        if options(1)==1
            disp([' - Selecting minimal (absolute) slope of desorption part (removing the time mode)'])
        end
        
    else
        error(' The fourth input FEATURE has to be 0, 1, 2, 3, or 4. Type HELP NOSEPROC for more info')
    end
    
elseif length(size(S)) == 2 & size(id3,1) == 1;
    if feature(1) == 0  % No feature
        if options(1)==1
            disp([' - No features selected (transformed data kept)'])
        end
    elseif feature(1) == 1  % Every w'th from v
        if length(feature)<3
            warning(' When the fourth input (feature) is set to 1, additional v and w must be given [1 v w], where w is the interval with which to select variables and v the start time')
            feature(2) = 1;
            feature(3) = 3;
        end
        if length(feature) == 3;
            v = feature(2);
            w = feature(3);      
        end;
        Snew = Snew(v:w:end,:);
        if options(1)==1
            disp([' - Selecting every ',num2str(feature(3)),' variable in the time mode from time ',num2str(feature(2)),' to time ',num2str(J),''])
        end
    elseif feature(1) == 2  % Absolute maximum
        for i = 1:I
            s = squeeze(Snew(i,:));
            [a,b]=max(abs(s));
            Snew2(i) = Snew(i,b);
        end
        Snew = Snew2;
        if options(1)==1
            disp([' - Selecting the absolute maximum (removing the time mode)'])
        end
    elseif feature(1) == 3  % Maximum slope of adsorption part
        for i = 1:I
            s = squeeze(Snew(i,:));
            [a,b]=max(abs(s));
            sg = gradient(Snew(i,1:b));
            Snew2(i) = max(sg);
        end
        Snew = Snew2;
        if options(1)==1
            disp([' - Selecting maximal slope of adsorption part (removing the time mode)'])
        end
    elseif feature(1) == 4  % Minimum slope of desorption part
        for i = 1:I
            s = squeeze(Snew(i,:));
            [a,b]=max(abs(s));
            sg = gradient(Snew(i,b:end));
            Snew2(i) = min(sg);
        end
        Snew = Snew2;
        if options(1)==1
            disp([' - Selecting minimal (absolute) slope of desorption part (removing the time mode)'])
        end
        
    else
        error(' The fourth input FEATURE has to be 0, 1, 2, 3, or 4. Type HELP NOSEPROC for more info')
    end
end
    
    
%%%%%%%%%%%%%%%% CENTERING AND SCALING
if length(size(S)) == 3    
    show = -1; % Parameter for nprocess in order not to show screen output from nprocess
    if centscal(1) == 0  % None
        if options(1)==1
            disp([' - No centering/scaling chosen'])
        end
    elseif centscal(1) == 1  % center the data
        if length(size(Snew))==3 % Snew still three-way => do unfold centering
            [Snew]=nprocess(Snew,[1 0 0],[0 0 0],[],[],1,show,1);  
        else % Snew two-way
            [Snew]=nprocess(Snew,[1 0],[0 0],[],[],1,show,1);  
        end
        if options(1)==1
            disp([' - Centering the data'])
        end
    elseif centscal(1) == 2  % scale the data
        if length(size(Snew))==3 % Snew still three-way => do unfold centering
            [Snew]=nprocess(Snew,[0 0 0],[0 0 1],[],[],1,show,1);  
        else % Snew two-way
            [Snew]=nprocess(Snew,[0 0],[0 1],[],[],1,show,1);  
        end
        if options(1)==1
            disp([' - Scaling the sensor mode'])
        end
    elseif centscal(1) == 3  % center and scale the data
        if length(size(Snew))==3 % Snew still three-way => do unfold centering
            [Snew]=nprocess(Snew,[1 0 0],[0 0 1],[],[],1,show,1);  
        else % Snew two-way
            [Snew]=nprocess(Snew,[1 0],[0 1],[],[],1,show,1);  
        end
        if options(1)==1
            disp([' - Centering samples and scaling the sensor mode'])
        end
        
    else
        error(' The fifth input CENTSCAL has to be 0, 1, 2, or 3. Type HELP NOSEPROC for more info')
    end
    
elseif length(size(S)) == 2 & size(id1,1) == 1;
    show = -1; % Parameter for nprocess in order not to show screen output from nprocess
    if centscal(1) == 0  % None
        if options(1)==1
            disp([' - No centering/scaling chosen'])
        end
    elseif centscal(1) == 1  % center the data
        [Snew]=nprocess(Snew,[0 0],[0 0],[],[],1,show,1);  
        if options(1)==1
            disp([' - Centering the data'])
        end
    elseif centscal(1) == 2  % scale the data
        [Snew]=nprocess(Snew,[0 0],[0 1],[],[],1,show,1);  
        if options(1)==1
            disp([' - Scaling the sensor mode'])
        end
    elseif centscal(1) == 3  % center and scale the data
        [Snew]=nprocess(Snew,[0 0],[0 1],[],[],1,show,1);  
        if options(1)==1
            disp([' - Centering samples and scaling the sensor mode'])
        end    
    else
    error(' The fifth input CENTSCAL has to be 0, 1, 2, or 3. Type HELP NOSEPROC for more info')
end

elseif length(size(S)) == 2 & size(id3,1) == 1;
    show = -1; % Parameter for nprocess in order not to show screen output from nprocess
    if centscal(1) == 0  % None
        if options(1)==1
            disp([' - No centering/scaling chosen'])
        end
    elseif centscal(1) == 1  % center the data
        [Snew]=nprocess(Snew,[1 0],[0 0],[],[],1,show,1);  
        if options(1)==1
            disp([' - Centering the data'])
        elseif centscal(1) == 2  % scale the data
            [Snew]=nprocess(Snew,[0 0],[0 0],[],[],1,show,1);  
        end
        if options(1)==1
            disp([' - Scaling the sensor mode'])
        end
    elseif centscal(1) == 3  % center and scale the data
        [Snew]=nprocess(Snew,[1 0],[0 0],[],[],1,show,1);  
        if options(1)==1
            disp([' - Centering samples and scaling the sensor mode'])
        end    
    else
        error(' The fifth input CENTSCAL has to be 0, 1, 2, or 3. Type HELP NOSEPROC for more info')
    end
end

    
    
    
    










%%%%%%%%%%%%%%%%%%%%   ADDITIONAL HELP FUNCTIONS


function [Xnew,mX,sX]=nprocess(X,Cent,Scal,mX,sX,reverse,show,usemse);

%NPROCESS pre and postprocessing of multiway arrays
%
%
% CENTERING AND SCALING OF N-WAY ARRAYS
%
% This m-file works in two ways
%    I.  Calculate center and scale parameters and preprocess data
%    II. Use given center and scale parameters for preprocessing data
% 
% %%% I. Calculate center and scale parameters %%%
% 
%     [Xnew,Means,Scales]=nprocess(X,Cent,Scal);
% 
%     INPUT
%     X       Data array
%     Cent    is binary row vector with as many elements as DimX.
%             If Cent(i)=1 the centering across the i'th mode is performed
%             I.e cnt = [1 0 1] means centering across mode one and three.
%     Scal    is defined likewise. Scal(i)=1, means scaling to standard  
%             deviation one within the i'th mode
% 
%     OUTPUT
%     Xnew    The preprocessed data
%     mX      Sparse vector holding the mean-values 
%     sX      Sparse vector holding the scales
%
% %%% II. Use given center and scale parameters %%%
% 
%     Xnew=nprocess(X,Cent,Scal,mX,sX,reverse);
% 
%     INPUT
%     X       Data array
%     Cent    is binary row vector with as many elements as DimX.
%             If Cent(i)=1 the centering across the i'th mode is performed
%             I.e Cent = [1 0 1] means centering across mode one and three.
%     Scal    is defined likewise. Scal(i)=1, means scaling to standard  
%             deviation one within the i'th mode
%     mX      Sparse vector holding the mean-values 
%     sX      Sparse vector holding the scales
%     reverse Optional input
%             if reverse = 1 normal preprocessing is performed (default)
%             if reverse = -1 inverse (post-)processing is performed
%
%     OUTPUT
%     Xnew    The preprocessed data
%
% For convenience this m-file does not use iterative 
% preprocessing, which is necessary for some combinations of scaling
% and centering. Instead the algorithm first standardizes the modes
% successively and afterwards centers. The prior standardization ensures
% that the individual variables are on similar scale (this might be slightly
% disturbed upon centering - unlike for two-way data).
%
% The full I/O for nprocess is
% [Xnew,mX,sX]=nprocess(X,Cent,Scal,mX,sX,reverse,show,usemse);
% where show set to zero avoids screen output and where usemse set 
% to one uses RMSE instead of STD for scaling (more appropriate 
% in some settings)

% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% Phone  +45 35283296
% Fax    +45 35283245
% E-mail rb@kvl.dk


% $ Version 1.03 $ Date 6. May 1998 $ Drastic error in finding scale parameters corrected $ Not compiled $
% $ Version 1.031 $ Date 25. January 2000 $ Error in scaling part $ Not compiled $
% $ Version 1.032 $ Date 28. January 2000 $ Minor bug$ Not compiled $
% $ Version 1.033 $ Date 14. April 2001 $ Incorrect backscaling fixed.
% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $
% $ Version 2.00 $ May 2001 $ rewritten by Giorgio Tomasi $ RB $ Not compiled $
% $ Version 2.01 $ Feb 2002 $ Fixed errors occuring with one-slab inputs $ RB $ Not compiled $
% $ Version 2.02 $ Oct 2003 $ Added possibility for sclaing with RMSE $ RB $ Not compiled $


ord  = ndims(X);
DimX = size(X);
Xnew = X;

if nargin<3
   error(' Three input arguments must be given')
end

if nargin==4
   error(' You must input both mX and sX even if you are only doing centering')
end

if nargin<8
  usemse=0;
end

if ~exist('mX','var')
   mX = [];
end
if ~exist('sX','var')
   sX = [];
end

MODE = isa(mX,'cell')&isa(sX,'cell');

if ~exist('show')==1
   show=1;
end

if ~exist('reverse')==1
   reverse=1;
end

if ~any([1 -1]==reverse)
   error( 'The input <<reverse>> must be one or minus one')
end

if show~=-1
   if ~MODE
      disp(' Calculating mean and scale and processing data')
   else
      if reverse==1
         disp(' Using given mean and scale values for preprocessing data')
      elseif reverse==-1
         disp(' Using given mean and scale values for postprocessing data')
      end
   end
end

for i=1:ndims(X)
   Inds{i} = ones(size(Xnew,i),1);
end
Indm = repmat({':'},ndims(Xnew) - 1,1); 

out=0;
if ~MODE
   mX = cell(ord,1);
   sX = cell(ord,1);
end
Ord2Patch = [2,1;1,2];
if reverse == 1
   %Standardize
   for j = ord:-1:1
      o = [j 1:j-1 j+1:ord];
      if Scal(j)
         if show~=-1
            disp([' Scaling mode ',num2str(j)])
         end
         if ~MODE
           if ~usemse
             sX{j} = (stdnan(nshape(Xnew,j)')').^-1;
           else
             sX{j} = (rmsenan(nshape(Xnew,j)')').^-1;
           end
         end
         Xnew = Xnew.*ipermute(sX{j}(:,Inds{o(2:end)}),o);
      end
   end
   %Center
   for j = ord:-1:1
      o = [1:j-1 j+1:ord,j];
      if Cent(j)
         if show~=-1
            if ~MODE
               disp([' Centering mode ',num2str(j)])
            else
               disp([' Subtracting off-sets in mode ',num2str(j)])
            end
         end
         if ~MODE
            if ord ~= 2
               mmm = nshape(Xnew,j);
               if min(size(mmm))==1
                  mmm = mmm;
               else
                  mmm = missmean(mmm);
               end
               mX{j} = reshape(mmm,DimX(o(1:end-1)));
            else
               mX{j} = reshape(missmean(nshape(Xnew,j)),DimX(o(1)),1);
            end
         end
         Xnew = Xnew - ipermute(mX{j}(Indm{:},Inds{j}),o);
      end
   end
   
else

   %Center
   for j = 1:ord
      
      if Cent(j)
         if show~=-1
            disp([' Adding off-sets in mode ',num2str(j)])
         end
         Xnew = Xnew + ipermute(mX{j}(Indm{:},Inds{j}),[1:j-1 j+1:ord,j]);
      end
   end
   %Standardize
   for j = 1:ord
      o = [1:j-1 j+1:ord];
      if Scal(j)
         if show~=-1
            disp([' Rescaling back to original domain in mode ',num2str(j)])
         end
         Xnew = Xnew ./ ipermute(sX{j}(:,Inds{o}),[j o]);
      end
   end
   
end

function st=rmsenan(X);

%RMSENAN estimate RMSE with NaN's
%
% Estimates the RMSE of each column of X
% when there are NaN's in X.
%
% Columns with only NaN's get a standard deviation of zero


% $ Version 1.02 $ Date 28. July 1998 $ Not compiled $
% 
%
% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% E-mail: rb@kvl.dk

[I,J]=size(X);

st=[];
for j=1:J
  id=find(~isnan(X(:,j)));
  if length(id)
    st=[st sqrt(mean(X(id,j).^2))];
  else
    st=[st 0];
  end
end



function varargout=nshape(X,f);

%NSHAPE rearrange a multi-way array
%
% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro & Claus A. Andersson 1995
% Royal Veterinary and Agricultutal University, Denmark
% E-mail rb@kvl.dk
%
% [Xf,DimXf] = nshape(X,f);
%
% Refolds an N-way array so that Xf is X with index
% f as row-index, and the remaining in succesive order. For an 
% I x J x K x L four-way array this means X1 is I x JKL, X2 is
% J x IKL, X3 is K x IJL, and X4 is L x IJK
%
%
%    K  _______             
%      /      /|           1      J     2J    JK
%     /______/ |         1  _____________________
%    |      |  |           |      |      |      |
%    |      | /    -->     |      |      |      |        f = (Mode) 1 (same as original array)
% I  |______|/          I  |______|______|______|
%           J
%
%                          1      I     2I    KI
%                        1  _____________________
%                          |      |      |      |
%                  -->     |      |      |      |        f = (Mode) 2
%                        J |______|______|______|
%
%  
%                          1      I     2I    IJ
%                        1  _____________________
%                          |      |      |      |
%                  -->     |      |      |      |        f = (Mode) 3
%                        K |______|______|______|
%
%
% f can also indicate the order (meaning the sequence) of the modes
% [Xf,DimXf] = nshape(X,[3 2 1 4]);
% will return Xf as K x JIL
%
% If the last input is not given all rearrangements are given.
% For a fourway array this would read
% [X1,X2,X3,X4]=nshape(X);
%

% $ Version 1.03 $ Date 18. July 1999 $ Not compiled $
% $ Version 1.031 $ Date 18. July 1999 $ Error in help figure and now outputs new DimX $ Not compiled $
% $ Version 2.0 $ Jan 2002 $ Not compiled $ Improved speed and added permute functionality Giorgio Tomasi


ord       = ndims(X);
DimX      = size(X);
varargout = [];
if nargin < 2
   f     = 0;
   do_it = ones(1,nargout);
else
   if length(f) == 1
      do_it = [1:ord] == f;
   end
end
if length(f) == 1
   for i = 1:ord
      if do_it(i)
         varargout{end+1} = reshape(permute(X,[i 1:i-1 i+1:ord]),DimX(i),prod(DimX([1:i-1 i+1:ord])));
      end
   end
   if nargin == 2
      varargout{2} = [DimX(f) DimX([1:f-1 f+1:ord])];
   end
else
   if length(f)==ord
      DimX         = DimX(f);
      varargout{1} = reshape(permute(X,f),DimX(1),prod(DimX(2:end)));
      if nargin == 2
         varargout{2} = DimX;
      end
   else
      error(['f can either be the dimension to be put first or',char(10),...
            'a vector containing the new order of the dimensions']);
   end
end


function mm=missmean(X)

%MISSMEAN mean of a matrix X with NaN's
%
%[mm]=missmean(X)
%
%This function calculates the mean of a matrix X.
%X may hold missing elements denoted by NaN's which
%are ignored (weighted to zero).
%
%Check that for no column of X, all values are missing

% Copyright
% Claus A. Andersson 1996-
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% E-mail: claus@andersson.dk


%Insert zeros for missing, correct afterwards
missidx = isnan(X);
i = find(missidx);
X(i) = 0;

%Find the number of real(non-missing objects)
if min(size(X))==1,
   n_real=length(X)-sum(missidx);
else
   n_real=size(X,1)-sum(missidx);
end

i=find(n_real==0);
if isempty(i) %All values are real and can be corrected
   mm=sum(X)./n_real;
else %There are columns with all missing, insert missing
   n_real(i)=1;
   mm=sum(X)./n_real;
   mm(i)=i + NaN;
end
