function []=noseplot(model,NoseData,PreprocessedData,method);

%
% NOSEplot for plotting parameters of NOSEMODEL
%
% I/O NOSEplot(model,NoseData,PreprocessedData,method)
% 
% INPUT
% model     Structure array with model parameters
% NoseData  Structure array with data, labels, author etc. (see help file
% Method    Vector with settings for the individual plot type
% Coloum 1  Plot type                   PCA   PLS   PARAFAC2   N-PLS   PARAFAC
%   1       Scores, loadings, Bi-plot    x     x       x         x        x 
%   2       Loadings (line plot)         x             x                  x
%   3       Expl. Var. for X and/or Y    x     x       x         x        x
%   4       Influence plot               x
%   5       Modelling plot                             x                  x   
%   6       T vs. U plot                       x                 x
%   7       Predicted vs. measured             x                 x
%
% For column # further settings is needed according to the following (F is
% the number of components, Variable is the Y-variable, k the slab number)
% Model    Column 1  Coloumn 2    Coloumn 3   Coloumn 4    Example
% ALL         1      1: Scores    PC X-axis   PC Y-axis    [1 1 1 2]: Score plot with Score1 vs. Score2
%                    2: Loadings  PC X-axis   PC Y-axis    PC: Score or loading 
%                    3: Bi-plot   PC X-axis   PC Y-axis
%   
% PCA         2      F                                     [2 1]: Line plot for the first component (PC1) 
% PARAFAC/2   2      1: Mode A            
%                    2: Mode B    k(only for PARAFAC2)     [2 2 7]: Line plot of the k'th slab of mode  
%                    3: Mode C
%
% PLS/NPLS    3      1: X                                  [3 1]: Exp. Var. for X matrix
%                    2: Y   
%
% PARAFAC/2   5      1: Prepro data  k                     [5 1 7]: Preprocessed data of the k'th slab 
%                    2: Residuals    k 
%                    3: Model        k
%
% PLS/NPLS    6         F                                  [6 1]: T vs. U plot for the first component
%
% PLS/NPLS    7         F         Variable                 [7 1 3]: Pred. vs meas. for the third Y-variable using component two
%

if strcmp(model.name,'PCA')
    PCAplot(model,NoseData,PreprocessedData,method)
% Method    
% [1 v PC_X PC_Y] Score (v=1), loading (v=2) and bi-plot (v=3). 
%                 PC_X: first axis PC number
%                 PC_Y: second axis PC number
% [2 F]           Line plot with loadings for component F
% [3]             Fitted and crossvalidated model for 1:F components
% [4]             Influence plot (leverage vs. residuals)
    

elseif strcmp(model.name,'PLS')
    PLSplot(model,NoseData,PreprocessedData,method)
% Method
% [1 v PC_X PC_Y] Score (v=1), loading (v=2) and bi-plot (v=3). 
%                 PC_X: first axis PC number
%                 PC_Y: second axis PC number
% [3 v]           Fitted and crossvalidated model for 1:F components for X (v=1)
%                 and Y(v=2).
% [6 F]           T/U plot for component F
% [7 F Variable]  Predicted vs. measured Y for PLS model for the given variable 
%                 (indicated with a number from 1 to # variables)
%                 If one Y-variable (PLS1) exclude the Variable input        
%                 Both fitted and crossvalidated predictions are shown   


elseif strcmp(model.name,'PARAFAC2')
    PARAFAC2plot(model,NoseData,PreprocessedData,method)
% Method
% [1 v PC_X PC_Y] Score (mode 1) and loading (mode 3) plot. PC_X: first axis PC
%                 number and PC_Y: second axis PC number
% [2 v k]         Loading line plot for the v'th mode for component 1:F 
%                 k should only be included when v = 2
% [3]             Fitted and crossvalidated model for 1:F components
% [5 v k]         Modelling plot for the k'th slab
%                 [2 1 k]: Raw data
%                 [2 2 k]: Modelled data
%                 [2 3 k]: Residuals
%                 where k is the slab to plot
    
elseif strcmp(model.name,'NPLS')
    NPLSplot(model,NoseData,PreprocessedData,method)
% Method
% [1 v PC_X PC_Y] Score (v=1), loading (v=2) and bi-plot (v=3). 
%                 PC_X: first axis PC number
%                 PC_Y: second axis PC number
% [3 v]           Fitted and crossvalidated model for 1:F components for X (v=1)
%                 and Y(v=2).    
% [6 F]           T/U plot for component F
% [7 F Variable]  Predicted vs. measured Y for PLS model for the given variable 
%                 (indicated with a number from 1 to # variables)
%                 If one Y-variable (PLS1) exclude the Variable input        
%                 Both fitted and crossvalidated predictions are shown                         
   
elseif strcmp(model.name,'PARAFAC')
    PARAFACplot(model,NoseData,PreprocessedData,method)
% Method
% [1 v PC_X PC_Y] Score (mode 1) and loading (mode 3) plot. PC_X: first axis PC
%                 number and PC_Y: second axis PC number
% [2 v]           Loading line plot for the v'th mode for component 1:F 
% [3]             Fitted and crossvalidated model for 1:F components
% [5 v k]         Modelling plot for the k'th slab
%                 [2 1 k]: Raw data
%                 [2 2 k]: Modelled data
%                 [2 3 k]: Residuals
%                 where k is the slab to plot

else
   text(0,0,'Cannot plot model. Type not recognized'),axis off
end


%%%%%%%%%%PLOTTING FUNCTIONS%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%PCAplot%%%%%%%%%%%%%%%%%%%%%

function []=PCAplot(model,NoseData,PreprocessedData,method)

% PCAplot for plotting parameters of PCA model
%
% I/O PCAplot(model,NoseData,method)
% 
% INPUT
% model     Structure array with model parameters
% NoseData  Structure array with data, labels, author etc. (see help file
%           for further information)
% Method    Defines which plot is created
%           [1 v PC_X PC_Y] Score (v=1), loading (v=2) and bi-plot (v=3). 
%                           PC_X: first axis PC number
%                           PC_Y: second axis PC number
%           [2 F]           Line plot with loadings for component F
%           [3]             Fitted and crossvalidated model for 1:F components
%           [4]             Influence plot (leverage vs. residuals)
%
% Copyright, Thomas Skov (thsk@kvl.dk), 2004
%

for i=1:3;
id=find(NoseData.IncludedX{i});
LabelX{i}=NoseData.labels{i}(id,:);
end

T=model.parameters{model.numbcomp}.loads{1};
P=model.parameters{model.numbcomp}.loads{2};
FacMax=model.numbcomp;

if method(1) == 1 %Score plot, Loading plot and Bi-plot
    if length(method) == 1;
        warning('When method is set to 1, additional v for plot type must be given');
        text(0,0,'Cannot plot model. Type not recognized'),axis off
    elseif length(method) == 3;
        warning('PC_Y is set to 2')
        PC_Y = 2;
        PC_X = method(3); 
        v = method(2);
    elseif length(method) == 2;
        warning('PC_X is set to 1 and PC_Y is set to 2')
        PC_X = 1;
        PC_Y = 2;
        v = method(2);
    elseif length(method) == 4;   
        PC_X = method(3);
        PC_Y = method(4);
        v = method(2);
    end
    
    if length(method) > 1 & v == 1; %Score plot
        plot(T(:,PC_X),T(:,PC_Y),'.')
        text(T(:,PC_X),T(:,PC_Y),LabelX(1))
        grid;
        title('Score plot')
        xlabel(['PC ' num2str(PC_X)])
        ylabel(['PC ' num2str(PC_Y)])
        
     elseif length(method) > 1 & v == 2; %Loading plot
        plot(P(:,PC_X),P(:,PC_Y),'.')
        if size(P,1) > size(LabelX{3},1)
           text(P(:,PC_X),P(:,PC_Y),num2str([1:size(P,1)]'))
        else 
           text(P(:,PC_X),P(:,PC_Y),LabelX(3))
        end
        grid;
        title('Loading plot')
        xlabel(['PC ' num2str(PC_X)])
        ylabel(['PC ' num2str(PC_Y)])
        
    elseif length(method) > 1 & v == 3; %Bi-plot        
        plot(T(:,PC_X),T(:,PC_Y),'.'); 
        text(T(:,PC_X),T(:,PC_Y),LabelX(1))
        hold on;
        plot(P(:,PC_X),P(:,PC_Y),'.')
        if size(P,1) > size(LabelX{3},1)
           text(P(:,PC_X),P(:,PC_Y),num2str([1:size(P,1)]'))
        else 
           text(P(:,PC_X),P(:,PC_Y),LabelX(3))
        end
        grid;
        title('Bi-plot')
        xlabel(['PC ' num2str(PC_X)])
        ylabel(['PC ' num2str(PC_Y)])
    end
       
elseif method(1) == 2 %Line plot with loadings
    if length(method) < 2;
        warning('When method is set to 2, additional F must be given')
        F = 1;
    elseif length(method) == 2;
        F = method(2);
    end
    
    plot(1:(size(model.parameters{1}.loads{2},1)),P(:,F)');grid;
    title('Loading line plot')
    ylabel(['Loading ' num2str(F)])
       
    if size(model.parameters{1}.loads{2},1) == size(LabelX{3},1)
        xlabel([num2str(NoseData.mode{3})]);
    elseif size(model.parameters{1}.loads{2},1) ~= size(LabelX{3},1)
        xlabel('-');
    end
    
elseif method(1) == 3 %Fit and crossvalidated fit for F components
    if size(model.crossval,1) == 0;    
        V = [model.residuals.fitpercent(1:FacMax)]';
        bar((1:FacMax),[V],.76)';
        axis([0 FacMax+1 0 100]);
        grid;
               
        legend('Fitted')
        title('Explained variance - not cross-validated')
        xlabel('Number of components')
        ylabel('Percent variance explained')
        
    elseif size(model.crossval,1) == 1;
        V = [model.residuals.fitpercent(1:FacMax)]';   
        W = [model.crossval.fitpercent(1:FacMax)]';
        Z = [V W];
        
        if FacMax > 1;
            bar((1:FacMax),[Z],.76,'grouped')
            axis([0.5 FacMax+0.5 0 100]);
            grid;
            legend('Fitted','Cross-validated') 
            xlabel('Number of components')
        elseif FacMax == 1;
            bar((0:1),[Z],.50,'grouped')            
            set(gca,'XTickLabel',{'Fitted';'Cross-validated'})
            grid
        end
                   
        title('Explained variance')
        ylabel('Percent variance explained')        
    end
    
elseif method(1) == 4 %Influence plot (leverage vs. residuals)
    for i=1:size(model.parameters{model.numbcomp}.loads{1},1);
        for f=1:model.numbcomp;
            res_total=PreprocessedData.data(i,:)-(T(i,1:f)*P(:,1:f)');
            for j=1:size(model.parameters{model.numbcomp}.loads{2},1);
                res_sample(i,f)=sqrt(sum(res_total(j)*res_total(j)));
            end
        end
    end
    
    for f=1:model.numbcomp;
        for i=1:size(model.parameters{2}.loads{1},1);
            lev(i,f) = T(i,1:f)*inv(T(:,1:f)'*T(:,1:f))*T(i,1:f)';
        end
    end
    
    for k=1:FacMax;
        Q = char('blue','green','red','k','cyan','yellow','magenta');
        set(gcf,'DefaultTextColor',Q(k))
        plot(lev(:,k),res_sample(:,k),[Q(k),'.'])
        text(lev(:,k),res_sample(:,k),LabelX(1))
        hold on
    end
    set(gcf,'DefaultTextColor','black')
    grid;
    title('Influence plot - colored after # PC')
    xlabel('Leverage')
    ylabel('Residuals')
    legend(num2str([1:FacMax]'))
end


%%%%%%%%%%%PLSplot%%%%%%%%%%%%%%%%%%%%%

function []=PLSplot(model,NoseData,PreprocessedData,method)

% PLSplot for plotting parameters of PLS model
%
% I/O PLSplot(model,NoseData,method)
% 
% INPUT
% model     Structure array with model parameters
% NoseData  Structure array with data, labels, author etc. (see help file
%           for further information)
% Method    Defines which plot is created
%           [1 v PC_X PC_Y] Score (v=1), loading (v=2) and bi-plot (v=3). 
%                           PC_X: first axis PC number
%                           PC_Y: second axis PC number
%           [3 v]           Fitted and crossvalidated model for 1:F components for X (v=1)
%                           and Y(v=2).
%           [6 F]           T/U plot for component F
%           [7 F Variable]  Predicted vs. measured Y for PLS model for the given variable 
%                           (indicated with a number from 1 to # variables)
%                           If one Y-variable (PLS1) exclude the Variable input        
%                           Both fitted and crossvalidated predictions are shown   
%
% Copyright, Rasmus Bro & Thomas Skov (thsk@kvl.dk), 2004
%

for i=1:3;
id=find(NoseData.IncludedX{i});
LabelX{i}=NoseData.labels{i}(id,:);
end

id1=find(NoseData.IncludedX{1});
id2=find(NoseData.IncludedY);
RefData=NoseData.ref.data(id1,id2);

T=model.parameters.x{1};
P=model.parameters.x{2};
B=model.parameters.B;
U=model.parameters.y{1};
FacMax=model.numbcomp;

if method(1) == 1 %Score plot, Loading plot and Bi-plot
    if length(method) == 1;
        warning('When method is set to 1, additional v for plot type must be given');
        text(0,0,'Cannot plot model. Type not recognized'),axis off
    elseif length(method) == 3;
        warning('PC_Y is set to 2')
        PC_Y = 2;
        PC_X = method(3); 
        v = method(2);
    elseif length(method) == 2;
        warning('PC_X is set to 1 and PC_Y is set to 2')
        PC_X = 1;
        PC_Y = 2;
        v = method(2);
    elseif length(method) == 4;   
        PC_X = method(3);
        PC_Y = method(4);
        v = method(2);
    end
    
    if length(method) > 1 & v == 1; %Score plot
        plot(T(:,PC_X),T(:,PC_Y),'.')
        text(T(:,PC_X),T(:,PC_Y),LabelX(1))
        grid;
        title('Score plot')
        xlabel(['PC ' num2str(PC_X)])
        ylabel(['PC ' num2str(PC_Y)])
        
    elseif length(method) > 1 & v == 2; %Loading plot
        plot(P(:,PC_X),P(:,PC_Y),'.')
        if size(P,1) > size(LabelX{3},1)
           text(P(:,PC_X),P(:,PC_Y),num2str([1:size(P,1)]'))
        else 
           text(P(:,PC_X),P(:,PC_Y),LabelX(3))
        end
        grid;
        title('Loading plot')
        xlabel(['PC ' num2str(PC_X)])
        ylabel(['PC ' num2str(PC_Y)])
        
    elseif length(method) > 1 & v == 3; %Bi-plot        
        plot(T(:,PC_X),T(:,PC_Y),'.'); 
        text(T(:,PC_X),T(:,PC_Y),LabelX(1))
        hold on;
        plot(P(:,PC_X),P(:,PC_Y),'.')
        if size(P,1) > size(LabelX{3},1)
           text(P(:,PC_X),P(:,PC_Y),num2str([1:size(P,1)]'))
        else 
           text(P(:,PC_X),P(:,PC_Y),LabelX(3))
        end
        grid;
        title('Bi-plot')
        xlabel(['PC ' num2str(PC_X)])
        ylabel(['PC ' num2str(PC_Y)])
    end
       
elseif method(1) == 6; %T vs. U
    if length(method) < 2;
       warning('When method is set to 6, additional F must be given');
       F = 1;
    elseif length(method) == 2;
       F = method(2);
    end
       b = pinv([ones(size(T,1),1) T(:,F)])*U(:,F);
       plot(T(:,F),U(:,F),'.');
       text(T(:,F),U(:,F),LabelX(1));
       hold on
       plot([min(T(:,F)) max(T(:,F))],[min(T(:,F))*b(2)+b(1) max(T(:,F))*b(2)+b(1)])
       grid;
       title(['T/U plot for component ', num2str(F)])
       xlabel('T')
       ylabel('U')
       
elseif method(1) == 7 %Predicted vs. measured Y
    if size(model.crossval,1) == 0;
        if size(RefData,2) == 1
            if (length(method) < 2) %PLS1 model - one Y      
                F = 1;
            elseif (length(method) < 3) & (size(RefData,2) == 1)
                F = method(2);
            elseif (length(method) == 3);
                F = method(2);
                Variable = 1;
            end
        
            b = pinv([ones(size(RefData,1),1) RefData(:,1)])*model.ypredictions(:,F);    
            plot(RefData(:,1),model.ypredictions(:,F),'r.')
            hold on
            plot([min(RefData(:,1)) max(RefData(:,1))],[min(RefData(:,1))*b(2)+b(1) max(RefData(:,1))*b(2)+b(1)],'r')
            text(RefData(:,1),model.ypredictions(:,F),LabelX(1))
            grid;
          
            title(['Predicted vs. measured for component ', num2str(F)])
            xlabel('Measured Y')
            ylabel('Predicted Y')
        
            c = predplot(RefData,model.ypredictions(:,F),0,model);
            
        elseif size(RefData,2) > 1;
            if length(method) < 3;
                warning('When more Y-variables in PLS model an additional number must be given indicating after which variable to predict')
                F = 1;
                Variable = 1;
            elseif length(method) == 3; %PLS2 model - more Y's
                F = method(2); 
                Variable = method(3);
            end
        
            b(:,F) = pinv([ones(size(RefData,1),1) RefData(:,Variable)])*model.ypredictions(:,Variable,F);  
            plot(RefData(:,Variable),model.ypredictions(:,Variable,F),'r.')
            hold on
            plot([min(RefData(:,Variable)) max(RefData(:,Variable))],[min(RefData(:,Variable))*b(2,F)+b(1,F) max(RefData(:,Variable))*b(2,F)+b(1,F)],'r')
            text(RefData(:,Variable),model.ypredictions(:,Variable,F),LabelX(1))
            grid;
            
            title(['Predicted vs. measured for variable ', num2str(Variable), ' component ', num2str(F)])
            xlabel('Measured Y')
            ylabel('Predicted Y') 
        
            hold on
            c = predplot(RefData(:,Variable),model.ypredictions(:,Variable,F),0,model);
        end
        
    elseif size(model.crossval,1) == 1;
        if size(RefData,2) == 1
            if (length(method) < 2) %PLS1 model - one Y      
                F = 1;
            elseif (length(method) < 3) & (size(RefData,2) == 1)
                F = method(2);
            elseif (length(method) == 3);
                F = method(2);
                Variable = 1;
            end
        
            b = pinv([ones(size(RefData,1),1) RefData(:,1)])*model.ypredictions(:,F);    
            plot(RefData(:,1),model.ypredictions(:,F),'r.')
            hold on
            plot([min(RefData(:,1)) max(RefData(:,1))],[min(RefData(:,1))*b(2)+b(1) max(RefData(:,1))*b(2)+b(1)],'r')
            text(RefData(:,1),model.ypredictions(:,F),LabelX(1))
            grid;
          
            b = pinv([ones(size(RefData,1),1) RefData(:,1)])*model.crossval.ypredictions(F,:)';    
            plot(RefData(:,1),model.crossval.ypredictions(F,:)','b.')
            plot([min(RefData(:,1)) max(RefData(:,1))],[min(RefData(:,1))*b(2)+b(1) max(RefData(:,1))*b(2)+b(1)],'b')
            text(RefData(:,1),model.crossval.ypredictions(F,:)',LabelX(1))
            title(['Predicted vs. measured for component ', num2str(F)])
            xlabel('Measured Y')
            ylabel('Predicted Y')
        
            c = predplot(RefData,model.crossval.ypredictions(F,:)',0,model);
            
        elseif size(RefData,2) > 1;
            if length(method) < 3;
                warning('When more Y-variables in PLS model an additional number must be given indicating after which variable to predict')
                F = 1;
                Variable = 1;
            elseif length(method) == 3; %PLS2 model - more Y's
                F = method(2); 
                Variable = method(3);
            end
        
            b(:,F) = pinv([ones(size(RefData,1),1) RefData(:,Variable)])*model.ypredictions(:,Variable,F);  
            plot(RefData(:,Variable),model.ypredictions(:,Variable,F),'r.')
            hold on
            plot([min(RefData(:,Variable)) max(RefData(:,Variable))],[min(RefData(:,Variable))*b(2,F)+b(1,F) max(RefData(:,Variable))*b(2,F)+b(1,F)],'r')
            text(RefData(:,Variable),model.ypredictions(:,Variable,F),LabelX(1))
            grid;
            
            b(:,F) = pinv([ones(size(RefData,1),1) RefData(:,Variable)])*model.crossval.ypredictions(F,:,Variable)';  
            plot(RefData(:,Variable),model.crossval.ypredictions(F,:,Variable)','b.')
            hold on
            plot([min(RefData(:,1)) max(RefData(:,Variable))],[min(RefData(:,Variable))*b(2,F)+b(1,F) max(RefData(:,Variable))*b(2,F)+b(1,F)],'b')
            text(RefData(:,Variable),model.crossval.ypredictions(F,:,Variable)',LabelX(1))
           
            title(['Predicted vs. measured for variable ', num2str(Variable), ' component ', num2str(F)])
            xlabel('Measured Y')
            ylabel('Predicted Y') 
        
            hold on
            c = predplot(RefData(:,Variable),model.crossval.ypredictions(F,:,Variable)',0,model);
        end
    end
    
elseif method(1) == 3 %Fitted and crossvalidated model for 1:F components for X and Y
    if length(method) < 2
        warning('An additional number v for which dimension to plot - X: v=1, Y:v=2 - must be given');
    elseif length(method) == 2;
        if method(2) == 1;
            if size(model.crossval,1) == 0;
                V = [model.residuals.fitpercent_x(1:FacMax)]';
                bar((1:FacMax),[V],.76,'grouped') %X (fitted)
                axis([0.5 FacMax+0.5 0 100])
                grid;
                legend('Fitted')
                title('Explained variance - not cross-validated')
                xlabel('Number of components')
                ylabel('Percent variance explained of X')
                
            elseif size(model.crossval,1) == 1;
                V = [model.residuals.fitpercent_x(1:FacMax)]';
                W = [model.crossval.fitpercent_x(1:FacMax)]';
                Z = [V W];
                
                if FacMax > 1;
                    bar((1:FacMax),[Z],.76,'grouped')
                    axis([0.5 FacMax+0.5 0 100]);
                    grid;
                    legend('Fitted','Cross-validated') 
                    xlabel('Number of components')
                elseif FacMax == 1;
                    bar((0:1),[Z],.50,'grouped')            
                    set(gca,'XTickLabel',{'Fitted';'Cross-validated'})
                    grid
                end
                title('Explained variance')
                ylabel('Percent variance explained of X')
            end
            
        elseif method(2) == 2;
            if size(model.crossval) == 0;
                V = [model.residuals.fitpercent_y(1:FacMax)]';
                bar((1:FacMax),[V],.76,'grouped') %Y (fitted)
                axis([0.5 FacMax+0.5 0 100]) 
                grid;
                legend('Fitted')
                title('Explained variance - not cross-validated')
                xlabel('Number of components')
                ylabel('Percent variance explained of Y')
                
            elseif size(model.crossval) == 1;
                V = [model.residuals.fitpercent_y(1:FacMax)]';       
                W = [model.crossval.fitpercent_y(1:FacMax)]';
                Z = [V W];
                
                if FacMax > 1;
                    bar((1:FacMax),[Z],.76,'grouped')
                    axis([0.5 FacMax+0.5 0 100]);
                    grid;
                    legend('Fitted','Cross-validated') 
                    xlabel('Number of components')
                elseif FacMax == 1;
                    bar((0:1),[Z],.50,'grouped')            
                    set(gca,'XTickLabel',{'Fitted';'Cross-validated'})
                    grid
                end
                title('Explained variance')
                ylabel('Percent variance explained of Y')
            end
        end
    end
end


%%%%%%%%%%%PARAFAC2plot%%%%%%%%%%%%%%%%%%%%%

function []=PARAFAC2plot(model,NoseData,PreprocessedData,method)

% PARAFAC2plot for plotting parameters of PARAFAC2 model
%
% I/O PARAFAC2plot(model,NoseData,method)
% 
% INPUT
% model     Structure array with model parameters
% NoseData  Structure array with data, labels, author etc. (see help file
%           for further information)
% Method    Defines which plot is created
%           [1 v PC_X PC_Y] Score (mode 1) and loading (mode 3) plot. PC_X: first axis PC
%                           number and PC_Y: second axis PC number
%           [2 v k]         Loading line plot for the v'th mode for component 1:F 
%                           k should only be included when v = 2
%           [3]             Fitted and crossvalidated model for 1:F components
%           [5 v k]         Modelling plot for the k'th slab
%                           [2 1 k]: Raw data
%                           [2 2 k]: Modelled data
%                           [2 3 k]: Residuals
%                           where k is the slab to plot
%
% Copyright, Thomas Skov (thsk@kvl.dk), 2004
%

for i=1:3;
id=find(NoseData.IncludedX{i});
LabelX{i}=NoseData.labels{i}(id,:);
end

Load1=model.parameters{model.numbcomp}.loads{1};
Load2=model.parameters{model.numbcomp}.loads{2};
Load3=model.parameters{model.numbcomp}.loads{3};
P=model.parameters{model.numbcomp}.loads{2}(1).P;
H=model.parameters{model.numbcomp}.loads{2}(1).H;
FacMax=model.numbcomp;

if method(1) == 1 %Score plot, Loading plot and Bi-plot
    if length(method) == 1;
        warning('When method is set to 1, additional v for plot type must be given');
        text(0,0,'Cannot plot model. Type not recognized'),axis off
    elseif length(method) == 3;
        warning('PC_Y is set to 2')
        PC_Y = 2;
        PC_X = method(3); 
        v = method(2);
    elseif length(method) == 2;
        warning('PC_X is set to 1 and PC_Y is set to 2')
        PC_X = 1;
        PC_Y = 2;
        v = method(2);
    elseif length(method) == 4;   
        PC_X = method(3);
        PC_Y = method(4);
        v = method(2);
    end
    
    if length(method) > 1 & v == 1; %Score plot
        plot(Load1(:,PC_X),Load1(:,PC_Y),'.')
        text(Load1(:,PC_X),Load1(:,PC_Y),LabelX(1))
        grid;
        title('Score plot')
        xlabel(['Score ' num2str(PC_X)])
        ylabel(['Score ' num2str(PC_Y)])
        
    elseif length(method) > 1 & v == 2; %Loading plot
        plot(Load3(:,PC_X),Load3(:,PC_Y),'.')
        text(Load3(:,PC_X),Load3(:,PC_Y),LabelX(3))
        grid;
        title('Loading plot')
        xlabel(['Loading ' num2str(PC_X)])
        ylabel(['Loading ' num2str(PC_Y)])
        
    elseif length(method) > 1 & v == 3; %Bi-plot        
        plot(Load1(:,PC_X),Load1(:,PC_Y),'.'); 
        text(Load3(:,PC_X),Load3(:,PC_Y),LabelX(3))
        hold on;
        plot(Load3(:,PC_X),Load3(:,PC_Y),'.')
        text(Load1(:,PC_X),Load1(:,PC_Y),LabelX(1))
        grid;
        title('Bi-plot')
        xlabel(['Score/loading ' num2str(PC_X)])
        ylabel(['Score/loading ' num2str(PC_Y)])
    end

elseif method(1) == 5 %Modelling plot (pre-processed data, modelled data and residuals for the k'th slab)
    if length(method) < 3;
        warning('When method is set to 5, additional v (kind of plot) and k (# slab) must be given - default is raw data for slab 1')
        v = 1;
        k = 1;
    elseif length(method) == 3;
        v = method(2);
        k = method(3);
    end
    
    if v == 1; %Pre-processed data
        plot(squeeze(PreprocessedData.data(:,:,k)'))
        grid;
        title(['Pre-processed data for slab number ',num2str(k)])
        xlabel('Time(s)')
        ylabel('Signal')
    elseif v == 2; %Modelled data
        X1=Load1*diag(Load3(k,:))*(P{k}*H)';
        plot(X1')
        grid;
        title(['Modelled data for slab number ',num2str(k)])
        xlabel('Time(s)')
    elseif v == 3; %Residuals
        E=PreprocessedData.data(:,:,k)-(Load1*diag(Load3(k,:))*(P{k}*H)');
        plot(E')
        grid;
        title(['Residuals for slab number ',num2str(k)])
        xlabel('Time(s)')
    end
            
elseif method(1) == 2 %Line plot with loadings
    if length(method) < 3;
        warning('When method is set to 2, additional v (loadings for mode v) must be given. If v = 2 k (# slab) must be given as well')
    end
    if length(method) == 3;
        v = method(2);
        k = method(3);
        if v == 1;
            plot(Load1);
            grid;
            title('Loading line plot - colored after # component')
            xlabel([num2str(NoseData.mode{1})])
        elseif v == 2;
            B=P{k}*H;
            plot(B)
            grid;
            title(['Loading line plot - colored after # component - for slab ', num2str(k)])
            xlabel([num2str(NoseData.mode{2})])
        elseif v == 3;
            plot(Load3);
            grid;
            title('Loading line plot - colored after # component')
            xlabel([num2str(NoseData.mode{3})])
        end
        legend(num2str([1:FacMax]'),1)
    end
                       
elseif method(1) == 3 %Fit and crossvalidated fit for F components
    if size(model.crossval) == 0;
        V = [model.residuals.fitpercent(1:FacMax)]';
        bar((1:FacMax),[V],.76,'grouped')
        axis([0.5 FacMax+0.5 0 100])
        grid;
        legend('Fitted')
        title('Explained variance - not cross-validated')
        xlabel('Number of components')
        ylabel('Percent variance explained') 
    elseif size(model.crossval) == 1;
        V = [model.residuals.fitpercent(1:FacMax)]';
        W = [model.crossval.fitpercent(1:FacMax)]';
        Z = [V W];
                
        if FacMax > 1;
            bar((1:FacMax),[Z],.76,'grouped')
            axis([0.5 FacMax+0.5 0 100]);
            grid;
            legend('Fitted','Cross-validated') 
            xlabel('Number of components')
        elseif FacMax == 1;
            bar((0:1),[Z],.50,'grouped')            
            set(gca,'XTickLabel',{'Fitted';'Cross-validated'})
            grid
        end
                   
        title('Explained variance')
        ylabel('Percent variance explained')  
    end
end


%%%%%%%%%%%%%%%%%NPLSplot%%%%%%%%%%%%%%

function []=NPLSplot(model,NoseData,PreprocessedData,method)

% NPLSplot for plotting parameters of N-PLS model
%
% I/O NPLSplot(model,NoseData,method)
% 
% INPUT
% model     Structure array with model parameters
% NoseData  Structure array with data, labels, author etc. (see help file
%           for further information)
% Method    Defines which plot is created
%           [1 v PC_X PC_Y] Score (v=1), loading (v=2) and bi-plot (v=3). 
%                           PC_X: first axis PC number
%                           PC_Y: second axis PC number
%           [3 v]           Fitted and crossvalidated model for 1:F components for X (v=1)
%                           and Y(v=2).    
%           [6 F]           T/U plot for component F
%           [7 F Variable]  Predicted vs. measured Y for PLS model for the given variable 
%                           (indicated with a number from 1 to # variables)
%                           If one Y-variable (PLS1) exclude the Variable input        
%                           Both fitted and crossvalidated predictions are shown                         
%
% Copyright, Rasmus Bro & Thomas Skov (thsk@kvl.dk), 2004
%

for i=1:3;
id=find(NoseData.IncludedX{i});
LabelX{i}=NoseData.labels{i}(id,:);
end

id1=find(NoseData.IncludedX{1});
id2=find(NoseData.IncludedY);
RefData=NoseData.ref.data(id1,id2);

T=model.parameters.x{1};
P1=model.parameters.x{2};
P2=model.parameters.x{3};
B=model.parameters.B;
U=model.parameters.y{1};
FacMax=model.numbcomp;

if method(1) == 1 %Score plot, Loading plot and Bi-plot
    if length(method) == 1;
        warning('When method is set to 1, additional v for plot type must be given');
        text(0,0,'Cannot plot model. Type not recognized'),axis off
    elseif length(method) == 3;
        warning('PC_Y is set to 2')
        PC_Y = 2;
        PC_X = method(3); 
        v = method(2);
    elseif length(method) == 2;
        warning('PC_X is set to 1 and PC_Y is set to 2')
        PC_X = 1;
        PC_Y = 2;
        v = method(2);
    elseif length(method) == 4;   
        PC_X = method(3);
        PC_Y = method(4);
        v = method(2);
    end
    
    if length(method) > 1 & v == 1; %Score plot
        plot(T(:,PC_X),T(:,PC_Y),'.')
        text(T(:,PC_X),T(:,PC_Y),LabelX(1))
        grid;
        title('Score plot')
        xlabel(['PC ' num2str(PC_X)])
        ylabel(['PC ' num2str(PC_Y)])
        
    elseif length(method) > 1 & v == 2; %Loading plot
        plot(P2(:,PC_X),P2(:,PC_Y),'.')
        text(P2(:,PC_X),P2(:,PC_Y),LabelX(3))
        grid;
        title('Loading plot')
        xlabel(['PC ' num2str(PC_X)])
        ylabel(['PC ' num2str(PC_Y)])
        
    elseif length(method) > 1 & v == 3; %Bi-plot        
        plot(T(:,PC_X),T(:,PC_Y),'.'); 
        text(T(:,PC_X),T(:,PC_Y),LabelX(1))
        hold on;
        plot(P2(:,PC_X),P2(:,PC_Y),'.')
        text(P2(:,PC_X),P2(:,PC_Y),LabelX(3))
        grid;
        title('Bi-plot')
        xlabel(['PC ' num2str(PC_X)])
        ylabel(['PC ' num2str(PC_Y)])
    end
    
elseif method(1) == 6 %T vs. U
    if length(method) < 2;
       warning('When method is set to 6, additional F must be given');
       F = 1;
   elseif length(method) == 2;
       F = method(2);
   end
       b = pinv([ones(size(T,1),1) T(:,F)])*U(:,F);
       plot(T(:,F),U(:,F),'.');
       text(T(:,F),U(:,F),LabelX(1));
       hold on
       plot([min(T(:,F)) max(T(:,F))],[min(T(:,F))*b(2)+b(1) max(T(:,F))*b(2)+b(1)])
       grid;
       title(['T/U plot for component ', num2str(F)])
       xlabel('T')
       ylabel('U')
      
elseif method(1) == 7 %Predicted vs. measured Y
    if size(model.crossval,1) == 0;
        if size(RefData,2) == 1
            if (length(method) < 2) %PLS1 model - one Y      
                F = 1;
            elseif (length(method) < 3) & (size(RefData,2) == 1)
                F = method(2);
            elseif (length(method) == 3);
                F = method(2);
                Variable = 1;
            end
        
            b = pinv([ones(size(RefData,1),1) RefData(:,1)])*model.ypredictions(:,F);    
            plot(RefData(:,1),model.ypredictions(:,F),'r.')
            hold on
            plot([min(RefData(:,1)) max(RefData(:,1))],[min(RefData(:,1))*b(2)+b(1) max(RefData(:,1))*b(2)+b(1)],'r')
            text(RefData(:,1),model.ypredictions(:,F),LabelX(1))
            grid;
          
            title(['Predicted vs. measured for component ', num2str(F)])
            xlabel('Measured Y')
            ylabel('Predicted Y')
        
            c = predplot(RefData,model.ypredictions(:,F),0,model);
            
        elseif size(RefData,2) > 1;
            if length(method) < 3;
                warning('When more Y-variables in PLS model an additional number must be given indicating after which variable to predict')
                F = 1;
                Variable = 1;
            elseif length(method) == 3; %PLS2 model - more Y's
                F = method(2); 
                Variable = method(3);
            end
        
            b(:,F) = pinv([ones(size(RefData,1),1) RefData(:,Variable)])*model.ypredictions(:,Variable,F);  
            plot(RefData(:,Variable),model.ypredictions(:,Variable,F),'r.')
            hold on
            plot([min(RefData(:,Variable)) max(RefData(:,Variable))],[min(RefData(:,Variable))*b(2,F)+b(1,F) max(RefData(:,Variable))*b(2,F)+b(1,F)],'r')
            text(RefData(:,Variable),model.ypredictions(:,Variable,F),LabelX(1))
            grid;
            
            title(['Predicted vs. measured for variable ', num2str(Variable), ' component ', num2str(F)])
            xlabel('Measured Y')
            ylabel('Predicted Y') 
        
            hold on
            c = predplot(RefData(:,Variable),model.ypredictions(:,Variable,F),0,model);
        end
        
    elseif size(model.crossval,1) == 1;
        if size(RefData,2) == 1
            if (length(method) < 2) %PLS1 model - one Y      
                F = 1;
            elseif (length(method) < 3) & (size(RefData,2) == 1)
                F = method(2);
            elseif (length(method) == 3);
                F = method(2);
                Variable = 1;
            end
        
            b = pinv([ones(size(RefData,1),1) RefData(:,1)])*model.ypredictions(:,F);    
            plot(RefData(:,1),model.ypredictions(:,F),'r.')
            hold on
            plot([min(RefData(:,1)) max(RefData(:,1))],[min(RefData(:,1))*b(2)+b(1) max(RefData(:,1))*b(2)+b(1)],'r')
            text(RefData(:,1),model.ypredictions(:,F),LabelX(1))
            grid;
          
            b = pinv([ones(size(RefData,1),1) RefData(:,1)])*model.crossval.ypredictions(F,:)';    
            plot(RefData(:,1),model.crossval.ypredictions(F,:)','b.')
            plot([min(RefData(:,1)) max(RefData(:,1))],[min(RefData(:,1))*b(2)+b(1) max(RefData(:,1))*b(2)+b(1)],'b')
            text(RefData(:,1),model.crossval.ypredictions(F,:)',LabelX(1))
            title(['Predicted vs. measured for component ', num2str(F)])
            xlabel('Measured Y')
            ylabel('Predicted Y')
        
            c = predplot(RefData,model.crossval.ypredictions(F,:)',0,model);
            
        elseif size(RefData,2) > 1;
            if length(method) < 3;
                warning('When more Y-variables in PLS model an additional number must be given indicating after which variable to predict')
                F = 1;
                Variable = 1;
            elseif length(method) == 3; %PLS2 model - more Y's
                F = method(2); 
                Variable = method(3);
            end
        
            b(:,F) = pinv([ones(size(RefData,1),1) RefData(:,Variable)])*model.ypredictions(:,Variable,F);  
            plot(RefData(:,Variable),model.ypredictions(:,Variable,F),'r.')
            hold on
            plot([min(RefData(:,Variable)) max(RefData(:,Variable))],[min(RefData(:,Variable))*b(2,F)+b(1,F) max(RefData(:,Variable))*b(2,F)+b(1,F)],'r')
            text(RefData(:,Variable),model.ypredictions(:,Variable,F),LabelX(1))
            grid;
            
            b(:,F) = pinv([ones(size(RefData,1),1) RefData(:,Variable)])*model.crossval.ypredictions(F,:,Variable)';  
            plot(RefData(:,Variable),model.crossval.ypredictions(F,:,Variable)','b.')
            hold on
            plot([min(RefData(:,1)) max(RefData(:,Variable))],[min(RefData(:,Variable))*b(2,F)+b(1,F) max(RefData(:,Variable))*b(2,F)+b(1,F)],'b')
            text(RefData(:,Variable),model.crossval.ypredictions(F,:,Variable)',LabelX(1))
           
            title(['Predicted vs. measured for variable ', num2str(Variable), ' component ', num2str(F)])
            xlabel('Measured Y')
            ylabel('Predicted Y') 
        
            hold on
            c = predplot(RefData(:,Variable),model.crossval.ypredictions(F,:,Variable)',0,model);
        end
    end
            
elseif method(1) == 3 %Fitted and crossvalidated model for 1:F components for X and Y
    if length(method) < 2
        warning('An additional number v for which dimension to plot - X: v=1, Y:v=2 - must be given');
    elseif length(method) == 2;
        if method(2) == 1;
            if size(model.crossval,1) == 0;
                V = [model.residuals.fitpercent_x(1:FacMax)]';
                bar((1:FacMax),[V],.76,'grouped') %X (fitted)
                axis([0.5 FacMax+0.5 0 100])
                grid;
                legend('Fitted')
                title('Explained variance - not cross-validated')
                xlabel('Number of components')
                ylabel('Percent variance explained of X')
                
            elseif size(model.crossval,1) == 1;
                V = [model.residuals.fitpercent_x(1:FacMax)]';
                W = [model.crossval.fitpercent_x(1:FacMax)]';
                Z = [V W];
                
                if FacMax > 1;
                    bar((1:FacMax),[Z],.76,'grouped')
                    axis([0.5 FacMax+0.5 0 100]);
                    grid;
                    legend('Fitted','Cross-validated') 
                    xlabel('Number of components')
                elseif FacMax == 1;
                    bar((0:1),[Z],.50,'grouped')            
                    set(gca,'XTickLabel',{'Fitted';'Cross-validated'})
                    grid
                end
                title('Explained variance')
                ylabel('Percent variance explained of X')
            end
            
        elseif method(2) == 2;
            if size(model.crossval) == 0;
                V = [model.residuals.fitpercent_y(1:FacMax)]';
                bar((1:FacMax),[V],.76,'grouped') %Y (fitted)
                axis([0.5 FacMax+0.5 0 100]) 
                grid;
                legend('Fitted')
                title('Explained variance - not cross-validated')
                xlabel('Number of components')
                ylabel('Percent variance explained of Y')
                
            elseif size(model.crossval) == 1;
                V = [model.residuals.fitpercent_y(1:FacMax)]';       
                W = [model.crossval.fitpercent_y(1:FacMax)]';
                Z = [V W];
                if FacMax > 1;
                    bar((1:FacMax),[Z],.76,'grouped')
                    axis([0.5 FacMax+0.5 0 100]);
                    grid;
                    legend('Fitted','Cross-validated') 
                    xlabel('Number of components')
                elseif FacMax == 1;
                    bar((0:1),[Z],.50,'grouped')            
                    set(gca,'XTickLabel',{'Fitted';'Cross-validated'})
                    grid
                end
                title('Explained variance')
                ylabel('Percent variance explained of Y')
            end
        end
    end
end


%%%%%%%%%%%PARAFACplot%%%%%%%%%%%%%%%%%%%%%

function []=PARAFACplot(model,NoseData,PreprocessedData,method)

% PARAFACplot for plotting parameters of PARAFAC model
%
% I/O PARAFACplot(model,NoseData,method)
% 
% INPUT
% model     Structure array with model parameters
% NoseData  Structure array with data, labels, author etc. (see help file
%           for further information)
% Method    Defines which plot is created
%           [1 v PC_X PC_Y] Score (mode 1) and loading (mode 3) plot. PC_X: first axis PC
%                           number and PC_Y: second axis PC number
%           [2 v k]         Loading line plot for the v'th mode for component 1:F 
%           [3]             Fitted and crossvalidated model for 1:F components
%           [5 v k]         Modelling plot for the k'th slab
%                           [2 1 k]: Raw data
%                           [2 2 k]: Modelled data
%                           [2 3 k]: Residuals
%                           where k is the slab to plot
%
% Copyright, Thomas Skov (thsk@kvl.dk), 2004
%

for i=1:3;
id=find(NoseData.IncludedX{i});
LabelX{i}=NoseData.labels{i}(id,:);
end

Load1=model.parameters{model.numbcomp}.loads{1};
Load2=model.parameters{model.numbcomp}.loads{2};
Load3=model.parameters{model.numbcomp}.loads{3};
FacMax=model.numbcomp;

if method(1) == 1 %Score plot, Loading plot and Bi-plot
    if length(method) == 1;
        warning('When method is set to 1, additional v for plot type must be given');
        text(0,0,'Cannot plot model. Type not recognized'),axis off
    elseif length(method) == 3;
        warning('PC_Y is set to 2')
        PC_Y = 2;
        PC_X = method(3); 
        v = method(2);
    elseif length(method) == 2;
        warning('PC_X is set to 1 and PC_Y is set to 2')
        PC_X = 1;
        PC_Y = 2;
        v = method(2);
    elseif length(method) == 4;   
        PC_X = method(3);
        PC_Y = method(4);
        v = method(2);
    end
    
    if length(method) > 1 & v == 1; %Score plot
        plot(Load1(:,PC_X),Load1(:,PC_Y),'.')
        text(Load1(:,PC_X),Load1(:,PC_Y),LabelX(1))
        grid;
        title('Score plot')
        xlabel(['Score ' num2str(PC_X)])
        ylabel(['Score ' num2str(PC_Y)])
        
    elseif length(method) > 1 & v == 2; %Loading plot
        plot(Load3(:,PC_X),Load3(:,PC_Y),'.')
        text(Load3(:,PC_X),Load3(:,PC_Y),LabelX(3))
        grid;
        title('Loading plot')
        xlabel(['Loading ' num2str(PC_X)])
        ylabel(['Loading ' num2str(PC_Y)])
        
    elseif length(method) > 1 & v == 3; %Bi-plot        
        plot(Load1(:,PC_X),Load1(:,PC_Y),'.'); 
        text(Load3(:,PC_X),Load3(:,PC_Y),LabelX(3))
        hold on;
        plot(Load3(:,PC_X),Load3(:,PC_Y),'.')
        text(Load1(:,PC_X),Load1(:,PC_Y),LabelX(1))
        grid;
        title('Bi-plot')
        xlabel(['Score/loading ' num2str(PC_X)])
        ylabel(['Score/loading ' num2str(PC_Y)])
    end

elseif method(1) == 5 %Modelling plot (pre-processed data, modelled data and residuals for the k'th slab)
    if length(method) < 2;
        warning('When method is set to 5, additional v (kind of plot) and k (# slab) must be given - default is raw data for slab 1')
        v = 1;
        k = 1;
    elseif length(method) == 3;
        v = method(2);
        k = method(3);
    end
    
    if v == 1; %Pre-processed data
        plot(squeeze(PreprocessedData.data(:,:,k)'))
        grid;
        title(['Pre-processed data for slab number ',num2str(k)])
        xlabel('Time(s)')
        ylabel('Signal')
    elseif v == 2; %Modelled data
        X1=Load1*diag(Load3(k,:))*Load2';
        plot(X1')
        grid;
        title(['Modelled data for slab number ',num2str(k)])
        xlabel('Time(s)')
    elseif v == 3; %Residuals
        E=PreprocessedData.data(:,:,k)-(Load1*diag(Load3(k,:))*Load2');
        plot(E')
        grid;
        title(['Residuals for slab number ',num2str(k)])
        xlabel('Time(s)')
    end
            
elseif method(1) == 2 %Line plot with loadings
    if length(method) < 2;
        warning('When method is set to 2, additional v (loadings for mode v) must be given')
    end
    if length(method) == 2;
        v = method(2);
        if v == 1;
            plot(Load1);
            grid;
            title('Loading line plot - colored after # component')
            xlabel([num2str(NoseData.mode{1})])
        elseif v == 2;
            plot(Load2)
            grid;
            title('Loading line plot - colored after # component')
            xlabel([num2str(NoseData.mode{2})])
        elseif v == 3;
            plot(Load3);
            grid;
            title('Loading line plot - colored after # component')
            xlabel([num2str(NoseData.mode{3})])
        end
        legend(num2str([1:FacMax]'),1)
    end
                       
elseif method(1) == 3 %Fit and crossvalidated fit for F components
    if size(model.crossval) == 0;
        V = [model.residuals.fitpercent(1:FacMax)]';
        bar((1:FacMax),[V],.76,'grouped')
        axis([0.5 FacMax+0.5 0 100])
        grid;
        legend('Fitted')
        title('Explained variance - not cross-validated')
        xlabel('Number of components')
        ylabel('Percent variance explained') 
    elseif size(model.crossval) == 1;
        V = [model.residuals.fitpercent(1:FacMax)]';
        W = [model.crossval.fitpercent(1:FacMax)]';
        Z = [V W];
        
        if FacMax > 1;
            bar((1:FacMax),[Z],.76,'grouped')
            axis([0.5 FacMax+0.5 0 100]);
            grid;
            legend('Fitted','Cross-validated') 
            xlabel('Number of components')
        elseif FacMax == 1;
            bar((0:1),[Z],.50,'grouped')            
            set(gca,'XTickLabel',{'Fitted';'Cross-validated'})
            grid
        end
        title('Explained variance')
        ylabel('Percent variance explained')  
    end
end


%%%%%%%%%%%Predplot%%%%%%%%%%%%%%%%%%%%%%%

function c = predplot(y,ypred,LocationOfLegend,model);

% c = predplot(y,ypred,LocationOfLegend);


c=corrcoef(y,ypred);
c=round(1000*c(2,1))/1000;

SI = polyfit(y,ypred,1);
slope=round(100*SI(1))/100;
intercept=round(100*SI(2))/100;
rmsep = std(y-ypred,1);

if size(model.crossval,1) == 0;
   text(((max(y)-min(y))/2),min(ypred),{['r = ',num2str(c)],...
      ['slope = ',num2str(slope)],...
      ['intercept = ',num2str(intercept)],...
      ['RMSEP = ',num2str(rmsep)]},'BackgroundColor',[1 1 1],'Margin',1,....
      'VerticalAlignment','bottom','HorizontalAlignment','center')

else text(((max(y)-min(y))/2),min(ypred),{['r = ',num2str(c)],...
      ['slope = ',num2str(slope)],...
      ['intercept = ',num2str(intercept)],...
      ['RMSEP = ',num2str(rmsep)],...
      ['Statistics for the cross-validated model (blue line)']},....
      'BackgroundColor',[1 1 1],'Margin',1,....
      'VerticalAlignment','bottom','HorizontalAlignment','center')
end