function model = nosemodel(X,Y,comps,val,silent,options);

% NOSEMODEL For modeling preprocessed nose data
%
% Input
% X       Preprocessed data
% Y       Preprocessed Y (dependent) data
%         Leave empty ([]) if not applicable
% comps   Number of components
% val     if set to one, crossvalidation is performed
% silent  Set to 1 for avoiding all outputs
% options options(1)=1 => PARAFAC preferred
%         options(1)=0 => PARAFAC2 preferred (default)
%
% Output
% model   A structure holding the parameters and 
%         other information of the model
%
% I/O model = nosemodel(X,Y,LV,val,silent);
%     alternative I/O for saving data to ascii files
%             nosemodel(model,filename)
%
% Copyright, Thomas Skov & Rasmus Bro (rb@kvl.dk), 2004


if nargin==2 % Save model to files
  % First input model
  % Second input is a string with the modelname
  model = X;
  savename = [Y model.name];
  
  time = model.time;
  save([savename ,'_date.txt'],'time','-ascii')
  numbcomp = model.numbcomp;
  save([savename ,'_numb_comp.txt'],'numbcomp','-ascii')
  
  for lv = 1:length(model.parameters)
    p = model.parameters{lv}.loads;
    load1 = p{1};
    save([savename ,'_loads_comp',num2str(lv),'_mode_1.txt'],'load1','-ascii')
    load2 = p{2};
    load2
    save([savename ,'_loads_comp',num2str(lv),'_mode_2.txt'],'load2','-ascii')
    if length(model.parameters)==3
      load3 = model.parameters{3};
      save([savename ,'_loads_comp',num2str(lv),'_mode_3.txt'],'load3','-ascii')
    end    
  end
  disp(' Only limited functionality for saving and only for PCA')
  return
  
  
else
  
  if nargin<6
    options(1)=0;
  elseif ~any([0 1]==options(1)) % Check if incorrect value
    options(1)=0;
  end
  if nargin<5
    silent = 1;
  end
  if nargin<4
    val = 1;
  end
  
  component_mode = 0; % Use regression mode
  if isempty(Y)
    component_mode = 1;
  elseif size(Y,1)~=size(X,1);
    component_mode = 1;
  end
  
  if length(size(X))>2
    DoThreeway = 1;
  else
    DoThreeway = 0;
  end
  
  if val
    cwait = cwaitbar([0 0 0],{'Fitting full model';'Cross-validation - latent variables';'Cross-validation - segments'});
  else
    cwait = cwaitbar([0],{'Latent variables'});
  end
  
  if ~silent
    if component_mode
      if DoThreeway
        if options(1)==0
          disp([' Fitting a PARAFAC2 model'])
        else
          disp([' Fitting a PARAFAC model'])
        end
      else
        disp(' Fitting a PCA model')
      end
    else
      if DoThreeway
        disp(' Fitting an N-PLS model')
      else
        disp(' Fitting a PLS model')
      end
    end
  end
  
  
  
  %%%%%%%%%%%%%%%%%% FIT THE ACTUAL MODELS
  
  if component_mode
    if DoThreeway
      
      if options(1)==0
        %%%%%%%%%%%%%%% PARAFAC2 
        
        ssX = sum(X(:).^2);
        % Fit the model    
        PF2_options = [1e-7 2000 0 0 1];
        %[A,H,C,P,fit]=parafac2(X,comps,[0 0],PF2_options);
        
        model.name       = 'PARAFAC2';
        model.date       = date;
        model.time       = clock;
        model.numbcomp   = comps;
        
        for f = 1:comps
          cwaitbar([1 f/comps]);
          [A,H,C,P,fit]=parafac2(X,f,[0 0],PF2_options);
          m.loads{1}   = A;
          m.loads{2}.H = H;
          m.loads{2}.P = P;
          m.loads{3}   = C;
          model.parameters{f}=m;
          model.residuals.sumsofsquares_x      = ssX;      
          model.residuals.residual_sumsofsquares_x(f)=fit;
          model.residuals.fitpercent(f) = 100*(1-fit/ssX);       
        end
        
        if val == 1
          % DO PARAFAC2 CROSS-VALIDATION
          PF2_options = [1e-7 2000 0 1 1];
          [xval]=parafac2(X,comps,[0 0],PF2_options);
          model.crossval.fitpercent = 100*(1-xval.SS/ssX);
          model.crossval.residual_sumsofsquares_x = xval.SS;
          model.crossval.numbsegments = xval.NumberOfSegments;
          model.crossval.parameters_subsampled.A = xval.A_xval;
          model.crossval.parameters_subsampled.H = xval.H_xval;
          model.crossval.parameters_subsampled.C = xval.C_xval;
          model.crossval.parameters_subsampled.P = xval.P_xval;
        else
          model.crossval   = [];
        end
      else % Do PARAFAC
        %%%%%%%%%%%%%%% PARAFAC
        
        ssX = sum(X(:).^2);
        % Fit the model    
        DimX = size(X);
        model.name       = 'PARAFAC';
        model.date       = date;
        model.time       = clock;
        model.numbcomp   = comps;
        
        for f = 1:comps
          cwaitbar([1 f/comps]);
          [A,B,C,fit,it] = compparafac(reshape(X,DimX(1),DimX(2)*DimX(3)),DimX,f);
          m.loads{1}   = A;
          m.loads{2}   = B;
          m.loads{3}   = C;
          model.parameters{f}=m;
          model.residuals.sumsofsquares_x      = ssX;      
          model.residuals.residual_sumsofsquares_x(f)=fit;
          model.residuals.fitpercent(f) = 100*(1-fit/ssX);       
        end
        
        if val == 1
          % DO PARAFAC CROSS-VALIDATION
          [xval]=crossvalparafac(X,comps);
          model.crossval.fitpercent = 100*(1-xval.SS/ssX);
          model.crossval.residual_sumsofsquares_x = xval.SS;
          model.crossval.numbsegments = xval.NumberOfSegments;
          model.crossval.parameters_subsampled.A = xval.A_xval;
          model.crossval.parameters_subsampled.B = xval.B_xval;
          model.crossval.parameters_subsampled.C = xval.C_xval;
        else
          model.crossval   = [];
        end
      
      end
    else
      %%%%%%%%%%%%%%% PCA    
      
      ssX = sum(X(:).^2);
      % Fit the model    
      PCA_options = [1e-7 2000 0 0 1];
      [T,P,fit]=pcasvd(X,comps,[0 0],PCA_options);
      
      model.name       = 'PCA';
      model.date       = date;
      model.time       = clock;
      model.numbcomp   = comps;
      
      for f = 1:comps
        cwaitbar([1 f/comps]);
        [T,P,fit]=pcasvd(X,f,[0 0],PCA_options);
        m.loads{1}   = T;
        m.loads{2}   = P;
        model.parameters{f}=m;
        model.residuals.sumsofsquares_x      = ssX;      
        model.residuals.residual_sumsofsquares_x(f)    = fit;
        model.residuals.fitpercent(f) = 100*(1-fit/ssX);       
      end
      
      if val == 1
        % DO PCA CROSS-VALIDATION
        PCA_options = [1e-7 2000 0 1 1];
        [xval]=pcasvd(X,comps,[0 0],PCA_options);
        model.crossval.fitpercent = 100*(1-xval.SS/ssX);
        model.crossval.residual_sumsofsquares_x = xval.SS;
        model.crossval.numbsegments = xval.NumberOfSegments;
        model.crossval.parameters_subsampled.T = xval.T_xval;
        model.crossval.parameters_subsampled.P = xval.P_xval;
      else
        model.crossval   = [];
      end
    end
  else
    
    if DoThreeway
      model.name       = 'NPLS';
    else
      model.name       = 'PLS';
    end
    model.date       = date;
    model.time       = clock;
    
    
    ssX = sum(X(:).^2);
    ssY = sum(Y(:).^2);
    model.numbcomp   = comps;
    
    [XValResult,Model]=ncrossreg(X,Y,comps,0,eye(size(X,1)),val);
    Model = rmfield(Model,'MeanX');Model = rmfield(Model,'MeanY');
    
    model.parameters.x = Model.Xfactors;
    model.parameters.y = Model.Yfactors;
    model.parameters.core = Model.Core;
    model.parameters.B = Model.B;
    
    model.residuals.sumsofsquares_x      = ssX;
    model.residuals.sumsofsquares_y      = ssY;
    model.residuals.residual_sumsofsquares_x = XValResult.ssX_Fit;
    model.residuals.residual_sumsofsquares_y = XValResult.ssY_Fit;  
    model.residuals.fitpercent_x = XValResult.Percent.Xexp(:,1)';
    model.residuals.fitpercent_y = XValResult.Percent.Yexp(:,1)';
    
    model.ypredictions = Model.Yfitted;
    if val
      model.crossval.fitpercent_x = XValResult.Percent.Xexp(:,2)';
      model.crossval.fitpercent_y = XValResult.Percent.Yexp(:,2)';
      model.crossval.residual_sumsofsquares_x = XValResult.ssX_Xval;
      model.crossval.residual_sumsofsquares_y = XValResult.ssY_Xval;
      model.crossval.ypredictions = XValResult.ypred;
      model.crossval.PRESS = XValResult.PRESS;
      model.crossval.RMSEP = XValResult.RMSEP;
      model.crossval.definition_segments = XValResult.DefSegments;
    else
      model.crossval = [];
    end
  end
  
end
close(cwait)


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 
%%%%%%%%%%%%%%%%%%%% PCA
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 

function [T,P,fit,AddiOutput]=pcasvd(X,F,Constraints,Options,A,H,C,P);


% PCA - based on the PARAFAC2 algorithm below.
%
%
% Constraints Not Active
%
% Options
%   An optional vector of length 3
%   Options(1) Convergence criterion
%            1e-7 if not given or given as zero
%   Options(2) Maximal iterations
%            default 2000 if not given or given as zero
%   Options(3) Not Active
%   Options(4) Cross-validation
%            0 => no cross-validation
%            1 => cross-validation splitting in 7 segments
%            If cross-validation is chosen, the result is given 
%            the first output (A). No more outputs are given
%   Options(5) show output
%            0 => show standard output on screen
%            1 => hide all output to screen
%
% Copyright
% Rasmus Bro
% KVL, DK, 1998
% rb@kvl.dk

ShowFit  = 1000; % Show fit every 'ShowFit' iteration
NumRep   = 10; %Number of repetead initial analyses
NumItInRep = 80; % Number of iterations in each initial fit
if ~(length(size(X))==2|iscell(X))
  error(' X must be a two-way array or a cell array')
end
%set random number generators
randn('state',sum(100*clock));
rand('state',sum(100*clock));

if nargin < 4
  Options = zeros(1,5);
end
if length(Options)<5
  Options = Options(:);
  Options = [Options;zeros(5-length(Options),1)];
end

% Convergence criterion
if Options(1)==0
  ConvCrit = 1e-7;
else
  ConvCrit = Options(1);
end
if Options(5)==0
  disp(' ')
  disp(' ')
  disp([' Convergence criterion        : ',num2str(ConvCrit)])
end

% Maximal number of iterations 
if Options(2)==0
  MaxIt = 2000;
else
  MaxIt = Options(2);
end

% Initialization method
initi = Options(3);


[I,J] = size(X);

% CROSS-VALIDATION
if Options(4)==1
  Opt = Options;
  Opt(4) = 0;
  splits = 7;
  while rem(I,splits)==0 % Change the number of segments if 7 is a divisor in prod(size(X))
    splits = splits + 2;
  end
  AddiOutput.NumberOfSegments = splits;
  if Options(5)==0
    disp(' ')
    disp([' Cross-validation will be performed using ',num2str(splits),' segments'])
    disp([' and using from 1 to ',num2str(F),' components'])
    XvalModel = [];
  end
  SS = zeros(1,F);
  for f = 1:F
    cwaitbar([2 f/F]);
    Trep = [];Prep = [];
    M = repmat(NaN,[I J]);
    for s = 1:splits
      cwaitbar([3 s/splits]);
      Xmiss = X;
      Xmiss(s:splits:end)=NaN;
      [t,p]=pcasvd(Xmiss,f,Constraints,Opt);
      Trep(:,:,s)=t;Prep(:,:,s)=p;
      m    =t*p';
      M(s:splits:end)=m(s:splits:end);
      SS(f) = SS(f) + sum(sum(((X(s:splits:end)-m(s:splits:end)).^2))); 
    end
    XvalModel{f} = M;
    %AddiOutput.XvalModels=XvalModel;
    AddiOutput.SS = SS;
    AddiOutput.T_xval{f}=Trep;
    AddiOutput.P_xval{f}=Prep;
    A = AddiOutput;
  end
  T = AddiOutput;
  return
end

% Find missing and replace with average 
MissingElements = 0;
MissNum=0;AllNum=0;


x=X;
miss = sparse(isnan(x));
MissingOnes = miss;
if any(miss(:))
  MissingElements = 1;
  % Replace missing with mean over slab (not optimal but what the heck)
  % Iteratively they'll be replaced with model estimates
  x(find(miss)) = mean(x(find(~miss)));
  X = x;
  MissNum = MissNum + prod(size(find(miss)));
  AllNum = AllNum + prod(size(x));
end

if MissingElements
  if Options(5)==0
    PercMiss = 100*MissNum/AllNum;
    RoundedOf = .1*round(PercMiss*10);
    disp([' Missing data handled by EM   : ',num2str(RoundedOf),'%'])
  end
end
clear x

% Initialize by ten small runs
if nargin<5
  if initi==0
    if Options(5)==0
      disp([' Use best of ',num2str(NumRep)]) 
      disp(' initially fitted models')
    end
    Opt = Options;
    Opt = Options(1)/20;
    Opt(2) = NumItInRep; % Max NumItInRep iterations
    Opt(3) = 1;  % Init with SVD
    Opt(4) = 0;
    Opt(5) = 1;
    [T,P,bestfit]=pcasvd(X,F,Constraints,Opt);
    AllFit = bestfit;
    for i = 2:NumRep
      Opt(3) = 2;   % Init with random
      [t,p,fit]=pcasvd(X,F,Constraints,Opt);
      AllFit = [AllFit fit];
      if fit<bestfit
        T=t;P=p;
        bestfit = fit;
      end
    end
    AddiOutput.AllFit = AllFit;
    if Options(5)==0
      for ii=1:length(AllFit)
        disp([' Initial Model Fit            : ',num2str(AllFit(ii))])
      end
    end
    % Initialize by SVD
  elseif initi==1
    if Options(5)==0
      disp(' SVD based initialization')
    end
    [T,s,P]=svd(X,0);  
    T=T(:,1:F);
    P=P(:,1:F);
  elseif initi==2
    if Options(5)==0
      disp(' Random initialization')
    end
    T = rand(I,F);
    P = rand(J,F);
  else
    error(' Options(2) wrongly specified')
  end
end

XtX=X*X'; % Calculate for evaluating fit (but if initi = 1 it has been calculated)
fit    = sum(diag(XtX));
oldfit = fit*2;
fit0   = fit;
it     = 0;
Delta = 1;


% Iterative part
while abs(fit-oldfit)>oldfit*ConvCrit & it<MaxIt & fit>1000*eps
  oldfit = fit;
  it   = it + 1;
  
  
  [T,s,P]=svd(X,0);
  T = T(:,1:F)*s(1:F,1:F);
  P = P(:,1:F);
  
  M = T*P';
  E = X-M;
  fit = sum(E(:).^2);
  
  if MissingElements
    X(find(miss)) = M(find(miss));
  end
  
  % Print interim result
  if rem(it,ShowFit)==0|it == 1
    if Options(5)==0
      fprintf(' %12.10f       %g        %3.4f \n',fit,it,100*(1-fit/fit0));
      subplot(2,2,1)
      plot(A),title('First mode')
      subplot(2,2,2)
      plot(C),title('Third mode')
      subplot(2,2,3)
      plot(P{1}*H),title('Second mode (only first k-slab shown)')
      drawnow
    end
  end
  
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 
%%%%%%%%%% END PCA
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 













%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%% HELP FUNCTIONS BELOW %%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%




function [A,H,C,P,fit,AddiOutput]=parafac2(X,F,Constraints,Options,A,H,C,P);


%     ___________________________________________________
%
%                  THE PARAFAC2 MODEL
%     ___________________________________________________
% 
%
%
% Algorithm to fit the PARAFAC2 model which is an advanced variant of the 
% normal PARAFAC1 model. It handles slab-wise deviations between components
% in one mode as long as the cross-product of the components stays 
% reasonably fixed. This can be utilized for modeling chromatographic 
% data with retention time shifts, modeling certain batch data of 
% varying length etc. See Bro, Kiers & Andersson, Journal of Chemometrics,
% 1999, 13, 295-309 for details on application and Kiers, ten Berge & 
% Bro, Journal of Chemometrics, 1999, 13, 275-294, for details on the algorithm
% 
%
% The PARAFAC2 model is given
% 
% Xk = A*Dk*(Pk*H)' + Ek, k = 1, .., K
% 
% Xk is a slab of data (I x J) in which J may actually vary with K. K 
% is the number of slabs. A (I x F) are the scores or first-mode loadings. Dk 
% is a diagonal matrix that holds the k'th row of C in its diagonal. C 
% (K x F) is the third mode loadings, H is an F x F matrix, and Pk is a
% J x F orthogonal matrix (J may actually vary from k to k. The output here
% is given as a cell array of size J x F x K. Thus, to get e.g. the second P
% write P(:,:,2), and to get the estimate of the second mode loadings at this
% second frontal slab (k = 2), write P(:,:,2)*H. The matrix Ek holds the residuals.
% 
% INPUT
% 
% X
%   Holds the data.
%   If all slabs have similar size, X is an array:
%      X(:,:,1) = X1; X(:,:,2) = X2; etc.  
%   If the slabs have different size X is a cell array (type <<help cell>>)
%      X{1} = X1; X{2} = X2; etc.
%   If you have your data in an 'unfolded' two-way array of size
%   I x JK (the three-way array is I x J x K), then simply type
%   X = reshape(X,[I J K]); to convert it to an array.
%
% F
%   The number of components to extract
% 
% Constraints
%   Vector of length 2. The first element defines constraints
%   imposed in the first mode, the second defines contraints in
%   third mode (the second mode is not included because constraints
%   are not easily imposed in this mode)
% 
%   If Constraints = [a b], the following holds. If 
%   a = 0 => no constraints in the first mode
%   a = 1 => nonnegativity in the first mode
%   a = 2 => orthogonality in the first mode
%   a = 3 => unimodality (and nonnegativity) in the first mode
%   same holds for b for the third mode
%
% Options
%   An optional vector of length 3
%   Options(1) Convergence criterion
%            1e-7 if not given or given as zero
%   Options(2) Maximal iterations
%            default 2000 if not given or given as zero
%   Options(3) Initialization method
%            A rather slow initialization method is used per default
%            but it pays to investigate in avoiding local minima.
%            Experience may point to faster methods (set Options(3)
%            to 1 or 2). You can also change the number of refits etc.
%            in the beginning of the m-file
%            0 => best of 10 runs of maximally 80 iterations (default)
%            1 => based on SVD
%            2 => random numbers
%   Options(4) Cross-validation
%            0 => no cross-validation
%            1 => cross-validation splitting in 7 segments
%            If cross-validation is chosen, the result is given 
%            the first output (A). No more outputs are given
%   Options(5) show output
%            0 => show standard output on screen
%            1 => hide all output to screen
%
% AUXILIARY
% - Missing elements: Use NaN for missing elements
% - You can input initial values by using the input argument
%           (X,F,Constraints,Options,A,H,C,P);
%
% OUTPUT
% See right above INPUT
% 
% I/O
% 
% Demo
% parafac2('demo')
% 
% Short 
% [A,H,C,P]=parafac2(X,F);
%
% Long
% [A,H,C,P,fit]=parafac2(X,F,Constraints,Options);
%
% Copyright
% Rasmus Bro
% KVL, DK, 1998
% rb@kvl.dk
%
% Reference to algorithm
% Bro, Kiers & Andersson, PARAFAC2 - Part II. Modeling chromatographic 
% data with retention time shifts, Journal of Chemometrics, 1999, 13, 295-309

% TO DO:
% Set the algorithm to handle fixed modes as in PARALIN
% Make it N-way
% Incorporate ulsr

% $ Version 1.01 $ Date 28. December 1998 $ Not compiled $ RB
% $ Version 1.02 $ Date 31. March    1999 $ Added X-validation and added function $ Not compiled $ RB
% $ Version 1.03 $ Date 20. April    1999 $ Cosmetic changes $ Not compiled $ RB
% $ Version 1.04 $ Date 25. April    1999 $ Cosmetic changes $ Not compiled $ RB
% $ Version 1.05 $ Date 18. May      1999 $ Added orthogonality constraints $ Not compiled $ RB
% $ Version 1.06 $ Date 14. September1999 $ Changed helpfile $ Not compiled $ RB
% $ Version 1.07 $ Date 20. October  1999 $ Added unimodality $ Not compiled $ RB
% $ Version 1.08 $ Date 27. March    2000 $ Optimized handling of missing dat $ Not compiled $ RB
% $ Version 1.09 $ Date 27. January  2003 $ fixed output in cross-validation and removed breaks $ Not compiled $ RB
% $ Version 1.010 $ Date 27. January  2003 $ fixed error in cross-validation - not thoroughly tested! $ Not compiled $ RB
% $ Version 1.011 $ Date 8. April     2003 $ fixed yet an error in cross-validation - not thoroughly tested! $ Not compiled $ RB
%



ShowFit  = 1000; % Show fit every 'ShowFit' iteration
NumRep   = 10; %Number of repetead initial analyses
NumItInRep = 80; % Number of iterations in each initial fit
if ~(length(size(X))==3|iscell(X))
  error(' X must be a three-way array or a cell array')
end
%set random number generators
randn('state',sum(100*clock));
rand('state',sum(100*clock));

if nargin < 4
  Options = zeros(1,5);
end
if length(Options)<5
  Options = Options(:);
  Options = [Options;zeros(5-length(Options),1)];
end

% Convergence criterion
if Options(1)==0
  ConvCrit = 1e-7;
else
  ConvCrit = Options(1);
end
if Options(5)==0
  disp(' ')
  disp(' ')
  disp([' Convergence criterion        : ',num2str(ConvCrit)])
end

% Maximal number of iterations 
if Options(2)==0
  MaxIt = 2000;
else
  MaxIt = Options(2);
end

% Initialization method
initi = Options(3);

if nargin<3
  Constraints = [0 0];
end
if length(Constraints)~=2
  Constraints = [0 0];
  disp(' Length of Constraints must be two. It has been set to zeros')
end
% Modify to handle GPA (Constraints = [10 10]);
if Constraints(2)==10
  Constraints(1)=0;
  ConstB = 10;
else
  ConstB = 0;
end


ConstraintOptions=[ ...
    'Fixed                     ';...
    'Unconstrained             ';...
    'Non-negativity constrained';...
    'Orthogonality constrained ';...
    'Unimodality constrained   ';...
    'Not defined               ';...
    'Not defined               ';...
    'Not defined               ';...
    'Not defined               ';...
    'Not defined               ';...
    'Not defined               ';...
    'GPA                       '];


if Options(5)==0
  disp([' Maximal number of iterations : ',num2str(MaxIt)])
  disp([' Number of factors            : ',num2str(F)])
  disp([' Loading 1. mode, A           : ',ConstraintOptions(Constraints(1)+2,:)])
  disp([' Loading 3. mode, C           : ',ConstraintOptions(Constraints(2)+2,:)])
  disp(' ')
end


% Make X a cell array if it isn't
if ~iscell(X)
  for k = 1:size(X,3)
    x{k} = X(:,:,k);
  end
  X = x;
  clear x
end
I = size(X{1},1);
K = max(size(X));

% CROSS-VALIDATION
if Options(4)==1
  Opt = Options;
  Opt(4) = 0;
  splits = 7;
  while rem(I,splits)==0 % Change the number of segments if 7 is a divisor in prod(size(X))
    splits = splits + 2;
  end
  AddiOutput.NumberOfSegments = splits;
  if Options(5)==0
    disp(' ')
    disp([' Cross-validation will be performed using ',num2str(splits),' segments'])
    disp([' and using from 1 to ',num2str(F),' components'])
    XvalModel = [];
  end
  SS = zeros(1,F);
  for f = 1:F
    cwaitbar([2 f/F]);
    Arep = [];Hrep = [];Crep = [];clear Prep;
    for s = 1:splits
      cwaitbar([3 s/splits]);
      Xmiss = X;
      for k = 1:K 
        Xmiss{k}(s:splits:end)=NaN;
      end
      [a,h,c,p]=parafac2(Xmiss,f,Constraints,Opt);
      Arep(:,:,s)=a;Hrep(:,:,s)=h;Crep(:,:,s)=c;Prep(s,:)=p;
      for k = 1:K
        m    = a*diag(c(k,:))*(p{k}*h)';
        M{k} = m;
        SS(f) = SS(f) + sum(sum(((X{k}(s:splits:end)-m(s:splits:end)).^2))); 
      end
      XvalModel{f} = M;
    end
    %AddiOutput.XvalModels=XvalModel;
    AddiOutput.SS = SS;
    AddiOutput.A_xval{f}=Arep;
    AddiOutput.H_xval{f}=Hrep;
    AddiOutput.C_xval{f}=Crep;
    AddiOutput.P_xval{f}=Prep;
    A = AddiOutput;
  end
  
  
  [a,b]=min(SS);
  a=AddiOutput.A_xval{b};
  c = AddiOutput.C_xval{b};
  A = AddiOutput;
  return
end

% Find missing and replace with average 
MissingElements = 0;
MissNum=0;AllNum=0;
for k = 1:K
  x=X{k};
  miss = sparse(isnan(x));
  MissingOnes{k} = miss;
  if any(miss(:))
    MissingElements = 1;
    % Replace missing with mean over slab (not optimal but what the heck)
    % Iteratively they'll be replaced with model estimates
    x(find(miss)) = mean(x(find(~miss)));
    X{k} = x;
    MissNum = MissNum + prod(size(find(miss)));
    AllNum = AllNum + prod(size(x));
  end
end
if MissingElements
  if Options(5)==0
    PercMiss = 100*MissNum/AllNum;
    RoundedOf = .1*round(PercMiss*10);
    disp([' Missing data handled by EM   : ',num2str(RoundedOf),'%'])
  end
end
clear x

% Initialize by ten small runs
if nargin<5
  if initi==0
    if Options(5)==0
      disp([' Use best of ',num2str(NumRep)]) 
      disp(' initially fitted models')
    end
    Opt = Options;
    Opt = Options(1)/20;
    Opt(2) = NumItInRep; % Max NumItInRep iterations
    Opt(3) = 1;  % Init with SVD
    Opt(4) = 0;
    Opt(5) = 1;
    [A,H,C,P,bestfit]=parafac2(X,F,Constraints,Opt);
    AllFit = bestfit;
    for i = 2:NumRep
      Opt(3) = 2;   % Init with random
      [a,h,c,p,fit]=parafac2(X,F,Constraints,Opt);
      AllFit = [AllFit fit];
      if fit<bestfit
        A=a;H=h;C=c;P=p;
        bestfit = fit;
      end
    end
    AddiOutput.AllFit = AllFit;
    if Options(5)==0
      for ii=1:length(AllFit)
        disp([' Initial Model Fit            : ',num2str(AllFit(ii))])
      end
    end
    % Initialize by SVD
  elseif initi==1
    if Options(5)==0
      disp(' SVD based initialization')
    end
    XtX=X{1}*X{1}';
    for k = 2:K
      XtX = XtX + X{k}*X{k}';
    end
    [A,s,v]=svd(XtX,0);  
    A=A(:,1:F);
    C=ones(K,F)+randn(K,F)/10;
    H = eye(F);
  elseif initi==2
    if Options(5)==0
      disp(' Random initialization')
    end
    A = rand(I,F);
    C = rand(K,F);
    H = eye(F);
  else
    error(' Options(2) wrongly specified')
  end
end

if initi~=1
  XtX=X{1}*X{1}'; % Calculate for evaluating fit (but if initi = 1 it has been calculated)
  for k = 2:K
    XtX = XtX + X{k}*X{k}';
  end
end  
fit    = sum(diag(XtX));
oldfit = fit*2;
fit0   = fit;
it     = 0;
Delta = 1;

if Options(5)==0
  disp(' ')
  disp(' Fitting model ...')
  disp(' Loss-value      Iteration     %VariationExpl')
end

% Iterative part
while abs(fit-oldfit)>oldfit*ConvCrit & it<MaxIt & fit>1000*eps
  oldfit = fit;
  it   = it + 1;
  
  % Update P
  for k = 1:K
    Qk       = X{k}'*(A*diag(C(k,:))*H');
    P{k}     = Qk*psqrt(Qk'*Qk);
    %  [u,s,v]  = svd(Qk.');P{k}  = v(:,1:F)*u(:,1:F)';
    Y(:,:,k) = X{k}*P{k};
  end
  
  % Update A,H,C using PARAFAC-ALS
  [A,H,C,ff]=compparafac(reshape(Y,I,F*K),[I F K],F,1e-4,[Constraints(1) ConstB Constraints(2)],A,H,C,5);
  [fit,X] = pf2fit(X,A,H,C,P,K,MissingElements,MissingOnes);
  
  % Print interim result
  if rem(it,ShowFit)==0|it == 1
    if Options(5)==0
      fprintf(' %12.10f       %g        %3.4f \n',fit,it,100*(1-fit/fit0));
      subplot(2,2,1)
      plot(A),title('First mode')
      subplot(2,2,2)
      plot(C),title('Third mode')
      subplot(2,2,3)
      plot(P{1}*H),title('Second mode (only first k-slab shown)')
      drawnow
    end
  end
  
end

if rem(it,ShowFit)~=0 %Show final fit if not just shown
  if Options(5)==0
    fprintf(' %12.10f       %g        %3.4f \n',fit,it,100*(1-fit/fit0));
  end
end



function [fit,X]=pf2fit(X,A,H,C,P,K,MissingElements,MissingOnes);

% Calculate fit and impute missing elements from model

fit = 0;
for k = 1:K
  M   = A*diag(C(k,:))*(P{k}*H)';
  % if missing values replace missing elements with model estimates
  if nargout == 2 
    if any(MissingOnes{k})
      x=X{k};
      x(find(MissingOnes{k})) = M(find(MissingOnes{k}));
      X{k} = x;
    end
  end
  fit = fit + sum(sum(abs (X{k} - M ).^2));
end


function X = psqrt(A,tol)

% Produces A^(-.5) even if rank-problems

[U,S,V] = svd(A,0);
if min(size(S)) == 1
  S = S(1);
else
  S = diag(S);
end
if (nargin == 1)
  tol = max(size(A)) * S(1) * eps;
end
r = sum(S > tol);
if (r == 0)
  X = zeros(size(A'));
else
  S = diag(ones(r,1)./sqrt(S(1:r)));
  X = V(:,1:r)*S*U(:,1:r)';
end


function [A,B,C,fit,it] = compparafac(X,DimX,Fac,crit,Constraints,A,B,C,maxit,DoLineSearch);

% Complex PARAFAC-ALS
% Fits the PARAFAC model Xk = A*Dk*B.' + E
% where Dk is a diagonal matrix holding the k'th
% row of C.
%
% Uses on-the-fly projection-compression to speed up 
% the computations. This requires that the first mode 
% is the largest to be effective
% 
% INPUT
% X          : Data
% DimX       : Dimension of X
% Fac        : Number of factors
% OPTIONAL INPUT
% crit       : Convergence criterion (default 1e-6)
% Constraints: [a b c], if e.g. a=0 => A unconstrained, a=1 => A nonnegative
% A,B,C      : Initial parameter values
%
% I/O
% [A,B,C,fit,it]=parafac(X,DimX,Fac,crit,A,B,C);
%
% Copyright 1998
% Rasmus Bro
% KVL, Denmark, rb@kvl.dk

% Initialization
if nargin<9
  maxit   = 2500;      % Maximal number of iterations
end
showfit = pi;         % Show fit every 'showfit'th iteration (set to pi to avoid)

if nargin<4
  crit=1e-6;
end

if crit==0
  crit=1e-6;
end

I = DimX(1);
J = DimX(2);
K = DimX(3);

InitWithRandom=0;
if nargin<8
  InitWithRandom=1;
end
if nargin>7 & size(A,1)~=I
  InitWithRandom=1;
end

if nargin<5
  ConstA = 0;ConstB = 0;ConstC = 0;
else
  ConstA = Constraints(1);ConstB = Constraints(2);ConstC = Constraints(3);
end

if InitWithRandom
  
  if I<Fac
    A = rand(I,Fac);
  else
    A = orth(rand(I,Fac));
  end
  if J<Fac
    B = rand(J,Fac);
  else
    B = orth(rand(J,Fac));
  end
  if K<Fac
    C = rand(K,Fac);
  else
    C = orth(rand(K,Fac));
  end
end

SumSqX = sum(sum(abs(X).^2));
fit    = SumSqX;
fit0   = fit;
fitold = 2*fit;
it     = 0;
Delta  = 5;

while abs((fit-fitold)/fitold)>crit&it<maxit&fit>10*eps
  it=it+1;
  fitold=fit;
  
  % Do line-search
  if rem(it+2,2)==-1
    [A,B,C,Delta]=linesrch(X,DimX,A,B,C,Ao,Bo,Co,Delta);
  end
  
  Ao=A;Bo=B;Co=C;
  % Update A
  Xbc=0;
  for k=1:K
    Xbc = Xbc + X(:,(k-1)*J+1:k*J)*conj(B*diag(C(k,:)));
  end
  if ConstA == 0 % Unconstrained
    A = Xbc*pinv((B'*B).*(C'*C)).';
  elseif ConstA == 1 % Nonnegativity, requires reals
    Aold = A;
    for i = 1:I
      ztz = (B'*B).*(C'*C);
      A(i,:) = fastnnls(ztz,Xbc(i,:)')';
    end
    if any(sum(A)<100*eps*I)
      A = .99*Aold+.01*A; % To prevent a matrix with zero columns
    end
  elseif ConstA == 2 % Orthogonality
    A = Xbc*(Xbc'*Xbc)^(-.5);
  elseif ConstA == 3 % Unimodality
    A = unimodalcrossproducts((B'*B).*(C'*C),Xbc',A);
  end
  
  % Project X down on orth(A) - saves time if first mode is large
  [Qa,Ra]=qr(A,0);
  x=Qa'*X;
  
  % Update B
  if ConstB == 10 % Procrustes
    B = eye(Fac);
  else
    Xac=0;
    for k=1:K
      Xac = Xac + x(:,(k-1)*J+1:k*J).'*conj(Ra*diag(C(k,:)));
    end
    if ConstB == 0 % Unconstrained
      B = Xac*pinv((Ra'*Ra).*(C'*C)).';
    elseif ConstB == 1 % Nonnegativity, requires reals
      Bold = B;
      for j = 1:J
        ztz = (Ra'*Ra).*(C'*C);
        B(j,:) = fastnnls(ztz,Xac(j,:)')';
      end
      if any(sum(B)<100*eps*J)
        B = .99*Bold+.01*B; % To prevent a matrix with zero columns
      end
    end
  end
  
  % Update C
  if ConstC == 0 % Unconstrained
    ab=pinv((Ra'*Ra).*(B'*B));
    for k=1:K 
      C(k,:) = (ab*diag(Ra'* x(:,(k-1)*J+1:k*J)*conj(B))).';
    end
  elseif ConstC == 1  % Nonnegativity, requires reals
    Cold = C;
    ztz = (Ra'*Ra).*(B'*B);
    for k = 1:K
      xab = diag(Ra'* x(:,(k-1)*J+1:k*J)*B);
      C(k,:) = fastnnls(ztz,xab)';
    end
    if any(sum(C)<100*eps*K)
      C = .99*Cold+.01*C; % To prevent a matrix with zero columns
    end
  elseif ConstC == 2 % Orthogonality
    Z=(Ra'*Ra).*(B'*B);
    Y=[];
    for k=1:K
      d=diag(Ra'*x(:,(k-1)*J+1:k*J)*B)'; 
      Y=[Y;d];
    end;
    [P,D,Q]=svd(Y,0);
    C=P*Q';
  elseif ConstC == 3 % Unimodality
    xab = [];
    for k = 1:K
      xab = [xab diag(Ra'* x(:,(k-1)*J+1:k*J)*B)];
    end
    C = unimodalcrossproducts((Ra'*Ra).*(B'*B),xab,C);
  elseif ConstC == 10 % GPA => Isotropic scaling factor
    ab=(Ra'*Ra).*(B'*B);
    ab = pinv(ab(:));
    C(1,:) = 1;
    for k=2:K 
      yy = [];
      yyy = diag(Ra'* x(:,(k-1)*J+1:k*J)*conj(B)).';
      for f=1:Fac
        yy = [yy;yyy(:)];
      end
      C(k,:) = ab*yy;
    end
  end
  
  % Calculating fit. Using orthogonalization instead
  %fit=0;for k=1:K,residual=X(:,(k-1)*J+1:k*J)-A*diag(C(k,:))*B.';fit=fit+sum(sum((abs(residual).^2)));end
  [Qb,Rb]=qr(B,0);
  [Z,Rc]=qr(C,0);
  fit=SumSqX-sum(sum(abs(Ra*ppp(Rb,Rc).').^2));
  
  if rem(it,showfit)==0
    fprintf(' %12.10f       %g        %3.4f \n',fit,it,100*(1-fit/fit0));
  end
end

% ORDER ACCORDING TO VARIANCE
Tuck     = diag((A'*A).*(B'*B).*(C'*C));
[out,ID] = sort(Tuck);
A        = A(:,ID);
if ConstB ~= 10 % Else B is eye
  B        = B(:,ID);
end
C        = C(:,ID);
% NORMALIZE A AND C (variance in B)
if ConstB ~= 10 % Then B is eye
  for f=1:Fac,normC(f) = norm(C(:,f));end
  for f=1:Fac,normA(f) = norm(A(:,f));end
  B        = B*diag(normC)*diag(normA);  
  A        = A*diag(normA.^(-1));
  C        = C*diag(normC.^(-1));
  
  % APPLY SIGN CONVENTION
  SignA = sign(sum(sign(A))+eps);
  SignC = sign(sum(sign(C))+eps);
  A = A*diag(SignA);
  C = C*diag(SignC);
  B = B*diag(SignA)*diag(SignC);
end

function [NewA,NewB,NewC,DeltaMin] = linesrch(X,DimX,A,B,C,Ao,Bo,Co,Delta);

dbg=0;

if nargin<5
  Delta=5;
else
  Delta=max(2,Delta);
end

dA=A-Ao;
dB=B-Bo;
dC=C-Co;
Fit1=sum(sum(abs(X-A*ppp(B,C).').^2));
regx=[1 0 0 Fit1];
Fit2=sum(sum(abs(X-(A+Delta*dA)*ppp((B+Delta*dB),(C+Delta*dC)).').^2));
regx=[regx;1 Delta Delta.^2 Fit2];

while Fit2>Fit1
  if dbg
    disp('while Fit2>Fit1')
  end
  Delta=Delta*.6;
  Fit2=sum(sum(abs(X-(A+Delta*dA)*ppp((B+Delta*dB),(C+Delta*dC)).').^2));
  regx=[regx;1 Delta Delta.^2 Fit2];
end

Fit3=sum(sum(abs(X-(A+2*Delta*dA)*ppp((B+2*Delta*dB),(C+2*Delta*dC)).')^2));
regx=[regx;1 2*Delta (2*Delta).^2 Fit3];

while Fit3<Fit2
  if dbg
    disp('while Fit3<Fit2')
  end
  Delta=1.8*Delta;
  Fit2=Fit3;
  Fit3=sum(sum(abs(X-(A+2*Delta*dA)*ppp((B+2*Delta*dB),(C+2*Delta*dC)).')^2));
  regx=[regx;1 2*Delta (2*Delta).^2 Fit2];
end

% Add one point between the two smallest fits
[a,b]=sort(regx(:,4));
regx=regx(b,:);
Delta4=(regx(1,2)+regx(2,2))/2;
Fit4=sum(sum(abs(X-(A+Delta4*dA)*ppp((B+Delta4*dB),(C+Delta4*dC)).').^2));
regx=[regx;1 Delta4 Delta4.^2 Fit4];

%reg=pinv([1 0 0;1 Delta Delta^2;1 2*Delta (2*Delta)^2])*[Fit1;Fit2;Fit3]
reg=pinv(regx(:,1:3))*regx(:,4);
%DeltaMin=2*reg(3);

DeltaMin=-reg(2)/(2*reg(3));

%a*x2 + bx + c = fit
%2ax + b = 0
%x=-b/2a

NewA=A+DeltaMin*dA;
NewB=B+DeltaMin*dB;
NewC=C+DeltaMin*dC;
Fit=sum(sum(abs(X-NewA*ppp(NewB,NewC).').^2));

if dbg
  regx
  plot(regx(:,2),regx(:,4),'o'),
  hold on
  x=linspace(0,max(regx(:,2))*1.2);
  plot(x',[ones(100,1) x' x'.^2]*reg),
  hold off
  drawnow
  [DeltaMin Fit],pause
end

[minfit,number]=min(regx(:,4));
if Fit>minfit
  DeltaMin=regx(number,2);
  NewA=A+DeltaMin*dA;
  NewB=B+DeltaMin*dB;
  NewC=C+DeltaMin*dC;
end

function AB=ppp(A,B);

% $ Version 1.02 $ Date 28. July 1998 $ Not compiled $
%
% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% Phone  +45 35283296
% Fax    +45 35283245
% E-mail rb@kvl.dk
%
% The parallel proportional profiles product - triple-P product
% For two matrices with similar column dimension the triple-P product
% is ppp(A,B) = [kron(B(:,1),A(:,1) .... kron(B(:,F),A(:,F)]
% 
% AB = ppp(A,B);
%
% Copyright 1998
% Rasmus Bro
% KVL,DK
% rb@kvl.dk

[I,F]=size(A);
[J,F1]=size(B);

if F~=F1
  error(' Error in ppp.m - The matrices must have the same number of columns')
end

AB=zeros(I*J,F);
for f=1:F
  ab=A(:,f)*B(:,f).';
  AB(:,f)=ab(:);
end



function [x,w] = fastnnls(XtX,Xty,tol)
%NNLS	Non-negative least-squares.
%	b = fastnnls(XtX,Xty) returns the vector b that solves X*b = y
%	in a least squares sense, subject to b >= 0, given the inputs
%       XtX = X'*X and Xty = X'*y.
%
%	A default tolerance of TOL = MAX(SIZE(X)) * NORM(X,1) * EPS
%	is used for deciding when elements of b are less than zero.
%	This can be overridden with b = fastnnls(X,y,TOL).
%
%	[b,w] = fastnnls(XtX,Xty) also returns dual vector w where
%	w(i) < 0 where b(i) = 0 and w(i) = 0 where b(i) > 0.
%
%	See also LSCOV, SLASH.

%	L. Shure 5-8-87
%	Revised, 12-15-88,8-31-89 LS.
%	Copyright (c) 1984-94 by The MathWorks, Inc.

%       Revised by:
%	Copyright
%	Rasmus Bro 1995
%	Denmark
%	E-mail rb@kvl.dk
%       According to Bro & de Jong, J. Chemom, 1997

% initialize variables


if nargin < 3
  tol = 10*eps*norm(XtX,1)*max(size(XtX));
end
[m,n] = size(XtX);
P = zeros(1,n);
Z = 1:n;
x = P';
ZZ=Z;
w = Xty-XtX*x;

% set up iteration criterion
iter = 0;
itmax = 30*n;

% outer loop to put variables into set to hold positive coefficients
while any(Z) & any(w(ZZ) > tol)
  [wt,t] = max(w(ZZ));
  t = ZZ(t);
  P(1,t) = t;
  Z(t) = 0;
  PP = find(P);
  ZZ = find(Z);
  nzz = size(ZZ);
  z(PP')=(Xty(PP)'/XtX(PP,PP)');
  z(ZZ) = zeros(nzz(2),nzz(1))';
  z=z(:);
  % inner loop to remove elements from the positive set which no longer belong
  
  while any((z(PP) <= tol)) & iter < itmax
    
    iter = iter + 1;
    QQ = find((z <= tol) & P');
    alpha = min(x(QQ)./(x(QQ) - z(QQ)));
    x = x + alpha*(z - x);
    ij = find(abs(x) < tol & P' ~= 0);
    Z(ij)=ij';
    P(ij)=zeros(1,max(size(ij)));
    PP = find(P);
    ZZ = find(Z);
    nzz = size(ZZ);
    z(PP)=(Xty(PP)'/XtX(PP,PP)');
    z(ZZ) = zeros(nzz(2),nzz(1));
    z=z(:);
  end
  x = z;
  w = Xty-XtX*x;
end

x=x(:);


function B=unimodalcrossproducts(XtX,XtY,Bold)

% Solves the problem min|Y-XB'| subject to the columns of 
% B are unimodal and nonnegative. The algorithm is iterative and
% only one iteration is given, hence the solution is only improving 
% the current estimate
%
% I/O B=unimodalcrossproducts(XtX,XtY,Bold)
% Modified from unimodal.m to handle crossproducts in input 1999
%
% Copyright 1997
%
% Rasmus Bro
% Royal Veterinary & Agricultural University
% Denmark
% rb@kvl.dk
%
% Reference
% Bro and Sidiropoulos, "Journal of Chemometrics", 1998, 12, 223-247. 


B=Bold;
F=size(B,2);
for f=1:F
  xty = XtY(f,:)-XtX(f,[1:f-1 f+1:F])*B(:,[1:f-1 f+1:F])';
  beta=pinv(XtX(f,f))*xty;
  B(:,f)=ulsr(beta',1);
end


function [b,All,MaxML]=ulsr(x,NonNeg);

% ------INPUT------
%
% x          is the vector to be approximated
% NonNeg     If NonNeg is one, nonnegativity is imposed
%
%
%
% ------OUTPUT-----
%
% b 	     is the best ULSR vector
% All 	     is containing in its i'th column the ULSRFIX solution for mode
% 	     location at the i'th element. The ULSR solution given in All
%            is found disregarding the i'th element and hence NOT optimal
% MaxML      is the optimal (leftmost) mode location (i.e. position of maximum)
%
% ___________________________________________________________
%
%
%               Copyright 1997
%
% Nikos Sidiroupolos
% University of Maryland
% Maryland, US
%
%       &
%
% Rasmus Bro
% Royal Veterinary & Agricultural University
% Denmark
%
% 
% ___________________________________________________________


% This file uses MONREG.M

x=x(:);
I=length(x);
xmin=min(x);
if xmin<0
  x=x-xmin;
end


% THE SUBSEQUENT 
% CALCULATES BEST BY TWO MONOTONIC REGRESSIONS

% B1(1:i,i) contains the monontonic increasing regr. on x(1:i)
[b1,out,B1]=monreg(x);

% BI is the opposite of B1. Hence BI(i:I,i) holds the monotonic
% decreasing regression on x(i:I)
[bI,out,BI]=monreg(flipud(x));
BI=flipud(fliplr(BI));

% Together B1 and BI can be concatenated to give the solution to
% problem ULSR for any modloc position AS long as we do not pay
% attention to the element of x at this position


All=zeros(I,I+2);
All(1:I,3:I+2)=B1;
All(1:I,1:I)=All(1:I,1:I)+BI;
All=All(:,2:I+1);
Allmin=All;
Allmax=All;
% All(:,i) holds the ULSR solution for modloc = i, disregarding x(i),


iii=find(x>=max(All)');
b=All(:,iii(1));
b(iii(1))=x(iii(1));
Bestfit=sum((b-x).^2);
MaxML=iii(1);
for ii=2:length(iii)
  this=All(:,iii(ii));
  this(iii(ii))=x(iii(ii));
  thisfit=sum((this-x).^2);
  if thisfit<Bestfit
    b=this;
    Bestfit=thisfit;
    MaxML=iii(ii);
  end
end

if xmin<0
  b=b+xmin;
end


% Impose nonnegativity
if NonNeg==1
  if any(b<0)
    id=find(b<0);
    % Note that changing the negative values to zero does not affect the
    % solution with respect to nonnegative parameters and position of the
    % maximum.
    b(id)=zeros(size(id))+0;
  end
end

function [b,B,AllBs]=monreg(x);

% Monotonic regression according
% to J. B. Kruskal 64
%
% b     = min|x-b| subject to monotonic increase
% B     = b, but condensed
% AllBs = All monotonic regressions, i.e. AllBs(1:i,i) is the 
%         monotonic regression of x(1:i)
%
%
% Copyright 1997
%
% Rasmus Bro
% Royal Veterinary & Agricultural University
% Denmark
% rb@kvl.dk
%


I=length(x);
if size(x,2)==2
  B=x;
else
  B=[x(:) ones(I,1)];
end

AllBs=zeros(I,I);
AllBs(1,1)=x(1);
i=1;
while i<size(B,1)
  if B(i,1)>B(min(I,i+1),1)
    summ=B(i,2)+B(i+1,2);
    B=[B(1:i-1,:);[(B(i,1)*B(i,2)+B(i+1,1)*B(i+1,2))/(summ) summ];B(i+2:size(B,1),:)];
    OK=1;
    while OK
      if B(i,1)<B(max(1,i-1),1)
        summ=B(i,2)+B(i-1,2);
        B=[B(1:i-2,:);[(B(i,1)*B(i,2)+B(i-1,1)*B(i-1,2))/(summ) summ];B(i+1:size(B,1),:)];
        i=max(1,i-1);
      else
        OK=0;
      end
    end
    bInterim=[];
    for i2=1:i
      bInterim=[bInterim;zeros(B(i2,2),1)+B(i2,1)];
    end
    No=sum(B(1:i,2));
    AllBs(1:No,No)=bInterim;
  else
    i=i+1;
    bInterim=[];
    for i2=1:i
      bInterim=[bInterim;zeros(B(i2,2),1)+B(i2,1)];
    end
    No=sum(B(1:i,2));
    AllBs(1:No,No)=bInterim;
  end
end

b=[];
for i=1:size(B,1)
  b=[b;zeros(B(i,2),1)+B(i,1)];
end




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%% NCROSSREG

function [XValResult,Model]=ncrossreg(X,y,MaxFac,Centering,SegmentsID,val);
%NCROSSREG cross-validation of regression model
%
% See also:
% 'ncrossdecomp'
% 
% CROSS-VALIDATION OF BI- & MULTILINEAR REGRESSION MODELS
% Performs cross-validation of 
% - NPLS  Input multi-way array X
% - PLS   Input two-way X
% 
% The data are by default centered across the first mode, but no scaling
% is applied (this must be done beforehand)
% 
% I/O
% [XValResult,Model]=ncrossreg(X,y,MaxFac,Centering);
%
% INPUT
% X         : X array 
% y         : y array 
% MaxFac    : Maximal number of factors (from one to MaxFac factors will be investigated)
%
% OPTIONAL INPUT
% Centering : If not zero, centering is performed on every segment
% SegmentsID: Optional binary matrix. Rows as rows in X and i'th column defines i'th segment
%             Rows in i'th column set to one are left out at 
% val       : Extra parameter added to skip crossvalidation
%
% OUTPUT
% XValResult
%  Structured array with the following elements
%  ypred    : Cross-validated predictions
%  ssX_Xval : Cross-validated sum of squares of residuals in X (f'th element for f-component model)
%  ssX_Fit  : Fitted sum of squares of residuals in X (f'th element for f-component model)
%  ssY_Xval : Cross-validated sum of squares of residuals in Y (f'th element for f-component model)
%  ssY_Fit  : Fitted sum of squares of residuals in Y (f'th element for f-component model)
%  Percent  : Structured array holding Xexp and Yexp, each with fitted and X-validated % Variance captured.
%  PRESS    : Predicted REsidual Sum of Squares in Y
%  RMSEP    : Root Mean Square Error of Prediction (cross-validation)
%
% Model
%  Structured array holding the NPLS model



% $ Version 1.0301 $ Date 26. June 1999  
% $ Version 1.0302 $ Date 1. January 2000
% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $
% $ Version 2.01 $ jan 2003 $ Added option for skipping centering and added percentages in output $ RB $ Not compiled $
% $ Version 2.02 $ Sep 2003 $ Fixed bug in non-center option $ RB $ Not compiled $

%
% Copyright, 1997 - 
% This M-file and the code in it belongs to the holder of the copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added. The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'. In case of doubt, contact the holder of the copyrights.
% 
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% Phone  +45 35283296
% Fax    +45 35283245  
% E-mail rb@kvl.dk

I = size(X,1);

DimX = size(X);
DimY = size(y);
X = reshape(X,DimX(1),prod(DimX(2:end)));
y = reshape(y,DimY(1),prod(DimY(2:end)));

Ypred     = zeros([MaxFac DimY]);
Ex        = zeros([MaxFac DimX]);
Ey        = zeros([MaxFac DimY]);

if nargin<4
  Centering = 1;
elseif isempty(Centering)
  Centering = 1;
end

%%%%%%%%%%%%%%%%%
%%MAKE SEGMENTS%%
%%%%%%%%%%%%%%%%%
if exist('SegmentsID')~=1
  SegmentsID = MakeSegments(I);
end

%%%%%%%%%%%%%%%%%%%
%%MAKE SUB-MODELS%%
%%%%%%%%%%%%%%%%%%%

if val
  for i=1:size(SegmentsID,2)
    In = find(~SegmentsID(:,i));
    Out = find(SegmentsID(:,i));
    % Offsets
    if Centering
      Mx = nanmean(X(In,:));
      My = nanmean(y(In,:));
    else
      Mx = zeros(1,prod(DimX(2:end)));
      My = zeros(1,prod(DimY(2:end)));
    end
    %Centered data
    Xc = X(In,:)-repmat(Mx,length(In),1);
    yc = y(In,:)-repmat(My,length(In),1);
    
    %   %Centered data
    %  Xc = X(In,:)-ones(length(In),1)*Mx;
    %  yc = y(In,:)-ones(length(In),1)*My;
    
    % Calculate model 
    DimXc = DimX;DimXc(1)=size(Xc,1);
    Xc = reshape(Xc,DimXc);
    DimYc = DimY;DimYc(1)=size(yc,1);
    yc = reshape(yc,DimYc);
    [Xfactors,Yfactors,Core,B] = npls(Xc,yc,MaxFac,NaN);
    
    
    %Predict left-out samples
    for f=1:MaxFac
      Xc = X(Out,:)-ones(length(Out),1)*Mx;
      DimXc = DimX;DimXc(1)=size(Xc,1);
      Xc = reshape(Xc,DimXc);
      
      [ypr,T,ssx,Xres] = npred(Xc,f,Xfactors,Yfactors,Core,B,NaN);   
      Ex(f,Out,:)    = reshape(Xres,DimXc(1),prod(DimXc(2:end)));
      Ypred(f,Out,:) = ypr+ones(length(Out),1)*My;
      Ypredf = squeeze(Ypred(f,:,:));
      if size(y,2) == 1
        YpredfOut=Ypredf(Out);
      else
        YpredfOut=Ypredf(Out,:);
      end
      %size(Ey(f,Out,:)),size(y(Out,:)),size(YpredfOut)
      if size(y,2)==1
        Ey(f,Out,:)    = squeeze(y(Out,:))'-squeeze(YpredfOut);
      else
        Ey(f,Out,:)    = squeeze(y(Out,:))-squeeze(YpredfOut);
      end
    end
  end
end

if Centering
  Mx = nanmean(X(In,:));
  My = nanmean(y(In,:));
else
  Mx = zeros(1,prod(DimX(2:end)));
  My = zeros(1,prod(DimY(2:end)));
end
%Centered data
Xc = X-repmat(Mx,size(X,1),1);
yc = y-repmat(My,size(y,1),1);

%%Centered data
%Xc = X-ones(I,1)*Mx;
%yc = y-ones(I,1)*My;
[Xfactors,Yfactors,Core,B,ypred,ssx,ssy] = npls(reshape(Xc,DimX),reshape(yc,DimY),MaxFac,NaN);
Model.Xfactors = Xfactors;
Model.Yfactors = Yfactors;
Model.Core     = Core;
Model.B        = B;
Model.Yfitted  = ypred;
Model.MeanX    = Mx;
Model.MeanY    = My;

sseX_fit  = ssx(2:end,1);
sseY_fit  = ssy(2:end,1);
for f=1:MaxFac
  id=find(~isnan(Ex(f,:)));sseX_xval(f) = sum(Ex(f,id).^2);
  id=find(~isnan(Ey(f,:)));sseY_xval(f) = sum(Ey(f,id).^2);
  PRESS(f) = sum(Ey(f,id).^2);
end
RMSEP = sqrt(PRESS/I);

Xval = [sseX_fit sseX_xval'];
Yval = [sseY_fit sseY_xval'];
Xval = 100*(1-Xval/sum(Xc(find(~isnan(X))).^2));
Yval = 100*(1-Yval/sum(yc(find(~isnan(y))).^2));

XValResult.ypred       = Ypred;
XValResult.ssX_Xval    = sseX_xval;
XValResult.ssX_Fit     = sseX_fit';
XValResult.ssY_Xval    = sseY_xval;
XValResult.ssY_Fit     = sseY_fit';
XValResult.Percent.Xexp = Xval;
XValResult.Percent.Yexp = Yval;
XValResult.PRESS       = PRESS;
XValResult.RMSEP       = RMSEP;
XValResult.DefSegments = sparse(SegmentsID);


function SegmentsID = MakeSegments(I);

XvalMeth=questdlg('Which type of validation do you want to perform (ENTER => full Xval)?','Choose validation','Full X-validation','Segmented','Prespecified','Full X-validation');

switch XvalMeth
  case 'Full X-validation'
    SegmentsID = speye(I);
    
  case 'Segmented'
    prompt={'Enter the number of segments:'};
    eval(['def={''',num2str(min(I,max(3,round(I/7)))),'''};'])
    dlgTitle='Number of segments';
    lineNo=1;
    answer=inputdlg(prompt,dlgTitle,lineNo,def);
    NumbSegm=eval(answer{1});
    
    % Make sure the number of segments is OK
    while NumbSegm<2|NumbSegm>I
      prompt={'INCONSISTENT NUMBER CHOSEN (must be > 1 and <= samples)'};
      eval(['def={''',num2str(min(I,max(3,round(I/7)))),'''};'])
      dlgTitle='Number of segments';
      lineNo=1;
      answer=inputdlg(prompt,dlgTitle,lineNo,def);
      NumbSegm=eval(answer{1})
      NumbSegm<2|NumbSegm>I
    end
    
    XvalSegm=questdlg('How should segments be chosen?','Choose segmentation','111222333...','123123123...','Random','123123123...');
    switch XvalSegm
      
      case '111222333...'
        SegmentsID = sparse(I,NumbSegm);
        NumbInEachSegm = floor(I/NumbSegm);
        Additional = I-NumbInEachSegm*NumbSegm;
        currentsample = 1;
        for i=1:NumbSegm
          if i <=Additional
            add = NumbInEachSegm+1;
          elseif i<NumbSegm
            add = NumbInEachSegm;
          else
            add = I-currentsample+1;
          end
          SegmentsID(currentsample:currentsample+add-1,i)=1;
          currentsample = currentsample + add;
        end
      case '123123123...'
        SegmentsID = sparse(I,NumbSegm);
        NumbInEachSegm = floor(I/NumbSegm);
        for i=1:NumbSegm
          SegmentsID(i:NumbSegm:end,i)=1;
        end
      case 'Random'
        % Make nonrandom and then randomize order
        SegmentsID = sparse(I,NumbSegm);
        NumbInEachSegm = floor(I/NumbSegm);
        for i=1:NumbSegm
          SegmentsID(i:NumbSegm:end,i)=1;
        end
        rand('state',sum(100*clock)) %Randomize randomizer
        [a,b] = sort(rand(I,1))
        SegmentsID = SegmentsID(b,:);
    end
    
  case 'Prespecified' 
    prompt={'Enter the name of the file defining the subsets'};
    def={'SegmentsID'};
    dlgTitle='Import definition';
    lineNo=1;
    answer=inputdlg(prompt,dlgTitle,lineNo,def);
    SegmentsID=eval(answer{1});
    
end % switch

function y = nanmean(x)
%NANMEAN Average or mean ignoring NaNs.
%   NANMEAN(X) returns the average treating NaNs as missing values.  
%   For vectors, NANMEAN(X) is the mean value of the non-NaN
%   elements in X.  For matrices, NANMEAN(X) is a row vector
%   containing the mean value of each column, ignoring NaNs.
%
%   See also NANMEDIAN, NANSTD, NANMIN, NANMAX, NANSUM.

%   Copyright (c) 1993-98 by The MathWorks, Inc.
%   $Revision: 2.8 $  $Date: 1997/11/29 01:45:53 $

if isempty(x) % Check for empty input.
  y = NaN;
  return
end

% Replace NaNs with zeros.
nans = isnan(x);
i = find(nans);
x(i) = zeros(size(i));

if min(size(x))==1,
  count = length(x)-sum(nans);
else
  count = size(x,1)-sum(nans);
end

% Protect against a column of all NaNs
i = find(count==0);
count(i) = ones(size(i));
y = sum(x)./count;
y(i) = i + NaN;



















%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%   NPLS


function [Xfactors,Yfactors,Core,B,ypred,ssx,ssy,reg] = npls(X,Y,Fac,show);

%NPLS multilinear partial least squares regression
%
% See also:
% 'parafac' 'tucker'
%
%
% MULTILINEAR PLS  -  N-PLS
%
% INPUT
% X        Array of independent variables
% Y        Array of dependent variables
% Fac      Number of factors to compute
% 
% OPTIONAL
% show	   If show = NaN, no outputs are given
%
%
% OUTPUT
% Xfactors Holds the components of the model of X in a cell array.
%          Use fac2let to convert the parameters to scores and
%          weight matrices. I.e., for a three-way array do
%          [T,Wj,Wk]=fac2let(Xfactors);
% Yfactors Similar to Xfactors but for Y
% Core     Core array used for calculating the model of X
% B        The regression coefficients from which the scores in
%          the Y-space are estimated from the scores in the X-
%          space (U = TB);
% ypred    The predicted values of Y for one to Fac components
%          (array with dimension Fac in the last mode)
% ssx      Variation explained in the X-space.
%          ssx(f+1,1) is the sum-squared residual after first f factors.
%          ssx(f+1,2) is the percentage explained by first f factors.
% ssy      As above for the Y-space
% reg      Cell array with regression coefficients for raw (preprocessed) X
%
%
% AUXILIARY
%
% If missing elements occur these must be represented by NaN.
%
%
% [Xfactors,Yfactors,Core,B,ypred,ssx,ssy,reg] = npls(X,y,Fac);
% or short
% [Xfactors,Yfactors,Core,B] = npls(X,y,Fac);
%

% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% Phone  +45 35283296
% Fax    +45 35283245
% E-mail rb@kvl.dk
%


% $ Version 1.02 $ Date July 1998 $ Not compiled $
% $ Version 1.03 $ Date 4. December 1998 $ Not compiled $ Cosmetic changes
% $ Version 1.04 $ Date 4. December 1999 $ Not compiled $ Cosmetic changes
% $ Version 1.05 $ Date July 2000 $ Not compiled $ error caused weights not to be normalized for four-way and higher
% $ Version 1.06 $ Date November 2000 $ Not compiled $ increase max it and decrease conv crit to better handle difficult data
% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $
% $ Version 2.01 $ June 2001 $ Changed to handle new core in X $ RB $ Not compiled $
% $ Version 2.02 $ January 2002 $ Outputs all predictions (1 - LV components) $ RB $ Not compiled $

if ~exist('show')==1|nargin<4
  show=1;
end

maxit=120;

DimX = size(X);
X = reshape(X,DimX(1),prod(DimX(2:end)));
ordX = length(DimX);if ordX==2&size(X,2)==1;ordX = 1;end
DimY = size(Y);
Y = reshape(Y,DimY(1),prod(DimY(2:end)));
ordY = length(DimY);if ordY==2&size(Y,2)==1;ordY = 1;end


[I,Jx]=size(X);
[I,Jy]=size(Y);

missX=0;
missy=0;
MissingX = 0;
MissingY = 0;
if any(isnan(X(:)))|any(isnan(Y(:)))
  if any(isnan(X(:)))
    MissingX=1;
  else
    MissingX=0;
  end
  if any(isnan(Y(:)))
    MissingY=1;
  else
    MissingY=0;
  end
  if show~=0&~isnan(show)
    disp(' ')
    disp(' Don''t worry, missing values will be taken care of')
    disp(' ')
  end
  missX=abs(1-isnan(X));
  missy=abs(1-isnan(Y));
end
crit=1e-10;
B=zeros(Fac,Fac);
T=[];
U=[];
Qkron =[];
if MissingX
  SSX=sum(sum(X(find(missX)).^2));
else
  SSX=sum(sum(X.^2));
end
if MissingY
  SSy=sum(sum(Y(find(missy)).^2));
else
  SSy=sum(sum(Y.^2));
end
ssx=[];
ssy=[];
Xres=X;
Yres=Y;
xmodel=zeros(size(X));
Q=[];
W=[];

for num_lv=1:Fac
  
  %init
  u=rand(DimX(1),1);t=rand(DimX(1),1);tgl=t+2;it=0;
  while (norm(t-tgl)/norm(t))>crit&it<maxit
    tgl=t;
    it=it+1;
    
    % w=X'u
    [wloads,wkron] = Xtu(X,u,MissingX,missX,Jx,DimX,ordX);
    
    % t=Xw
    if MissingX
      for i=1:I,
        m = find(missX(i,:));
        t(i)=X(i,m)*wkron(m)/(wkron(m)'*wkron(m));
      end
    else
      t=X*wkron;
    end
    
    % w=X'u
    [qloads,qkron] = Xtu(Yres,t,MissingY,missy,Jy,DimY,ordY);
    % u=yq
    if MissingY
      for i=1:I
        m = find(missy(i,:));
        u(i)=Yres(i,m)*qkron(m)/(qkron(m)'*qkron(m));
      end
    else
      u=Yres*qkron;
    end
  end
  
  T=[T t];
  for i = 1:ordX-1
    if num_lv == 1
      W{i} = wloads{i};
    else
      W{i} = [W{i} wloads{i}];
    end
  end
  U=[U u];
  for i = 1:max(ordY-1,1)
    if num_lv == 1
      Q{i} = qloads{i};
    else
      Q{i} = [Q{i} qloads{i}];
    end
  end
  Qkron = [Qkron qkron];
  
  % Make core arrays
  if ordX>1
    Xfac{1}=T;Xfac(2:ordX)=W;
    Core{num_lv} = calcore(reshape(X,DimX),Xfac,[],0,1);
  else
    Core{num_lv} = 1;
  end
  %   if ordY>1
  %      Yfac{1}=U;Yfac(2:ordY)=Q;
  %      Ycore{num_lv} = calcore(reshape(Y,DimY),Yfac,[],0,1);
  %   else
  %      Ycore{num_lv} = 1;
  %   end
  
  
  B(1:num_lv,num_lv)=inv(T'*T)*T'*U(:,num_lv);
  
  if Jy > 1
    if show~=0&~isnan(show)
      disp(' ') 
      fprintf('number of iterations: %g',it);
      disp(' ')
    end
  end
  
  % Make X model
  if ordX>2
    Wkron = kron(W{end},W{end-1});
  else
    Wkron = W{end};
  end
  for i = ordX-3:-1:1
    Wkron = kron(Wkron,W{i});
  end
  if num_lv>1
    xmodel=T*reshape(Core{num_lv},num_lv,num_lv^(ordX-1))*Wkron';
  else
    xmodel = T*Core{num_lv}*Wkron';
  end
  
  % Make Y model   
  %  if ordY>2
  %     Qkron = kron(Q{end},Q{end-1});
  %  else
  %     Qkron = Q{end};
  %  end
  %  for i = ordY-3:-1:1
  %     Qkron = kron(Qkron,Q{i});
  %  end
  %  if num_lv>1
  %     ypred=T*B(1:num_lv,1:num_lv)*reshape(Ycore{num_lv},num_lv,num_lv^(ordY-1))*Qkron';
  %  else
  %     ypred = T*B(1:num_lv,1:num_lv)*Ycore{num_lv}*Qkron';
  %  end
  ypred=T*B(1:num_lv,1:num_lv)*Qkron';
  Ypred(:,num_lv) = ypred(:); % Vectorize to avoid problems with different orders and the de-vectorize later on
  
  Xres=X-xmodel; 
  Yres=Y-ypred;
  if MissingX
    ssx=[ssx;sum(sum(Xres(find(missX)).^2))];
  else
    ssx=[ssx;sum(sum(Xres.^2))];
  end
  if MissingY
    ssy=[ssy;sum(sum((Y(find(missy))-ypred(find(missy))).^2))];
  else
    ssy=[ssy;sum(sum((Y-ypred).^2))];
  end
end
ypred = reshape(Ypred',[size(Ypred,2) DimY]);
ypred = permute(ypred,[2:ordY+1 1]);

ssx= [ [SSX(1);ssx] [0;100*(1-ssx/SSX(1))]];
ssy= [ [SSy(1);ssy] [0;100*(1-ssy/SSy(1))]];


Xfactors{1}=T;
for j = 1:ordX-1
  Xfactors{j+1}=W{j};
end

Yfactors{1}=U;
for j = 1:max(ordY-1,1)
  Yfactors{j+1}=Q{j};
end


% Calculate regression coefficients that apply directly to X
if nargout>7
  if length(DimY)>2
    error(' Regression coefficients are only calculated for models with vector Y or multivariate Y (not multi-way Y)')
  end
  R = outerm(W,0,1);
  for iy=1:size(Y,2)
    if length(DimX) == 2
      dd = [DimX(2) 1];
    else
      dd = DimX(2:end);
    end
    for i=1:Fac
      sR = R(:,1:i)*B(1:i,1:i)*diag(Q{1}(iy,1:i));
      ssR = sum( sR',1)';
      reg{iy,i} = reshape( ssR ,dd);
    end       
  end
  
end




function [wloads,wkron] = Xtu(X,u,Missing,miss,J,DimX,ord);


% w=X'u
if Missing
  for i=1:J
    m = find(miss(:,i));
    if (u(m)'*u(m))~=0
      ww=X(m,i)'*u(m)/(u(m)'*u(m));
    else
      ww=X(m,i)'*u(m);
    end
    if length(ww)==0
      w(i)=0;
    else
      w(i)=ww;
    end
  end
else
  w=X'*u;
end

% Reshape to array
if length(DimX)>2
  w_reshaped=reshape(w,DimX(2),prod(DimX(3:length(DimX))));
else
  w_reshaped = w(:);
end


% Find one-comp decomposition
if length(DimX)==2
  wloads{1} = w_reshaped/norm(w_reshaped);
  
elseif length(DimX)==3&~any(isnan(w_reshaped))
  [w1,s,w2]=svd(w_reshaped);
  wloads{1}=w1(:,1);
  wloads{2}=w2(:,1);
else
  wloads=parafac(reshape(w_reshaped,DimX(2:length(DimX))),1,[0 2 0 0 NaN]');
  for j = 1:length(wloads);
    wloads{j} = wloads{j}/norm(wloads{j});
  end
end

% Apply sign convention
for i = 1:length(wloads)
  sq = (wloads{i}.^2).*sign(wloads{i});
  wloads{i} = wloads{i}*sign(sum(sq));
end


% Unfold solution
if length(wloads)==1
  wkron = wloads{1};
else
  wkron = kron(wloads{end},wloads{end-1});
  for o = ord-3:-1:1
    wkron = kron(wkron,wloads{o});
  end
end


function [Factors,it,err,corcondia]=parafac(X,Fac,Options,const,OldLoad,FixMode,Weights);

% PARAFAC multiway parafac model
%
% See also:
% 'npls' 'tucker' 'dtld' 'gram'
%
%
%     ___________________________________________________
%
%                  THE PARAFAC MODEL
%     ___________________________________________________
% 
% [Factors,it,err,corcondia,Weights] = parafac(X,Fac,Options,const,OldLoad,FixMode,Weights);
%
% or skipping optional in/outputs
%
% Factors = parafac(X,Fac);
%
% Algorithm for computing an N-way PARAFAC model. Optionally
% constraints can be put on individual modes for obtaining 
% orthogonal, nonnegative, or unimodal solutions. The algorithm
% also handles missing data. For details of PARAFAC 
% modeling see R. Bro, Chemom. Intell. Lab. Syst., 1997.
%
% Several possibilities exist for speeding up the algorithm. 
% Compressing has been incorporated, so that large arrays can be
% compressed by using Tucker (see Bro & Andersson, Chemom. 
% Intell. Lab. Syst., 1998).
% Another acceleration method incorporated here is to 
% extrapolate the individual loading elements a number of 
% iterations ahead after a specified number of iterations.
%
% A temporary MAT-file called TEMP.mat is saved for every 
% 50 iterations. IF the computer breaks down or the model 
% seems to be good enough, one can break the program and 
% load the last saved estimate. The loadings in TEMP.MAT
% are given a cell array as described below and can be 
% converted to A, B, C etc. by FAC2LET.M
% 
% All loading vectors except in first mode are normalized, 
% so that all variance is kept in the first mode (as is 
% common in two-way PCA). The components are arranged as
% in PCA. After iterating, the most important component is
% made the first component etc.
%
%
%
% ----------------------INPUT---------------------
%
% X          X is the input array, which can be from three- to N-way (also
%            twoway if the third mode is interpreted as a onedimensional
%            mode). 
%
% Fac		    No of factors/components sought.
%
%
% ----------------OPTIONAL INPUT---------------------
%
% Options    Optional parameters. If not given or set to zero or [], 
%            defaults will be used. If you want Options(5) to be 2 and
%            not change others, simply write Options(5)=2. Even if Options
%            hasn't been defined Options will contain zeros except its
%            fifth element.
%
%            Options(1) - Convergence criterion
%            The relative change in fit for which the algorithm stops.
%            Standard is 1e-6, but difficult data might require a lower value.
%  
%            Options(2) - Initialization method
%            This option is ignored if PARAFAC is started with old values.
%            If no default values are given the default Options(2) is 0.
%            The advantage of using DTLD or SVD for initialization is that
%            they often provide good starting values. However, since the 
%            initial values are then fixed, repeating the fitting will give
%            the exact same solution. Therefore it is not possible to substantiate
%            if a local minimum has been reached. To avoid that use an initialization
%            based on random values (2).
%
%            0  = fit using DTLD/GRAM for initialization (default if three-way and no missing)
%            1  = fit using SVD vectors for initialization (default if higher than three-way or missing)
%            2  = fit using random orthogonalized values for initialization
%            10 = fit using the best-fitting models of several models
%            fitted using a few iterations
%
%            Options(3) - Plotting options
%            2=produces several graphical outputs (loadings shown during iterations)
%            1=as above, but graphics only shown after convergence
%            0=no plots
%
%            Options(4) - Not user-accesible
% 
%            Options(5) - How often to show fit
%            Determines how often the deviation between the model and the data
%            is shown. This is helpful for adjusting the output to the number
%            of iterations. Default is 10. If showfit is set to NaN, almost no
%            outputs are given 
%
%            Options(6) - Maximal number of iterations
%            Maximal number of iterations allowed. Default is 2500.
%
% const      A vector telling type of constraints put on the loadings of the
%            different modes. Same size as DimX but the i'th element tells
%            what constraint is on that mode.
%            0 => no constraint,
%            1 => orthogonality
%            2 => nonnegativity
%            3 => unimodality (and nonnegativitiy)
%            If const is not defined, no constraints are used.
%            For no constraints in a threeway problem const = [0 0 0]
%
% OldLoad    If initial guess of the loadings is available. OldLoad should be
%            given a cell array where OldLoad{1}=A,OldLoad{2}=B etc.
%
% FixMode    FixMode is a binary vector of same sixe as DimX. If 
%            FixMode(i) = 1 => Mode i is fixed (requires old values given)
%            FixMode(i) = 0 => Mode i is not fixed hence estimated
%            Ex.: FixMode = [0 1 1] find the scores of a data set given the loadings.
%            When some modes are fixed, the numbering of the components will 
%            also be fixed. Normally components are sorted according to variance
%            as in PCA, but this will not be performed if some modes are fixed.
%
% Weights    If a matrix of the same size as X is given, weighted regression
%            is performed using the weights in the matrix Weights. 
%
% ---------------------OUTPUT---------------------
%
% Factors    PARAFAC estimate of loadings in one matrix. For a 3 component
%            solution to a 4 x 3 x 3 array the loadings A, B & C will be
%            stored in a 3 element cell vector:
%            Factors{1}=A,
%            Factors{2}=B
%            Factors{3}=C
%            etc.
%
%            Use FAC2LET.M for converting to "normal" output.
%
% it         Number of iterations used. Can be helpful for checking if the algorithm
%            has converged or simply hit the maximal number of iterations (default 2500).
%
% err        The fit of the model = the sum of squares of errors (not including missing
%            elements).
%
% Corcondia  Core consistency test. Should ideally be 100%. If significantly below
%            100% the model is not valid
%
%
%
% OTHER STUFF
%  
%  Missing values are handled by expectation maximization only. Set all 
%  missing values to NaN
%
%  COMMAND LINE (SHORT)
%
%  Factors = parafac(X,Fac);
%

% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% Phone  +45 35283296
% Fax    +45 35283245
% E-mail rb@kvl.dk

% $ Version 1.03 $ Date 1. October   1998 $ Not compiled $ Changed sign-convention because of problems with centered data
% $ Version 1.04 $ Date 18. February 1999 $ Not compiled $ Removed auxiliary line
% $ Version 1.06 $ Date 1. December  1999 $ Not compiled $ Fixed bug in low fit error handling
% $ Version 1.07 $ Date 17. January  2000 $ Not compiled $ Fixed bug in nnls handling so that the algorithm is not stopped until nonnegative appear
% $ Version 1.08 $ Date 21. January  2000 $ Not compiled $ Changed init DTLD so that primarily negative loadings are reflected if possible
% $ Version 1.09 $ Date 30. May 2000 $ Not compiled $ changed name noptioPF to noptiopf
% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $
% $ Version 2.001 $ June 2001 $ Fixed error in weighted regression $ RB $ Not compiled $

NumbIteraInitia=20;

if nargin==0
  disp(' ')
  disp(' ')
  disp(' THE PARAFAC MODEL')
  disp(' ')
  disp(' Type <<help parafac>> for more info')
  disp('  ')
  disp(' [Factors,it,err,Corcondia] = parafac(X,Fac,Options,const,OldLoad,FixMode,Weights);')
  disp(' or short')
  disp(' Factors = parafac(X,Fac);')
  disp(' ')
  disp(' Options=[Crit Init Plot NotUsed ShowFit MaxIt]')
  disp(' ')
  disp(' ')
  disp(' EXAMPLE:')
  disp(' To fit a four-component PARAFAC model to X of size 6 x 2 x 200 x 3 type')
  disp(' Factors=parafac(X,4)')
  disp(' and to obtain the scores and loadings from the output type')
  disp(' [A,B,C,D]=fac2let(Factors);')
  return
elseif nargin<2
  error(' The inputs X, and Fac must be given')
end

DimX = size(X);
X = reshape(X,DimX(1),prod(DimX(2:end)));

if nargin<3
  Options = [1e-6 0 0 1 10 2500]';
  OptionsDefault=Options;
else
  % Call the current Options OptionsHere and load default to use if some of the current settings should be default
  Options=Options(:);
  I=length(Options);
  if I==0
    Options=zeros(8,1);
  end
  I=length(Options);
  if I<8
    Options=[Options;zeros(8-I,1)];
  end
  OptionsHere=Options;
  Options = [1e-6 0 0 1 10 2500]';
  OptionsDefault=Options;
  Options=OptionsHere;
end

if ~exist('OldLoad')==1
  OldLoad=0;
elseif length(OldLoad)==0
  OldLoad=0;
end

% Convergence criteria
if Options(1,1)==0
  Options(1,1)=OptionsDefault(1,1);
end
crit=Options(1);

% Initialization
if ~any(Options(2))
  Options(2)=OptionsDefault(2);
end
Init=Options(2);

% Interim plotting
Plt=Options(3,1);
if ~any([0 1 2]==Plt)
  error(' Options(3,1) - Plotting - not set correct; must be 0,1, or 2')
end

if Options(5,1)==0
  Options(5,1)=OptionsDefault(5,1);
end
showfit=Options(5,1);
if isnan(showfit)
  showfit=-1;
end
if showfit<-1|round(showfit)~=showfit
  error(' Options(5,1) - How often to show fit - not set correct; must be positive integer or -1')
end

if Options(6,1)==0
  Options(6,1)=OptionsDefault(6,1);
  maxit=Options(6,1);
elseif Options(6)>0&round(Options(6))==Options(6)
  maxit=Options(6,1);
else
  error(' Options(6,1) - Maximal number of iterations - not set correct; must be positive integer')
end

ShowPhi=0; % Counter. Tuckers congruence coef/Multiple cosine/UUC shown every ShowPhiWhen'th time the fit is shown
ShowPhiWhen=10;
MissConvCrit=1e-4; % Convergence criterion for estimates of missing values
NumberOfInc=0; % Counter for indicating the number of iterations that increased the fit. ALS algorithms ALLWAYS decrease the fit, but using outside knowledge in some sense (approximate equality or iteratively reweighting might cause the algorithm to diverge

% INITIALIZE 
if showfit~=-1
  disp(' ') 
  disp(' PRELIMINARY')
  disp(' ')
end
ord=length(DimX);

if showfit~=-1
  disp([' A ',num2str(Fac),'-component model will be fitted'])
end

if exist('const')~=1
  const=zeros(size(DimX));
elseif length(const)~=ord
  const=zeros(size(DimX));
  if showfit~=-1
    disp(' Constraints are not given properly')
  end
end

if showfit~=-1
  for i=1:ord
    if const(i)==0
      disp([' No constraints on mode ',num2str(i)])
    elseif const(i)==1
      disp([' Orthogonality on mode ',num2str(i)])
    elseif const(i)==2
      disp([' Nonnegativity on mode ',num2str(i)])
    elseif const(i)==3
      disp([' Unimodality on mode ',num2str(i)])
    end
  end
end

% Check if orthogonality required on all modes
if max(max(const))==1
  if min(min(const))==1,disp(' ')
    disp(' Not possible to orthogonalize all modes in this implementation.')
    error(' Contact the authors for further information')
  end
end

if exist('FixMode')==1
  if length(FixMode)~=ord
    FixMode = zeros(1,ord);
  end
else
  FixMode = zeros(1,ord);
end

if showfit~=-1
  if any(FixMode)
    disp([' The loadings of mode : ',num2str(find(FixMode(:)')),' are fixed']) 
  end
end
if exist('Weights')~=1
  Weights=[];
end

% Display convergence criterion
if showfit~=-1
  disp([' The convergence criterion is ',num2str(crit)]) 
end

% Define loading as one ((r1*r2*r3*...*r7)*Fac x 1) vector [A(:);B(:);C(:);...].
% The i'th loading goes from lidx(i,1) to lidx(i,2)
lidx=[1 DimX(1)*Fac];
for i=2:ord
  lidx=[lidx;[lidx(i-1,2)+1 sum(DimX(1:i))*Fac]];
end

% Check if weighted regression required
if size(Weights,1)==size(X,1)&prod(size(Weights))/size(X,1)==size(X,2)
  Weights = reshape(Weights,size(Weights,1),prod(size(Weights))/size(X,1));
  if showfit~=-1
    disp(' Given weights will be used for weighted regression')
  end
  DoWeight=1;
else
  if showfit~=-1
    disp(' No weights given')
  end
  DoWeight=0;
end

% Make idx matrices if missing values
if any(isnan(X(:)))
  MissMeth=1;
else
  MissMeth=0;
end
if MissMeth
  id=sparse(find(isnan(X)));
  idmiss2=sparse(find(~isnan(X)));
  if showfit~=-1
    disp([' ', num2str(100*(length(id)/prod(DimX))),'% missing values']);
    disp(' Expectation maximization will be used for handling missing values')
  end
  SSX=sum(sum(X(idmiss2).^2)); % To be used for evaluating the %var explained
  % If weighting to zero should be used
  % Replace missing with mean values or model estimates initially
  if length(OldLoad)==sum(DimX)*Fac
    model=nmodel(OldLoad);
    model = reshape(model,DimX);
    X(id)=model(id);
  else
    meanX=mean(X(find(~isnan(X))));
    meanX=mean(meanX);
    X(id)=meanX*ones(size(id));
  end
else
  if showfit~=-1
    disp(' No missing values')
  end
  SSX=sum(sum(X.^2)); % To be used for evaluating the %var explained
end

% Check if weighting is tried used together with unimodality or orthogonality
if any(const==3)|any(const==1)
  if DoWeight==1
    disp(' ')
    disp(' Weighting is not possible together with unimodality and orthogonality.')
    disp(' It can be done using majorization, but has not been implemented here')
    disp(' Please contact the authors for further information')
    error
  end
end

% Acceleration
acc=-5;     
do_acc=1;   % Do acceleration every do_acc'th time
acc_pow=2;  % Extrapolate to the iteration^(1/acc_pow) ahead
acc_fail=0; % Indicate how many times acceleration have failed 
max_fail=4; % Increase acc_pow with one after max_fail failure
if showfit~=-1
  disp(' Line-search acceleration scheme initialized')
end

% Find initial guesses for the loadings if no initial values are given

% Use old loadings
if length(OldLoad)==ord % Use old values
  if showfit~=-1
    disp(' Using old values for initialization')
  end
  Factors=OldLoad;
  % Use DTLD
elseif Init==0
  if min(DimX)>1&ord==3&MissMeth==0
    if showfit~=-1
      disp(' Using direct trilinear decomposition for initialization')
    end
    [A,B,C]=dtld(reshape(X,DimX),Fac);
    A=real(A);B=real(B);C=real(C);
    % Check for signs and reflect if appropriate
    for f=1:Fac
      if sign(sum(A(:,f)))<0
        if sign(sum(B(:,f)))<0
          B(:,f)=-B(:,f);
          A(:,f)=-A(:,f);
        elseif sign(sum(C(:,f)))<0
          C(:,f)=-C(:,f);
          A(:,f)=-A(:,f);
        end
      end
      if sign(sum(B(:,f)))<0
        if sign(sum(C(:,f)))<0
          C(:,f)=-C(:,f);
          B(:,f)=-B(:,f);
        end
      end
    end
    Factors{1}=A;Factors{2}=B;Factors{3}=C;
    
  else
    if showfit~=-1
      disp(' Using singular values for initialization')
    end
    Factors=ini(X,Fac,2);
  end
  
  % Use SVD 
elseif Init==1
  if showfit~=-1
    disp(' Using singular values for initialization')
  end
  Factors=ini(X,Fac,2);
  
  % Use random (orthogonal)
elseif Init==2
  if showfit~=-1
    disp(' Using orthogonal random for initialization')
  end
  Factors=ini(X,Fac,1);
  
elseif Init==3
  error(' Initialization option set to three has been changed to 10')
  
  % Use several small ones of the above
elseif Init==10
  if showfit~=-1
    disp(' Using several small runs for initialization')
  end
  Opt=Options;
  Opt(5) = NaN;
  Opt(6) = NumbIteraInitia;
  Opt(2) = 0;
  ERR=[];
  [Factors,it,err] = parafac(X,Fac,Opt,const,[],[],Weights);
  ERR = [ERR;err];
  Opt(2) = 1;
  [F,it,Err] = parafac(X,Fac,Opt,const,[],[],Weights);
  ERR=[ERR;Err];
  if Err<err
    Factors=F;
    err=Err;
  end
  Opt(2)=2;
  for rep=1:3
    [F,it,Err]=parafac(X,Fac,Opt,const,[],[],Weights);
    ERR=[ERR;Err];
    if Err<err
      Factors=F;
      err=Err;
    end
  end
  if showfit~=-1
    disp(' ')
    disp(' Obtained fit-values')
    disp([' Method   Fit'])
    disp([' DTLD     ',num2str(ERR(1))])
    disp([' SVD      ',num2str(ERR(2))])
    disp([' RandOrth ',num2str(ERR(3))])
    disp([' RandOrth ',num2str(ERR(4))])
    disp([' RandOrth ',num2str(ERR(5))])
  end
else
  error(' Problem in PARAFAC initialization - Not set correct')
end
% Convert to old format
ff = [];
for f=1:length(Factors)
  ff=[ff;Factors{f}(:)];
end
Factors = ff;

% ALTERNATING LEAST SQUARES
err=SSX;
f=2*crit;
it=0;
connew=2;conold=1; % for missing values
ConstraintsNotRight = 0; % Just to ensure that iterations are not stopped if constraints are not yet fully imposed

if showfit~=-1
  disp(' ')
  disp(' Sum-of-Squares   Iterations  Explained')
  disp(' of residuals                 variation')
end

while ((f>crit) | (norm(connew-conold)/norm(conold)>MissConvCrit) | ConstraintsNotRight) & it<maxit
  conold=connew; % for missing values
  it=it+1;
  acc=acc+1; 
  if acc==do_acc;
    Load_o1=Factors;
  end
  if acc==do_acc+1;
    acc=0;Load_o2=Factors;
    Factors=Load_o1+(Load_o2-Load_o1)*(it^(1/acc_pow));
    % Convert to new format
    clear ff,id1 = 0;
    for i = 1:length(DimX) 
      id2 = sum(DimX(1:i).*Fac);ff{i} = reshape(Factors(id1+1:id2),DimX(i),Fac);id1 = id2;
    end
    model=nmodel(ff);
    model = reshape(model,DimX(1),prod(DimX(2:end)));
    
    if MissMeth
      connew=model(id);
      errX=X-model;
      if DoWeight==0
        nerr=sum(sum(errX(idmiss2).^2));
      else
        nerr=sum(sum((Weights(idmiss2).*errX(idmiss2)).^2));
      end
    else
      if DoWeight==0
        nerr=sum(sum((X-model).^2));
      else
        nerr=sum(sum((X.*Weights-model.*Weights).^2));
      end
    end
    if nerr>err
      acc_fail=acc_fail+1;
      Factors=Load_o2;
      if acc_fail==max_fail,
        acc_pow=acc_pow+1+1;
        acc_fail=0;
        if showfit~=-1
          disp(' Reducing acceleration');
        end
      end
    else
      if MissMeth
        X(id)=model(id);
      end
    end
  end
  
  
  if DoWeight==0
    for ii=ord:-1:1
      if ii==ord;
        i=1;
      else
        i=ii+1;
      end
      idd=[i+1:ord 1:i-1];
      l_idx2=lidx(idd,:);
      dimx=DimX(idd);
      if ~FixMode(i)
        L1=reshape(Factors(l_idx2(1,1):l_idx2(1,2)),dimx(1),Fac);
        if ord>2
          L2=reshape(Factors(l_idx2(2,1):l_idx2(2,2)),dimx(2),Fac);
          Z=kr(L2,L1);
        else
          Z = L1;
        end
        for j=3:ord-1
          L1=reshape(Factors(l_idx2(j,1):l_idx2(j,2)),dimx(j),Fac);
          Z=kr(L1,Z);
        end
        ZtZ=Z'*Z;
        ZtX=Z'*X';
        OldLoad=reshape(Factors(lidx(i,1):lidx(i,2)),DimX(i),Fac);
        L=pfls(ZtZ,ZtX,DimX(i),const(i),OldLoad,DoWeight,Weights);
        Factors(lidx(i,1):lidx(i,2))=L(:);
      end
      x=zeros(prod(DimX([1:ii-1 ii+1:ord])),DimX(ii));  % Rotate X so the current last mode is the first
      x(:)=X;
      X=x';
    end
  else
    for ii=ord:-1:1
      if ii==ord;
        i=1;
      else
        i=ii+1;
      end
      idd=[i+1:ord 1:i-1];
      l_idx2=lidx(idd,:);
      dimx=DimX(idd);
      if ~FixMode(i)
        L1=reshape(Factors(l_idx2(1,1):l_idx2(1,2)),dimx(1),Fac);
        if ord>2
          L2=reshape(Factors(l_idx2(2,1):l_idx2(2,2)),dimx(2),Fac);
          Z=kr(L2,L1);
        else
          Z = L1;
        end
        for j=3:ord-1
          L1=reshape(Factors(l_idx2(j,1):l_idx2(j,2)),dimx(j),Fac);
          Z=kr(L1,Z);
        end
        OldLoad=reshape(Factors(lidx(i,1):lidx(i,2)),DimX(i),Fac);
        L=pfls(Z,X,DimX(i),const(i),OldLoad,DoWeight,Weights);
        Factors(lidx(i,1):lidx(i,2))=L(:);
      end
      x=zeros(prod(DimX([1:ii-1 ii+1:ord])),DimX(ii));
      x(:)=X;
      X=x';
      x(:)=Weights;
      Weights=x';
    end
  end
  
  % EVALUATE SOFAR
  % Convert to new format
  clear ff,id1 = 0;
  for i = 1:length(DimX) 
    id2 = sum(DimX(1:i).*Fac);
    ff{i} = reshape(Factors(id1+1:id2),DimX(i),Fac);
    id1 = id2;
  end
  model=nmodel(ff);
  model = reshape(model,DimX(1),prod(DimX(2:end)));
  if MissMeth  % Missing values present
    connew=model(id);
    X(id)=model(id);
    errold=err;
    errX=X-model;
    if DoWeight==0
      err=sum(sum(errX(idmiss2).^2));
    else
      err=sum(sum((Weights(idmiss2).*errX(idmiss2)).^2));
    end
  else
    errold=err;
    if DoWeight==0
      err=sum(sum((X-model).^2));
    else
      err=sum(sum((Weights.*(X-model)).^2));
    end
  end
  
  if err<1000*eps, % Getting close to the machine uncertainty => stop
    disp(' WARNING')
    disp(' The misfit is approaching the machine uncertainty')
    disp(' If pure synthetic data is used this is OK, otherwise if the')
    disp(' data elements are very small it might be appropriate ')
    disp(' to multiply the whole array by a large number to increase')
    disp(' numerical stability. This will only change the solution ')
    disp(' by a scaling constant')
    f = 0;
  else
    f=abs((err-errold)/err);
    if f<crit % Convergence: then check that constraints are fulfilled
      if any(const==2)|any(const==3) % If nnls or unimodality imposed
        for i=1:ord % Extract the 
          if const(i)==2|const(i)==3 % If nnls or unimodality imposed
            Loadd = Factors(sum(DimX(1:i-1))*Fac+1:sum(DimX(1:i))*Fac);
            if any(Loadd<0)
              ConstraintsNotRight=1;
            else
              ConstraintsNotRight=0;
            end
          end
        end
      end
    end
  end
  
  if it/showfit-round(it/showfit)==0
    if showfit~=-1,
      ShowPhi=ShowPhi+1;
      if ShowPhi==ShowPhiWhen,
        ShowPhi=0;
        if showfit~=-1,
          disp(' '),
          disp('    Tuckers congruence coefficient'),
          % Convert to new format
          clear ff,id1 = 0;
          for i = 1:length(DimX) 
            id2 = sum(DimX(1:i).*Fac);ff{i} = reshape(Factors(id1+1:id2),DimX(i),Fac);id1 = id2;
          end
          [phi,out]=ncosine(ff,ff);
          disp(phi),
          if MissMeth
            fprintf(' Change in estim. missing values %12.10f',norm(connew-conold)/norm(conold));
            disp(' ')
            disp(' ')
          end
          disp(' Sum-of-Squares   Iterations  Explained')
          disp(' of residuals                 variation')
        end
      end
      if DoWeight==0
        PercentExpl=100*(1-err/SSX);
      else
        PercentExpl=100*(1-sum(sum((X-model).^2))/SSX);
      end
      fprintf(' %12.10f       %g        %3.4f    \n',err,it,PercentExpl);
      if Plt==2
        % Convert to new format
        clear ff,id1 = 0;
        for i = 1:length(DimX) 
          id2 = sum(DimX(1:i).*Fac);ff{i} = reshape(Factors(id1+1:id2),DimX(i),Fac);id1 = id2;
        end
        pfplot(reshape(X,DimX),ff,Weights',[0 0 0 0 0 0 0 1]);
        drawnow
      end
    end
  end
  
  
  
  % Make safety copy of loadings and initial parameters in temp.mat
  if it/50-round(it/50)==0
    save temp Factors
  end
  
  % JUDGE FIT
  if err>errold
    NumberOfInc=NumberOfInc+1;
  end
  
end % while f>crit


% CALCULATE TUCKERS CONGRUENCE COEFFICIENT
if showfit~=-1 & DimX(1)>1
  disp(' '),disp('   Tuckers congruence coefficient')
  % Convert to new format
  clear ff,id1 = 0;
  for i = 1:length(DimX) 
    id2 = sum(DimX(1:i).*Fac);ff{i} = reshape(Factors(id1+1:id2),DimX(i),Fac);id1 = id2;
  end
  [phi,out]=ncosine(ff,ff);
  disp(phi)
  disp(' ')
  if max(max(abs(phi)-diag(diag(phi))))>.85
    disp(' ')
    disp(' ')
    disp(' WARNING, SOME FACTORS ARE HIGHLY CORRELATED.')
    disp(' ')
    disp(' You could decrease the number of components. If this')
    disp(' does not help, try one of the following')
    disp(' ')
    disp(' - If systematic variation is still present you might')
    disp('   wanna decrease your convergence criterion and run')
    disp('   one more time using the loadings as initial guess.')
    disp(' ')
    disp(' - Or use another preprocessing (check for constant loadings)')
    disp(' ')
    disp(' - Otherwise try orthogonalising some modes,')
    disp(' ')
    disp(' - Or use Tucker3/Tucker2,')
    disp(' ')
    disp(' - Or a PARAFAC with some modes collapsed (if # modes > 3)')
    disp(' ')
  end
end


% SHOW FINAL OUTPUT

if DoWeight==0
  PercentExpl=100*(1-err/SSX);
else
  PercentExpl=100*(1-sum(sum((X-model).^2))/SSX);
end
if showfit~=-1
  fprintf(' %12.10f       %g        %3.4f \n',err,it,PercentExpl);
  if NumberOfInc>0
    disp([' There were ',num2str(NumberOfInc),' iterations that increased fit']);
  end
end


% POSTPROCES LOADINGS (ALL VARIANCE IN FIRST MODE)
A=reshape(Factors(lidx(1,1):lidx(1,2)),DimX(1),Fac);
for i=2:ord
  B=reshape(Factors(lidx(i,1):lidx(i,2)),DimX(i),Fac);
  for ff=1:Fac
    A(:,ff)=A(:,ff)*norm(B(:,ff));
    B(:,ff)=B(:,ff)/norm(B(:,ff));
  end
  Factors(lidx(i,1):lidx(i,2))=B(:);
end
Factors(lidx(1,1):lidx(1,2))=A(:);
if showfit~=-1
  disp(' ')
  disp(' Components have been normalized in all but the first mode')
end

% PERMUTE SO COMPONENTS ARE IN ORDER AFTER VARIANCE DESCRIBED (AS IN PCA) IF NO FIXED MODES
if ~any(FixMode)
  A=reshape(Factors(lidx(1,1):lidx(1,2)),DimX(1),Fac);
  [out,order]=sort(diag(A'*A));
  order=flipud(order);
  A=A(:,order);
  Factors(lidx(1,1):lidx(1,2))=A(:);
  for i=2:ord
    B=reshape(Factors(lidx(i,1):lidx(i,2)),DimX(i),Fac);
    B=B(:,order);
    Factors(lidx(i,1):lidx(i,2))=B(:);
  end  
  if showfit~=-1
    disp(' Components have been ordered according to contribution')
  end
elseif showfit ~= -1
  disp(' Some modes fixed hence no sorting of components performed')
end

% APPLY SIGN CONVENTION IF NO FIXED MODES


%  FixMode=1
if ~any(FixMode)&~(any(const==2)|any(const==3))
  Sign = ones(1,Fac);
  for i=ord:-1:2
    A=reshape(Factors(lidx(i,1):lidx(i,2)),DimX(i),Fac);
    Sign2=ones(1,Fac);
    for ff=1:Fac
      [out,sig]=max(abs(A(:,ff)));
      Sign(ff) = Sign(ff)*sign(A(sig,ff));
      Sign2(ff) = sign(A(sig,ff));
    end
    A=A*diag(Sign2);
    Factors(lidx(i,1):lidx(i,2))=A(:);
  end 
  A=reshape(Factors(lidx(1,1):lidx(1,2)),DimX(1),Fac);
  A=A*diag(Sign);
  Factors(lidx(1,1):lidx(1,2))=A(:);
  if showfit~=-1
    disp(' Components have been reflected according to convention')
  end
  
end 

% TOOLS FOR JUDGING SOLUTION
if nargout>3      
  x=X;
  if MissMeth
    x(id)=NaN*id;
  end
  % Convert to new format
  clear ff,id1 = 0;
  for i = 1:length(DimX) 
    id2 = sum(DimX(1:i).*Fac);ff{i} = reshape(Factors(id1+1:id2),DimX(i),Fac);id1 = id2;
  end
  corcondia=corcond(reshape(x,DimX),ff,Weights,1);
end

if Plt==1|Plt==2
  % Convert to new format
  clear ff,id1 = 0;
  for i = 1:length(DimX) 
    id2 = sum(DimX(1:i).*Fac);ff{i} = reshape(Factors(id1+1:id2),DimX(i),Fac);id1 = id2;
  end
  
  pfplot(reshape(X,DimX),ff,Weights,ones(1,8));
end

% Show which criterion stopped the algorithm
if showfit~=-1
  if ((f<crit) & (norm(connew-conold)/norm(conold)<MissConvCrit))
    disp(' The algorithm converged')
  elseif it==maxit
    disp(' The algorithm did not converge but stopped because the')
    disp(' maximum number of iterations was reached')
  elseif f<eps
    disp(' The algorithm stopped because the change in fit is now')
    disp(' smaller than the machine uncertainty.')
  else
    disp(' Algorithm stopped for some mysterious reason')
  end
end

% Convert to new format
clear ff,id1 = 0;
for i = 1:length(DimX) 
  id2 = sum(DimX(1:i).*Fac);ff{i} = reshape(Factors(id1+1:id2),DimX(i),Fac);id1 = id2;
end
Factors = ff;


function [A,B,C,fit]=dtld(X,F,SmallMode);

%DTLD direct trilinear decomposition
%
% See also:
% 'gram', 'parafac'
%
% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% Phone  +45 35283296
% Fax    +45 35283245
% E-mail rb@kvl.dk
%
%
% DIRECT TRILINEAR DECOMPOSITION
%
% calculate the parameters of the three-
% way PARAFAC model directly. The model
% is not the least-squares but will be close
% to for precise data with little model-error
%
% This implementation works with an optimal
% compression using least-squares Tucker3 fitting
% to generate two pseudo-observation matrices that
% maximally span the variation of all samples. per
% default the mode of smallest dimension is compressed
% to two samples, while the remaining modes are 
% compressed to dimension F.
% 
% For large arrays it is fastest to have the smallest
% dimension in the first mode
%
% INPUT
% [A,B,C]=dtld(X,F);
% X is the I x J x K array
% F is the number of factors to fit
% An optional parameter may be given to enforce which
% mode is to be compressed to dimension two
%
% Copyright 1998
% Rasmus Bro, KVL
% rb@kvl.dk

% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $
% $ Version 1.02 $ Date 28. July 1998 $ Not compiled $
% $ Version 1.03 $ Date 25. April 1999 $ Not compiled $

DimX = size(X);
X = reshape(X,DimX(1),prod(DimX(2:end)));

DontShowOutput = 1;

%rearrange X so smallest dimension is in first mode


if nargin<4
  [a,SmallMode] = min(DimX);
  X = nshape(reshape(X,DimX),SmallMode);
  DimX = DimX([SmallMode 1:SmallMode-1 SmallMode+1:3]);
  Fac   = [2 F F];
else
  X = nshape(reshape(X,DimX),SmallMode);
  DimX = DimX([SmallMode 1:SmallMode-1 SmallMode+1:3]);
  Fac   = [2 F F];
end
f=F;
if F==1;
  Fac   = [2 2 2];
  f=2;
end 


if DimX(1) < 2
  error(' The smallest dimension must be > 1')
end

if any(DimX(2:3)-Fac(2:3)<0)
  error(' This algorithm requires that two modes are of dimension not less the number of components')
end



% Compress data into a 2 x F x F array. Only 10 iterations are used since exact SL fit is insignificant; only obtaining good truncated bases is important
[Factors,Gt]=tucker(reshape(X,DimX),Fac,[0 0 0 0 NaN 10]);
% Convert to old format
Gt = reshape(Gt,size(Gt,1),prod(size(Gt))/size(Gt,1));

[At,Bt,Ct]=fac2let(Factors);

% Fit GRAM to compressed data
[Bg,Cg,Ag]=gram(reshape(Gt(1,:),f,f),reshape(Gt(2,:),f,f),F);

% De-compress data and find A


BB = Bt*Bg;
CC = Ct*Cg;
AA = X*pinv(kr(CC,BB)).';

if SmallMode == 1
  A=AA;
  B=BB;
  C=CC;
elseif SmallMode == 2 
  A=BB;
  B=AA;
  C=CC;
elseif SmallMode == 3
  A=BB;
  B=CC;
  C=AA;
end

fit = sum(sum(abs(X - AA*kr(CC,BB).').^2));
if ~DontShowOutput
  disp([' DTLD fitted raw data with a sum-squared error of ',num2str(fit)])
end


function mwa = outerm(facts,lo,vect)

if nargin < 2
  lo = 0;
end
if nargin < 3
  vect = 0;
end
order = length(facts);
if lo == 0
  mwasize = zeros(1,order);
else
  mwasize = zeros(1,order-1);
end
k = 0;
for i = 1:order
  if i ~= lo
    [m,n] = size(facts{i});
    k = k + 1;
    mwasize(k) = m;
    if k > 1
      if nofac ~= n
        error('All orders must have the same number of factors')
      end
    else
      nofac = n;
    end
  end
end
mwa = zeros(prod(mwasize),nofac);

for j = 1:nofac
  if lo ~= 1
    mwvect = facts{1}(:,j);
    for i = 2:order
      if lo ~= i
        %mwvect = kron(facts{i}(:,j),mwvect);
        mwvect = mwvect*facts{i}(:,j)';
        mwvect = mwvect(:);
      end
    end
  elseif lo == 1
    mwvect = facts{2}(:,j);
    for i = 3:order
      %mwvect = kron(facts{i}(:,j),mwvect);
      mwvect = mwvect*facts{i}(:,j)';
      mwvect = mwvect(:);
    end
  end
  mwa(:,j) = mwvect;
end
% If vect isn't one, sum up the results of the factors and reshape
if vect ~= 1
  mwa = sum(mwa,2);
  mwa = reshape(mwa,mwasize);
end



function [G]=calcore(X,Factors,Options,O,MissingExist);

%CALCORE Calculate the Tucker core
%
%	
% [G]=calcore(X,Factors,Options);
% [G]=calcore(X,Factors);
%
% This algorithm applies to the general N-way case, so
% the unfolded X can have any number of dimensions. The principles of
% 'projections' and 'systematic unfolding methodology (SUM)' are used
% in this algorithm so orthogonality is required.
% This algorithm can handle missing values in X and
% also allows for TUCKER2 models using the an empty matrix in the
% corresponding cell of Factors.
% The variable 'Factors' must contain the stringed-out factors.

%	Copyright
%	Claus A. Andersson 1995-1997
%	Chemometrics Group, Food Technology
%	Department of Food and Dairy Science
%	Royal Veterinary and Agricultutal University
%	Rolighedsvej 30, T254
%	DK-1958 Frederiksberg
%	Denmark
%
%	Phone 	+45 35283502
%	Fax	   +45 35283245
%	E-mail	claus@andersson.dk
%

format compact
format long

DimX = size(X);
X = reshape(X,DimX(1),prod(DimX(2:end)));
ff = [];
for f=1:length(Factors)
  ff=[ff;Factors{f}(:)];
  Fac(f)=size(Factors{f},2);
  if isempty(Factors{f}) % 'Tucker2' - i.e. no compression in that mode
    Fac(f) = -1;
  end
end
Factors = ff;

% Initialize system variables
if length(Fac)==1,
  Fac=Fac*ones(size(DimX));
end;

Fac_orig=Fac;
i=find(Fac==-1);
Fac(i)=zeros(1,length(i));
N=size(Fac,2);
FIdx0=zeros(1,N);
FIdx1=zeros(1,N);
if ~exist('MissingExist')
  if sum(isnan(X(:)))>0,
    MissingExist=1;
  else
    MissingExist=0;
  end;
end;
FIdx0=cumsum([1 DimX(1:N-1).*Fac(1:N-1)]);
FIdx1=cumsum([DimX.*Fac]);
if ~exist('O') | isempty(O),
  O=1;
end;


if O, %means orthogonality
  CurDimX=DimX;
  RedData=X;
  for c=1:N,
    
    if Fac_orig(c)==-1,
      kthFactor=eye(DimX(c));
      CurDimX(c)=DimX(c);
    else
      kthFactor=reshape(Factors(FIdx0(c):FIdx1(c)),DimX(c),Fac(c));
      CurDimX(c)=Fac(c);
    end;      
    if MissingExist
      RedData=missmult(kthFactor',RedData);
    else
      RedData=kthFactor'*RedData;
    end;
    
    if c~=N,
      newi=CurDimX(c+1);
      newj=prod(CurDimX)/CurDimX(c+1);
    else
      newi=CurDimX(1);
      newj=prod(CurDimX)/CurDimX(1);
    end;
    
    RedData=reshape(RedData',newi,newj);
  end;
  G=RedData;
else %oblique factors
  
  LMatTmp=1;
  if Fac_orig(1)==-1,
    LMatTmp=eye(DimX(c));
  else
    LMatTmp=reshape(Factors(FIdx0(1):FIdx1(1)),DimX(1),Fac(1));
  end;    
  
  RMatTmp=1;
  for c=2:N,
    if Fac_orig(c)==-1,
      kthFactor=eye(DimX(c));
    else
      kthFactor=reshape(Factors(FIdx0(c):FIdx1(c)),DimX(c),Fac(c));
    end;    
    RMatTmp=ckron(kthFactor',RMatTmp);
  end;
  
  if MissingExist
    RedData=missmult(pinv(LMatTmp),X);
    RedData=missmult(RedData,pinv(RMatTmp));
  else
    RedData=LMatTmp\X;
    RedData=RedData/RMatTmp;
  end;
  
  G=RedData;
  
end;    

for i = 1:length(Fac)
  if Fac(i)==0
    Fac(i) = DimX(i);
  end
end
G = reshape(G,Fac);

return



function C=ckron(A,B)
%CKRON
% C=ckron(A,B)
%
% Claus Andersson, Jan. 1996

% Should not be compiled to overwrite ckron.mex

[mA,nA] = size(A);
[mB,nB] = size(B);

C = zeros(mA*mB,nA*nB);
if mA*nA <= mB*nB
  for i = 1:mA
    iC = 1+(i-1)*mB:i*mB;
    for j = 1:nA
      jC = 1+(j-1)*nB:j*nB;
      C(iC,jC) = A(i,j)*B;
    end
  end
else
  for i = 1:mB
    iC = i:mB:(mA-1)*mB+i;
    for j = 1:nB
      jC = j:nB:(nA-1)*nB+j;
      C(iC,jC) = B(i,j)*A;
    end
  end
end

function [X]=missmult(A,B)

%MISSMULT product of two matrices containing NaNs
%
%[X]=missmult(A,B)
%This function determines the product of two matrices containing NaNs
%by finding X according to
%     X = A*B
%If there are columns in A or B that are pur missing values,
%then there will be entries in X that are missing too.
%
%The result is standardized, that is, corrected for the lower
%number of contributing terms.
%
%Missing elements should be denoted by 'NaN's


% Copyright
% Claus A. Andersson 1996-
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% E-mail: claus@andersson.dk

%INBOUNDS
%REALONLY

[ia ja]=size(A);
[ib jb]=size(B);
X=zeros(ia,jb);

one_arry=ones(ia,1);
for j=1:jb,
  p=one_arry*B(:,j)';
  tmpMat=A.*p;
  X(:,j)=misssum(tmpMat')';
end;



function [mm]=misssum(X,def)
%MISSSUM sum of a matrix X with NaN's
%
%[mm]=misssum(X,def)
%
%This function calculates the sum of a matrix X.
%X may hold missing elements denoted by NaN's which
%are ignored.
%
%The result is standardized, that is, corrected for the lower
%number of contributing terms.
%
%Check that for no column of X, all values are missing

% Copyright
% Claus A. Andersson 1996-
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% E-mail: claus@andersson.dk

%Insert zeros for missing, correct afterwards
missidx = isnan(X);
i = find(missidx);
if ~isempty(i),
  X(i) = zeros(size(i));
end;

%Find the number of real(non-missing objects)
if min(size(X))==1,
  n_real=length(X)-sum(missidx);
  weight=length(X);
else
  n_real=size(X,1)-sum(missidx);
  weight=size(X,1);
end

i=find(n_real==0);
if isempty(i) %All values are real and can be corrected
  mm=weight*sum(X)./n_real;
else %There are columns with all missing, insert missing
  n_real(i)=1;
  mm=weight*sum(X)./n_real;
  mm(i)=i + NaN;
end



function [ypred,T,ssX,Xres]=npred(X,Fac,Xfactors,Yfactors,Core,B,show)

%NPRED prediction with NPLS model
%
% See also:
% 'npls' 'testreg'
%
%
% Predict Y for a new set of samples using an N-PLS model
%
% [ypred,T,ssx,Xres] = npred(X,Fac,Xfactors,Yfactors,Core,B,show);
%
% INPUT
% X        The array to be predicted
% Fac      Number of factors to use in prediction
% Xfactors Parameters of the calibration model of X (incl. scores) in a cell array
% Yfactors Parameters of the calibration model of Y (incl. scores) in a cell array
% Core     Core array used for calculating the model of X
% B        Regression matrix of calibration model
%
% OUTPUT
% ypred    the predictions of y
% T        is the scores of the new samples
% ssX      sum-of-squares of x residuals
%          ssX(1,1)  sum-of-squares of X
%          ssX(2,1)  sum-of-squares of residuals
%          ssX(1,2)  percentage variation explained after 0 component (=0)
%          ssX(2,2)  percentage variation explained after Fac component
%
% Xres     is the residuals of X
%


%	Copyright
%	Rasmus Bro 1995
%	Denmark
%	E-mail rb@kvl.dk
% $ Version 2.01 $ June 2001 $ Changed to handle new core in X $ RB $ Not compiled $
% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $
% $ Version 1.02 $ Date 28. July 1998 $ Not compiled $
%
% Copyright, 1995 - 2001
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% Phone  +45 35283296
% Fax    +45 35283245
% E-mail rb@kvl.dk
%


% Convert to old notation

Dimx = size(X);
X = reshape(X,Dimx(1),prod(Dimx(2:end)));

DimX = size(Xfactors{1},1);
for j = 2:length(Xfactors)
  DimX(j) = size(Xfactors{j},1);
end
DimY = size(Yfactors{1},1);
for j = 2:length(Yfactors)
  DimY(j) = size(Yfactors{j},1);
end


if nargin==0
  disp(' ')
  disp(' NPRED')
  disp('[ypred,T,ssx,Xres] = npred(X,Fac,Xfactors,Yfactors,Core,B,show);')
  disp(' ')
  return
end

ord=length(DimX);


if ~exist('show')==1
  show=1;
end

maxit=20;


[I,not]=size(X);
if any(isnan(X(:)))
  miss=1-isnan(X);
  Missing=1;
else
  Missing=0;
end

Xres=X;
T=zeros(I,Fac);

W = [];
for f = 1:Fac
  w = Xfactors{end}(:,f);
  for o = length(DimX)-1:-1:2
    w = kron(w,Xfactors{o}(:,f));
  end
  W = [W w];
end
Q = [];
for f = 1:Fac
  q = Yfactors{end}(:,f);
  for o = length(DimY)-1:-1:2
    q = kron(q,Yfactors{o}(:,f));
  end
  Q = [Q q];
end


for f=1:Fac
  if Missing
    for i=1:I
      m = find(miss(i,:));
      T(i,f)=Xres(i,m)*W(m,f)/(W(m,f)'*W(m,f));
    end
  else
    T(:,f)=Xres*W(:,f);
  end
  if f==Fac
    Wkron = Xfactors{end}(:,1:Fac);
    for o = length(DimX)-1:-1:2
      Wkron = kron(Wkron,Xfactors{o}(:,1:Fac));
    end
    Xres=Xres-T(:,1:Fac)*reshape(Core{Fac},Fac,Fac^(length(DimX)-1))*Wkron';
  end
end

ypred=T*B(1:Fac,1:Fac)*Q(:,1:Fac)';

ssx=sum(Xres(find(~isnan(Xres))).^2);
ssX=sum(X(find(~isnan(Xres))).^2);
ssX=[ssX 0;ssx 100*(1-ssx/ssX)];

Xres = reshape(Xres,[size(X,1) DimX(2:end)]);


function [result]=crossvalparafac(X,F,A,H,C,P);

[I,J,K] = size(X);
randn('state',sum(100*clock));
rand('state',sum(100*clock));
ConvCrit = 1e-7;

% CROSS-VALIDATION

splits = 7;
while rem(I,splits)==0 % Change the number of segments if 7 is a divisor in prod(size(X))
  splits = splits + 2;
end
result.NumberOfSegments = splits;
SS = zeros(1,F);
for f = 1:F
  cwaitbar([2 f/F]);
  Arep = [];Brep = [];Crep = [];
  for s = 1:splits
    cwaitbar([3 s/splits]);
    Xmiss = X;
    Xmiss(s:splits:end)=NaN;
    Factors=parafac(Xmiss,f,[0 0 0 0 NaN]);
    a = Factors{1};b=Factors{2};c = Factors{3};
    %[a,b,c] = compparafac(reshape(Xmiss,I,J*K),[I J K],f);
    Arep(:,:,s)=a;Brep(:,:,s)=b;Crep(:,:,s)=c;
    for k = 1:K
      m(:,:,k)    = a*diag(c(k,:))*b';
    end
    SS(f) = SS(f) + sum(sum(((X(s:splits:end)-m(s:splits:end)).^2))); 
    XvalModel{f} = m;
  end
  %AddiOutput.XvalModels=XvalModel;
  result.SS = SS;
  result.A_xval{f}=Arep;
  result.B_xval{f}=Brep;
  result.C_xval{f}=Crep;
end

function fout = cwaitbar(x,name,col)
%CWAITBAR Display compound wait bar.
%   H = CWAITBAR(X,TITLE) creates and displays wait bars of 
%   fractional lengths X and with one title string TITLE.
%   The handle to the compound waitbar figure is returned in H.
%   X values should be between 0 and 1.
%   Each subsequent call to cwaitbar, CWAITBAR([BAR X]),
%   extends the length of the bar BAR to the new position X.
%   The first bar is the topmost bar and is BAR = 1 which
%   corresponds to the outermost loop.
%   H = CWAITBAR(X,TITLE) where TITLE is a cellstring with same
%   number of titles as there are fractional lengths in X.
%   Suitable for showing the bars' corresponding loop indices.
%   H = CWAITBAR(X,TITLE,COLOR) where COLOR is the color of the
%   bars. COLOR is either a color code character (see PLOT) or
%   an RGB vector. The default color is red. COLOR can also be 
%   a cell array with same number of elements as there are bars
%   in the cwaitbar figure.
%
%   The order of the elements in vector X and cell arrays TITLE
%   and COLOR which is consistent with the bar number BAR is:
%   The first element corresponds to the first bar at the top
%   of the figure which in turn corresponds to the outermost loop.
%
%   CWAITBAR is typically used inside nested FOR loops that
%   performs lengthy computations.
%
%      Examples:
%         cwaitbar([.3 .2 .7],'Please wait...');     %single title
%
%         h = cwaitbar([0 0 0],{'i','j','k'},{[.8 .2 .8],'b','r'});
%         for i=1:5,
%            % computations %
%            for j=1:10
%               % computations %
%               for k=1:100
%                  % computations %
%                  cwaitbar([3 k/100])
%               end
%               cwaitbar([2 j/10])
%            end
%            cwaitbar([1 i/5])
%         end
%         close(h)
%
%   See also WAITBAR.

% Based on matlab's WAITBAR. See help for WAITBAR.
% Copyright (c) 2003-11-02, B. Rasmus Anthin.
% Revision 2003-11-03 - 2003-11-06.
% GPL license.

xline = [100 0 0 100 100];
yline = [0 0 1 1 0];


switch nargin
case 1   % waitbar(x)    update
   bar=x(1);
   x=max(0,min(100*x(2),100));
   f = findobj(allchild(0),'flat','Tag','CWaitbar');
   if ~isempty(f), f=f(1);end
   a=sort(get(f,'child'));                         %axes objects
   if isempty(f) | isempty(a), 
      error('Couldn''t find waitbar handles.'); 
   end
   bar=length(a)+1-bar;        %first bar is the topmost bar instead
   if length(a)<bar
      error('Bar number exceeds number of available bars.')
   end
   for i=1:length(a)
      p(i)=findobj(a(i),'type','patch');
      l(i)=findobj(a(i),'type','line');      
   end
   %rewind upper bars when they are full
%   if bar==1
%      for i=2:length(a)
%         xpatchold=get(p(i),'xdata');
%         xold=xpatchold(2);
%         if xold==100
%            set(p,'erase','normal')
%            xpatch=[0 0 0 0];
%            set(p(i),'xdata',xpatch,'erase','none')
%            set(l(i),'xdata',xline)
%         end
%      end
%   end
   
   a=a(bar);
   p=p(bar);
   l=l(bar);
   xpatchold=get(p,'xdata');
   xold=xpatchold(2);
   if xold>x                      %erase old patches (if bar is shorter than before)
      set(p,'erase','normal')
      %xold=0;
   end
   xold=0;
        %previously: (continue on old patch)
   xpatch=[xold x x xold];
   set(p,'xdata',xpatch,'erase','none')
   set(l,'xdata',xline)

case 2   % waitbar(x,name)  initialize
   x=fliplr(max(0,min(100*x,100)));

   oldRootUnits = get(0,'Units');
   set(0, 'Units', 'points');
   pos = get(0,'ScreenSize');
   pointsPerPixel = 72/get(0,'ScreenPixelsPerInch');
   
   L=length(x)*.6+.4;
   width = 360 * pointsPerPixel;
   height = 75 * pointsPerPixel * L;
   pos = [pos(3)/2-width/2 pos(4)/2-height/2 width height];

   f = figure(...
      'Units', 'points', ...
      'Position', pos, ...
      'Resize','off', ...
      'CreateFcn','', ...
      'NumberTitle','off', ...
      'IntegerHandle','off', ...
      'MenuBar', 'none', ...
      'Tag','CWaitbar');
   colormap([]);
   
   for i=1:length(x)
      h = axes('XLim',[0 100],'YLim',[0 1]);
      if ~iscell(name)
         if i==length(x), title(name);end
      else
         if length(name)~=length(x)
            error('There must be equally many titles as waitbars, or only one title.')
         end
         title(name{end+1-i})
      end
      set(h, ...
         'Box','on', ...
         'Position',[.05 .3/L*(2*i-1) .9 .2/L],...
         'XTickMode','manual',...
         'YTickMode','manual',...
         'XTick',[],...
         'YTick',[],...
         'XTickLabelMode','manual',...
         'XTickLabel',[],...
         'YTickLabelMode','manual',...
         'YTickLabel',[]);
      
      xpatch = [0 x(i) x(i) 0];
      ypatch = [0 0 1 1];
      
      patch(xpatch,ypatch,'r','edgec','r','erase','none')
      line(xline,yline,'color','k','erase','none');
      
   end
   set(f,'HandleVisibility','callback');
   set(0, 'Units', oldRootUnits);
   
case 3
   if iscell(col) & length(col)~=length(x)
      error('There must be equally many colors as waitbars, or only one color.')
   end
   f=cwaitbar(x,name);
   a=get(f,'child');
   p=findobj(a,'type','patch');
   l=findobj(a,'type','line');
   if ~iscell(col)
      set(p,'facec',col,'edgec',col)
   else
      for i=1:length(col)
         set(p(i),'facec',col{i},'edgec',col{i})
      end
   end
   set(l,'xdata',xline')
end  % case
drawnow
figure(f)

if nargout==1,
  fout = f;
end


function [Factors]=ini(X,Fac,MthFl,IgnFl)
%INI initialization of loadings
%
% function [Factors]=ini(X,Fac,MthFl,IgnFl)
%
% This algorithm requires access to:
% 'gsm' 'fnipals' 'missmult'
%
% Copyright
% Claus A. Andersson 1995-1999
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, T254
% DK-1958 Frederiksberg
% Denmark
% Phone  +45 35283788
% Fax    +45 35283245
% E-mail claus@andersson.dk
%
% ---------------------------------------------------------
%                    Initialize Factors 
% ---------------------------------------------------------
%
% [Factors]=ini(X,Fac,MthFl,IgnFl);
% [Factors]=ini(X,Fac,MthFl);
%
% X        : The multi-way data.
% Fac      : Vector describing the number of factors
%            in each of the N modes.
% MthFl    : Method flag indicating what kind of
%            factors you want to initiate Factors with:
%            '1' : Random values, orthogonal
%            '2' : Normalized singular vectors, orthogonal
% IgnFl    : This feature is only valid with MthFl==2.
%            If specified, these mode(s) will be ignored,
%            e.g. IgnFl=[1 5] or IgnFl=[3] will
%            respectively not initialize modes one and 
%            five, and mode three.
% Factors  : Contains, no matter what method, orthonormal
%            factors. This is the best general approach to
%            avoid correlated, hence ill-posed, problems.
%
% Note that it IS possible to initialize the factors to have
% more columns than rows, since this may be required by some
% PARAFAC models. If this is required, the 'superfluos' 
% columns will be random and orthogonal columns.
% This algorithm automatically arranges the sequence of the
% initialization to minimize time and memory consumption.
% Note, if you get a warning from NIPALS about convergence has
% not been reached, you can simply ignore this. With regards 
% to initialization this is not important as long as the
% factors being returned are in the range of the eigensolutions.

% $ Version 1.02 $ Date 30 Aug 1999 $ Not compiled $
% $ Version 1.0201 $ Date 21 Jan 2000 $ Not compiled $ RB removed orth of additional columns
% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $

format long
format compact

DimX = size(X);
X = reshape(X,DimX(1),prod(DimX(2:end)));

% Assign intermediaries
Show=0;
rand('seed',sum(100*clock));
MissingExist=any(isnan(X(:)));

% Initialize system variables
N=size(Fac,2);
if N==1,
    Fac=Fac*ones(1,size(DimX,2));
end;
N=size(Fac,2);

FIdx0=zeros(1,N);
FIdx1=zeros(1,N);
latest=1;
for c=1:N,
    if Fac(c)==-1,
        FIdx0(c)=0;
    else
        FIdx0(c)=latest;
        latest=latest+Fac(c)*DimX(c);
        FIdx1(c)=latest-1;
    end;
end;

% Check inputs
if ~exist('IgnFl'),
    IgnFl=[0];
end;

%Random values
if MthFl==1,
    for c=1:N,
        A=orth(rand( DimX(c) , min([Fac(c) DimX(c)]) ));
        %B=[A orth(rand(DimX(c),Fac(c)-DimX(c)))]; 
        B=[A rand(DimX(c),Fac(c)-DimX(c))]; 
        Factors(FIdx0(c):FIdx1(c))=B(:)';
    end;
    if Show>=1,
        fprintf('ini.m : Initialized using random values.\n');
    end;
else
    %Singular vectors
    Factors=rand(1,sum(~(Fac==-1).*DimX.*Fac));
    if MthFl==2 | MthFl==3 
        [A Order]=sort(Fac);
        RedData=X;
        CurDimX=DimX;
        for k=1:N,
            c=Order(k);
            if Fac(c)>0,
                for c1=1:c-1;
                    newi=CurDimX(c1+1);
                    newj=prod(CurDimX)/CurDimX(c1+1);
                    RedData=reshape(RedData',newi,newj);
                end;
                Op=0;
                if MissingExist | (Op==0 & Fac(c)<=5 & (50<min(size(RedData)) & min(size(RedData))<=120)),
                    %Need to apply NIPALS
                    t0=clock;
                    A=reshape(Factors(FIdx0(c):FIdx1(c)),DimX(c),Fac(c));
                    if MissingExist
                        MissIdx=find(isnan(RedData));
                        [A,P]=fnipals(RedData,min([Fac(c) DimX(c)]),A);
                        Xm=A*P';
                        RedData(MissIdx)=Xm(MissIdx);
                        MissingExist=0;
                     else
                        [A]=fnipals(RedData,min([Fac(c) DimX(c)]),A);
                    end;
                    B=[A orth(rand(DimX(c),Fac(c)-DimX(c)))];
                    Factors(FIdx0(c):FIdx1(c))=B(:)';
                    t1=clock;
                    if Show>=2,
                        disp(['ini.m: NIPALS used ' num2str(etime(t1,t0)) ' secs. on mode ' int2str(c)]),
                    end;
                    Op=1;
                end;
                if Op==0 & (120<min(size(RedData)) & min(size(RedData))<Inf),
                    %Need to apply Gram-Schmidt
                    t0=clock;
                    C=RedData*RedData';
                    A=reshape(Factors(FIdx0(c):FIdx1(c)),DimX(c),Fac(c));
                    for i=1:3,
                        A=gsm(C*A);
                    end;
                    B=[A orth(rand(DimX(c),Fac(c)-DimX(c)))];
                    Factors(FIdx0(c):FIdx1(c))=B(:)';
                    t1=clock;
                    if Show>=2,
                        disp(['ini.m: GS used ' num2str(etime(t1,t0)) ' secs. on mode ' int2str(c)]),
                    end;
                    Op=1;
                end;
                if Op==0 & (0<min(size(RedData)) & min(size(RedData))<=200),
                    %Small enough to apply SVD
                    t0=clock;
                    if max(size(RedData))<1000
                       [U S A]=svd(RedData',0);
                    else
                       [U S A]=svds(RedData');
                    end
                    A=A(:,1:min(size(A,2),min([Fac(c) DimX(c)])));
                    if size(A,2)<Fac(c)
                      A = [A rand(size(A,1),Fac(c)-size(A,2))];
                    end
                    n_ = Fac(c)- min([Fac(c) DimX(c)]);
                    if n_>0,
                       a = rand(DimX(c),n_);
                       if DimX(c)>=n_
                          a = orth(a);
                       else
                          a = orth(a')';
                       end;
                       B=[A a];
                    else 
                       Factors(FIdx0(c):FIdx1(c))=A(:)';
                    end;
                    
                    t1=clock;
                    if Show>=2,
                        disp(['ini.m: SVD used ' num2str(etime(t1,t0)) ' secs. on mode ' int2str(c)]),
                    end;
                    Op=1;
                end;
                CurDimX(c)=min([Fac(c) DimX(c)]);
                if MissingExist,
                    RedData=missmult(A',RedData);
                else
                    RedData=A'*RedData;
                end;
                %Examine if re-ordering is necessary
                if c~=1,
                    for c1=c:N,
                        if c1~=N,
                            newi=CurDimX(c1+1);
                            newj=prod(CurDimX)/newi;
                        else
                           newi=CurDimX(1);
                            newj=prod(CurDimX)/newi;
                        end;
                        RedData=reshape(RedData',newi,newj);
                    end;
                end;
            end;
        end;
        if Show>=1,
            fprintf('ini.m : Initialized using SVD and projection.\n');
        end;
    end;
end,
format
% Convert to new format
clear ff,id1 = 0;
for i = 1:length(DimX) 
   id2 = sum(DimX(1:i).*Fac(1:i));
   ff{i} = reshape(Factors(id1+1:id2),DimX(i),Fac(i));id1 = id2;
end
Factors = ff;


function [E]=gsm(V);
%GSM orthogonalization
%
% [E]=GSM(V);
% GS   Gram-Schmidt Method for orthogonalisation
%      An orthonormal basis spanning the columns of V is returned in E.
% 
%      This algorithm does not use pivoting or any other
%      stabilization scheme. For a completely safe orthogonalization
%      you should use 'ORTH()' though is may take triple the time.
%      'GSM()' is optimized for speed and requies only minimum storage
%      during iterations. No check of rank is performed on V!
%
%      Claus Andersson, 1996, KVL

[m n]=size(V);

%Allocate space for the basis
E=zeros(m,n);

%The first basis vector is taken directly from V
s=sqrt(sum(V(:,1).^2));
E(:,1)=V(:,1)/s;

%Find the other basis vectors as orthogonals to
%the already determined basis by projection
for k=2:n,
  f=V(:,k)-E(:,1:(k-1))*(E(:,1:(k-1))'*V(:,k));
  s=sqrt(sum(f.^2));
  if s<eps,
    E(:,k)=0*f;   %set to zeros
  else
    E(:,k)=f/s;   %normalize
  end;
end;


function [T,P]=fnipals(X,w,T)

%FNIPALS nipals algorithm for PCA
% 
% function [T,P]=fnipals(X,w,T)
%
% 'fnipals.m'
%
% This algorithm requires the presence of:
% 'missmean.m' 
%
% Copyright
% Claus A. Andersson 1995-
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, T254
% DK-1958 Frederiksberg
% Denmark
% E-mail: claus@andersson.dk
%
% ----------------------------------------------------
%        Find eigenvectors according to NIPALS
% ----------------------------------------------------
%
% [T,P]=fnipals(X,w,T);
% [T,P]=fnipals(X,w);
%
% T is found so that X = T*P', s.t ||T||=1 and T'T=I
%
% X        : The matrix to be decomposed.
% w        : Number of factors to extract.
%            If w is high (perhaps>20) consider using SVD.
% T        : Initial guess of the solution, optional.
%            If T is not specified, a little time will
%            be used on finding orthogonal random 
%            starting values.
%
% You may want to calculate P afterwards by typing 'P=X*T'.
% Note that the T returned is orthonormal.
% Calculation of P is left of this implementation to save FLOP's.
% It handles missing values NaNs (very dispersed, less than 15%)
% If the problem is small enough you would prefer the SVD rather
% than NIPALS for finding T. NIPALS may be inaccurate when
% extracting too many factors, i.e., many more than the rank 
% of X. 

%scalar ConvLim WarnLim ItMax a b i

% $ Version 1.01 $ Date 18. June 1998 $ Not compiled $

ConvLim=1e-12;
WarnLim=1e-4;
ConvLimMiss=100*ConvLim;
ItMax=100;

filename='fnipals.m';

[a b]=size(X);

if (w>a | w>b) | w<1,
    help(filename);
    error(['Error in ' filename ': Number of factors to extract is invalid!'])
end;

np=isnan(X);
MissingExist=any(np);

if ~exist('T'),
    T=orth(randn(a,w));
end;

if exist('P'),
    P=[];
end;

if ~MissingExist
    if (size(T) == [a w]),
        if a>b,
            P=X'*T;
            l2=Inf;
            Z=X'*X;
            for i=1:w,
                p=P(:,i);
                d=1;
                it=0;
                while (d>ConvLim) & (it<ItMax),
                    it=it+1;
                    p=Z*p;
                    l1=sqrt(p'*p);
                    p=p/l1;
                    d=(l1-l2)^2;
                    l2=l1;
                end;
                P(:,i)=sqrt(l1)*p;
                Z=Z-P(:,i)*P(:,i)';
                WarnLim=sqrt(l1)/1000;
                if it>=ItMax & d>WarnLim,
                    disp('FNIPALS, High-X: Iterated up to the ItMax limit!')
                    disp('FNIPALS, High-X: The solution has not converged!')
                end;
            end;
            T=X*P;
        else
            P=[];
            l2=Inf;
            Z=X*X';
            for i=1:w,
                t=T(:,i); 
                d=1;
                it=0;
                while (d>ConvLim) & (it<ItMax),
                    it=it+1;
                    t=Z*t;
                    l1=sqrt(t'*t);
                    t=t/l1;
                    d=(l1-l2).^2;
                    l2=l1;
                end;
                T(:,i)=sqrt(l1)*t;
                Z=Z-T(:,i)*T(:,i)';
                WarnLim=sqrt(l1)/1000;
                if it>=ItMax & d>WarnLim,
                    disp('FNIPALS, Wide-X: Iterated up to the ItMax limit!')
                    disp('FNIPALS, Wide-X: The solution has not converged!')
                end;
            end;
        end;
        T=gsm(T);
    else
        error(['Error in ' filename ': Number of factors to extract is invalid!'])
    end;
else
    MissIdx=find(np);
    [i j]=find(np);
    mnx=missmean(X)/2;
    mny=missmean(X')/2;
    n=size(i,1);
    for k=1:n,
        i_i=i(k);
        j_j=j(k);
        X(i_i,j_j) = mny(i_i) + mnx(j_j);
    end;
    mnz=(missmean(mnx)+missmean(mny))/2;
    
    ssmisold=sum(sum( X(MissIdx).^2 ));
    sstotold=sum(sum( X.^2 ));
    ssrealold=sstotold-ssmisold;
    iterate=1;
    while iterate
        
        if (size(T) == [a w]),
            if a>b,
                P=X'*T;
                l2=Inf;
                Z=X'*X;
                for i=1:w,
                    p=P(:,i);
                    d=1;
                    it=0;
                    while (d>ConvLim) & (it<ItMax),
                        it=it+1;
                        p=Z*p;
                        l1=sqrt(p'*p);
                        p=p/l1;
                        d=(l1-l2)^2;
                        l2=l1;
                    end;
                    P(:,i)=sqrt(l1)*p;
                    Z=Z-P(:,i)*P(:,i)';
                    WarnLim=sqrt(l1)/1000;
                    if it>=ItMax & d>WarnLim,
                        disp('FNIPALS, High-X: Iterated up to the ItMax limit!')
                        disp('FNIPALS, High-X: The solution has not converged!')
                    end;
                end;
                T=X*P;
            else
                P=[];
                l2=Inf;
                Z=X*X';
                for i=1:w,
                    t=T(:,i); 
                    d=1;
                    it=0;
                    while (d>ConvLim) & (it<ItMax),
                        it=it+1;
                        t=Z*t;
                        l1=sqrt(t'*t);
                        t=t/l1;
                        d=(l1-l2).^2;
                        l2=l1;
                    end;
                    T(:,i)=sqrt(l1)*t;
                    Z=Z-T(:,i)*T(:,i)';
                    WarnLim=sqrt(l1)/1000;
                    if it>=ItMax & d>WarnLim,
                        disp('FNIPALS, Wide-X: Iterated up to the ItMax limit!')
                        disp('FNIPALS, Wide-X: The solution has not converged!')
                    end;
                end;
            end;
            T=gsm(T);
        else
            error(['Error in ' filename ': Number of factors to extract is invalid!'])
        end;
        
        P=X'*T;
        Xm=T*P';
        X(MissIdx)=Xm(MissIdx);
        ssmis=sum(sum( Xm(MissIdx).^2 ));
        sstot=sum(sum( X.^2 ));
        ssreal=sstot-ssmis;
        if abs(ssreal-ssrealold)<ConvLim*ssrealold & abs(ssmis-ssmisold)<ConvLimMiss*ssmisold,
            iterate=0;
        end;
        ssrealold=ssreal;
        ssmisold=ssmis;   
    end;
end;
T=gsm(T);


function mm=missmean(X)

%MISSMEAN mean of a matrix X with NaN's
%
%[mm]=missmean(X)
%
%This function calculates the mean of a matrix X.
%X may hold missing elements denoted by NaN's which
%are ignored (weighted to zero).
%
%Check that for no column of X, all values are missing

% Copyright
% Claus A. Andersson 1996-
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% E-mail: claus@andersson.dk


%Insert zeros for missing, correct afterwards
missidx = isnan(X);
i = find(missidx);
X(i) = 0;

%Find the number of real(non-missing objects)
if min(size(X))==1,
   n_real=length(X)-sum(missidx);
else
   n_real=size(X,1)-sum(missidx);
end

i=find(n_real==0);
if isempty(i) %All values are real and can be corrected
   mm=sum(X)./n_real;
else %There are columns with all missing, insert missing
   n_real(i)=1;
   mm=sum(X)./n_real;
   mm(i)=i + NaN;
end


function AB = kr(A,B);
%KR Khatri-Rao product
%
% The Khatri - Rao product
% For two matrices with similar column dimension the khatri-Rao product
% is kr(A,B) = [kron(B(:,1),A(:,1) .... kron(B(:,F),A(:,F)]
% 
% I/O AB = ppp(A,B);
%
% kr(A,B) equals ppp(B,A) - where ppp is the triple-P product = 
% the parallel proportional profiles product which was originally 
% suggested in Bro, Ph.D. thesism, 1998

% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% Phone  +45 35283296
% Fax    +45 35283245
% E-mail rb@kvl.dk
%
% $ Version 1.02 $ Date 28. July 1998 $ Not compiled $
% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $

[I,F]=size(A);
[J,F1]=size(B);

if F~=F1
   error(' Error in kr.m - The matrices must have the same number of columns')
end

AB=zeros(I*J,F);
for f=1:F
   ab=B(:,f)*A(:,f).';
   AB(:,f)=ab(:);
end


function load=pfls(ZtZ,ZtX,dimX,cons,OldLoad,DoWeight,W);

%PFLS
%
% See also:
% 'unimodal' 'monreg' 'fastnnls'
%
% 
% Calculate the least squares estimate of
% load in the model X=load*Z' => X' = Z*load'
% given ZtZ and ZtX
% cons defines if an unconstrained solution is estimated (0)
% or an orthogonal (1), a nonnegativity (2), or a unimodality (3)
%
%
% Used by PARAFAC.M

% $ Version 1.02 $ Date 28. July 1998 $ Not compiled $
%
% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% Phone  +45 35283296
% Fax    +45 35283245
% E-mail rb@kvl.dk
%

% Apr 2002 - Fixed error in weighted ls $ rb

if ~DoWeight

  if cons==0 % No constr
    %load=((Z'*Z)\Z'*Xinuse)';
    load=(pinv(ZtZ)*ZtX)';
  
  elseif cons==1 % Orthogonal loadings acc. to Harshman & Lundy 94
    load=ZtX'*(ZtX*ZtX')^(-.5);

  elseif cons==2 % Nonnegativity constraint
    load=zeros(size(OldLoad));
    for i=1:dimX
       load(i,:)=fastnnls(ZtZ,ZtX(:,i))';
%       if min(load(i,:))<-eps*1000
%          load(i,:)=OldLoad(i,:);
%       end
    end

  elseif cons==3 % Unimodality & NNLS
     load=OldLoad;
     F=size(OldLoad,2);
     if F>1
       for i=1:F
        ztz=ZtZ(i,i);
        ztX=ZtX(i,:)-ZtZ(i,[1:i-1 i+1:F])*load(:,[1:i-1 i+1:F])';
        beta=(pinv(ztz)*ztX)';
        load(:,i)=ulsr(beta,1);
       end
     else
       beta=(pinv(ZtZ)*ZtX)';
       load=ulsr(beta,1);
     end
  end

elseif DoWeight
  Z=ZtZ;
  X=ZtX;
  if cons==0 % No constr
    load=OldLoad;
    one=ones(1,size(Z,2));
    for i=1:dimX
      ZW=Z.*(W(i,:).^2'*one);
      %load(i,:)=(pinv(Z'*diag(W(i,:))*Z)*(Z'*diag(W(i,:))*X(i,:)'))';
      load(i,:)=(pinv(ZW'*Z)*(ZW'*X(i,:)'))';
    end

  elseif cons==2 % Nonnegativity constraint
    load=OldLoad;
    one=ones(1,size(Z,2));
    for i=1:dimX
      ZW=Z.*(W(i,:).^2'*one);
      load(i,:)=fastnnls(ZW'*Z,ZW'*X(i,:)')';
    end

  elseif cons==1
    disp(' Weighted orthogonality not implemented yet')
    disp(' Please contact the authors for further information')
    error

  elseif cons==3
    disp(' Weighted unimodality not implemented yet')
    disp(' Please contact the authors for further information')
    error

  end

end


% Check that NNLS and alike do not intermediately produce columns of only zeros
if cons==2|cons==3
  if any(sum(load)==0)  % If a column becomes only zeros the algorithm gets instable, hence the estimate is weighted with the prior estimate. This should circumvent numerical problems during the iterations
    load = .9*load+.1*OldLoad;
  end
end


function [Xm]=nmodel(Factors,G,Om);

%NMODEL make model of data from loadings
%
% function [Xm]=nmodel(Factors,G,Om);
%
% This algorithm requires access to:
% 'neye.m'
%
%
% [Xm]=nmodel(Factors,G,Om);
%
% Factors  : The factors in a cell array. Use any factors from 
%            any model. 
% G        : The core array. If 'G' is not defined it is assumed
%            that a PARAFAC model is being established.
%            Use G = [] in the PARAFAC case.
% Om       : Oblique mode.
%            'Om'=[] or 'Om'=0, means that orthogonal
%                   projections are requsted. (default)
%            'Om'=1 means that the factors are oblique.  
%            'Om'=2 means that the ortho/oblique is solved automatically.  
%                   This takes a little additional time.
% Xm       : The model of X.
%
% Using the factors as they are (and the core, if defined) the general N-way model
% is calculated. 


% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $
% $ Version 1.02 $ Date 17. Apr 1999 $ Not compiled $
%
%
% Copyright
% Claus A. Andersson 1995-1999
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, T254
% DK-1958 Frederiksberg
% Denmark
% E-mail claus@andersson.dk


for i = 1:length(Factors);
   DimX(i)=size(Factors{i},1);
end
i = find(DimX==0);
for j = 1:length(i)
   DimX(i(j)) = size(G,i(j));
end



if nargin<2, %Must be PARAFAC
   Fac=size(Factors{1},2);
   G=[];
else
   for f = 1:length(Factors)
      if isempty(Factors{f})
         Fac(f) = -1;
      else
         Fac(f) = size(Factors{f},2);
      end;
   end
end

if ~exist('Om')
    Om=[];
end;

if isempty(Om)
    Om=0;
end;

if size(Fac,2)==1,
    Fac=Fac(1)*ones(1,size(DimX,2));
end;
N=size(Fac,2);

if size(DimX,2)>size(Fac,2),
    Fac=Fac*ones(1,size(DimX,2));
end;  
N=size(Fac,2);

Fac_orig=Fac;
i=find(Fac==-1);
if ~isempty(i)
    Fac(i)=zeros(1,length(i));
    Fac_ones(i)=ones(1,length(i));
end;
DimG=Fac;
i=find(DimG==0);
DimG(i)=DimX(i);

if isempty(G),
   G=neye(DimG);
end;   
G = reshape(G,size(G,1),prod(size(G))/size(G,1));

% reshape factors to old format
ff = [];
for f=1:length(Factors)
 ff=[ff;Factors{f}(:)];
end
Factors = ff;


if DimG(1)~=size(G,1) | prod(DimG(2:N))~=size(G,2),

    help nmodel

    fprintf('nmodel.m   : ERROR IN INPUT ARGUMENTS.\n');
    fprintf('             Dimension mismatch between ''Fac'' and ''G''.\n\n');
    fprintf('Check this : The dimensions of ''G'' must correspond to the dimensions of ''Fac''.\n');
    fprintf('             If a PARAFAC model is established, use ''[]'' for G.\n\n');
    fprintf('             Try to reproduce the error and request help at rb@kvl.dk\n');
    return;
end;

if sum(DimX.*Fac) ~= length(Factors),
    help nmodel
    fprintf('nmodel.m   : ERROR IN INPUT ARGUMENTS.\n');
    fprintf('             Dimension mismatch between the number of elements in ''Factors'' and ''DimX'' and ''Fac''.\n\n');
    fprintf('Check this : The dimensions of ''Factors'' must correspond to the dimensions of ''DimX'' and ''Fac''.\n');
    fprintf('             You may be using results from different models, or\n');
    fprintf('             You may have changed one or more elements in ''Fac'' or ''DimX'' after ''Factors'' have been calculated.\n\n');
    fprintf('             Read the information above for information on arguments.\n');
    return;
end;

FIdx0=cumsum([1 DimX(1:N-1).*Fac(1:N-1)]);
FIdx1=cumsum([DimX.*Fac]);

if Om==0,
    Orthomode=1;
end;

if Om==1,
    Orthomode=0;
end;

if Om==2,
    Orthomode=1;
    for c=1:N,
        if Fac_orig(c)~=-1,
            A=reshape(Factors(FIdx0(c):FIdx1(c)),DimX(c),Fac(c));
            AA=A'*A;
            ssAA=sum(sum(AA.^2));
            ssdiagAA=sum(sum(diag(AA).^2));
            if abs(ssAA-ssdiagAA) > 100*eps;
                Orthomode=0;
            end;
        end;
    end;
end;

if Orthomode==0,
    Zmi=prod(abs(Fac_orig(2:N)));
    Zmj=prod(DimX(2:N));
    Zm=zeros(Zmi,Zmj);
    DimXprodc0 = 1;
    Facprodc0 = 1;
    Zm(1:Facprodc0,1:DimXprodc0)=ones(Facprodc0,DimXprodc0);
    for c=2:N,
        if Fac_orig(c)~=-1,
            A=reshape(Factors(FIdx0(c):FIdx1(c)),DimX(c),Fac(c));
            DimXprodc1 = DimXprodc0*DimX(c);
            Facprodc1 = Facprodc0*Fac(c);
            Zm(1:Facprodc1,1:DimXprodc1)=ckron(A',Zm(1:Facprodc0,1:DimXprodc0));
            DimXprodc0 = DimXprodc1;
            Facprodc0 = Facprodc1;
        end;
    end;
    if Fac_orig(1)~=-1,
        A=reshape(Factors(FIdx0(1):FIdx1(1)),DimX(1),Fac(1));
        Xm=A*G*Zm;
    else 
        Xm=G*Zm;
    end;
elseif Orthomode==1,
    CurDimX=DimG;
    Xm=G;
    newi=CurDimX(2);
    newj=prod(CurDimX)/CurDimX(2);
    Xm=reshape(Xm',newi,newj);
    for c=2:N,
        if Fac_orig(c)~=-1,
            A=reshape(Factors(FIdx0(c):FIdx1(c)),DimX(c),Fac(c));
            Xm=A*Xm;
            CurDimX(c)=DimX(c);
        else
            CurDimX(c)=DimX(c);
        end;
        if c~=N,
            newi=CurDimX(c+1);
            newj=prod(CurDimX)/CurDimX(c+1);
        else,
				newi=CurDimX(1);
            newj=prod(CurDimX)/CurDimX(1);
        end;
        Xm=reshape(Xm',newi,newj);
    end;
    if Fac_orig(1)~=-1,
        A=reshape(Factors(FIdx0(1):FIdx1(1)),DimX(1),Fac(1));
        Xm=A*Xm;
    end;
end;    

Xm = reshape(Xm,DimX);


function [MultPhi,Phis] = ncosine(factor1,factor2);

%NCOSINE multiple cosine/Tuckers congruence coefficient
%
% [MultPhi,Phis] = ncosine(factor1,factor2,DimX,Fac);
%
% ----------------------INPUT---------------------
%
% factor1   = cell array with loadings of one model
% factor2   = cell array with loadings of one (other) model
%     If factor1 and factor2 are identical then
%        the multiple cosine of a given solution is
%          estimated; otherwise the similarity of the
%          two different solutions is given
%
% ----------------------OUTPUT---------------------
%
% MultPhi   Is the multiple cosine of the model
% Phis      Is the cosine between components in
%          individual component matrices arranged
%          as [PhiA;PhiB ...]

% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $
% $ Version 1.02 $ Date 28. July 1998 $ Not compiled $
%
% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% Phone  +45 35283296
% Fax    +45 35283245
% E-mail rb@kvl.dk
%

% Convert to old format
Fac = size(factor1,2);
for i = 1:length(factor1)
   DimX(i) = size(factor1{i},1);
end

ff = [];
for f=1:length(factor1)
 ff=[ff;factor1{f}(:)];
end
factor1 = ff;

ff = [];
for f=1:length(factor2)
 ff=[ff;factor2{f}(:)];
end
factor2 = ff;


if length(factor1)~=length(factor2)
  error(' factor1 and factor2 must hold components of same sizes in NCOSINE.M')
end
ord=length(DimX);
l_idx=0;
Fac=length(factor1)/sum(DimX);
for o=1:ord
  l_idx=[l_idx sum(DimX(1:o))*Fac];
end
L1=reshape(factor1(1:DimX(1)*Fac),DimX(1),Fac);
L2=reshape(factor2(1:DimX(1)*Fac),DimX(1),Fac);
for f=1:Fac
  L1(:,f)=L1(:,f)/norm(L1(:,f));
  L2(:,f)=L2(:,f)/norm(L2(:,f));
end
%GT correction
Phis=L1'*L2;
%Previously: Phis=L2'*L2;
%End GT correction
MultPhi=Phis;

for i=2:ord
  L1=reshape(factor1(l_idx(i)+1:l_idx(i+1)),DimX(i),Fac);
  L2=reshape(factor2(l_idx(i)+1:l_idx(i+1)),DimX(i),Fac);
  for f=1:Fac
    L1(:,f)=L1(:,f)/norm(L1(:,f));
    L2(:,f)=L2(:,f)/norm(L2(:,f));
  end
  phi=(L1'*L2);
  MultPhi=MultPhi.*phi;
  Phis=[Phis;phi];
end


function [Consistency,G,stdG,Target]=corcond(X,Factors,Weights,Plot);

%CORCOND Core consistency for PARAFAC model
%
% See also:
% 'unimodal' 'monreg' 'fastnnls'
%
% CORe CONsistency DIAgnostics (corcondia)
% Performs corcondia of a PARAFAC model and returns the cocote plot
% as well as the degree of consistency (100 % is max).
%
% Consistency=corcond(X,Factors,Weights,Plot);
% 
% INPUT
% X        : Data array 
% Factors  : Factors given in standard format as a cell array
% Weights  : Optional weights (otherwise skip input or give an empty array [])
% Plot     = 0 or not given => no plots are produced
%          = 1              => normal corcondia plot
%          = 2              => corcondia plot with standard deviations 
%
% OUTPUT
% The core consistency given as the percentage of variation in a Tucker3 core
% array consistent with the theoretical superidentity array. Max value is 100%
% Consistencies well below 70-90% indicates that either too many components
% are used or the model is otherwise mis-specified.
%

%
% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% Phone  +45 35283296
% Fax    +45 35283245
% E-mail rb@kvl.dk
%

% $ Version 1.02 $ Date 28. July 1998 $ Not compiled $
% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $
% $ Version 2.01 $ Feb 2003 $ replaced regg with t3core when weights are used $ RB $ Not compiled $

DimX = size(X);
X = reshape(X,DimX(1),prod(DimX(2:end)));
Fac = size(Factors{1},2);

if nargin<4
  Plot=0;
end
if nargin<3
  Weights=0;
end

ord=length(DimX);
l_idx=0;
for i=1:ord
  l_idx=[l_idx sum(DimX(1:i))*Fac];
end


% Scale all loadings to same magnitude
magn=ones(Fac,1);
for i=1:ord
   L=Factors{i};
   for f=1:Fac
     magn(f)=magn(f)*norm(L(:,f));
     L(:,f)=L(:,f)/norm(L(:,f));
   end
   Factors{i}=L;
end
% Magn holds the singular value of each component. Scale each loading vector by 
% the cubic root (if three-way) so all loadings of a component have the same variance

magn = magn.^(1/ord);
for i=1:ord
   L=Factors{i};
   for f=1:Fac
     L(:,f)=L(:,f)*magn(f);
   end
   Factors{i}=L;
end


% Make diagonal array holding the magnitudes
Ident=nident(Fac,ord);
if Fac>1
   DimIdent=ones(1,ord)*Fac;
   Ident=nshape(reshape(Ident,DimIdent),ord);
end

% Make matrix of Kronecker product of all loadings expect the large; Z = kron(C,B ... )
  NewFac=[];
  NewFacNo=[];
  for i=ord:-1:1
    Z=Factors{i};
    % Check its of full rank or adjust core and use less columns
    rankZ=rank(Z);
    if rankZ<Fac
       %OLD out=Z(:,rankZ+1:Fac);Z=Z(:,1:rankZ);H=[[eye(rankZ)] pinv(Z)*out];Ident=H*Ident;
       [q,r]=qr(Z);
       Ident=r*Ident;
       Z=q;
       DimIdent(i)=size(r,1);
    end
    if i>1&Fac>1
      Ident=nshape(reshape(Ident,DimIdent([i:ord 1:i-1])),ord);
    end
    NewFac{i}=Z;
    NewFacNo=[rankZ NewFacNo];
  end
Factors=NewFac;
Fac=NewFacNo;
if nargin<3
  [G,stdG]=regg(reshape(X,DimX),Factors,Weights); %Doesn't work with weights
else
  G=T3core(reshape(X,DimX),Factors,Weights);
  stdG = G; % Arbitrary (not used)
end

DimG = size(G);
G = G(:);

 Ident=Ident(:);
 Target=Ident;
 [a,b]=sort(abs(Ident));
 b=flipud(b);
 Ident=Ident(b);
 GG=G(b);
 stdGG=stdG(b);
 bNonZero=find(Ident);
 bZero=find(~Ident);

 ssG=sum(G(:).^2);
 Consistency=100*(1-sum((Target-G).^2)/ssG);
 
 
 if Plot
    clf
    Ver=version;
    Ver=Ver(1);
    if Fac>1
       eval(['set(gcf,''Name'',''Diagonality test'');']);
       if Ver>4
          plot([Ident(bNonZero);Ident(bZero)],'y','LineWidth',3)
          hold on
          plot(GG(bNonZero),'ro','LineWidth',3)
          plot(length(bNonZero)+1:prod(Fac),GG(bZero),'gx','LineWidth',3)
          if Plot==2
            line([[1:length(G)];[1:length(G)]],[GG GG+stdGG]','LineWidth',1,'Color',[0 0 0])
            line([[1:length(G)];[1:length(G)]],[GG GG-stdGG]','LineWidth',1,'Color',[0 0 0])
          end
          hold off
          title(['Core consistency ',num2str(Consistency),'% (yellow target)'],'FontWeight','bold','FontSize',12)          
       else
          plot([Ident(bNonZero);Ident(bZero)],'y')
          hold on
          plot(GG(bNonZero),'ro')
          plot(length(bNonZero)+1:prod(Fac),GG(bZero),'gx')
          if Plot==2
            line([[1:length(G)];[1:length(G)]],[GG GG+stdGG]','LineWidth',1,'Color',[0 0 1])
            line([[1:length(G)];[1:length(G)]],[GG GG-stdGG]','LineWidth',1,'Color',[0 0 1])
          end
          hold off
          title(['Core consistency ',num2str(Consistency),'% (yellow target)'])
       end
       xlabel('Core elements (green should be zero/red non-zero)')
       ylabel('Core Size')
    else
       eval(['set(gcf,''Name'',''Diagonality test'');']);
       title(['Core consistency ',num2str(Consistency),'% (yellow target)'])
       xlabel('Core elements (green should be zero/red non-zero)')
       ylabel('Size')
       plot(GG(bNonZero),'ro')
       title(['Core consistency ',num2str(Consistency),'%'])
       xlabel('Core elements (red non-zero)')
       ylabel('Core Size')
    end
 end

G = reshape(G,DimG);

function [G,stdG]=regg(X,Factors,Weights);

%REGG Calculate Tucker core
%
% Calculate Tucker3 core

% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $

DimX = size(X);
X = reshape(X,DimX(1),prod(DimX(2:end)));
Fac = size(Factors{1},2);

ord=length(DimX);
if ord<3
   disp(' ')
   disp(' !!Corcondia only applicable for three- and higher-way arrays!!')
   return
end

if length(Fac)==1
   for i=1:length(Factors)
      Fac(i) = size(Factors{i},2);
   end
end
vecX=X(:); % Vectorize X

% Make sure Weights are defined (as ones if none given)
if nargin<3
   Weights=ones(size(X));
end
if length(Weights(:))~=length(X(:));
   Weights=ones(size(X));
end
Weights=Weights(:);

% Set weights of missing elements to zero
id=find(isnan(vecX));
Weights(id)=zeros(size(id));
vecX(id)=zeros(size(id));

% Create Kronecker product of all but the last mode loadings
L2 = Factors{end-1};
L1 = Factors{end-2};
Z = kron(L2,L1);
for o=ord-3:-1:1
   Z = kron(Z,Factors{o});
end


% Make last mode loadings, L
L=Factors{end};

% We want to fit the model ||vecX - Y*vecG||, where Y = kron(L,Z), but 
% we calculate Y'Y and Y'vecX by summing over k
J=prod(DimX(1:ord-1));
Ytx = 0;
YtY = 0;
for k=1:DimX(ord)
   W=Weights((k-1)*J+1:k*J);
   WW=(W.^2*ones(1,prod(Fac)));
   Yk  = kron(L(k,:),Z);
   Ytx = Ytx + Yk'*(W.*vecX((k-1)*J+1:k*J));
   YtY = YtY + (Yk.*WW)'*Yk;
end

G=pinv(YtY)*Ytx;

if nargout>1
   se = (sum(vecX.^2) + G'*YtY*G -G'*Ytx);
   mse = se/(length(vecX)-length(G));
   stdG=sqrt(diag(pinv(YtY))*mse);
end
G = reshape(G,Fac);


function pfplot(X,Factors,Weights,Option);
%PFPLOT plot parafac model
%
% See also:
% 'parafac'
%
%
% pfplot(X,Factors,Weights,Option);
% Different aspects for evaluation of the solution.
%
% Option # = 1
% 1	NOT ACCESIBLE
% 2	NOT ACCESIBLE
% 3	DIAGONALITY PLOT
% 4	PLOTS OF RESIDUAL VARIANCE
% 5	PLOTS OF LEVERAGE
% 6	RESIDUALS (STANDARD DEVIATION) VERSUS LEVERAGE
% 7	NORMAL PROBABILITY PLOT
% 8	LOADING PLOT
% 
% You HAVE to input all four inputs. If you have no weights, just input [].
% The last input must be an 8-vector with ones if you want the plot and
% zeros else. E.g.
%
% pfplot(X,factors,[],[0 0 1 0 0 0 0 1]);
%
% to have the diagonality and the loading plot
%

% $ Version 1.02 $ Date 28. July 1998 $ Not compiled $
% $ Version 1.03 $ Date 6. October 1999 $ Changed to handle missing values correctly$
% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $
%
% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Rasmus Bro
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% Phone  +45 35283296
% Fax    +45 35283245
% E-mail rb@kvl.dk
%

DimX = size(X);
X = reshape(X,DimX(1),prod(DimX(2:end)));

% Convert to old format
NewLoad = Factors;
ff = [];
for f=1:length(Factors)
 ff=[ff;Factors{f}(:)];
end
Factors = ff;


factors = Factors;
ord=length(DimX);
Fac=length(factors)/sum(DimX);
lidx(1,:)=[1 DimX(1)*Fac];
for i=2:ord
  lidx=[lidx;[lidx(i-1,2)+1 sum(DimX(1:i))*Fac]];
end
if Option(3)==1
 % ESTIMATE DIAGONALITY OF T3-CORE
 diagonality=corcond(reshape(X,DimX),NewLoad,Weights,1);
end
model=nmodel(NewLoad);
model = reshape(model,DimX(1),prod(DimX(2:end)));
if Option(4)==1
% PLOTS OF RESIDUAL VARIANCE
  figure,eval(['set(gcf,''Name'',''Residual variance'');']);
  aa=ceil(sqrt(ord));bb=ceil(ord/aa);
  for i=1:ord
    r=nshape(reshape(X-model,DimX),i)';
    varian=stdnan(r).^2;
    subplot(aa,bb,i)
      plot(varian)
      if DimX(i)<30
        hold on
        plot(varian,'r+')
      end
      eval(['xlabel(''Mode ', num2str(i),''');']);
      ylabel('Residual variance');
  end
end
if Option(5)==1
% PLOTS OF LEVERAGE
figure
eval(['set(gcf,''Name'',''Leverage'');']);
aa=ceil(sqrt(ord));
bb=ceil(ord/aa);
for i=1:ord
    A=reshape(factors(lidx(i,1):lidx(i,2)),DimX(i),Fac);
    lev=diag(A*pinv(A'*A)*A');
    subplot(aa,bb,i)
    if std(lev)>eps
        plot(lev+100*eps,'+')
        for j=1:DimX(i)
            text(j,lev(j),num2str(j))
        end
    else
        warning('Leverage is constant')
    end
    eval(['xlabel(''Mode ', num2str(i),''');']);
    ylabel('Leverage');
end
end
if Option(6)==1
% RESIDUALS (STANDARD DEVIATION) VERSUS LEVERAGE
figure
eval(['set(gcf,''Name'',''Residuals vs. Leverages'');']);
aa=ceil(sqrt(ord));
bb=ceil(ord/aa);
for i=1:ord
  subplot(aa,bb,i)
  A=reshape(factors(lidx(i,1):lidx(i,2)),DimX(i),Fac);
  lev=diag(A*pinv(A'*A)*A')'+100*eps;
  r=nshape(reshape(X-model,DimX),i)';
  stand=stdnan(r);
  if std(lev)>eps
      plot(lev,stand,'+')
      for j=1:DimX(i)
          text(lev(j),stand(j),num2str(j))
      end
      eval(['xlabel(''Leverage in mode ', num2str(i),''');']);
      ylabel('Standard deviation');
  else
      warning('Leverage is constant')
  end
end
end
% if Option(7)==1
%    % NORMAL PROBABILITY PLOT
%    if exist('normplot')
%       disp(' ')
%       disp(' Normal probability plots are time-consuming')
%       disp(' They are made in the statistics toolbox though, so we can''t change that!')
%       figure,
%       eval(['set(gcf,''Name'',''Normal probability of residuals'');']);
%       aa=ceil(sqrt(ord));
%       bb=ceil(ord/aa);
%       r=nshape(reshape(X-model,DimX),i)';
%       r=r(:);
%       normplot(r(find(~isnan(r))))
%    end
% end
if Option(8)==1
% LOADING PLOT
  if sum(Option)>1
    figure
  end
  eval(['set(gcf,''Name'',''Loadings'');']);
  aa=ceil(sqrt(ord));
  bb=ceil(ord/aa);
  for i=1:ord
    subplot(aa,bb,i)
    A=reshape(factors(lidx(i,1):lidx(i,2)),DimX(i),Fac);
    plot(A)
    eval(['xlabel(''Mode ', num2str(i),''');']);
    ylabel('Loading');
  end
end
drawnow


function G=neye(Fac);
% NEYE  Produces a super-diagonal array
%
%function G=neye(Fac);
%
% $ Version 2.00 $ May 2001 $ Changed to array notation $ RB $ Not compiled $
% $ Version 1.00 $ Date 5. Aug. 1998 $ Not compiled $
%
% This algorithm requires access to:
% 'getindxn'
%
% See also:
% 'parafac' 'maxvar3' 'maxdia3'
%
% ---------------------------------------------------------
%             Produces a super-diagonal array
% ---------------------------------------------------------
%	
% G=neye(Fac);
%
% Fac      : A row-vector describing the number of factors
%            in each of the N modes. Fac must be a 1-by-N vector. 
%            Ex. [3 3 3] or [2 2 2 2]



% Copyright, 1998 - 
% This M-file and the code in it belongs to the holder of the
% copyrights and is made public under the following constraints:
% It must not be changed or modified and code cannot be added.
% The file must be regarded as read-only. Furthermore, the
% code can not be made part of anything but the 'N-way Toolbox'.
% In case of doubt, contact the holder of the copyrights.
%
% Claus A. Andersson
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, DK-1958 Frederiksberg, Denmark
% E-mail claus@andersson.dk

N=size(Fac,2);
if N==1,
   fprintf('Specify ''Fac'' as e vector to define the order of the core, e.g.,.\n')
   fprintf('G=eyecore([2 2 2 2])\n')
end;

G=zeros(Fac(1),prod(Fac(2:N)));

for i=1:Fac(1),
   [gi,gj]=getindxn(Fac,ones(1,N)*i);
   G(gi,gj)=1;
end;

G = reshape(G,Fac);


function [i,j]=getindxn(R,Idx);
%GETINDXN
%
%[i,j]=GetIndxn(R,Idx)
%
% Copyright
% Claus A. Andersson 1995-
% Chemometrics Group, Food Technology
% Department of Food and Dairy Science
% Royal Veterinary and Agricultutal University
% Rolighedsvej 30, T254
% DK-1958 Frederiksberg
% Denmark
% E-mail: claus@andersson.dk

l=size(Idx,2);

i=Idx(1);
j=Idx(2);

if l==3,
  j = j + R(2)*(Idx(3)-1);
 else
  for q = 3:l,
    j = j + prod(R(2:(q-1)))*(Idx(q)-1);
  end;
end;
