function result = eemizer(X,Factors,Rayl);

%EEMIZER (Ver. 2.0) for finding PARAFAC models of fluorescence EEM data
%
% result=eemizer(Xs,Factors,Rayl);
%
% Factors is the set of number of factors to look at. E.g. Factors = [2:4]
% Rayl is the widths investigated for Rayleigh. E.g. Rayl = [6 9 12 15];
% For e.g. a width of 9, emission 9 nm above and below excitation is set to
% missing.
%
% The data must be in a dataset object with samples in first mode, emission
% in the second mode and excitation in the third mode. Axis scales (in nm)
% must be given for mode 2 and 3. There must be more than 7 samples in the
% dataset.
%
% result=eemizer(X,Factors,Rayl);
% result=eemizer(X,[1:4],[5:5:20]);
%
% When done with the calculations, the result will give the best model for
% each number of components. Use the bar plot to view the models and save
% them to workspace.
%
% You can look into other models as well. The output result{f}.allmeasures
% gives [FIT COREC SPLTH EEMqual] for all models with three components
% (f is the number of components). To look at this, do
% plotgui(result{f}.allmeasures) where f is the number of components. If
% you are interested in a particular model, you can generate the data and
% look at this model by noting the sample number of the model in the plot
% produced by plotgui.
%
% [model,data]=eemizerview(X,result,factor,number);
%
% where X is the original data, result is the outcome of eemizer, factor is
% the number of factors and number is the sample number in the table
% result{factor}.allmeasures.
%
% If you stop the algorithm before it terminates, you can rerun from where
% you stopped by running
%
% result=eemizer('eemizertemp');
%
% You need to be able to load the temporary file eemizertemp.mat for this
% to work.
%
% I/O
% result=eemizer(X,Factors,Rayl);   % How to run it
% eemizer(result);                  % Reproduces the overall plot
% result=eemizer('eemizertemp');    % Resume from aborted run
% eemizerview(X,result,fact,numb);  % Plot another than the best model

if ~iscell(X) % Then it is default use
    if isstr(X) % Then its a filename containing a mat file to re-start from
        [out,models]=eemoptimize(X);
        load(X, 'Factors');
        load(X, 'Rayl');
        load(X, 'X');
    else
        [out,models]=eemoptimize(X,Factors,Rayl);
    end
    
    for f=Factors;
        % Find the best models for this number of components
        reslt=[];
        for Ex = 1:size(models(:,:,f),2);
            for Ra=1:length(Rayl);
                objtive=models{Ra,Ex,f}.R;
                d=repmat([Ex Ra],size(objtive,1),1);
                reslt=[reslt;[d objtive]];
            end
        end
        [a,b]=sort(reslt(:,end));
        b = flipud(b);
        reslt=reslt(b,:);
        % BEST MODEL WITH f COMPONENTS
        
        Ex = reslt(1,1);
        Ra = reslt(1,2);
        j=find(models{Ra,Ex,f}.R(:,end)==reslt(1,end));
        result{f}.bestmodel.model=models{Ra,Ex,f}.model{j};
        result{f}.bestmodel.LOWEXREMOVE=Ex;
        result{f}.bestmodel.SCATWIDTH=Rayl(Ra);
        s=models{Ra,Ex,f}.SMPS{j};
        result{f}.bestmodel.samplesused=s;
        r=Rayl(Ra);
        prestr = ['Xnew = flucut(X(:,:,',num2str(Ex),':end),',num2str(r),',',num2str(r),',struct(''LowZero'',',num2str(30),'));'];
        result{f}.bestmodel.dataprocessing=prestr;
        %result{f}.bestmodel.EEMqual=models{Ra,Ex,f}.R(j,7);
        %result{f}.bestmodel.FIT_SPLTH_COREC=models{Ra,Ex,f}.R(j,4:6);
        
        % OTHER CANDIDATES
        for jj=1:size(reslt,1)
            Ex = reslt(jj,1);
            Ra = reslt(jj,2);
            j=find(models{Ra,Ex,f}.R(:,end)==reslt(jj,end));
            result{f}.othermodels{jj}.model=models{Ra,Ex,f}.model{j};
            result{f}.othermodels{jj}.LOWEXREMOVE=Ex;
            result{f}.othermodels{jj}.SCATWIDTH=Rayl(Ra);
            s=models{Ra,Ex,f}.SMPS{j};
            result{f}.othermodels{jj}.samplesused=s;
            r=Rayl(Ra);
            prestr = ['Xnew = flucut(X(:,:,',num2str(Ex),':end),',num2str(r),',',num2str(r),',struct(''LowZero'',',num2str(30),'));'];
            result{f}.othermodels{jj}.dataprocessing=prestr;
            %result{f}.othermodels{jj}.EEMqual=models{Ra,Ex,f}.R(j,7);
            %result{f}.othermodels{jj}.FIT_COREC_SPLTH=models{Ra,Ex,f}.R(j,4:6);
        end
        allmeasures=dataset(reslt(:,end-3:end));
        allmeasures.label{2}={'Fit';'Splithalf quality';'Core consistency';'EEMQual - combined quality'};
        result{f}.allmeasures=allmeasures;
    end
    result{end}.X=X;
    result{end}.Factors = Factors;
    result{end}.Rayl = Rayl;
else % Replace best with newly chosen
    result=X;
    X = result{end}.X;
    Factors = result{end}.Factors;
    Rayl = result{end}.Rayl;
end

clf,hold on
for f=Factors
    %h=bar(f,result{f}.bestmodel.EEMqual);
    h=bar(f,result{f}.allmeasures.data(1,4));
    
    % Define loads
    tp.m   = result{f}.bestmodel.model; % Number of modes
    this=result{f}.bestmodel;
    smps = this.samplesused;
    lwex = this.LOWEXREMOVE;
    w = this.SCATWIDTH;
    data = flucut(X(smps,:,lwex:end),w,w,struct('LowZero',30));
    tp.x = data;
    tp.f=f;
    set(h,'userdata',tp);
    hcmenu = uicontextmenu;
    % Define callbacks for context menu
    hcb1 = ['tp=get(gco,''userdata'');Xeem=tp.x;Meem=tp.m;'];
    hcb2 = ['tp=get(gco,''userdata'');modelviewer(tp.m,tp.x)'];
    % Define the context menu items and install their callbacks
    item1 = uimenu(hcmenu, 'Label', 'Save model and data to workspace (Xeem & Meem)', 'Callback', hcb1);
    item2 = uimenu(hcmenu, 'Label', 'Plot this model', 'Callback', hcb2);
    set(h,'uicontextmenu',hcmenu)
    hold on
end
axis([min(Factors)-.5 max(Factors)+.5 0 1.05])
xlabel('Number of factors (right-click bar for options)','FontWeight','bold')
ylabel('EEMqual','FontWeight','bold')
hold off

function [re,result]=eemoptimize(X,Factors,Rayl);

% EEMOPTIMIZE
% [quality,results]=eemoptimize(X,Factors,Rayl);
% Factors is the set of number of factors to look at. E.g. Factors = [2:4]
% Rayl is the widths investigated for Rayliegh. E.g. Rayl = [6 9 12 15];

if isstr(X) % Then load the mat file and go from there
    load(X);
else
    Ex = X.axisscale{3};
    if min(Ex)>260
        exax = 1; % Don't reduce the axis
    else
        exax=find(Ex<240);
        if isempty(exax)
            exax = 1;
        end
    end
end

TORUN = NaN([length(Factors)*length(exax)*length(Rayl) 3]);
c=0; % Make a list of all metaparameters settings to run
for fct=Factors
    for ex = exax
        for ray=Rayl
            c=c+1;
            TORUN(c,:)=[fct ex ray];
        end
    end
end

try
    matlabpool
end
tempresult={};

opflu = flucut('options');
opflu.LowZero = 30;
opflu.plots = 'off';
I = size(TORUN,1);
parfor torun=1:I
    fct=TORUN(torun,1);
    ex=TORUN(torun,2);
    ray=TORUN(torun,3);
    [~,c2]=find(exax==ex);
    [~,c]=find(Rayl==ray);
    disp(['Working on factor ',num2str(fct),'; excitation ',num2str(c2),' of ',num2str(length(exax)),'; scatter ',num2str(c),' of ',num2str(length(c))]);
    Xnew = flucut(X(:,:,ex:end),ray,ray,opflu);
    tempresult{torun}=eemoutlier(Xnew,fct);
end

for torun=1:size(TORUN,1)
    fct=TORUN(torun,1);
    ex=TORUN(torun,2);
    ray=TORUN(torun,3);
    [~,c2]=find(exax==ex);
    [~,c]=find(Rayl==ray);
    result{c,c2,fct}=tempresult{torun};
    re(c,c2,fct)=max(result{c,c2,fct}.R(:,end));
end

function varargout = flucut(X,LowMiss,TopMiss,options)
%FLUCUT Remove scatter from fluorescence EEM data.
% from PLS_Toolbox 7.0.3

if nargin<1;
  X = 'io';
end

if ischar(X) 
  options = [];
  options.MakeWts = 'off';
  options.TopZero = 'off';
  options.LowZero = 'off';
  options.Blank = NaN;
  options.plots   = 'on';
  options.RamanCorrect = 'off';
  options.RamanWidth  = 10;
  options.RamanShift  = 3382;
  options.RemoveMissing = 'off';
  options.definitions = @optiondefs;
   if nargout==0; 
       evriio(mfilename,X,options); 
   else; 
       varargout{1} = evriio(mfilename,X,options); 
   end
  return;
end

if nargin<4
  options = [];
end

if nargin<2
  error('At least two inputs required.')
end
if nargin<3 | isempty(TopMiss)
  TopMiss = NaN;
end
if max(size(LowMiss))==1
  LowMiss=[LowMiss LowMiss];
end
if max(size(TopMiss))==1
  TopMiss=[TopMiss TopMiss];
end

try
  if ~isa(X,'dataset')
    error('Input (X) must be a dataset (see help).')
  end
catch
  error('Input (X) must be a dataset (see help for how to make a DataSet Object).')
end

xsz = size(X);
if length(xsz)<3
  error('Input (X) must be 3-dimensional EEM data organized as [samples x emission x excitation]')
end
if isempty(X.axisscale{2})
  EmAx = 1:xsz(2);
else
  EmAx = X.axisscale{2};
end
if isempty(X.axisscale{3})
  ExAx = 1:xsz(3);
else
  ExAx = X.axisscale{3};
end

% DO PLOTTING OF OLD RESULTS
if iscell(LowMiss) % Either filter values hence plotting
  clf
  Samplenumber = TopMiss(1);
  Rayl = LowMiss;
  
  %%%%%%% PLOTTING
  x = squeeze(X.data(Samplenumber,:,:));
  
  % Plot EEM
  p=surf(ExAx,EmAx,x);
  axis tight
  set(p,'facecolor',[.5 .5 .5],'edgecolor','none')
  camlight, lighting gouraud
  hold on
  set(gcf,'userdata',{X,Rayl,Samplenumber});
  
  % Make sample selector
  uicontrol('Style', 'popup',...
    'String', num2str([1:xsz(1)]'),...
    'Units','Normalized', ...
    'Position',[.01 .9 .2 .1], ...
    'Value',Samplenumber, ...
    'Callback', sprintf('s=get(gcf,''userdata'');%s(s{1},s{2},get(gco,''value''));',mfilename));
  
  return;
end

% DO EXCISING
mask = ones(xsz(2),xsz(3));
for j=1:length(ExAx)
  % First order missing
  if ~isnan(LowMiss(1))
    if length(LowMiss)==2
      k = (EmAx<=(ExAx(j)+LowMiss(1))&EmAx>=(ExAx(j)-LowMiss(2)));
    else
      k = (EmAx<=(ExAx(j)+LowMiss(1)));
    end
    mask(k,j)=NaN;
  end
  
  % Low zeros
  if strcmpi(options.LowZero,'on')
    k = (EmAx<(ExAx(j)-LowMiss(2)));
    mask(k,j)=0;
  end
  
  % High missing
  if ~isnan(TopMiss(1))
    if length(TopMiss)==2
      k  = (EmAx>=(2*ExAx(j)-TopMiss(1))&EmAx<=(2*ExAx(j)+TopMiss(2)));
    else
      k  = (EmAx>=(2*ExAx(j)-TopMiss(1)));
    end
    mask(k,j)=NaN;
  end
  
  % High zeros
%   if strcmpi(options.TopZero,'on')
%     k = (EmAx>(2*ExAx(j)+TopMiss(1)));
%     mask(k,j)=0;
%   end
  
end

%Raman removal
% if strcmpi(options.RamanCorrect,'on');
%   Emission = zeros(1,xsz(3));
%   for k = 1:xsz(3) % EVERY EXCITATION
%     Emission(k)=1e7/( (1e7/ExAx(k))-options.RamanShift);
%   end
%   for k=1:length(Emission);
%     kk = ((EmAx>Emission(k)-options.RamanWidth)& (EmAx<Emission(k)+options.RamanWidth));
%     mask(kk,k)=NaN;
%   end
% end

%apply mask to entire dataset (at once!)
masked = X.data;
mask = shiftdim(mask,-1);
for j=1:size(masked,1);
  masked(j,:,:) = masked(j,:,:).*mask;
  k = mask==0;
  maskedj = squeeze(masked(j,:));
  maskedj(k) = 0;
  masked(j,:) = maskedj;
end
X.data = masked;

%blank subtraction
if isfinite(options.Blank) ...
    & round(options.Blank)==options.Blank ...
    & options.Blank>=1 & options.Blank<=xsz(1);
  %if a valid integer
  blank = X.data(options.Blank,:,:);
  blanked = X.data;
  for i=1:xsz(1);
    blanked(i,:,:) = blanked(i,:,:) - blank;
  end
  X.data = blanked;
  % Soft delete blank
  X.include{1} = setdiff(X.include{1},options.Blank);
end

%%%%%%%%%%%%% MAKE WEIGHTS
if strcmp(options.MakeWts,'on')
  w = gauss(X,EmAx,ExAx,[max(LowMiss) max(LowMiss)/1.5 max(TopMiss)],350,options.RamanShift);
  w = w/max(w(:));
  w = abs(1-w)+.05;
  X.userdata.weights = w;
end

if strcmp(options.RemoveMissing,'on')
  % Remove areas with only missing
  n = squeeze(isnan(X.data(1,:,:)));
  i = (sum(n)<xsz(2));
  j = (sum(n,2)<xsz(3));
  if any(~i) | any(~j)
    X = X(:,j,i);
  end
end

if strcmp(options.plots,'on')
  figure;
  feval(mfilename,X,{},1);
end

varargout{1} = X;

function model = gauss(X,EmAx,ExAx,Width,Height,wavenumber_forraman)
% Calculates a gaussian profile as
% area * exp( -(emaxis-(Exc+DelPos)).^2/(2*Width^2)   );
%
% For gauss-fitting
% Assumes that emission is measured appr. every nm
% Width(1) = width of first Rayl. (2) of Raman, (3) of second Ray

[I J K] = size(X);
model = zeros(size(X));

for k = 1:K
  if ~isnan(Width(1))
    gauss_est = Height* exp( -(EmAx-(ExAx(k))).^2/(2*Width(1)^2)   );
    for i = 1:I
      model(i,:,k) = gauss_est;
    end
  end
  if ~isnan(Width(3))
    gauss_est = Height* exp( -(EmAx-(2*ExAx(k))).^2/(2*Width(3)^2)   );
    for i = 1:I
      model(i,:,k) = model(i,:,k)+gauss_est;
    end
  end
  
  if ~isnan(Width(2))
    Ex=1e7/( (1e7/ExAx(k))-wavenumber_forraman);
    gauss_est = Height* exp( -(EmAx-(Ex)).^2/(2*Width(2)^2)   );
    for i = 1:I
      model(i,:,k) = model(i,:,k)+gauss_est;
    end
  end
end
function m = nanmax(x)
m = max(x(~isnan(x)));
function m = nanmin(x)
m = min(x(~isnan(x)));
function out = optiondefs()
defs = {
  %Name                         Tab             Datatype        Valid                         Userlevel       %Description
  'MakeWts'                     'Display'       'select'        {'on' 'off'}                  'novice'        'If ''on'', weights will be given that can be used to downweight areas of scatter e.g., in PARAFAC models. The weights will be held in the output dataset field ''userdata''. {default = ''off'', do not create weights}'
  'plots'                       'Display'       'select'        {'final' 'all' 'off'}         'novice'        'Turn plotting of final model on or off. By choosing ''all'' you can choose to see the loadings as the iterations proceed. The final plot can also be produced using the function MODELVIEWER after the model has been fitted.';
  };

out = makesubops(defs);
function result = eemoutlier(X,F);

%EEMOUTLIER automatically remove outliers
%
% Provides an automated outlier selection procedure for a given dataset and
% given number of components. Samples with a high leverage or high
% sum-squared residual are removed one by one until no samples are assessed
% as outliers. The settings for making decisions are given in the top of
% the m-file. Outliers can only be removed until there are 8 samples left.
% Then the algorithm will stop.
%
% The model fit is PARAFAC with nonnegativity.
%
% The output is a file which contains the list of models. The first is the
% initial model and the last model is the final model (not necessarilly the
% right model, but the one where no more samples were removed)
%
% The output has the following fields
%   result.model % Cell of models
%   result.SMPS  % cell of sample sets
%   result.R     % Table of results
% 
%   This part is not currently implemented (rather - has been removed)
%   result.data  % Data used (the data are sample-wise normalized hence not
%                            identical to the raw data)
%
% R contains one row for each model with the following elements
% [number_samples ssqmax tsqsmax variance splthalf_quality coreconsist overall]
%
% ssqmax and tsqsmax are the values for residuals and leverage for the
% worst sample. variance, splthalf_quality and coreconsist are 100 for a
% perfect model. The last column 'overall' is the product of variance,
% splthalf_quality and coreconsist (as fractions). Hence if overall is 1,
% then all three are perfect
%
% I/O result = eemoutlier(X,factors);
%
% After analysis it is a good idea to look at the table results.R. If one
% row has a very high (close to one) element in the last column, then this
% model is probably good. If none have that, then maybe it is a good idea
% to look for a model with high variance which has either high core
% consistency or high splithalf quality
%
% If e.g. model number five looks good, you can plot it as
% modelviewer(result.model{5},result.data(result.SMPS{5},:,:))

samples = size(X,1); % Number of samples

% CRITERIA
critlevel = 5; % if any sample has a leverage or Q more than critlevel higher than the median, the sample is removed (one sample is removed at a time)
samplenumberfactor = 1;
if samples/20<1
    samplenumberfactor = 30/samples;
end; %If less than 20 samples increase the critical level of leverage and Q by this factor in order not to remove too many samples on small datasets
samplefraction = .10; % If more than samplefraction of the samples are removed, increase the critical level


op = parafac('options');
for i=1:3,op.constraints{i}.type = 'nonnegativity';end
op.stopcriteria.relativechange=1e-8;
op.stopcriteria.absolutechange=1e-8;

opt = nvalidate('options');
opt.plots = 'off';
opt.numbrepeats=1;
opt.modeloptions=op;

smps = [1:samples];
% normalize to handle weird samples
% This is now avoided
if 7==4
for i=1:size(X,1)
    xn = X.data(i,:,:);
    xn = xn(:);
    xn = xn(find(~isnan(xn)));
    X.data(i,:,:) = X.data(i,:,:)/sqrt(sum(xn.^2));
end
end
continu = 1;
cou = 0;
R = [];
clear model SMPS
while continu
    cou = cou+1;
    disp([' Calculating model number ',num2str(cou),' ...'])
    res = nvalidate(X(smps,:,:),F,'parafac',opt);
    corecon = res.models{F}.detail.coreconsistency.consistency;
    corecon(corecon<0)=0;
    var = res.models{F}.detail.ssq.perc;
    tsqsmax = max(res.models{F}.tsqs{1})/median(res.models{F}.tsqs{1});
    ssqmax = max(res.models{F}.ssqresiduals{1})/median(res.models{F}.ssqresiduals{1});
    numsam = length(res.models{F}.detail.includ{1});
    model{cou}=res.models{F};
    SMPS{cou}=smps;
    splthlf = res.splithalf.quality(F);
    
    % Adjust so we don't have zeros (will screw up sorting)
    splthlf(splthlf<0.01)=.01;
    var(var<1)=1;
    corecon(corecon<1)=1;
    
    R = [R;[numsam ssqmax tsqsmax var splthlf*100 corecon var*splthlf*corecon/10000]];
    if length(smps)<9
        fracfractor = 1e100;
    elseif (1-(length(smps)/samples))>samplefraction
        fracfactor = ((1-(length(smps)/samples))/samplefraction).^2;
    else
        fracfactor = 1;
    end
    
    if max(tsqsmax,ssqmax)>critlevel*samplenumberfactor*fracfactor
        continu = 1;
        if tsqsmax>ssqmax
            [a,b]=max(res.models{F}.tsqs{1});
        else
            [a,b]=max(res.models{F}.ssqresiduals{1});
        end
        smps(b) = [];
    else
        continu = 0;
    end
end
result.model = model;
result.SMPS = SMPS;
result.R = R;
result.data = X;
function result = nvalidate(x,ncomp,method,options)

%NVALIDATE the number PARAFAC or Tucker components
%  Used for testing the appropriate number of components in e.g.
%  a PARAFAC model. Input the data, x, and the number of components,
%  ncomp. E.g. set ncomp to [1:4] to test one- to four-component
%  models. For Tucker, e.g. choose [2 2 2;4 5 4] to check from 2 to
%  4/5 components depending on mode.
%  The third input is either 'parafac' or 'tucker'
%  An additional input allows to select additional options.
%
% Outputs are given in a struct that contains the fields
% ssq:           the sum-squared error
%     What-To-Look-For:
%        look for sudden changes as in a Scree-plot (often difficult)
%        and look for sudden increase in number of local minima (replicate
%        points for a number of components are not identical). This is
%        often a good indication that noise is being modeled.
%
% FOR PARAFAC, special plots are given:
% consistency:   The core consistency (corcondia)
%     What-To-Look-For:
%        CORCONDIA is a percentage below or equal to 100%. A value of 80-100%
%        means that the model is valid, while a value below, say 40% means that
%        the model is not valid. A value between 40 and 80% means that the model
%        is probably valid but somehow difficult to estimate, e.g., due to
%        slight misspecification or correlations. The Corcondia will mostly
%        decrease with number of components but very sharply where the correct
%        number of components are exceeded. Hence, the appropriate number of
%        components is the model with the highest number of components and a
%        valid CORCONDIA.
%
% iterations:    Number of iterations
%     What-To-Look-For:
%        A sudden increase in the number of iterations needed
%        suggests that too many components may be used (but could also
%        just be due to a difficult fitting problem).
%
% splithalf:     Stability of loadings
%     What-To-Look-For:
%        A high value (close to 100%) means that when the data is split in
%        two, both sets will yield the same parameters. May give
%        meaningless results for few samples or if the underlying variables
%        are not equally distributed over samples.
%
% models:        Cell of models
%        All models are saved in a cell where element {f,r} is the r'th
%        model replicate fit with f components.
%
%I/O: result = nvalidate(x,ncomp,method,options); % validate model of x
%I/O:          nvalidate(result,x);               % Plot earlier results
%I/O:          nvalidate(x,3,'parafac');          % validate 1-3 comp parafac
%I/O:          nvalidate(x,[1 1 1;3 3 3],'tucker');% validate [1 1 1] to [3 3 3] comp Tucker3 models
%I/O:          nvalidate demo
%
%See also: CORCONDIA, MODELVIEWER, PARAFAC, PARAFAC2, TUCKER


if nargin == 0; x = 'io'; end

if nargin>2
    method = lower(method);
    % Generate standard options
    if strcmp(method,'parafac')
        standardoptions = struct('name','options','display','on','plots','on','waitbar','off',...
            'numbrepeats',4,'modeloptions',parafac('options'));
    elseif strcmp(method,'tucker')
        standardoptions = struct('name','options','display','on','plots','on','waitbar','off',...
            'numbrepeats',1,'modeloptions',tucker('options'));
    else
        error(' Third input not correct. Must be either ''parafac'' or ''tucker''')
    end
else % Assume parafac
    standardoptions = struct('name','options','display','on','plots','on','waitbar','on',...
        'numbrepeats',4,'modeloptions',parafac('options'));
end
if ischar(x)
    options=standardoptions;
    if nargout==0;
        clear varargout;
        evriio(mfilename,x,options);
    else
        result = evriio(mfilename,x,options);
    end
    return
end


if ~isstruct(x) % Otherwise earlier results are given which should just be shown
    
    % Filter standard options for possible user-defined modifications
    try
        if ~isstruct(options.modeloptions)
            options = evriio('nvalidate','options',standardoptions);
        end
    catch
        options = evriio('nvalidate','options',standardoptions);
    end
    
    ssX   = zeros(ncomp(end),options.numbrepeats);
    Corco = zeros(ncomp(end),options.numbrepeats);
    It    = zeros(ncomp(end),options.numbrepeats);
    T3table = [];
    
    
    if strcmp(method,'parafac')
        modeloptions = evriio('parafac','options',options.modeloptions);
    else strcmp(method,'tucker')
        modeloptions = evriio('tucker','options',options.modeloptions);
    end
    modeloptions.display='off';
    modeloptions.plots='off';
    modeloptions.waitbar='off';
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    %%%%%% FIT PARAFAC %%%%%%%%%%%%%
    if strcmp(method,'parafac')
        FactorsOut=[];
        ssX=repmat(NaN,ncomp(end),options.numbrepeats);
        Corco=repmat(NaN,ncomp(end),options.numbrepeats);
        perc=repmat(NaN,ncomp(end),options.numbrepeats);
        It = repmat(NaN,ncomp(end),options.numbrepeats);
        for f=ncomp
            
            modeloptions.init = 1; % DTLD init
            model{f,1}=parafac(x,f,modeloptions);
            ssX(f,1)=model{f,1}.detail.ssq.residual;
            BestErr=ssX(f,1);
            Bestnumb = 1;
            Corco(f,1)=model{f,1}.detail.coreconsistency.consistency;
            It(f,1)=model{f,1}.detail.critfinal(3);
            perc(f,1)=model{f,1}.detail.ssq.perc;
            
            if options.numbrepeats>1,
                modeloptions.init = 4; % SVD init
                model{f,2}=parafac(x,f,modeloptions);
                ssX(f,2)=model{f,2}.detail.ssq.residual;
                Corco(f,2)=model{f,2}.detail.coreconsistency.consistency;
                It(f,2)=model{f,2}.detail.critfinal(3);
                perc(f,2)=model{f,2}.detail.ssq.perc;
                if ssX(f,2)<BestErr
                    BestErr=ssX(f,2);
                    Bestnumb = 2;
                end
            end
            
            for rep=3:options.numbrepeats
                modeloptions.init = 3; % SVD init
                model{f,rep}=parafac(x,f,modeloptions);
                ssX(f,rep)=model{f,rep}.detail.ssq.residual;
                perc(f,rep)=model{f,rep}.detail.ssq.perc;
                Corco(f,rep)=model{f,rep}.detail.coreconsistency.consistency;
                It(f,rep)=model{f,rep}.detail.critfinal(3);
                if ssX(f,rep)<BestErr
                    BestErr=ssX(f,rep);
                    Bestnumb = rep;
                end
            end
            splitresult{f} = splithalf('default',x,f,modeloptions);
            splitqual(f)   = splitresult{f}.overall.quality;
            
        end
        result.method      = 'parafac';
        result.ssq         = ssX;
        result.perc        = perc;
        result.consistency = Corco;
        result.iterations  = It;
        result.ncomp       = ncomp;
        result.options     = options;
        result.models      = model;
        result.splithalf.quality = splitqual;
        result.splithalf.details = splitresult;
        
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        %%%%%% FIT TUCKER  %%%%%%%%%%%%%
        
    elseif  strcmp(method,'tucker')
        
        if size(ncomp,2)~=ndims(x)
            error(' The number of columns in second input NCOMPS must equal the order of the array')
        end
        PossibleNumber = [min(ncomp):max(ncomp)]'*ones(1,ndims(x));
        possibleCombs = unique(nchoosek(PossibleNumber(:),ndims(x)),'rows');
        %remove useless
        f2 = [];
        for f1 = 1:size(possibleCombs,1)
            if (prod(possibleCombs(f1,:))/max(possibleCombs(f1,:)))<max(possibleCombs(f1,:)) % Check that the largest mode is larger than the product of the other
                f2 = [f2;f1];
            elseif any(possibleCombs(f1,:)>max(ncomp))  % Chk the model is desired,
                f2 = [f2;f1];
            end
        end
        possibleCombs(f2,:)=[];
        [f1,f2]=sort(sum(possibleCombs'));
        possibleCombs = [possibleCombs(f2,:) f1'];
        
        for f=1:size(possibleCombs,1)
            tucker(x,possibleCombs(f,1:end-1),modeloptions);
            save testing
            model{f,1}=tucker(x,possibleCombs(f,1:end-1),modeloptions);
            ssX(f,1)=model{f,1}.detail.ssq.residual;
            perc(f,1)=model{f,1}.detail.ssq.perc;
            It(f,1)=model{f,1}.detail.critfinal(3);
        end
        result.method      = 'tucker';
        result.ssq         = ssX;
        result.perc        = perc;
        result.iterations  = It;
        result.ncomp       = possibleCombs;
        result.options     = options;
        result.models      = model;
    end
else
    % rearrange so the results input fit the notation
    result  = x;
    x       = ncomp;
    method  = result.method;
end


if strcmpi(result.options.plots,'on')|isstruct(x)
    
    if strcmp(method,'parafac')
        figure
        set(gcf,'userdata',{result.models;x});
        delta=.25/result.options.numbrepeats; % shift points a little horizontally for appearance
        set(gcf,'userdata',{result.models;x});
        subplot(3,1,1)
        for r=1:result.options.numbrepeats
            for f=result.ncomp
                result.perc(f,r)
                gg=plot(f+(r-1)*delta,result.perc(f,r), ...
                    'MarkerEdgeColor','k','MarkerFaceColor','r', ...
                    'LineWidth',2,'Marker','o','LineStyle','none', ...
                    'MarkerSize',8);
                set(gg,'userdata',[f r]);
                callb = ...
                    ['model=get(gcf,''userdata'');f=[',num2str([f r]),'];modelviewer(model{1}{f(1),f(2)},model{2});'];
                cmenu = uicontextmenu;
                set(gg,'UIContextMenu', cmenu);
                ctxt = num2str(f);
                ctxt = ['[PARAFAC (',ctxt,')] - Press to view model'];
                uimenu(cmenu, 'Label',ctxt,'callback',callb);
                hold on
            end
        end
        axis([1-.1 result.ncomp(end)+1 0-.05*max(result.perc(:)) 1.05*max(result.perc(:)) ])
        
        set(gca,'XTick',result.ncomp(end));
        ylabel('Percentage explained','FontWeight','bold')
        title('PARAFAC TEST - Right-click circle in top plot to see models','FontWeight','bold')
        hold off
        subplot(3,1,2)
        for r=1:result.options.numbrepeats
            plot(1+(r-1)*delta:1:result.ncomp(end)+(r-1)*delta,result.consistency(:,r), ...
                'MarkerEdgeColor','k','MarkerFaceColor','r', ...
                'LineWidth',2,'Marker','o','LineStyle','none', ...
                'MarkerSize',8)
            hold on
        end
        hold off
        MinCo=min(result.consistency(:));
        try
            axis([1 result.ncomp(end)+1 max([MinCo 0]) 100 ]);
        catch
            axis([1 result.ncomp(end)+1 0 100 ]);
        end
        set(gca,'XTick',result.ncomp(end));
        ylabel('Core consistency','FontWeight','bold')
        
        subplot(3,1,3)
        for f=result.ncomp
            h=bar(f,result.splithalf.quality(f)*100,.3);
            
            % Define loads
            toplot.smd = result.splithalf.details{f}.overallmodel.detail.options.samplemode;
            toplot.sgm  = result.splithalf.details{f}.overall.split; % Is it segment one or two that was best
            toplot.idx   = result.splithalf.details{f}.overall.index; % How to order the second split to match the first
            toplot.l1    = result.splithalf.details{f}.splitmodels{(toplot.sgm-1)*2+1}.loads;
            toplot.l2    = result.splithalf.details{f}.splitmodels{(toplot.sgm-1)*2+2}.loads;
            toplot.n   = length(toplot.l1); % Number of modes
            set(h,'userdata',toplot);
            set(h,'ButtonDownFcn',['f=get(gco,''userdata'');figure,c1 = 0;for f1=1:f.n,if f1~=f.smd,c1=c1+1;subplot(1,f.n-1,c1),plot(f.l1{f1}),hold on,plot(f.l2{f1}(:,f.idx(2,:)),''linewidth'',2),hold off,axis tight,title([''Mode '',num2str(f1)],''fontweight'',''bold''),end,end']);
            hold on
        end
        axis([1-.5 result.ncomp(end)+.5 -10 100])
        set(gca,'XTick',result.ncomp(end));
        ylabel('Splithalf quality - press bar for loads','FontWeight','bold')
        hold off
        xlabel('Number of components','FontWeight','bold')
        
        
        
    elseif  strcmp(method,'tucker')
        figure
        set(gcf,'userdata',{result.models;x});
        for f=1:size(result.ncomp,1)
            gg=plot(result.ncomp(f,end),result.perc(f), ...
                'MarkerEdgeColor',[1 .4 0],'MarkerFaceColor',[1 .4 0], ...
                'LineWidth',2,'Marker','o','LineStyle','none', ...
                'MarkerSize',6);
            set(gg,'userdata',result.ncomp(f,1:end-1));
            callb = ...
                ['model=get(gcf,''userdata'');f=',num2str(f),';modelviewer(model{1}{f},model{2});'];
            cmenu = uicontextmenu;
            set(gg,'UIContextMenu', cmenu);
            ctxt = num2str(result.ncomp(f,1:end-1));
            ctxt = ['[Tucker (',ctxt,')] - Press to view model'];
            uimenu(cmenu, 'Label',ctxt,'callback',callb);
            hold on
        end
        % Plot the good ones
        totcomps = unique(result.ncomp(:,end));
        for f=1:length(totcomps)
            tt = find(result.ncomp(:,end)==totcomps(f));
            [a,b]=max(result.perc(tt));
            gg=plot(totcomps(f),result.perc(tt(b)), ...
                'MarkerEdgeColor','k','MarkerFaceColor','g', ...
                'LineWidth',2,'Marker','o','LineStyle','none', ...
                'MarkerSize',8);
            set(gg,'userdata',result.ncomp(tt(b),1:end-1));
            callb = ...
                ['model=get(gcf,''userdata'');f=',num2str(tt(b)),';modelviewer(model{1}{f},model{2});'];
            cmenu = uicontextmenu;
            set(gg,'UIContextMenu', cmenu);
            ctxt = num2str(result.ncomp(tt(b),1:end-1));
            ctxt = ['[Tucker (',ctxt,')] - Press to view model'];
            uimenu(cmenu, 'Label',ctxt,'callback',callb);
            hold on
        end
        
        
        ylabel('Percentage explained','FontWeight','bold')
        xlabel('Total number of components','FontWeight','bold')
        title('TUCKER TEST - Right-click circle to see model','FontWeight','bold')
        hold off
        
    end
end
function fout = cwaitbar(x,name,col)
%CWAITBAR Display compound wait bar.
%   H = CWAITBAR(X,TITLE) creates and displays wait bars of
%   fractional lengths X and with one title string TITLE.
%   The handle to the compound waitbar figure is returned in H.
%   X values should be between 0 and 1.
%   Each subsequent call to cwaitbar, CWAITBAR([BAR X]),
%   extends the length of the bar BAR to the new position X.
%   The first bar is the topmost bar and is BAR = 1 which
%   corresponds to the outermost loop.
%   H = CWAITBAR(X,TITLE) where TITLE is a cellstring with same
%   number of titles as there are fractional lengths in X.
%   Suitable for showing the bars' corresponding loop indices.
%   H = CWAITBAR(X,TITLE,COLOR) where COLOR is the color of the
%   bars. COLOR is either a color code character (see PLOT) or
%   an RGB vector. The default color is red. COLOR can also be
%   a cell array with same number of elements as there are bars
%   in the cwaitbar figure.
%
%   The order of the elements in vector X and cell arrays TITLE
%   and COLOR which is consistent with the bar number BAR is:
%   The first element corresponds to the first bar at the top
%   of the figure which in turn corresponds to the outermost loop.
%
%   CWAITBAR is typically used inside nested FOR loops that
%   performs lengthy computations.
%
%      Examples:
%         cwaitbar([.3 .2 .7],'Please wait...');     %single title
%
%         h = cwaitbar([0 0 0],{'i','j','k'},{[.8 .2 .8],'b','r'});
%         for i=1:5,
%            % computations %
%            for j=1:10
%               % computations %
%               for k=1:100
%                  % computations %
%                  cwaitbar([3 k/100])
%               end
%               cwaitbar([2 j/10])
%            end
%            cwaitbar([1 i/5])
%         end
%         close(h)
%
%   See also WAITBAR.

% Based on matlab's WAITBAR. See help for WAITBAR.
% Copyright (c) 2003-11-02, B. Rasmus Anthin.
% Revision 2003-11-03 - 2003-11-06.
% GPL license.

xline = [100 0 0 100 100];
yline = [0 0 1 1 0];


switch nargin
    case 1   % waitbar(x)    update
        bar=x(1);
        x=max(0,min(100*x(2),100));
        f = findobj(allchild(0),'flat','Tag','CWaitbar');
        if ~isempty(f), f=f(1);end
        a=sort(get(f,'child'));                         %axes objects
        if isempty(f) | isempty(a),
            error('Couldn''t find waitbar handles.');
        end
        bar=length(a)+1-bar;        %first bar is the topmost bar instead
        if length(a)<bar
            error('Bar number exceeds number of available bars.')
        end
        for i=1:length(a)
            p(i)=findobj(a(i),'type','patch');
            l(i)=findobj(a(i),'type','line');
        end
        %rewind upper bars when they are full
        %   if bar==1
        %      for i=2:length(a)
        %         xpatchold=get(p(i),'xdata');
        %         xold=xpatchold(2);
        %         if xold==100
        %            set(p,'erase','normal')
        %            xpatch=[0 0 0 0];
        %            set(p(i),'xdata',xpatch,'erase','none')
        %            set(l(i),'xdata',xline)
        %         end
        %      end
        %   end
        
        a=a(bar);
        p=p(bar);
        l=l(bar);
        xpatchold=get(p,'xdata');
        xold=xpatchold(2);
        if xold>x                      %erase old patches (if bar is shorter than before)
            set(p,'erase','normal')
            %xold=0;
        end
        xold=0;
        %previously: (continue on old patch)
        xpatch=[xold x x xold];
        set(p,'xdata',xpatch,'erase','none')
        set(l,'xdata',xline)
        
    case 2   % waitbar(x,name)  initialize
        x=fliplr(max(0,min(100*x,100)));
        
        oldRootUnits = get(0,'Units');
        set(0, 'Units', 'points');
        pos = get(0,'ScreenSize');
        pointsPerPixel = 72/get(0,'ScreenPixelsPerInch');
        
        L=length(x)*.6+.4;
        width = 360 * pointsPerPixel;
        height = 75 * pointsPerPixel * L;
        pos = [pos(3)/2-width/2 pos(4)/2-height/2 width height];
        
        f = figure(...
            'Units', 'points', ...
            'Position', pos, ...
            'Resize','off', ...
            'CreateFcn','', ...
            'NumberTitle','off', ...
            'IntegerHandle','off', ...
            'MenuBar', 'none', ...
            'Tag','CWaitbar');
        colormap([]);
        
        for i=1:length(x)
            h = axes('XLim',[0 100],'YLim',[0 1]);
            if ~iscell(name)
                if i==length(x), title(name);end
            else
                if length(name)~=length(x)
                    error('There must be equally many titles as waitbars, or only one title.')
                end
                title(name{end+1-i})
            end
            set(h, ...
                'Box','on', ...
                'Position',[.05 .3/L*(2*i-1) .9 .2/L],...
                'XTickMode','manual',...
                'YTickMode','manual',...
                'XTick',[],...
                'YTick',[],...
                'XTickLabelMode','manual',...
                'XTickLabel',[],...
                'YTickLabelMode','manual',...
                'YTickLabel',[]);
            
            xpatch = [0 x(i) x(i) 0];
            ypatch = [0 0 1 1];
            
            patch(xpatch,ypatch,'r','edgec','r','erase','none')
            line(xline,yline,'color','k','erase','none');
            
        end
        set(f,'HandleVisibility','callback');
        set(0, 'Units', oldRootUnits);
        
    case 3
        if iscell(col) & length(col)~=length(x)
            error('There must be equally many colors as waitbars, or only one color.')
        end
        f=cwaitbar(x,name);
        a=get(f,'child');
        p=findobj(a,'type','patch');
        l=findobj(a,'type','line');
        if ~iscell(col)
            set(p,'facec',col,'edgec',col)
        else
            for i=1:length(col)
                set(p(i),'facec',col{i},'edgec',col{i})
            end
        end
        set(l,'xdata',xline')
end  % case
drawnow
figure(f)

if nargout==1,
    fout = f;
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%% SPLITHALF %%%%%%%%%%%%%%%%%%%%%%%%%
function result = splithalf(varargin);

%SPLITHALF performs splihalf validation of PARAFAC models
% Input resulting structure to plot the results
%
%I/O result = splithalf(mode,data,...)
%             splithalf(result)
% mode = 'default' => Normalized loadings shown

if nargin>1
    samplemode = 1;
    order = length(size(varargin{2}));
    split = varargin{1};
    
    splitopt.display = 'off';
    if nargin<4 % Options definitely not given, then turn off display and screen output
        splitopt = parafac('options');
        splitopt.display = 'off';
        splitopt.plots  = 'off';
        inputs = varargin;
        inputs{end+1}=splitopt;
    else
        inputs = varargin;
    end
    
    % Make sure data are in a dataset
    if ~isa(varargin{2},'dataset')% Then it's a SDO
        varargin{2}=dataset(varargin{2});
    end
    
    % Check if option struct is given and pick the samplemode from that then
    for i=1:length(varargin)
        if isstruct(varargin{i})
            if isfield(varargin{i},'samplemode')
                samplemode = varargin{i}.samplemode;
                if ~any(1:length(size(varargin{2}.data)) == samplemode)
                    error([' The definition of the samplemode in the PARAFAC options has to be an integer between 1 (first mode) and ',num2str(length(size(varargin{2}))),' (last mode)'])
                end
            end
        end
    end
    if size(varargin{2}.data,samplemode)<9
        error(' More than 8 samples are needed in order to perform splithalf analysis')
    end
    % Define split sets
    if isstr(split) %#ok<FDEPR>
        if strcmpi(split,'default')
            I  = size(varargin{2}.data,samplemode);
            i1 = round(I/4);
            i2 = round(I/2);
            i3 = i1+i2;
            idx{1} = 1:i2;             % Split 1, A
            idx{2} = i2+1:I;           % Split 1, B
            idx{3} = [1:i1 i2+1:i3];   % Split 2, c
            idx{4} = [i1+1:i2 i3+1:I]; % Split 2, d
        end
    end
    
    % Do overall model
    if strcmp(splitopt.display,'on')
        disp('Fitting overall model')
    end
    M = parafac(inputs{2:end});  % Overall model
    
    % Perform splithalf
    for i = 1:4
        if strcmp(splitopt.display,'on')
            disp(['Fitting split model ',num2str(i),' of 4']);
        end
        X = varargin{2};
        X.include{samplemode}=idx{i}; % Include only the relevant samples
        Model{i}=parafac(X,inputs{3:end});
    end
    for i=1:order
        if i~=samplemode
            %ccc = corrcoef([Model{1}.loads{i} Model{2}.loads{i}]);
            %correlations{i}.split1 = ccc(1:varargin{3},varargin{3}+1:2*varargin{3});
            % Do congruence instead of correlation
            ccc = nm(Model{1}.loads{i})'*nm(Model{2}.loads{i});
            ccc2 = nm(Model{1}.loads{i})'*nm(M.loads{i}); % ALso compare to overall model
            correlations{i}.split1 = ccc.*ccc2;
            %ccc = corrcoef([Model{3}.loads{i} Model{4}.loads{i}]);
            %correlations{i}.split2 = ccc(1:varargin{3},varargin{3}+1:2*varargin{3});
            ccc = nm(Model{3}.loads{i})'*nm(Model{4}.loads{i});
            ccc2 = nm(Model{3}.loads{i})'*nm(M.loads{i}); % ALso compare to overall model
            correlations{i}.split2 = ccc.*ccc2;
        end
    end
    
    % Make a combined overall quality index
    qual1=ones(inputs{3},inputs{3});
    qual2=ones(inputs{3},inputs{3});
    for i=1:order
        if i~=samplemode
            qual1=qual1.*correlations{i}.split1;
            qual2=qual2.*correlations{i}.split2;
        end
    end
    % Chk which combination of factors in the two splits are best
    quality1 = 0;ind1 = repmat(NaN,1,inputs{3});
    quality2 = 0;ind2 = repmat(NaN,1,inputs{3});
    permu = perms([1:inputs{3}]);
    for j = 1:size(permu,1)
        %check
        thisqual = 1;
        for k =1:inputs{3}
            thisqual = thisqual*qual1(k,permu(j,k));
        end
        if abs(thisqual)>quality1
            quality1 = thisqual;
            ind1 = [1:inputs{3};permu(j,:)];
        end
        
        thisqual = 1;
        for k =1:inputs{3}
            thisqual = thisqual*qual2(k,permu(j,k));
        end
        if abs(thisqual)>quality2
            quality2 = thisqual;
            ind2 = [1:inputs{3};permu(j,:)];
        end
    end
    if abs(quality2)>abs(quality1)
        result.overall.quality = abs(quality2);
        result.overall.index = ind2;
        result.overall.split = 2;
    else
        result.overall.quality = abs(quality1);
        result.overall.index = ind1;
        result.overall.split = 1;
    end
    
    result.overallmodel = M;
    result.samplesets=idx;
    result.splitmodels = Model;
    result.correlations = correlations;
    
elseif nargin==1 % Results are input for plotting
    result = varargin{1};
    samplemode = result.overallmodel.detail.options.samplemode;
    order = length(result.overallmodel.loads);
    
    for i=1:order
        if i~=samplemode
            figure
            a=plot(nm(result.overallmodel.loads{i}),'k','linewidth',4);
            hold on;
            b=plot(nm(result.splitmodels{1}.loads{i}),'bo-');
            c=plot(nm(result.splitmodels{2}.loads{i}),'b^-');
            d=plot(nm(result.splitmodels{3}.loads{i}),'ro-');
            e=plot(nm(result.splitmodels{4}.loads{i}),'r^-');
            axis tight
            xlabel(['Mode ',num2str(i),' - black:overall model,blue:split 1,red:split 2']);
            
            set(a,'tag','a')
            set(b,'tag','b')
            set(c,'tag','c')
            set(d,'tag','d')
            set(e,'tag','e')
            
            ha = uicontrol('Parent',gcf, ...
                'Units','normalized', ...
                'callback',['aa = findobj(''tag'',''a'');try,for i=1:length(aa),if strcmp(lower(get(aa(i),''visible'')),''on''),set(aa(i),''visible'',''off'');else,set(aa(i),''visible'',''on'');end,end,end'],...
                'Style','pushbutton', ...
                'Tag','StaticText1', ...
                'HandleVisibility','off', ...
                'HorizontalAlignment','center', ...
                'Position',[0.01 0.95 0.18 0.05], ...
                'FontSize',10, ...
                'ToolTipString',['Toggle the loadings of the overall model'], ...
                'handlevisibility','on', ...
                'String','Overall model');
            
            hb = uicontrol('Parent',gcf, ...
                'Units','normalized', ...
                'callback',['bb = findobj(''tag'',''b'');try,for i=1:length(bb),if strcmp(lower(get(bb(i),''visible'')),''on''),set(bb(i),''visible'',''off'');else,set(bb(i),''visible'',''on'');end,end,end;cc = findobj(''tag'',''c'');try,for i=1:length(cc),if strcmp(lower(get(cc(i),''visible'')),''on''),set(cc(i),''visible'',''off'');else,set(cc(i),''visible'',''on'');end,end,end'],...
                'Style','pushbutton', ...
                'Tag','StaticText1', ...
                'HandleVisibility','off', ...
                'HorizontalAlignment','center', ...
                'Position',[0.2 0.95 0.1 0.05], ...
                'FontSize',10, ...
                'ToolTipString',['Toggle the loadings (two sets) of the first split'], ...
                'handlevisibility','on', ...
                'String','Split 1');
            
            hd = uicontrol('Parent',gcf, ...
                'Units','normalized', ...
                'callback',['dd = findobj(''tag'',''d'');try,for i=1:length(dd),if strcmp(lower(get(dd(i),''visible'')),''on''),set(dd(i),''visible'',''off'');else,set(dd(i),''visible'',''on'');end,end,end;ee = findobj(''tag'',''e'');try,for i=1:length(ee),if strcmp(lower(get(ee(i),''visible'')),''on''),set(ee(i),''visible'',''off'');else,set(ee(i),''visible'',''on'');end,end,end'],...
                'Style','pushbutton', ...
                'Tag','StaticText1', ...
                'HandleVisibility','off', ...
                'HorizontalAlignment','center', ...
                'Position',[0.32 0.95 0.1 0.05], ...
                'FontSize',10, ...
                'ToolTipString',['Toggle the loadings (two sets) of the second split'], ...
                'handlevisibility','on', ...
                'String','Split 2');
        end
    end
end
function Xn = nm(X);

Xn=X;
for i=1:size(X,2)
    Xn(:,i)=Xn(:,i)/norm(Xn(:,i));
end

Xn = Xn * diag(sign(sum(Xn.^3)));
