function ahimbu_str(model, format, X, classset)
% AHIMBU_STR Visualise the structure of an AHIMBU model
%
% Usage: AHIMBU_STR(model, format, X, classset)
%
% model     a hierarchical model built by AHIMBU
% format:   'plain' for plain text output or 'dot' for output suitable
%           for Graphviz, defaults to 'plain'
% X:        a dataset containing the predictors and class labels.
%           Not required if model was built with the
%           preserve_class_labels option.
% classset: the number of the row label set to consider, defaults to 1.
%           Not required if model was built with the
%           preserve_class_labels option.
%
% See also: AHIMBU
    if nargin < 2
        format = 'plain';
    end

    if nargin < 3
        X = [];
    end

    if nargin < 4
        classset = 1;
    end

    start_output(format)
    traverse(model, format, X, classset, '  ', 'G');
    end_output(format)
end

function start_output(format)
    switch format
        case 'dot'
            fprintf('digraph G {\n')
        case 'plain'
            % do nothing
        otherwise
            error('The ''format'' argument must be one of ''plain'', ''dot''')
    end
end

function end_output(format)
    switch format
        case 'dot'
            fprintf('}\n')
    end
end

function output_trigger(format, indent, id, label, lvs)
    plural = cell2mat(getfield({'', 's'}, {1 + (lvs > 1)}));
    switch format
        case 'dot'
            fprintf('%s%s [label="%s\\n%d LV%s"];\n', ...
                indent, id, label, lvs, plural ...
            )
        case 'plain'
            fprintf('%s[%d LV%s]\n', indent, lvs, plural)
    end
end

function output_link(format, indent, id, num)
    switch format
        case 'dot'
            fprintf('%s%s -> %s_%d\n', indent, id, id, num)
    end
end

function output_target(format, indent, id, num, label)
    switch format
        case 'dot'
            fprintf('%s%s_%d [label="%s"];\n', indent, id, num, label)
        case 'plain'
            fprintf('%s`-%s\n', indent, label)
    end
end

function traverse(m, format, X, cl, indent, name)
    % define the node for the current trigger
    output_trigger(format, indent, name, m.trigger.modeltype, m.trigger.ncomp)
    % define the nodes for the targets, recursively
    for i=1:length(m.targets)
        if ismodel(m.targets{i})
            traverse(m.targets{i}, format, X, cl, [indent '  '], [name '_' num2str(i)])
            output_link(format, indent, name, i)
        elseif isnumeric(m.targets{i}) && m.targets{i} == 0
            % well, do nothing: the "no answer" target doesn't need
            % external definition
        else
            if ischar(m.targets{i})
                label = m.targets{i};
            else
                assert(~isempty(X), ...
                    'Please provide the X argument to resolve integer outcomes' ...
                )
                label = X.classlookup{1,cl}.find(m.targets{i});
                label = label{1};
            end
            output_target(format, indent, name, i, label)
            output_link(format, indent, name, i)
        end
    end
end
