# Script to generate a plot and database with all the surface atoms for a
# nanoparticle of approximately D = 3.53 nm, Fig. 3c
from ase.neighborlist import NeighborList
from ase.db import connect
import csv
import matplotlib.pyplot as plt
import pandas as pd

def count_surface_atoms(db_file):
    """
    Count the exposed Pd and Au atoms at the surface for a given database file.
    Returns the percentage of Pd atoms at the surface.
    Here, IF the neighbor atoms to an atoms is <= 11, they are considered as surface atoms
    because they are exposed, bulk atoms have 12 neighbor atoms.
    """
    auS = []  # List with the exposed Au atoms at the surface
    pdS = []  # List with the exposed Pd atoms at the surface
    for row in connect(db_file).select():
        slab = connect(db_file).get_atoms(row.id)
        neighbors = NeighborList([1.43 for a in range(len(slab))], self_interaction=False, bothways=True)
        neighbors.update(slab)
        for i in range(len(slab)):
            indices, offsets = neighbors.get_neighbors(i)
            if len(indices) <= 11:
                if slab[i].symbol == 'Au':
                    auS.append(i)
                elif slab[i].symbol == 'Pd':
                    pdS.append(i)
    tot = len(auS) + len(pdS)
    if tot == 0:
        return 0
    else:
        return round((len(pdS) / tot) * 100, 2)

# List of database files
db_files = ['au10/au10end.db', 'au15/au15end.db', 'au20/au20end.db', 'au25/au25end.db', 'au30/au30end.db', 'au35/au35end.db', 'au40/au40end.db', 'au45/au45end.db', 'au50/au50end.db']

# Generate data
data = []
for db_file in db_files:
    percentage = int(db_file[2:4])
    pdS = count_surface_atoms(db_file)
    data.append((percentage, pdS))

# Save data in a CSV file
with open('dataSurfAtoms3.60nm.csv', 'w', newline='') as csvfile:
    writer = csv.writer(csvfile)
    writer.writerow(['Au Fraction', '% of Pd atoms at surface'])
    for row in data:
        writer.writerow(row)
