# Script to make and dissolve AuPd nanoparticles of D=1.8 nm, Au = 40%
from ase.io import *
import numpy as np
from ase.cluster import wulff_construction
from ase.neighborlist import NeighborList
# from ase.db import connect

# Open a database file to store each step (Here we do not save them)
# db = connect('geoms.db')

# === Step 1 - Generate a random nanoparticle by using the average bulk lattice parameters
# of the elements in the binary nanoparticle ===
surfaces = [(1, 1, 1), (1, 0, 0)]  # Exposed crystallographic facets
esurf = [1.00, 1.00]  # Surface energy of each crystallographic facet
lc = 4.0233  # Average lattice parameter

"""we used lc equal to the average bulk value of the optimized structures for each element in the alloy
nanoparticle (RPBE functional)"""

size = 201

"""Size approximate the size of the NPs because the Wulff construction being 
built from whole layers, meaning that it's impossible to hit the desired NP size"""

# Make a Pd nanoparticle with D ≈ 1.18 nm
nano = wulff_construction('Pd', surfaces, esurf, size, 'fcc', rounding='above', latticeconstant=lc)

# Define elements in the binary nanoparticles
metals = ['Au', 'Pd']

# Fix the gold molar fraction of each element 
au_c = 0.40
pd_c = 1 - au_c
composition = np.array([au_c, pd_c])

# Get the total number of atoms in the nanoparticle
natoms = len(nano)

# Calculate the number of atoms for each element
au_natoms = int(np.floor(natoms * composition[0]))
pd_natoms = natoms - au_natoms

# Create an array with the element labels for the nanoparticle
element_labels = np.zeros(natoms, dtype=int)
element_labels[:au_natoms] = metals.index('Au')
element_labels[au_natoms:] = metals.index('Pd')

# Shuffle the element labels randomly
np.random.shuffle(element_labels)
# Replace the element symbols in the ASE Atoms object with the labels from the array
for i, atom in enumerate(nano):
    atom.symbol = metals[element_labels[i]]
nano = nano
nano.write('startNP.traj') # write the start structure

# === Step 2 - Remove the highgest under-coordinated Au atoms and Pd atoms with coordination number <= 6 === # 
atoms = nano # Read the binary nanoparticle
# db.write(atoms)
cutoff = 1.43  # cutoff radii - one for each atom

while True:
    symbols = atoms.get_chemical_symbols()
    neighbors = NeighborList([cutoff for a in range(len(atoms))], self_interaction=False, bothways=True)
    neighbors.update(atoms)

    # Identify all the highest under-coordinated Au atoms (coordination number <=1)
    au_indices = [i for i in range(len(atoms)) if symbols[i] == 'Au' and len(neighbors.get_neighbors(i)[0]) <= 1]

    # Identify all Ir and Pt atoms with coordination number <=6
    irpt_indices = [i for i in range(len(atoms)) if symbols[i] == 'Pd'  and len(neighbors.get_neighbors(i)[0]) <= 6]

    # Combine the two lists and exit the loop if there are no more atoms to remove
    indices_to_remove = list(set(au_indices) | set(irpt_indices))
    if len(indices_to_remove) == 0:
        break

    # Change the chemical symbols of the atoms to be removed
    for i in indices_to_remove:
        symbols[i] = 'Mn'

    # Remove the atoms and update the neighbor list
    atoms.set_chemical_symbols(symbols)
    neighbors.update(atoms)
    mn_indices = [i for i in range(len(atoms)) if symbols[i] == 'Mn']
    del atoms[mn_indices]
    # db.write(atoms)

# Step 3 - write down the final structure
atoms.write('endNP.traj')
