function [S,W,wave,filelist]=ReadInScans(type,format,range,display,outdat,pattern)
% [S,W,wave,filelist]=ReadInScans(type,format,range,display,outdat,pattern);
% Reads in all files of specified format (csv, txt, xls, or xlsx) in the current directory
% and converts to a single 2D matrix of scans. Remove unwanted files of the same format
% from the current directory before running this program.
% The files will be loaded in alphabetical order (see output: filelist).
% All scans should be of the same size, type and format
%
%%%%%%%%%%%%%%%%
%Input variables
%%%%%%%%%%%%%%%%
%1. type = Up to 10 characters describing the type of data scans, used to name the output file. For example: 
%       'Absorbance':  Absorbance scans
%       'UV350':  UV scans at 350 nm excitation
%
%2. format = 'csv','xls','xlsx','txt'. 
%             Note 1. Importing from text files is supported only when the file exclusively 
%                   contains numbers with no text anywhere in the file.
%                   Range need not be specified for text files (use range = [])
%             Note 2. For xls or xls formats, data will be imported from the first worksheet of the excel file
%
%3. range = Range of cells in the raw EEMs that contain numbers (see Note 1 above for 'txt' formats).
%           e.g. range ='A3..B500' or range ='A2..BB3' (or range = [] for .txt files only)
%
%4. display = length of time to show plots of scans on screen (e.g. 1 = display for 1 second)
%             if display=0, no plots will be created.
%
%5. outdat = 1 to write S, W, filelist and wave to an Excel file in the current directory; 0 otherwise.
%             Data are saved in "OutData_xx.xls": where xx is the contents of 'type' above.
%
%6. pattern = a text string in the file name. E.g. 'QS*' means only filenames beginning with 'QS' will be read in.
%             If no pattern is specified the default will be to read in all files of the specified extension,
%             i.e. the same as reading in e.g. '*.csv' or '*.txt'.
%
%%%%%%%%%%%%%%%%
%Output variables
%%%%%%%%%%%%%%%%
%
%   S=2D matrix of scans
%   W=2D matrix of wavelengths corresponding with individual scans
%   wave=if all the rows of W are equal within 0.5 nm, wave is a single row of wavelengths rounded to 0.5nm
%   filelist=list of filenames
%   
%%%%%%%%%%%%%%%%
% Please cite this program as:
% ReadInScans.m (ver. 1.3: 2011-05-02)
%
% Copyright (C) 2010 KR Murphy,
% Water Research Centre
% The University of New South Wales
% Department of Civil and Environmental Engineering
% Sydney 2052, Australia
% krm@unsw.edu.au
%%%%%%%%%%%%%%%%

%% Initialize
error(nargchk(5,6,nargin))
error(nargoutchk(4,4,nargout))

%If files are in Excel format, delete old version of OutData.xls to avoid conflict.
if strcmp(format,'xls')==1;
    FN=['OutData_' type '.xls'];
    fprintf(['Any existing files of name ' FN ' in the current directory will be replaced. Press any key to continue...\n'])
    pause;
    eval(['delete ' FN])
end

if nargin == 5
    direc = dir(['*.' format]);
elseif nargin == 6
    direc = dir([pattern '.' format]);
end

filelist=[];S=[];W=[];frows=[];

%% Load Files
for i=1:size(direc,1)
    
    fprintf([direc(i).name '\n'])
    filelist=char(filelist,direc(i).name); %file list

    % LOAD FILES, excluding cells that contain text
    if strcmp(format,'csv')==1;
        x = dlmread(direc(i).name,',',range);
    elseif strcmp(format,'txt')==1;
        x = dlmread(direc(i).name);
    elseif strcmp(format,'xls')==1;
        x = xlsread(direc(i).name,1,range);
    elseif strcmp(format,'xlsx')==1;
        x = xlsread(direc(i).name,1,range);
    end   
    
    % convert to row vectors
    xsize=size(x);
    sizeerror=false;
    if min(xsize)==1 % single data pair
        if and(xsize(1)==1,xsize(2)==2); %x=[wave data]
            x=x';
        else
            sizeerror=true;
        end
    elseif min(xsize)==2 %data are in rows or in columns
        if xsize(1)>2; %data are in columns
            x=x';
        elseif xsize(1)==xsize(2); %data in rows or in columns
            if isempty(frows);
                fprintf(['input data are 2 rows x 2 colums, more information needed... \n'])
                while isempty(frows)
                    frows=input(['type 1 if wavelengths are in rows, or 2 if wavelengths are in columns: ']);
                    if ismember(frows,[1;2])==true
                    else frows=[];
                    end
                end
            end
            if frows==2;
              x=x';              
            end
        end
    elseif min(xsize)>2
        sizeerror=true;
    end
    if sizeerror==true;
        fprintf(['size of input data is ' num2str(xsize(1)) ' rows and ' num2str(xsize(2))  ' columns \n'])
        error(['Inappropriate data range. Scans must consist of a row or column ' ...
            'of wavelengths followed by 1 row or column of data']);
    end
    x=(sortrows(x',1))';    
    xsize=size(x);
    
    %check that x has no more than two rows (or columns)
    if i==1;
        xsize1=xsize;
    else
        %check that the size of x is consistent with the first scan
        if size(x)~=xsize1
            fprintf('The size of the input scans has changed!\n')
            fprintf(['size of first scan was ' num2str(xsize(1)) 'rows and ' num2str(xsize(2))  ' columns'])
            fprintf(['size of current scan is ' num2str(size(x,1)) 'rows and ' num2str(size(x,2))  ' columns'])
            error('Check raw data file and resize as necessary.')
        end
    end
    
    %% Assemble matrices
    W(i,:)=x(1,:);
    x=x(2,:);
    S(i,:)=x;
    
    %% DISPLAY a plot of each scan as it is loaded
    if display>0;
        plot(W(i,:),S(i,:),'ko'),
        title(direc(i).name,'interpreter','none')
        if i>1
            hold on
            plot(W(1:i,:)',S(1:i,:)','c-'),
            legend('current scan','prior scans')
            hold off
        end
       pause(display),
       if i<size(direc,1)
       close
       end
    end
end
%remove leading row of blank filenames
filelist=filelist(2:end,:); 

%Check for consistency among wavelengths
wave=NaN*ones(1,size(S,2));
rndW=round(W*2)/2;
rndW1=repmat(rndW(1,:),[size(rndW,1) 1]);
Wdiff=rndW-rndW1;
if max(max(Wdiff))==0;
    wave=rndW(1,:);
else
    fprintf('\n')
    fprintf('Warning: not all input wavelengths were the same! \n')
    fprintf('Check the rows of output variable W for non-standard wavelengths.\n')
    fprintf('The output variable "wave" is a vector of NaNs.\n')
end

%% EXPORT DATA 
% Export scans to an excel spreadsheet
if outdat==1;
    FN=['OutData_' type '.xls'];
    if size(wave,2)>255 %maximum number of columns allowed by Excel = 256
        wave_LowRes=wave(1:2:end);
        S_LowRes=S(:,1:2:end); 
        W_LowRes=W(:,1:2:end);
        while size(wave_LowRes,2)>255
            wave_LowRes=wave_LowRes(1:2:end);
            S_LowRes=S_LowRes(:,1:2:end);
            W_LowRes=W_LowRes(:,1:2:end);
        end
        xlswrite(FN,wave_LowRes,type,'A1')
        xlswrite(FN,S_LowRes,type,'A2')
        xlswrite(FN,W_LowRes,'Wavelengths')
        msg1='These worksheets contain data at lower wavelength resolution than the raw data files';
        msg2='Full-resolution data are retained in MATLAB';
        xlswrite(FN,cellstr(msg1),'Messages','A1')
        xlswrite(FN,cellstr(msg2),'Messages','A2')
    else
        xlswrite(FN,wave,type,'A1')
        xlswrite(FN,S,type,'A2')
        xlswrite(FN,W,'Wavelengths')
    end
    xlswrite(FN,cellstr(filelist),'Filenames','A2')
    fprintf([FN ' has been written to your current directory...\n'])
end

