% FDOMcorr template
% VERSION: 1.4 14-Nov-2011
% The template illustrates all the steps for assembling and correcting EEMs
% using the FDOMcorr toolbox for MATLAB.
% The directories and specifications shown (e.g. file types and sizes) are for demonstration purposes only
% Save this file with a new name and modify it as necessary to suit your data files
% Comment out unnecessary lines of code (select=> rightclick=> comment/uncomment)
% author: krm@unsw.edu.au
%
%%  ABOUT THE TOOLBOX
% The FDOMcorr toolbox is a formalisation of the EEM correction procedures resulting from discussions at the
% AGU Chapman Conference on organic matter fluorescence, held in Birmingham UK in October, 2008.
% Please cite as follows: 
%   The FDOMcorr toolbox for MATLAB (Murphy et al. 2010)
% 	'The measurement of dissolved organic matter fluorescence 
% 	in aquatic environments: An interlaboratory comparison'
% 	Environmental Science & Technology, doi: 10.1021/es102362t 
%
% Copyright (C) 2011 KR Murphy,
% Water Research Centre
% The University of New South Wales
% Department of Civil and Environmental Engineering
% Sydney 2052, Australia
% krm@unsw.edu.au

%% locate the FDOMcorr Toolbox on your computer (specify correct file path)
addpath('C:\EEMdata\FDOMcorr') 

%% Step 1:   read in data files
%% comment out lines of code that are not needed

cd 'C:\EEMdata\EEMs'
%each file is an EEM
filetype=2;ext = 'csv';RangeIn='A3..CD113';headers=[1 1];display_opt=1;outdat=1;
[X,Emmat,Exmat,filelist_eem,outdata]=ReadInEEMs(filetype,ext,RangeIn,headers,display_opt,outdat);
Ex=Exmat(1,:); %If these excitation wavelengths correspond to all scans, see step 2
Em=Emmat(:,1); %If these emission wavelengths correspond to all scans, see step 2

cd 'C:\EEMdata\Blanks'
%each file is an EEM
filetype=2;ext = 'csv';RangeIn='A3..CD113';headers=[1 1];display_opt=1;outdat=1;
[X_b,Emmat_b,Exmat_b,filelist_b,outdata_b]=ReadInEEMs(filetype,ext,RangeIn,headers,display_opt,outdat);
Exb=Exmat_b(1,:); %If these excitation wavelengths correspond to all scans, see step 2
Emb=Emmat_b(:,1); %If these emission wavelengths correspond to all scans, see step 2

cd 'C:\EEMdata\Raman'
%define landa - the excitation wavelength  for the water Raman scans, 350 nm is the default.
RamEx=350; %update as necessary. If landa <> 350, calculate RamOpt in Step 4

%each file consists of a column of em wavelengths and a column of data
filetype='Raman';ext = 'csv';RangeIn='A3..B83';display_opt=0.5;outdat=1;
[S_R,W_R,wave_R,filelist_R]=ReadInScans(filetype,ext,RangeIn,display_opt,outdat);

cd 'C:\EEMdata\Absscans'
%each file consists of a row of em wavelengths and a row of data
filetype='Abs';ext = 'txt';RangeIn=[];display_opt=0.5;outdat=1;
[S_abs,W_abs,wave_abs,filelist_abs]=ReadInScans(filetype,ext,RangeIn,display_opt,outdat);

cd 'C:\EEMdata\QS'
%each file consists of 1 em wavelength and 1 slope 
filetype='QS';ext = 'csv';RangeIn='A3..B3';display_opt=0.5;outdat=1;
[S_qs,W_qs,wave_qs,filelist_qs]=ReadInScans(filetype,ext,RangeIn,display_opt,outdat);
QSEx=350; %excitation wavelength used for the QS scans, 350 nm is the default.

cd 'C:\Fluorometers\FM3\CorrectionFiles'
%read in correction files
%each file consists of a column of wavelengths followed by a column of data
%stored in columns A-D of Excel workbook: CorrectionFactors.xls, Sheet: Oct30
Emcor=xlsread('CorrectionFactors.xls','Oct30','A2..B80')
Excor=xlsread('CorrectionFactors.xls','Oct30','C2..D46')

%% 1.b. read in Sample Log
% The sample log is used to match samples, and supply unique data 
% about the treatment of individual samples
% (e.g. adjust  for dilution, voltage, path length, etc.)
%
%read in log from SampleLog.xls, sheet=Cruise1.
%columns of LogTXT have complete filenames (i.e. with file extension) for: 
%1. EEMs, 2. Raman scans, 3. Abs scans, 4. QS scans, 5. blanks
%columns of LogNUM contain numerical data in 4 columns:
%1. dilution factors, 2. Absorbance path length, 3. Fluorescence path length, 4. coded error flags

cd 'C:\Data\Experiments\Cruises'
[LogNUM,LogTXT]=xlsread('SampleLog.xls','Cruise1','A2:K34')
Pair_EEM_log=[LogTXT(:,1) LogTXT(:,1)];
Pair_EEM_R=[LogTXT(:,1) LogTXT(:,2)];
Pair_EEM_abs=[LogTXT(:,1) LogTXT(:,3)];
Pair_EEM_qs=[LogTXT(:,1) LogTXT(:,4)];
Pair_EEM_bk=[LogTXT(:,1) LogTXT(:,5)];
Pair_BLK_R=[LogTXT(:,5) LogTXT(:,2)];
Pair_QS_R=[LogTXT(:,4) LogTXT(:,2)];

%import dilution factors from data log.
%DilFac(i)=10 indicates that sample i was diluted to 10% strength (1/10) before measurement
%DilFac(i)=2 indicates that sample i was diluted to 50% strength (1/2) before measurement
%DilFac(i)=1 indicates that sample i was undiluted
% Note DilFac must be sorted later using matchsamples to match them with the EEMs
DilFac=LogNUM(:,2); 

%import absorbance and fluorescence path lengths from the data log (see step 5). Path
%length is in cm. IFE correction in FDOMcorr is only performed for sample i 
%if pathlen(i)=pathlen_f(i)=1;
% Note pathlen_a and pathlen_f must be sorted later using matchsamples to match them with the EEMs
pathlen_a=LogNUM(:,2);  
pathlen_f=LogNUM(:,3);  

%import error flags and comments from the last two columns of the log (see step 5).
% Note flagcode and notes must be sorted later using matchsamples to match them with the EEMs
flagcode=LogNUM(:,4);  
notes=LogTXT(:,10);

%delete unnecessary variables.
clear filetype ext RangeIn headers display_opt outdat;

%% Step 3:   perform visual checks on input files
%Example: Check that the same wavelengths were used for all sample EEMs
%The plots below should produce a single line representing all scans.
figure, 
subplot(2,1,1),plot(Exmat'),
subplot(2,1,2),plot(Emmat),
title('Ex and Em wavelengths for samples')
legend(num2str((1:size(Emmat,1))'))
%The name of the ith sample loaded is filelist_eem(i,:)

%Example: Check that the same wavelengths were used for all Raman scans
%If no plot is produced, the wavelengths were the same - no further
%checking is needed.
if isempty(wave_R);
    figure, plot(W_R'),
    title('wavelengths for Raman scans')
    legend(num2str((1:size(W_R,1))')); %number of samples = no rows in W_R
    %The name of the ith sample loaded is filelist_R(i,:)
end

%Example: Plot 3D files (e.g. EEMs), one by one
This3D=X; %plot the EEMs
for i=1:size(This3D,1)
    figure, 
    contourf(Exmat(i,:),Emmat(:,i),squeeze(X(i,:,:)),30),
    c_peak=X(i,Em==450,Ex==350); %height of peak at 350/450
    caxis([0 c_peak*3]) %change the range of the contour axis (ignore scatter peaks)
    title(filelist_eem(i,:),'interpreter','none')
    pause,close
end

%% Step 3:   pair samples and scans
cd 'C:\Data\Experiments\Cruises' %error messages from matching will be stored in this folder
%[X2m,FL1,FL2,PL]=MatchSamples(filelist1,filelist2,PairedList,X1,X2);
% see FAQs for tips on dealing with missing data

[Sr,FL1,FL2,PLr]=MatchSamples(filelist_eem,filelist_R,Pair_EEM_R,X,S_R);
[Sabs,FL1,FL2,PLabs]=MatchSamples(filelist_eem,filelist_abs,Pair_EEM_abs,X,S_abs);
[Xbk,FL1,FL2,PLbk]=MatchSamples(filelist_eem,filelist_b,Pair_EEM_bk,X,X_b);
[Sqs,FL1,FL2,PLqs]=MatchSamples(filelist_eem,filelist_qs,Pair_EEM_qs,X,S_qs);
[Sqs_R,FL1,FL2,PLqs_R]=MatchSamples(filelist_qs,filelist_R,Pair_QS_R,S_qs,S_R);
[Sbk_R,FL1,FL2,PLbk_R]=MatchSamples(filelist_b,filelist_R,Pair_BLK_R,X_b,S_R);

%% Step 4:   define raman integration RangeIn
% if using the default Raman integration range in FDOMcorr:
RamOpt=[] %default Raman boundaries at 350 excitation: [350 381 426]; 

% alternatively, if using empirically determined raman boundaries (Murphy
% 2011, Appl. Spec. 65(2) 233-236), then define RamEx = excitation wavelength
% (landa) for Raman scans and execute the code below:
RamMat=[wave_R;S_R];
[IR,IRmed,IRdiff] = RamanIntegrationRange(RamMat,filelist_R,RamEx)

%Choose the integration range, based on the above results
%for example, if the best integration range is the median value IRmed, then:
RamOpt=[RamEx IRmed]; 

%% Step 5:   switch off IFE corrections for some samples
% (optional)
% IFE corrections invalid when absorbance & fluorescence pathlengths <>1cm
% in this case, create a new Absorbance file A_1cm to use in ReadInEEMs.
% see FAQs for more information.
[Sa,FL1,FL2,PLpatha]=MatchSamples(filelist_eem,LogTXT(:,1),Pair_EEM_log,X,pathlen_a); % sort pathlen_a to match filelist_eem
[Sf,FL1,FL2,PLpathf]=MatchSamples(filelist_eem,LogTXT(:,1),Pair_EEM_log,X,pathlen_f); % sort pathlen_f to match filelist_eem
A_on_path=ismember(Sf,1).*ismember(Sa,1);
Sabs_1cm=repmat(A_on_path,[1 length(wave_abs)]).*Sabs;
A_1cm=[wave_abs;Sabs_1cm];

% (optional) switch off IFE corrections for selected files with flagcode = 99;
% create a new Absorbance file A_val to use in ReadInEEMs.
% see FAQs for more information.
[Sflag,FL1,FL2,PLflag]=MatchSamples(filelist_eem,LogTXT(:,1),Pair_EEM_log,X,flagcode); % sort flagcode to match filelist_eem
[Snotes,FL1,FL2,PLnotes]=MatchSamples(filelist_eem,LogTXT(:,1),Pair_EEM_log,X,notes); % sort notes to match filelist_eem
A_on_flag=ones(size(Sflag,1),1)-ismember(flagcode,99);
Sabs_flag99=repmat(A_on_flag,[1 length(wave_abs)]).*Sabs;
A_val=[wave_abs;Sabs_flag99];

%% Step 6:   correct the EEMs
%Attach wavelengths to first row of 2D scans only
B=Xbk; %3D dataset of blank EEMs
W=[wave_R;Sr]; %2D Matrix of matched Raman scans
A=[wave_abs;Sabs]; %2D Matrix of matched absorbance scans.(optional - use A_1cm or A_val instead of A).
T=[wave_R;Sbk_R]; %2D Matrix of Raman scans corresponding to blanks
Q=Sqs; %Vector of uncorrected slopes of QS dilution series
Qw=[wave_R;Sqs_R]; %2D Matrix of Raman scans matched to QS dilution series

%Apply corrections: see three examples below. FDOMcorrect help has further examples.
%example 1 - full correction including blank subtraction with output in both RU and QSE
[XcRU Arp IFCmat BcRU XcQS QS_RU]=FDOMcorrect(X,Ex,Em,Emcor,Excor,W,RamOpt,A,B,T,Q,Qw);
%example 2 - full correction including blank subtraction with output only in RU
[XcRU Arp IFCmat BcRU XcQS QS_RU]=FDOMcorrect(X,Ex,Em,Emcor,Excor,W,RamOpt,A,B,T,[],[]);
%example 3 - no blank subtraction, no IFE correction, with output only in RU
[XcRU Arp]=FDOMcorrect(X,Ex,Em,Emcor,Excor,W,RamOpt);

%% Step 7:   (Optional) compensate for sample dilution
%If samples have been diluted, adjust fluorescence and absorbance back to 100% strength
%XcRU100,XcQS100 and A100 are the dilution compensated matrices
%Adjustment is based on values in the vector DilFac (from the Sample Log)

% First, sort DilFac to match filelist_eem
[Sdf,FL1,FL2,PLdf]=MatchSamples(filelist_eem,LogTXT(:,1),Pair_EEM_log,X,DilFac); % sort DilFac to match filelist_eem

% Adjust the corrected EEM matrices and the raw absorbance files. 
% Data previously written to Excel are not adjusted.
XcRU100=XcRU.*repmat(Sdf,[1 size(XcRU,2) size(XcRU,3)]);
XcQS100=XcQS.*repmat(Sdf,[1 size(XcRU,2) size(XcRU,3)]);
A100=[wave_abs;Sabs.*repmat(Sdf,[1 size(Sabs,2)])];

% Compare data before and after adjustment.
fprintf('\n\n Summary results for Ex/Em = 350/450 \n')
disp('    diluted            DF        restored')
disp([squeeze(XcRU(:,Em==450,Ex==350)) Sdf squeeze(XcRU100(:,Em==450,Ex==350))])

fprintf('\n\n Summary results for absorbance A254  \n')
disp('    diluted            DF        restored')
disp([A(:,(wave_abs==254)) Sdf A100(2:end,(wave_abs==254))])

%% Step 8:   save the data
% The easiest way is to delete any unwanted variables then save your entire workspace,
% especially because automatically saved variables may have different names to the ones you 
% gave them in this workspace. 
%
%If you chose to automatically output data from ReadInEEMs and FDOMcorrect, note that:
%(1) variables saved in RawData_FDOM.mat are always called: X Emmat Exmat filelist outdata 
%(2) variables saved in CorrData_FDOM.mat are always called: XcRU XcQS Arp IFCmat BcRU QS_RU X Ex Em Emcor Excor W RamOpt A B T Q Qw
%(3) each time these m-files are run they will overwrite any existing data files in the same folder 
%     with the same name (i.e. CorrData_FDOM and RawData_FDOM).
%(4) if you used ReadInEEMs to import both samples and blanks, then a file called RawData_FDOM.mat 
%    will be written to each the sample and the blank directory, and each will contain variables with the same names
%(5) automatically saved variables may have different names to the ones you gave them in this workspace. 
%
%Variables you might want that are not automatically saved include: XcRU100 XcQS100 A100 A_1cm A_val
%and the matched up filelists: filelist_eem PLr pathlen_abs PLbk PLqs PLqs_R PLbk_R

% Step 9: further processing
% There are numerous possibilities for further processing of these data. Below
% are some alternative sets of code to ready the data for PARAFAC analysis using:
%(A) Stedmon & Bro's PARAFAC tutorial (Limnol. Oceanogr.: Methods 6, 2008), using the NWAY toolbox
%(B) The PLS toolbox for MATLAB (Eigenvector Inc.)
%(C) Bro and Vidal's 2010 EEMIZER (PARAFAC based on the PLS toolbox for MATLAB)
% Note that scatter must be removed before modelling.
% Scatter removal is incorporated in (A) and (C)

%9A - Stedmon and Bro
%The corrected EEMs are stored in a data structure called "OriginalData".
OriginalData.Ex = Ex';                  %row vector => column vector
OriginalData.Em = Em;                   %col vector => column vector
OriginalData.X=XcRU; 
OriginalData.nEx=length(Ex);            %number of Excitation wavelengths
OriginalData.nEm=length(Em);            %number of Emission wavelengths
OriginalData.nSample=size(X,1);         %number of samples
OriginalData.filenames=filelist_eem;    
%now proceed as per the tutorial using your OriginalData

%9B,C - PLS toolbox (commercial software from Eigenvector Inc.)
%The following code uses the DataSet object in the PLS toolbox which is
%incompatible with a dataset object in the Mathworks Statistics Toolbox. Find information on this
%problem by typing 'Do PLS_Toolbox functions conflict with Mathworks Toolboxes?' 
%into the FAQ box at http://www.eigenvector.com/faq/

%check that MATLAB finds the right dataset object:
which dataset 
%if the file location is within the PLS toolbox, then continue
Xd = dataset(XcRU);     
Xd.title{1}='Samples';
Xd.title{2}='Em';
Xd.title{3}='Ex';
Xd.axisscale{2}=Em;
Xd.axisscale{3}=Ex;
Xd.label{1,1}=cellstr(filelist_eem);
Xd.labelname{1,1}='filenames';
%now perform eemizer modelling on Xd, OR
%remove scatter and model Xd with the PLS toolbox.
