function model=scream_mod(varargin)
%SCREAM for multi-way regression with shift and shape changes
% 
%Calculates a SCREAM model of desired complexity, as described in F. Marini
%& R. Bro, "SCREAM: a novel method method for multi-way regression problems 
%with shifts and shape changes in one mode", Chemometr. Intell. Lab. Syst.
%SCREAM is a multi-way regression method which allows to build calibration 
%models on tensors presenting shifts or shape changes along one of the
%modes. This is accomplished by coupling elements from PARAFAC2
%decomposition with Multiway Covariate Regression.
% 
%More specifically, the solution is found by minimizing the global least
%squares criterion:
% 
%            alpha*||X-CP'||^2+(1-alpha)*||Y-CR||^2         (1)
% 
%where X is the matricized version of the independent multiway array, C is 
%the matrix of X loadings along the third (sample) mode, P is the matrix
%containing the B and A loadings resulting from the PARAFAC2-like
%decomposition of X and R is the matrix of Y loadings, which are the
%regression coefficients in the score space. The constant alpha modulates
%the extent to which the two blocks contribute to the decomposition. 
%
%I/O:   model=scream_mod(X,Y,F, opts)
% 
%where 
% 
%X  is the independent multi-way data array, containing shifts or
%    shape changes. It can be input either as data cube arranged so that the
%    third mode is the sample one, the second the one where shifts are present
%    and the first one is the remaining, or, to allow also for the possibility 
%    of profiles of different length, as a cell array of dimension K, where K
%    is the number of samples. In the latter case, each element of K is a
%    matrix corresponding to the profile measured on a sample, and it is
%    arranged so that the second dimension is the one containing shifts, shape
%   changes, etc...
% 
%Y  is the vector or matrix of dependent variables
%
% F  is the desired number of SCREAM components to be calculated by the model
%
% opts is a structure containing all the relevant information for running
%     the algorithm. Its field are:
%
% - init:   defines the preferred kind of initialization. At the moment only
%           SVD-based initialization (=1), which is the default, and random
%           initialization (=2) are implemented.
%- alpha:   The value of alpha; alpha=1 indicates that only the X-block
%           governs the decomposition, alpha=0 means that only Y is
%           contributing. The default is alpha=0.5.
%- crit:    defines the convergence criterion; e.g., a value of 1e-6 means that
%           convergence is achieved and the algorithm is stopped when the relative
%           decrease of the least squares criterion in equation 1 is less
%           than 10^-6.
%- maxit:   Maximum number of iterations (a stopping criterion which is
%           alternative to the previous one). Default is 5000
%- prntlag: Defines how often the partial results are displayed on screen.
%           A value of 200 (the default) indicates that partial results are
%           displayed every 200 iterations.
%- plots:   Allows to select whether one wants a final plot ('on') or not ('off')
%
%The default options can be accessed by typing opts=scream_mod('options').
%If opts is not input as variable or corresponds to an empty variable, the
%default options will be used. 
%
%The results and the model parameters are collected in the structure model.
%In particular, the loadings are saved as cell array in model.Xloads:
%
%ATTENTION! Differently than with the input array X, to be consistent with
%other multi-way models, here model.Xloads{1} contains the loadings along
%the sample mode, model.Xloads{2} contains the loadings along the shifted
%mode (in turn accessible through model.Xloads{2}.P{i} - individual profiles
%and model.Xloads{2}.H) and model.Xloads{3} those along the remaining mode.
%The value of the global loss function and the contributions of X and Y
%blocks are reported in model.fit, while beta, i.e. the renormalized value of alpha,
%which takes into account the norm of the different blocks, is saved in
%model.beta. 
%
%The Y loadings (i.e. the regression coefficients linking the X scores to the 
%predicted Y are stored in model.Yloads, while the values of predicted Y based 
%on the model are saved in model.Ypred. Model bias and RMSEC for each Y
%variable are stored in model.bias and model.rmse, respectively.
%Lastly, the options used for model building are saved in model.opts
%
% I/O:   model=scream_mod(X,Y,F, opts)


if nargin==1&&strcmp(varargin{1}, 'options')   %Allows to retrieve standard options just by writing scream_mod('options');
    init=1; opts.init=init;                    % Type of initialization
    alpha=0.5; opts.alpha=alpha;               % Value of alpha
    crit=1e-7; opts.crit=crit;                 % Convergence criterion 
    maxit=5000; opts.maxit=maxit;              % Maximum number of iterations
    prntlag=200;opts.prntlag=prntlag;          % Display output every prntlag iterations
    plots='on'; opts.plots=plots;              % Governs plotting routine
    model=opts;
    
else
    X=varargin{1};                            %Independent block (X array)
    y=varargin{2};                            %Dependent block (Y array) 
    F=varargin{3};                            %Number of SCREAM components to be computed
    
    if nargin==3 || isempty(varargin{4})            %If no opts is input, then use the default options
        init=1; opts.init=init;
        alpha=0.5; opts.alpha=alpha;
        crit=1e-7; opts.crit=crit;
        maxit=5000; opts.maxit=maxit;
        prntlag=200;opts.prntlag=prntlag;
        plots='on'; opts.plots=plots;

    else
        opts=varargin{4};                    %Structure with options  
        init=opts.init;
        alpha=opts.alpha;
        prntlag=opts.prntlag;
        crit=opts.crit;
        maxit=opts.maxit;
        plots=opts.plots;
    end


if ~iscell(X)                  %If X is not input as cell array, then it needs to be transformed to it.
  x=cell(1,size(X,3));  
  for k = 1:size(X,3)
    x{k} = X(:,:,k);
  end
  X = x;
  clear x
end
I = size(X{1},1);              %Number of variables in the first ("unshifted") mode  
K = length(X);                 %Number of samples


ssy0=sum(sum(y.^2));           %Squared norm of Y (needed for the normalization of alpha)

if size(y,1)~=K,               %Check for consistency of dimensions
   fprintf('ERROR: The number of observations in X and y differ!! \n');
   keyboard
end

if init==1                      %Initialization of the loading matrices
    disp(' SVD based initialization')
    XtX=X{1}*X{1}';
    for k = 2:K
        XtX = XtX + X{k}*X{k}';
    end
    [A,~,~]=svd(XtX,0);  
    A=A(:,1:F);
    C=ones(K,F)+randn(K,F)/10;
    H = eye(F);
elseif init==2
    disp(' Random initialization')
    A = rand(I,F);
    C = rand(K,F);
    H = eye(F);
end

% Initialize Py
Py=pinv(C)*y;

fit=10^6;
fprintf('function value at Start is %12.8f \n',fit)

iter=0;
fitold=2*fit;


%Preallocation of variables
P=cell(1,K);
Xpcov=zeros(I,F,K);
f=nan(maxit);

while abs((fit-fitold)/fitold)>crit&&iter<maxit&&fit>10*eps  %Different convergence criteria (relative change in fit, max iterations, absolute fit)
   iter=iter+1;
   fitold=fit;

   
    % Compute P and create 3w array Xpcov
    for k = 1:K
        Qk=X{k}'*(A*diag(C(k,:))*H');
        P{k}= Qk*psqrt(Qk'*Qk);
        Xpcov(:,:,k) = X{k}*P{k};
    end

   
   % Update W and C (loadings in the sample mode)
   X3w=permute(Xpcov,[3,2,1]);
   Xha=reshape(X3w,K,I*F);
   if iter==1
       ssx0=sum(sum(Xha.^2));      %Sum of squares of the X block
       beta1=alpha./ssx0;
       beta2=(1-alpha)./ssy0;
       beta=beta1./(beta1+beta2);   %Normalized coefficient beta (beta is normalized to take into account the different variance in the blocks)
   end
   
   Z=kron(A,H);
   Z=Z(:,1:F+1:F*F);
   U=[sqrt(beta)*Z' sqrt(1-beta)*Py];
   V=[sqrt(beta)*Xha sqrt(1-beta)*y];
   W=pinv(Xha)*V*pinv(U);
   C=Xha*W;
   
   CC=C'*C;
   AA=A'*A;
   
   % Update Py
   Py=C\y;
   
   % Update H
   FF=AA.*CC;
   X3w=permute(X3w,[2,3,1]);
   Xac=reshape(X3w,F,I*K);
   Z=kron(C,A);
   Z=Z(:,1:F+1:F*F);
   H=Xac*Z/FF;
   HH=H'*H;
   
   % Update A
   FF=HH.*CC;
   X3w=permute(X3w,[2,3,1]);
   Xch=reshape(X3w,I,K*F);
   Z=kron(H,C);
   Z=Z(:,1:F+1:F*F);
   A=Xch*Z/FF;
   
   
   
   % Evaluate
   lossX=sum(sum((Xch-A*Z').^2));
   lossY=sum(sum((y-C*Py).^2));
   fit=beta*lossX+(1-beta)*lossY;
   if rem(iter,prntlag)==0
      fprintf('f = %12.8f after %g iters; lossX= %10.6f lossY=%10.6f\n',fit,iter,lossX,lossY)
   end
   f(iter)=fit;
end


model.fit.fit=f;
model.fit.lossX=lossX;
model.fit.lossY=lossY;
model.beta=beta;
model.opts=opts;
model.Xloads{2}.H=H;
model.Xloads{2}.P=P;
model.Xloads{3}=A;
model.Xloads{1}=C;
model.Yloads=Py;
model.Xweights=W;
model.Ypred=C*Py;
model.bias=mean(y-model.Ypred);
model.rmse=sqrt(sum((y-model.Ypred).^2)./K);

    
    
fprintf('function value is %12.8f after %g iterations \n',fit,iter)

if strcmp(plots,'on')
    %First figure: decomposition of the X block
    
    figure
    subplot(2,2,1)
    plot(C, 'LineWidth', 1.5)
    axis tight
    xlabel('Sample Index')
    ylabel('SCREAM components')
    title('Loadings along the sample mode')
    
    subplot(2,2,2)
    if F>1
        plot(C(:,1), C(:,2), '.r', 'MarkerSize', 8)
        xlabel('SCREAM comp.1')
        ylabel('SCREAM comp.2')
        title('Scores plot')
    else
        bar(C(:,1))
        xlabel('Sample Index')
        ylabel('SCREAM comp.1')
        title('Scores plot')
    end
    axis tight
    
    subplot(2,2,3)
    for i=1:K
        plot(P{i}*H, 'LineWidth', 1.5), hold on
    end
    hold off
    axis tight
    xlabel('Variable Index')
    ylabel('SCREAM components')
    title('Loadings along the shifted mode')
    
    
    subplot(2,2,4)
    plot(A, 'LineWidth', 1.5)
    axis tight
    xlabel('Variable Index')
    ylabel('SCREAM components')
    title('Loadings along the third mode')
    
    %Second figure: predictions
    ny1=ceil(sqrt(size(y,2)));    %1st dimension for subplot
    ny2=ceil(size(y,2)./ny1);     %2nd dimension for subplot
    figure
    for j=1:size(y,2)
        subplot(ny2,ny1,j)
        plot(y(:,j), model.Ypred(:,j), '.r', 'MarkerSize', 8)
        axis tight
        xlabel('Y measured')
        ylabel('Y predicted')
        title(['Y variable ' num2str(j)])
    end
    
    
    
end
    

end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function X = psqrt(A,tol)

   % Produces A^(-.5) even if rank-problems

   [U,S,V] = svd(A,0);
   if min(size(S)) == 1
     S = S(1);
   else
     S = diag(S);
   end
   if (nargin == 1)
     tol = max(size(A)) * S(1) * eps;
   end
   r = sum(S > tol);
   if (r == 0)
     X = zeros(size(A'));
   else
     S = diag(ones(r,1)./sqrt(S(1:r)));
     X = V(:,1:r)*S*U(:,1:r)';
    end

