function model=scream_cv(varargin)
%Calculates cross-validated SCREAM models and chooses the optimal
%complexity and the value of alpha as the one leading to the minimum error).
%Performs a grid search over predefined values of alpha and number of components
%and selects the best combination.
%
%     I/O: model=scream_cv(X,Y,options)
%
%  where X is the independent multi-way data array, containing shifts or
%shape changes. It can be input either as data cube arranged so that the
%third mode is the sample one, the second the one where shifts are present
%and the first one is the remaining, or, to allow also for the possibility 
%of profiles of different length, as a cell array of dimension K, where K
%is the number of samples. In the latter case, each element of K is a
%matrix corresponding to the profile measured on a sample, and it is
%arranged so that the second dimension is the one containing shifts, shape
%changes, etc...
% Y is the vector or matrix of dependent variables
% options is a structure of options containing the following fields:
%   - cv.cvtype:    Type of cross-validation. Can be 'loo' (Leave-one-out),
%                   'syst123' (Venetian blinds), 'syst111' (Contiguous
%                   blocks) or 'manual' (Custom)
%   - cv.segments:  Number of cross-validation splits. If 'manual' is selected
%                   as cvtype, segments is a vector having the same length
%                   as the number of samples and indicating, for each
%                   individual, the corresponding cancelation group (For
%                   example, if segments=[1 1 2 3 2 4 3 4 1], the 1st, 2nd
%                   and 9th samples will be put in the first cancelation
%                   group, the 3rd and the 5th in the second one and so on)
%   - alphaint:     Vector containing the values of alpha to be tested
%   - maxfact:      Maximum number of SCREAM components to be tested
%   - cvplots:      Governs the plotting routine for CV results (default='on')
%   - screamopts:   A structure containing the options for building the
%                   various SCREAM models (see the help of scream_mod for
%                   further details). 
% 
%Default options can be called writing scream_cv('options').
%If options is not input as variable, the
%default options will be used. 
%


if nargin==1&&strcmp(varargin{1}, 'options')    %Allows to retrieve standard options just by writing scream_mod('options');
    opts.cv.cvtype='syst123';               	%Venetian blinds
    opts.cv.segments=5;                         % 5 cancelation groups
    opts.alphaint=0:0.1:1;                      % grid search for alpha in steps of 0.1
    opts.maxfact=1;                             % Maximum number of SCREAM components explored 
    opts.cvplots='on';                          % Governs plotting routine (CV results)
    opts.screamopts.init=1;                     % Type of initialization
    opts.screamopts.alpha=0.5;                  % Value of alpha (To be updated during grid search);
    opts.screamopts.crit=1e-7;                  % Convergence criterion 
    opts.screamopts.maxit=5000;                 % Maximum number of iterations
    opts.screamopts.prntlag=200;                % Display output every prntlag iterations
    opts.screamopts.plots='off';                % Governs plotting routine (of the individual SCREAM models)
    model=opts;
    
else
    X=varargin{1}; 
    y=varargin{2};
    if nargin==3
        opts=varargin{3};
        cvtype=opts.cv.cvtype;
        segments=opts.cv.segments;
        alphaint=opts.alphaint;
        maxfact=opts.maxfact;
        cvplots=opts.cvplots;
        screamopts=opts.screamopts;             %Options for the calculation of the individual SCREAM models)
    else
        opts.cv.cvtype='syst123'; cvtype=opts.cv.cvtype;    %Default values are use if opts is not input	
        opts.cv.segments=5; segments=opts.cv.segments;      
        opts.alphaint=0:0.1:1; alphaint=opts.alphaint; 
        opts.maxfact=1; maxfact=opts.maxfact;
        opts.cvplots='on'; cvplots=opts.cvplots;
        opts.screamopts.init=1;                     
        opts.screamopts.alpha=0.5;                  
        opts.screamopts.crit=1e-7;                  
        opts.screamopts.maxit=5000;                 
        opts.screamopts.prntlag=200;                
        opts.screamopts.plots='off'; screamopts=opts.screamopts;  
    end
    
    
    if ~iscell(X)                  %If X is not input as cell array, then it needs to be transformed to it.
        x=cell(1,size(X,3));  
        for k = 1:size(X,3)
             x{k} = X(:,:,k);
        end
        X = x;
        clear x
    end
    ntot = length(X);                 %Total number of samples in the calibration set

    
    if strcmp(cvtype, 'loo')  %Loo cross-validation corresponds to syst123 with N segments 
        cvtype='syst123';
        segments=ntot; 
    elseif strcmp(cvtype, 'manual')
        cvsplits=segments;
        segments=max(cvsplits);
    end

    rmsecv=zeros(length(alphaint),maxfact);    %Initialization of the error matrix
    model.CV.splits=cell(1,segments);             %Structure field which stores the samples in each split
    model.CV.Ypreds=zeros(ntot,size(y,2),length(alphaint), maxfact);   %Stores the CV predictions as a function of the different parameter values
    
    for j=1:segments %Beginning of the crossvalidation loop

        switch cvtype  
            case 'syst123'    %venetian blind cross-validation 
                t=j:segments:ntot; %Validation set
                m=1:ntot; m(t)=[]; %Calibration set

            case 'syst111'    %contiguous blocks 
                ns=ceil(ntot./segments);  %number of samples in each group
                if j==segments
                    t=(j-1)*ns+1:ntot; %Validation set (last segment)
                else
                    t=(j-1)*ns+1:j*ns; %Validation set (all other segments)
                end
                m=[1:(j-1)*ns ns*j+1:ntot]; %Calibration set
            case 'manual'
                t=find(cvsplits==j); %Validation set is selected according to the cvsplits variable
                m=1:ntot; m(t)=[]; %Calibration set
        end

        Xm=X(m); Ym=y(m,:); %Calibration set
        Xt=X(t); Yt=y(t,:); %Validation set

        for a1=1:length(alphaint)         %loop over the values of alpha
            screamopts.alpha=alphaint(a1);
            for a2=1:maxfact     %loop over the number of SCREAM components
                sm=scream_mod(Xm,Ym,a2,screamopts);   %Calculates an a2 components SCREAM model on the jth calibration set
                sp=scream_pred(Xt,sm, 'off');                 %tests the model on the jth cancelation group
                rmsecv(a1,a2)=rmsecv(a1,a2)+sum(sum((Yt-sp.Ypred).^2));  %error matrix is updated
                model.CV.Ypreds(t,:,a1,a2)=sp.Ypred;     % CV  predictions are stored in the output
            end
        end
        model.CV.splits{j}=t;     %index of test samples is stored in the output
    end
    rmsecv=sqrt(rmsecv./ntot);    %Sum of squares is transformed to RMSECV
    model.CV.rmse=rmsecv;
    
    [er1,alpha1]=min(rmsecv);
    [~,optcomp]=min(er1);     %optimal number of SCREAM components
    alphaopt=alphaint(alpha1(optcomp));         %optimal Calue of alpha
    model.CV.alphaopt=alphaopt;
    model.CV.optcomp=optcomp;
    model.CV.options=opts;
    
    %Plotting routine
    if strcmp(cvplots,'on')
        figure
        if maxfact==1
            bar(alphaint, rmsecv)
            xlabel('\alpha')
            ylabel('RMSECV')
            legend(num2str((1:maxfact)'))
            plot(alphaint,rmsecv)
        else
            subplot(1,2,1)
            mesh(alphaint, 1:maxfact, rmsecv');
            xlabel('\alpha')
            ylabel('number of SCREAM components')
            zlabel('RMSECV')
            title('Error surface')
        
            subplot(1,2,2)
            bar(alphaint, rmsecv)
            xlabel('\alpha')
            ylabel('RMSECV')
            legend(num2str((1:maxfact)'))
            title('RMSECV plot - bars')
        
        end
        
        screamopts.plots='on';
    end
    
    screamopts.alpha=alphaopt;
    
    mfin=scream_mod(X,y,optcomp,screamopts);   %Calculation of the optimal model;
    
    model.OptModel=mfin;

end


