function [mzloads elloads]=loads2chrom(filename,model,el,mz,timeunits,min_mz,vec)
%Export PARAFAC, PARAFAC2 and MCR model loadings to *.mpl format
%negative parts of elution spectra are set to zero
%m/z below min_mz are ignored.
%
%Useage: [mzloads elloads]=loads2chrom(filename,model,el,mz,timeunits,min_mz,vec)
%
%INPUTS
% filename - Name of *.mpl file that will be created in the current directory
% model - PARAFAC,PARAFAC2 or MCR models (other models also accepted if
%           they can be presented in the generic format described below)
%         Accepts PARAFAC, PARAFAC2 and MCR model structures compatible with the PLS toolbox for MATLAB
%         Also accepts model loadings in the following generic format:
%           model={C;S;nSamples;modtype}
%           with C=contributions,S=mz spectra, 
%           matrix C is a single elution scan, or the column-augmented matrix [D1;D2;...;Dk) with Row Length: nSamples*length(el); #Columns: nFac 
%           matrix S has row length: length(mz); no. columns: nFac 
%           nSamples = no. of samples (nSamples must be a divisor of C)
%           modtype = short model description, e.g 'MCR','MCR-ALS','MCR-non-neg',etc.
% el- elution time axis values (the list of times)
% mz - m/z axis values (the list of ions, e.g. 35:550)
% timeunits - 'minutes', 'seconds', 'milliseconds'
% min_mz (optional)- minimum value of m/z; data below this will be ignored (default = 0)
% vec (optional, for PARAFAC2) = a vector of numbers, or else the word 'max', see below
%       For PARAFAC2, you can choose to export elution loadings from several samples or just the one with the largest peak (default).
%       e.g. [1 3] - first and third samples only
%       e.g. [1:NoSamples] - all samples
%       e.g. 'max' - elution profile from the sample which has the largest peak (with maximum score)
%        note that for MCR and Generic models, the elution loadings will be taken from the sample with the largest peak
%        while for PARAFAC, there is only one set of elution loadings representing all samples
%OUTPUTS
% mzloads - mz loadings, with comp1 in mzloads.a, comp2 in mzloads.b,.. etc   
% elloads - elution loadings, a matrix of size el x nFac
% also, an mpl file is created that can be imported to the OpenChrom chromatography software, available at http://www.openchrom.net/.
%
%EXAMPLES
%[mzloads elloads]=loads2chrom('P1frame11.mpl',P1model,el_axis,mz_axis,'minutes')
%[mzloads elloads]=loads2chrom('P2frame11.mpl',P2model,el_axis,mz_axis,'minutes',0,[1 3])
%[mzloads elloads]=loads2chrom('MCRframe11.mpl',{C;S;32;'MCR-ALS'},el_axis,mz_axis,'seconds')
%
% ----- REFERENCE -----
% K.R. Murphy, P. Wenig, G. Parcsi, T. Skov, R.M. Stuetz. Characterizing odorous emissions using new
% software for identifying peaks in chemometric models of gas chromatography - mass spectrometry datasets. 
% Chem Intel Lab Sys (in press). DOI: 10.1016/j.chemolab.2012.07.006
%
% ----- COPYRIGHT -----
% Copyright 2012 KR Murphy
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation version 3 of the License <http://www.gnu.org/licenses/>
%
% KR Murphy
% Dept. Civil & Environmental Engineering
% The University of New South Wales
% krm@unsw.edu.au

%%
%Perform checks on input variables
error(nargchk(5,7,nargin))
repeatoption=false; %default is to output a single elution peak
if size(el,1)<size(el,2),el=el';end %el is a column vector
if size(mz,2)<size(mz,1),mz=mz';end %mz is a row vector

if nargin >= 6
    if min_mz<0, error('m/z profile must be positive. specify a zero or positive value for min_mz'),end
    if nargin >=7; %PARAFAC2 with all scans outputed
        if isstruct(model)==false;
            fprintf('\n\nElution loadings for first sample used for an MCR model....\n\n');
            fprintf('\n ignoring the value of ''vec''....\n\n');
        elseif isstruct(model)==true;
            if isnumeric(vec)
                if strcmp(model.modeltype,'PARAFAC')==true
                    fprintf('\n\nValue of "vec" ignored when exporting a PARAFAC-1 model....\n\n');
                elseif strcmp(model.modeltype,'MCR')==true
                    fprintf('\n\nElution loadings for first sample used for an MCR model....\n\n');
                elseif strcmp(model.modeltype,'PARAFAC2')==true
                    checkvec(model.loads{3},vec) %check vec is compatible with NoSamples
                    if max(size(vec))>1,
                        repeatoption=true;
                    end
                end
            elseif strcmp(vec,'max')==true
                repeatoption=false;
            else error('value of input variable ''vec'' is not recognised')
            end
        end
    end
else
    min_mz=0; %default minimum m/z loading
end

%convert time to milliseconds
if strcmp(timeunits,'hours')==true
    el=round(el*60*60*1e3);
elseif strcmp(timeunits,'minutes')==true
    el=round(el*60*1e3);
elseif strcmp(timeunits,'seconds')==true
    el=round(el*1e3);
elseif strcmp(timeunits,'milliseconds')==true
else
    error('recognised units are: hours, minutes, seconds and milliseconds')
end

%% Obtain Model Loadings
AtoZ=char(97:122)'; %a:z ascii
mzax=[];mzloads=[];

if isstruct(model)==true %PLS toolbox model structure
    if strcmp(model.modeltype,'PARAFAC')==true
        A = model.loads{1};             %A, samples
        elloads = model.loads{2};       %B, elution times
        elloads(elloads<0)=0;           %set negative elution profiles to zero
        C = model.loads{3};             %C, m/z
        nFac=size(C,2);                 %number of factors
        for n=1:nFac
            mz_above_threshold=mz(C(:,n)>min_mz)';
            C_above_threshold=C(C(:,n)>min_mz,n);
            eval(['mzax.' AtoZ(n) ' = mz_above_threshold;']);                            %exclude m/z below threshold
            eval(['mzloads.' AtoZ(n) ' = [mz_above_threshold C_above_threshold];']);     %exclude m/z below threshold
        end
        fprintf(['\nM/Z loadings above threshold of ' num2str(min_mz) '\n']);
        disp(mzloads);
        %fprintf('\nElution loadings are of size:\n');
        %disp(size(elloads));
    elseif strcmp(model.modeltype,'PARAFAC2')==true
        A = model.loads{3};             %A, samples
        [yi,ii]=max(A);                 %iith sample has highest score
        
        if nargin>6,checkvec(A,vec),end
        P = model.loads{1}.P;           %B.P elution times
        H = model.loads{1}.H;           %B.H elution times
        C = model.loads{2};             %C, m/z
        nFac=size(C,2);                 %number of factors
        for n=1:nFac
            mz_above_threshold=mz(C(:,n)>min_mz)';
            C_above_threshold=C(C(:,n)>min_mz,n);
            eval(['mzax.' AtoZ(n) ' = mz_above_threshold;']);                            %exclude m/z below threshold
            eval(['mzloads.' AtoZ(n) ' = [mz_above_threshold C_above_threshold];']);     %exclude m/z below threshold
        end
        L=NaN*ones(size(P{1},1),size(A,2),size(A,1));
        for i = 1:length(P)              %length(P) = number of elution times
            L(:,:,i)=P{i}*H;             %L, elution mode loadings
        end
        
        if repeatoption==false;
            B=NaN*ones(size(A,2),size(P{1},1));
            %size(A),size(P{1}),size(B)
            for n=1:nFac
                B(n,:,:)=squeeze(L(:,n,ii(n)));     %elution mode loading for sample with highest score
            end
            elloads=B';
            elloads(elloads<0)=0;                   %set negative parts of elution profiles to zero
        elseif repeatoption==true;
            B3d=L;
            %size(B3d)
        end
        fprintf(['\nM/Z loadings above threshold of ' num2str(min_mz) '\n']);
        disp(mzloads);
    elseif strcmp(model.modeltype,'MCR')==true
        disp('MCR by PLStoolbox')
        Bstar = model.loads{1};                     %elution profiles scaled by concentration, B*
        nEl=length(el);
        lenBstar=size(Bstar,1);
        %pause
        nSamples=lenBstar/nEl;
        if (nSamples-floor(nSamples))>1e-16,
            disp(['No. elution times=' num2str(nEl) '; No. samples = ' num2str(lenBstar) '/' num2str(nEl) ' is not a whole number...!'])
            fprintf('\nlength of axis el is incompatible with the length of model elution loadings (first mode)\n')
            error('Could not calculate number of samples in MCR model')
        else 
            fprintf(['\n Number of samples detected is ' num2str(nSamples) '\n']);
        end
        nFac=size(Bstar,2);
        C = model.loads{2};                         %C, m/z
        elloads=NaN*ones(nEl,nFac);
        
        %use the sample with the largest contribution for each component for the elution profile
        [imax jmax]=max(Bstar);
        t=reshape(1:size(Bstar,1),[nEl,nSamples]); %a list of indices only, not data
        for n=1:nFac
            [dummy S_max]=find(t==jmax(n));
            data=Bstar(t(:,S_max),n);
            %plot(Bstar), hold on, plot(floor(lenBstar/2),imax(n),'ro')
            elloads(:,n)=data./repmat(max(data),[length(data) 1]);                      %normalized elution profiles
            mz_above_threshold=mz(C(:,n)>min_mz)';
            C_above_threshold=C(C(:,n)>min_mz,n);
            eval(['mzax.' AtoZ(n) ' = mz_above_threshold;']);                           %exclude m/z below threshold
            eval(['mzloads.' AtoZ(n) ' = [mz_above_threshold C_above_threshold];']);    %exclude m/z below threshold
        end
        elloads(elloads<0)=0;                       %set negative parts of elution profiles to zero
    end
elseif isstruct(model)==false  %Generic model loadings
    disp('model={C;S;nSamples;modtype};')           %C=contributions, S=spectra
    Bstar = model{1};                               %elution profiles scaled by concentration, B*
    C = model{2};                                   %C, m/z (note renaming of S matrix to be consistent with other models)
    nSamples = model{3};
    modtype= model{4};
    nFac=size(Bstar,2);
    nEl=size(Bstar,1)/nSamples;
    if ~isequal(nSamples,size(Bstar,1)/nEl)
        fprintf('length of axis el is incompatible with the length of model elution loadings (first mode)')
        error('error in input values')
    end
    elloads=NaN*ones(nEl,nFac);
    %use the sample with the largest contribution for each component for the elution profile
    t=reshape(1:size(Bstar,1),[nEl,nSamples]);
    [imax jmax]=max(Bstar);
    for n=1:nFac
        [dummy S_max]=find(t==jmax(n));
        data=Bstar(t(:,S_max),n);
        %plot(Bstar), hold on, plot(floor(lenBstar/2),imax(n),'ro')
        elloads(:,n)=data./repmat(max(data),[length(data) 1]);                      %normalized elution profiles
        mz_above_threshold=mz(C(:,n)>min_mz)';
        C_above_threshold=C(C(:,n)>min_mz,n);
        eval(['mzax.' AtoZ(n) ' = mz_above_threshold;']);                           %exclude m/z below threshold
        eval(['mzloads.' AtoZ(n) ' = [mz_above_threshold C_above_threshold];']);    %exclude m/z below threshold
    end
    elloads(elloads<0)=0;                            %set negative parts of elution profiles to zero
end

%% Optional Output Multiple Elution Profiles
%Create headers for PARAFAC2 loadings, when vec is numeric and greater than one
if isstruct(model)==true  %PLS toolbox
    if strcmp(model.modeltype,'PARAFAC2')==true
        if repeatoption==true;
            nSamples=size(B3d,3);
            nEl=size(B3d,1);
            B=reshape(B3d,[nEl,nSamples*nFac]);                                %nCol=nFac*nSample
            vec_cols=[];
            for i=1:length(vec)
                vec_cols=[vec_cols (vec(i)-1)*nFac+1:vec(i)*nFac];             %select columns for chosen samples
            end
            elloads=B(:,vec_cols);
                        
            elloads(elloads<0)=0;
%             %OPTIONAL -Plot the elution loadings that will be exported
%             figure,
%             plot(elloads),
%             title(['Elution loadings for exporting: ' filename], 'Interpreter', 'none'),
%             pause,close
            
            %%%%%%%%%%%
            k=0;head={};
            for i = 1:nSamples;
                for j=1:nFac
                    k=k+1;
                    head{k}=cellstr(char(['Peak ' num2str(j) ', Sample ' num2str(i)]));
                end
            end
            head=head(vec_cols);                                               %headers for selected samples
        end
    end
end

fprintf('\nElution loadings are of size:\n');
disp(size(elloads));

%% Export Model Loadings
%Write model output to text file ready for importing in OpenChrome
fid = fopen(filename, 'w');
k=0;
for i=1:size(elloads,2)
    k=k+1; 
    el_min=num2str(round(min(el)*100)/100);
    el_max=num2str(round(max(el)*100)/100);
    eval(['mz4chrome=mzloads.' AtoZ(k) ';']);       %only positive numbers
    if k==nFac; k=0;end
    %[mz4chrome(:,1) mz4chrome(:,2)]
    if repeatoption==true;
        peakmsg=char(head{i});
    else
        peakmsg=['Peak ' num2str(i)];
    end
    %[mz4chrome(:,1) mz4chrome(:,2)]'
    %size(el), size(elloads(:,i))
    %[el elloads(:,i)]'
    if isstruct(model)==true
        modtype=model.modeltype;
    end
    peakdata ={...
        '#------------------------------------------';...                                   %1
        '#';...                                                                             %2
        ['# ' peakmsg];...                                                                         %3
        '#';...                                                                             %4
        'description';...                                                                   %5
        [el_min ' to ' el_max ' ms, ' modtype ' model with '  num2str(nFac) ' components- ' peakmsg];...   %6
        '# mass spectrum (m/z - intensity)';...                                             %7
        [mz4chrome(:,1) mz4chrome(:,2)]';...                                                %8
        ['# elution profile (' timeunits ' converted to milliseconds - intensity)'];... 	%9
        [el elloads(:,i)]'}                                                                 %10
    
    [nrows,~]= size(peakdata);
    for row=1:nrows
        switch row
            case {1,2,3,4,6,7,9}
                fprintf(fid, '%s\r\n', peakdata{row,:});
            case {5}
                fprintf(fid, '%s\t', peakdata{row,:});
            case {8}
                fprintf(fid, '%d\t%-g\n', peakdata{row,:});
            case {10}
                fprintf(fid, '%d\t%-g\n', peakdata{row,:});
        end
    end
end
fclose(fid);

function checkvec(a,v)
if isnumeric(v)==1
    try
        a(v,:);
    catch err
        if (strcmp(err.identifier,'MATLAB:badsubscript'))
            msg = sprintf('%s', ...
                'input argument ''vec'' is incompatible with the number of samples in dataset');
            error('MATLAB:myCode:dimensions', msg);
        end
    end  % end try/catch
end
%%
function checkNsamples(model)
try
    model.NSamples;
catch err
    if (strcmp(err.identifier,'MATLAB:nonStrucReference'))
        msg = sprintf('%s', ...
            'Number of samples must be imported into the MCR model structure: model.Nsamples');
        error('MATLAB:myCode:nonStrucReference', msg);
    end
end  % end try/catch

