function model = prioritizer(X,target,variableprice,variablepriority,sampleprice,samplepriority,lambda)

%PRIORITIZER finds the right mix of samples to model a target considering
%price of samples, cost of using individual variables and cost of using
%many variables
%
% model = prioritizer(X,goal,variableprice,variablepriority,sampleprice,samplepriority,fewsamples)
%
%
% Inputs:
% X                 Dataset containing the model system. Each COLUMN contains one
%                   profile that is used to model the target
% target            The profile to be approximated by a linear combination of X
% variableprice     A vector of same length as the number of variables in
%                   the target. Each element contains the inverse cost of
%                   that variable. The higher the element, the more
%                   the model will tend to use that variable.
% variablepriority  A number between 0 and 1. set to 0 means that each 
%                   variable has price 1 and so are weighted equally 
%                   whereas setting to one means that the are weighted 
%                   according to price. Adjust to change how much 
%                   focus to pay on variable price
% sampleprice       See variableprice
% sampleprirority   See variablepriority
% fewsamples        A scalar used to downweight using many samples. Set it
%                   to higher values to have fewer samples make up
%                   the mixture
%
% Copyright 2012, Rasmus Bro, rb@life.ku.dk

% makes a vector with the relative importance of the thresholds
vp=1;
VP=zeros(length(variableprice),1);
while vp<=length(variableprice);
if variableprice(vp,1)<10;
    VP(vp)=0.1;
elseif variableprice(vp,1)>100;
        VP(vp)=10;
else VP(vp)=1;
    end
vp=vp+1;
end
variableprice=VP;

[I,J]=size(X);
if length(variableprice)~=I
    error(['the input variableprice must match the number of rows in X (',num2str(I),')'])
end
if length(sampleprice)~=J
    error(['the input sampleprice must match the number of columns in X (',num2str(J),')'])
end

Iw = ones(I,1)*(1-variablepriority) + (variableprice.^(-1))*variablepriority;
Jw = ones(J,1)*(1-samplepriority) + (sampleprice.^(-1))*samplepriority;
Xweighted = diag(Iw)*X.data*diag(Jw);

b_unconstrained = pinv(Xweighted)*diag(Iw)*target.data;

% Find sparse b
b = SMREW(diag(Iw)*target.data,Xweighted,b_unconstrained',lambda).';

% Now find best fit to target using the sparsely chosen variables
id = find(b>eps*1000);
b(id) = pinv(Xweighted(:,id))*diag(Iw)*target.data;
b(id) = fastnnls(Xweighted(:,id),diag(Iw)*target.data);
sb = diag(Jw)*b;

%Make some nice graphics

subplot(2,1,1)
plot(target.data,'linewidth',4,'color','blue');hold on
plot(X.data*b,'linewidth',2,'color','red');hold off
Totalcost=sum((b/sum(b)).*sampleprice);
title(['Matching profile using ',num2str(length(find(abs(b)>1000*eps))),' ingredients, the total price is ',num2str(Totalcost)])
legend({'Target';'Current match'})
axis tight

subplot(2,1,2)
bar(b)
%bperc=100*b/sum(b);
%bar(bperc)
id=find(b>0);

try
for i=1:length(id)
    g=text(id(i),b(id(i)),lower(deblank(X.label{2}(id(i),:))),'rotation',90,'fontsize',8)
end
end
title('Amount of ingredients')
ylabel('Percentage')
xlabel('Ingredient')
axis tight

model.b = b;
model.match = X.data*b;
model.Err1=target-model.match;
model.Err2=model.Err1.data.^2;
model.Fit=100*(1-sum(model.Err2)/sum(target.data.^2));

%data for the plt of the fit and cost
Err1 = target-model.match;
Err2 = Err1.data.^2;
Fit = 100*(1-sum(Err2)/sum(target.data.^2));
Totalcost=sum((model.b/sum(model.b)).*sampleprice);
Ning=length(find(abs(b)>1000*eps));

model.Xblendgrap=[Fit,Totalcost,Ning,samplepriority,variablepriority,lambda];

function B = SMREW(X,A,B,lambda)

% ell-1 based sparse matrix regression:
% Given X, A, find B to 
% min ||X-A*B.'||_2^2 + lambda*sum(sum(abs(B)))
%
% N. Sidiropoulos, July 2009
% nikos@telecom.tuc.gr
%
% This version: uses element-wise coordinate descent; 
% current B estimate given as input;
% Each element-wise update can be solved analytically (in closed form) 

[I,J]=size(X);
[I,F]=size(A);

DontShowOutput = 1;
maxit=1000;
convcrit = 1e-9;
showfitafter=1;
it=0;
Oldfit=1e100;
Diff=1e100;

while Diff>convcrit&it<maxit
    it=it+1;
    for j=1:J, 
        for f=1:F, 
            data = X(:,j) - A*B(j,:).' + A(:,f)*B(j,f); 
            alpha = A(:,f);
            if alpha.'*alpha > 10*eps                
                if alpha.'*data > lambda/2
                    B(j,f) = (alpha.'*data - lambda/2)/(alpha.'*alpha);
                elseif alpha.'*data <  -lambda/2 
                    B(j,f) = (alpha.'*data + lambda/2)/(alpha.'*alpha);
                else % Nikos fixed bug, Aug. 14, 2009
                    B(j,f) = 0; % Nikos fixed bug, Aug. 14, 2009
                end
            else
                B(j,f) = 0;
            end           
            if B(j,f)<0
                B(j,f)=0;
            end
        end
    end

    fit=norm(X-A*B.','fro')^2+lambda*sum(sum(abs(B)));
    if Oldfit < fit 
        disp(['*** bummer! *** ',num2str(Oldfit-fit)]) 
    end
    Diff=abs(Oldfit-fit);
    Oldfit=fit;
    
    if ~DontShowOutput
        % Output text
        if rem(it,showfitafter)==0
            disp([' SMREW Iterations:', num2str(it),' fit: ',num2str(fit)])
        end
    end
   
end