function [ModelIn,Content,X] = Initialise(Content,ModelIn,X,Xt,Xfil,Flags,varargin)
% function [ModelIn,Content,X] = Initialise (Content,ModelIn,X,Xt,Xfil,Flags,Matrices,Sizes);
% 
% Description:
% Set or updates the informations stored in ModelIn, Content and X.
% 
% 
% Inputs: 
% 
% Content, ModelIn, X: the current data structures ModelIn, Content and X 
% Xt                 : cell vector where each cell is a new loaded data set. Two classes of data are
%                      supported as elements: double arrays and cbdatasets. The treatment varies also
%                      depending on whether the data is "single set" or "multiple set". 
% 
%                      Double: Single set, the data are given default labels. The first mode has 'Sam.'
%                                         as default label, the others have 'Var.'.
%                              Multiple set, the sizes of the single sets are passed with the Sizes (non
%                                         optional) parameter. The name of the loaded sets are passed
%                                         with Matrices. This additional information is then saved in
%                                         the 'description' ( set names) and 'userdata' (set lengths)
%                                         fields of the CBDataSet object created from the data. The
%                                         labels in the combined mode (i.e. the first) are given in the
%                                         form 'Var. #/ Var. #' and the combined mode is given name
%                                         'Mode a/Mode b'. 
%                      CBDataSet: Single set, nothing is added or changed in the new data set. If the
%                                         labels in the first mode are different from those already
%                                         present in X, they are uniformated into the latters. 
%                                 Multiple set, the list of single sets is read via the 'description'
%                                         field while the sizes of the single sets are saved in the
%                                         'userdata' field. 
%                                         NB If this parameter is produced via concatenation of more
%                                         cbdatasets upon loading, the single files names ( passed with
%                                         Matrices) are automatically added to the corresponding labels
%                                         in the combined mode and the combined mode itself is given
%                                         name 'Mode a/Mode b'.
% 
% Xfil              : cell vector of length two. 
%                     Xfil{1} contains the name (full path) of the file holding the data
%                     Xfil{2} for single sets and multiple sets hold in  a cbdataset object is the name of the array
%                             within file Xfil{1}. In the case of double/multiple sets the second element is disregarded.
% Flags             : vector of length two. 
%                     Flags(1) is set to 1, all the data structures (i.e. ModelIn, Content and X) are
%                     reinitialised. 
%                     Flags(2) = 1 indicates that the loaded array is made up of several
%                     arrays/cbdatasets loaded separately.
%  
% Matrices(optional): name of the single sets constituing a "multiple set" data set. 
% Sizes (optional)  : vector holding the sizes of the single sets of a multiple set of class double 
% 
% 
% Outputs:
% ModelIn: ModelIn structure
% Content: Content data structure
% X: X data structure
% 
% 
% Called by:
% CuBatch, ImportData, PermuteWindow, ReshapeWindow, SelectSamples
% 
% Subroutines:
% Internal: None
% External: DSContent
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 26-Oct-2002 23:45:10
% 
% Contact: Giorgio Tomasi, gt@kvl.dk
%

%Some initial values
%Set flags, vector length = 5 only to allow further developments
if ~exist('Flags','var')
   Flags = [0 0 0 0 0];
else
   Flags = [Flags,zeros(1,5-length(Flags))];
end

%Reset X, Content and ModelIn
if Flags(1) 
   
   X       = [];
   Content = [];
   %ModelIn structure fields
   Constraints = {'closure','equality','non-negativity','orthogonality','threeway','unimodality'}';
   Data        = struct('augmentedmatrix',[],'dataparafac2',[],'initialinput',[],'sam',[],'samlabs',[],'sizes',[],'var',[]);
   Fact        = struct('min',1,'max',[]);
   Options     = struct('algorithm',struct('constants',[],'maxiter',[],'name',[],'value',[]),'commonmodes',1,'rotate',0,'species',[],'direction',[]);
   ValMethod   = struct('bootstrap',[],'name',[],'replicates',[],'segments',[]);
   VStruct     = struct('dims',[],'name',[],'sel',[],'cen',[],'scal',[]);
   SStruct     = struct('samples',[],'name',[],'sel',[],'cen',[],'scal',[]);   
   Constr      = struct('algorithm',[],'modes',0,'constants',[],'core',[],'factors',[],'name',Constraints,'value',[]);
   ModelIn     = struct('constraints',Constr,...
                        'data',Data,...
                        'file',[],...
                        'modelname',[],...
                        'nbfactors',Fact,...
                        'options',Options,...
                        'valmethod',ValMethod,...
                        'sam',SStruct,...
                        'samlabs',[],...
                        'var',VStruct); 
                        
end

%Convert Xt into cell if it is not
if ~isa(Xt,'cell')
   Xt = {Xt};
end

%Convert X into a cell if it is not
if ~isa(X,'cell')
   X = {X};
end

%Check if cbdataset or double
% Cu_FIX_2011  -- capital lettes CB
% orig if ~all( cellfun('isclass',Xt,'cbdataset') | cellfun('isclass',Xt,'double'))% | length(Xt) > 1

if ~all(cellfun('isclass',Xt,'CBdataset') | cellfun('isclass',Xt,'cbdataset') | cellfun('isclass',Xt,'double'))% | length(Xt) > 1
   error('Invalid data format')
end

%mark double array
MD     = isa(Xt{1},'double'); 
if isempty(X{1})
   
   MatrNum = 1;
   if MD
      X = {CBdataset(Xt{1})};
   else
      X = Xt;
   end
   
else
   MatrNum = length(X) + 1;
   %Automatically converts Xt in a cbdataset, if it is an array of doubles
   X = cat(2,X{:},Xt{:});
end

if strcmp(X{end}.name,'Matrix')
  
   X{end}.name = Xfil{2}  ;
end

%Set ModelIn.var structures and Content
Content{MatrNum}     = DSContent(X{MatrNum});
Labs                 = Content{MatrNum}.labobj;
ModelIn.var(MatrNum) = struct('dims',Content{MatrNum}.dims(2:end),...
   'name',Content{MatrNum}.name,...
   'sel',marklab(Labs(2:end),[' X *';' Y *']),...
   'cen',zeros(1,ndims(X{MatrNum}) - 1),...
   'scal',zeros(1,ndims(X{MatrNum}) - 1));
%Automatically select all variables
for i = 1:ndims(X{MatrNum}) - 1
   ModelIn.var(MatrNum).sel{i}.sel = MatrNum;
end
%Set constraints
for i = 1:length(ModelIn.constraints)
   ModelIn.constraints(i).modes = zeros(1,ndims(X{MatrNum}));
end

%Set the file field depending on single/multiple set
if Flags(2)
   
   %multiple set
   X{MatrNum}.type                = 'Batch';
   ModelIn.file(MatrNum).file     = Xfil{1};
   ModelIn.file(MatrNum).matrices = varargin{1};
   ModelIn.file(MatrNum).size     = varargin{2};
   X{MatrNum}.userdata            = varargin{2};
   X{MatrNum}.description         = varargin{1};
   %Combine the labels in the first/second mode
   if MatrNum == 1
   
      if MD

         for j = 1:length(varargin{2})
            a = label([1:varargin{2}(j)],[varargin{1}{j},'/ Var.']);
            b{j} = a.labels;
            c{j} = a.scalars;
         end
         Labs{1}              = label(cat(1,b{:}),'Sam. xxx/ Var.',cat(1,c{:}),[],'Mode a/Mode b');
         X{MatrNum}.labels{1} = Labs{1};
         Content{MatrNum}     = DSContent(X{MatrNum});

      else

         t = cumsum(X{MatrNum}.userdata);
         t = [0;t(:)];
         b = {};
         c = {};
         for j = 1:length(X{MatrNum}.userdata)
            a    = char(X{MatrNum}(t(j) + 1:t(j + 1)).labels{1});
            u    = X{MatrNum}.userdata;
            a    = [repmat([varargin{1}{j},'/'],u(j),1),a];
            b{j} = cellstr(a);
            c{j} = X{MatrNum}(t(j) + 1:t(j + 1)).scalars{1};
         end
         Labs{1}              = label(cat(1,b{:}),'Sam. xxx/ Var.',cat(1,c{:}),[],'Mode a/Mode b');
         X{MatrNum}.labels    = Labs;
         Content{MatrNum}     = DSContent(X{MatrNum});

      end
   
   else
      X{MatrNum}.labobj{1} = X{1}.labobj{1};
      Content{MatrNum}     = DSContent(X{MatrNum});
   end
 
else
   
   %single set
   if strcmp(X{MatrNum}.type,'Batch')
   
       ModelIn.file(MatrNum).file = Xfil{1};
       if length(X{MatrNum}.description) == length(X{MatrNum}.userdata)
          ModelIn.file(MatrNum).matrices = X{MatrNum}.description;
       else
          ModelIn.file(MatrNum).matrices = cellstr([repmat([X{MatrNum}.name ' '],length(X{MatrNum}.userdata),1),int2str([1:length(X{MatrNum}.userdata)]')]);
       end
       ModelIn.file(MatrNum).size     = X{MatrNum}.userdata;
   
   else
   
       X{MatrNum}.type                = 'Data';
       ModelIn.file(MatrNum).file     = Xfil{1};
       ModelIn.file(MatrNum).matrices = Xfil{2};
       ModelIn.file(MatrNum).size     = [];
   
   end

end
ModelIn.samlabs = Labs{1};

Sel         = marklab(Labs{1},['  * ';'T * ']);
%Automatically select all slabs in the first mode
Sel(:).sel  = 1;
ModelIn.sam = struct('samples',[1:X{1}.dims(1)]',...
   'name',Labs{1}.name,...
   'sel',Sel,...
   'cen',[0 0],...
   'scal',[0 0]);
