function Model = SelectVariables(Model,Way)
% function Model = SelectVariables (Model,Way);
% 
% Description:
% Opens the window for selecting variables in one of the loaded arrays X (and Y when it is the case)
% and handles the corresponding events. The selection window is assigned the following application-
% defined data:
% 'Model': the interim ModelIn structure
% 'Handles': window/object handles
% 'IO': index in the X data structure
% 
% Inputs: 
% Model: ModelIn structure (at the first call), NaN when handling callback events
% 
%      Way   Array InOut  Figure's title 
%    'In'      X     1   'X Variables' 
%    'Out'     Y     2   'Y Variables' 
%    'Matrix'  X     1    'Variables selection' 
% 
% 
% Outputs:
% Model: ModelIn structure
% 
% 
% Called by:
% CuBatch, ExportData
% 
% Subroutines:
% Internal: None
% External: charwidth, ctag, cval, getcvdata, guinames, screensize, setcvdata, skcell
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 21-Oct-2002 04:20:58
% 
% Contact: Giorgio Tomasi, gt@kvl.dk
% 
GuiNames
bgc      = [ 0.5029 0.5814 0.6402];
cbf      = 'SelectVariables(NaN);';
lbc      = [0.9 0.93 0.95];
pbc      = [0.1785 0.3706 0.5627];

if isa(Model,'struct')
   switch Way
      case 'In'
         InOut = 1;
         Title = 'X Variables';
      case 'Matrix'
         InOut = 1;
         Title = 'Variables selection';
      otherwise
         Title = 'Y Variables';
         InOut = 2; 
   end
   Mat1names = Model.var(1).sel.name;
   MatrNames = {Model.var.name};
   Vars      = Model.var(1).sel{1}.labels;
   
   %Define uicontrols offset
   if isa(Model.var(1).sel.name,'cell')
      ps1   = max(160,max([cellfun('length',Model.var(1).sel.name),cellfun('length',MatrNames)]) * charwidth);
   else
      ps1   = max(160,max([length(Model.var(1).sel.name),cellfun('length',MatrNames)]) * charwidth);
   end
   ps2   = max(160,charwidth * size(Model.var(1).sel{1}.labels,2));
   
   %Calculate window position for it to be at the centre of the screen
   ScSize = ScreenSize;
   vb     = ps1 + ps2 + 30 ;
   VPos   = fix((ScSize(3) - (vb + 140)) / 2);
   HPos   = fix((ScSize(4) - 390) / 2);
   FigPos = [VPos HPos (ps1 + ps2 + 170) 390];
   h         = figure(fip{:},'name',Title,'Color',bgc,crf,cbf,pos,FigPos);
   clf
   Help = uimenu(h,la,'&Help');
   uimenu(Help,la,'&About CuBatch',   cb,'AboutCuBatch')
   uimenu(Help,la,'&Help',            cb,cbf,tg,'Help')
   
   %Sample selection listbox context menu
   SelEvery  = SkCell(2,min([30,Model.var(1).dims(1),max(2,floor(Model.var(1).dims(1) / 4))]),'variable');
   cm        = uicontextmenu;
   uimenu(cm,la,'&Select all',tg,'SelAll',cb,cbf)
   VarHan.selevery = uimenu(cm,la,'&Select every');
   for i = 1:length(SelEvery)
      uimenu(VarHan.selevery,la,SelEvery{i},cb,cbf,tg,'SelectEvery',ud,i + 1);
   end
   uimenu(cm,la,'Remove all',cb,cbf,tg,'RemAll',sep{:})
   
   VarHan.modetxt  = uicontrol(ty,tx,pos,[      10 168 ps1  25],st,'Modes',       bd{:},                  bg,bgc,ari{:});
   VarHan.vartxt   = uicontrol(ty,tx,pos,[ps1 + 20 356 ps2  25],st,'Variables',   bd{:},                  bg,bgc,ari{:});
   VarHan.matrtxt  = uicontrol(ty,tx,pos,[      10 356 ps1  25],st,'Arrays',      bd{:},                  bg,bgc,ari{:});
   VarHan.assign   = uicontrol(      pos,[      vb  82 130  30],st,'Assign',      cb,cbf,tg,'Select',     bg,pbc,ari{:});
   VarHan.remove   = uicontrol(      pos,[      vb  46 130  30],st,'Remove',      cb,cbf,tg,'Deselect',   bg,pbc,ari{:});
   VarHan.ok       = uicontrol(      pos,[      vb  10 130  30],st,'Ok',          cb,cbf,tg,'Ok',         bg,pbc,ari{:});
   VarHan.matrin   = uicontrol(ty,lb,pos,[      10 200 ps1 160],st,MatrNames,     cb,cbf,tg,'MatrIn',va,1, ss{:},bg,lbc,cu{:});
   VarHan.modein   = uicontrol(ty,lb,pos,[      10  10 ps1 160],st,Mat1names,     cb,cbf,tg,'ModeIn',va,1, ss{:},bg,lbc,cu{:});
   VarHan.varin    = uicontrol(ty,lb,pos,[ps1 + 20  10 ps2 350],st,Vars,                 tg,'VarIn', va,[],ms{:},bg,lbc,cu{:},ucm,cm);
   setcvdata(h,'Model',Model,'Handles',VarHan,'IO',InOut);
   waitfor(h,'waitstatus','close')
   Model = getcvdata(h,'Model');
   delete(h)
   
else
   [Model,VarHan,InOut] = getcvdata(gcbf,'Model','Handles','IO');
   if ~strcmp(ctag,'window') & ~strcmp(get(gcbo,'type'),'uimenu');
      Selected = cval;
   end
   matn = get(VarHan.matrin,va);
   modn = get(VarHan.modein,va);
   varn = get(VarHan.varin,va);
   switch ctag
      case 'Help'
         a   = lower(which('CuBatch'));
         a   = strrep(a,'cubatch.m','Help\user\edit.html#SelectVariables');
         stat=web(['file:///' a],'-browser');
         
      case 'MatrIn'
         set(VarHan.varin,st,Model.var(matn).sel{1}.labels,va,1:Model.var(matn).dims(1));
         a  = SkCell(2,min([Model.var(matn).dims(1),30,max(2,floor(Model.var(matn).dims(1) / 4))]),'variable');
         b  = flipud(get(VarHan.selevery,'children'));
         if length(b) > length(a)
            delete(b(length(a) + 1:end))
         elseif length(a) > length(b)
            for i = length(b) + 1:length(a)
               uimenu(VarHan.selevery,la,a{i},cb,cbf,tg,'SelectEvery',ud,i + 1);
            end
         end
         set(VarHan.modein,st,Model.var(matn).sel.name,va,1);
         
      case 'ModeIn'
         a = SkCell(2,min([Model.var(matn).dims(modn),30,max(2,floor(Model.var(matn).dims(modn) / 4))]),'variable');
         b  = flipud(get(VarHan.selevery,'children'));
         if length(b) > length(a)
            delete(b(length(a) + 1:end))
         elseif length(a) > length(b)
            for i = length(b) + 1:length(a)
               uimenu(VarHan.selevery,la,a{i},cb,cbf,tg,'SelectEvery',ud,i + 1);
            end
         end
         set(VarHan.varin,st,Model.var(matn).sel{modn}.labels,va,1:Model.var(matn).dims(modn));   
         
      case 'SelectEvery'
         set(VarHan.varin,va,varn(1:get(gcbo,ud):end))
         
      case 'SelAll'
         set(VarHan.varin,va,1:size(get(VarHan.varin,st),1))
         
      case 'RemAll'
         Model.var(matn).sel{modn}.desel = InOut;
         set(VarHan.varin,st,Model.var(matn).sel{modn}.labels,va,[]);   
         set(VarHan.modein,st,Model.var(matn).sel.name,va,1);
         
      case 'Select'
         a = Model.var(matn).sel.sel{3 - InOut};
         if ~isa(a,'cell')
            a = {a};
         end
         b = cellfun('length',a);
         if any(b)
            if length(a) > 1
               uiwait(msgbox('You can''t assign slabs of the same multiway matrix to both X and Y'))
               return
            else
               c = Model.var(matn).sel(varn).sel{3 - InOut};
               if  ~isempty(c)
                  if length(c) == 1
                     Msg = {'Do you want to change the location of this variable?'};
                  else length(c) > 1
                     Msg = {'Do you want to change the location of these variables?'};
                  end
                  button = questdlg(Msg,'X/Y choice','Ok','Cancel','Ok');
                  if ~strcmp(button,'Ok')
                     return;
                  end
               end
            end
         end
         Model.var(matn).sel{modn}(varn).sel = InOut; 
         set(VarHan.varin,st,Model.var(matn).sel{modn}.labels)
         set(VarHan.modein,st,Model.var(matn).sel.name,va,modn);
         
      case 'Deselect'   
         Model.var(matn).sel{modn}(varn).desel = InOut; 
         set(VarHan.varin,st,Model.var(matn).sel{modn}.labels)
         set(VarHan.modein,st,Model.var(matn).sel.name,va,modn);
         
      case 'Ok'
         set(gcbf,'waitstatus','close')
         
      case 'window'
         setcvdata(gcbf,'Model',[])
         set(gcbf,'waitstatus','close')
         
   end
   setcvdata(gcbf,'Model',Model)
   %Define uicontrols offset 
   for i = 1:length(Model.var)
      if isa(Model.var(i).sel.name,'cell')
         ps1   = max(160,max([cellfun('length',Model.var(i).sel.name),length(Model.var(i).name)]) * charwidth);
      else
         ps1   = max(160,max([length(Model.var(i).sel.name),cellfun('length',{Model.var.name})]) * charwidth);
      end
   end
   ps2 = max(160,charwidth * size(Model.var(matn).sel{1}.labels,2));
   
   %Calculate window position for it to be at the centre of the screen
   ScSize = ScreenSize;
   vb     = ps1 + ps2 + 30 ;
   VPos   = fix((ScSize(3) - (vb + 140)) / 2);
   HPos   = fix((ScSize(4) - 390) / 2);
   FigPos = [VPos HPos (ps1 + ps2 + 170) 390];
   Han = struct2cell(VarHan);
   Pos = {FigPos,[10 168 ps1 25],[ps1 + 20 356 ps2 25],[10 356 ps1 25],[vb 82 130 30],...
         [vb 46 130 30],[vb 10 130 30],[10 200 ps1 160],[10 10 ps1 160],[ps1 + 20 10 ps2 350]}';
   set([gcbf,Han{2:end}],{pos},Pos);
   return
end