function ModelIn = SelectSamples(ModelIn,Flag)
% function ModelIn = SelectSamples (ModelIn,Flag);
% 
% Description:
% Opens the window for selecting batch/samples and handles the corresponding events. Two application-
% data structures are assigned to the opened figure: 'ModelIn', with the interim ModelIn structure and
% 'Handles', with the handles to the figure's objects.
% 
% Inputs: 
% ModelIn: ModelIn structure (at the first call), NaN when handling callback events
% Flag   : If it is equal to 1 it is possible to assign samples/batches to both the calibration set and
%          the test set. If it is equal to 2 the text above the "Assign to" button is changed into
%          'Choose samples'. For any other value the text is 'Exported set'.
% 
% 
% Outputs:
% ModelIn: ModelIn structure.
% 
% 
% Called by:
% CuBatch, ExportData
% 
% Subroutines:
% Internal: None
% External: charwidth, ctag, getcvdata, guinames, initialise, screensize, setcvdata, skcell
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 26-Oct-2002 17:42:36
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
% 

if nargin < 2
   Flag = 1;
end

GuiNames;
cbf     = 'SelectSamples(NaN);';
if isa(ModelIn,'struct')
   
   %Define uicontrols offset depending on sample and group names
   ps1   = max(length(ModelIn.sam(1).sel.name) * charwidth,180); %width for the samples listbox
   ps2   = max(size(ModelIn.sam(1).sel.labels,2) * charwidth,160); %width for the group listbox
   c     = round((ps2 - 150)/2);
   vbase = 17 + ps1;
   SG    = ModelIn.sam(1).sel.name;
   
   %Fetch sample labels
   SL          = ModelIn.sam(1).sel.labels;      

   %Define skipping parameters according to the number of samples present in the first group
   SkipSamTxt  = SkCell(2,min([size(SL,1),30,max(2,floor(size(SL,1) / 4))]),'sample');

   %Calculate window position for it to be at the centre of the screen
   ScSize = ScreenSize;
   VPos   = fix((ScSize(3) - (ps1 + ps2 + 20)) / 2);
   HPos   = fix((ScSize(4) - 491) / 2);
   FigPos = [VPos HPos (ps1 + ps2 + 20) 491];
   fig    = figure(pos,FigPos,fip{:},'name','Select Samples','Color',WIC,crf,cbf);
   clf
   
   %Help menu
   Help = uimenu(fig,la,'&Help');
   uimenu(Help,la,'&About CuBatch',   cb,'AboutCuBatch')
   uimenu(Help,la,'&Help',            cb,cbf,tg,'Help')
   
   %Sample selection listbox context menu
   cm = uicontextmenu;
   uimenu(cm,la,'&Select all',tg,'SelAll',cb,cbf)
   SamHan.selevery = uimenu(cm,la,'&Select every');
   for i = 1:length(SkipSamTxt)
      uimenu(SamHan.selevery,la,SkipSamTxt{i},cb,cbf,tg,'SelEvery',ud,i + 1);
   end
   uimenu(cm,la,'Remove all from both',cb,cbf,tg,'RemAll',sep{:})
   
   %Define uicontrols on the window
   uicontrol(ty,fr,                          pos,[4 + c 156 158 100],                           bg,FRC)
   if Flag
      uicontrol(ty,tx,st,'Calibration set', pos,[8 + c 225 150  25],       ari{:},             bg,FRC,fw,'Bold')
      uicontrol(      st,'Assign to',       pos,[8 + c 193 150  30],cb,cbf,ari{:},tg,'AddCal');
      uicontrol(      st,'Remove from',     pos,[8 + c 160 150  30],cb,cbf,ari{:},tg,'RemCal');
      uicontrol(ty,fr,                      pos,[4 + c  50 158 100],                           bg,FRC)
      uicontrol(ty,tx,st,'Test set',        pos,[8 + c 119 150  25],       ari{:},             bg,FRC,fw,'Bold')
      uicontrol(      st,'Assign to',       pos,[8 + c  87 150  30],cb,cbf,ari{:},tg,'AddTest');
      uicontrol(      st,'Remove from',     pos,[8 + c  54 150  30],cb,cbf,ari{:},tg,'RemTest');
   else 
      uicontrol(ty,tx,st,'Exported set', pos,[8 + c 225 150  25],       ari{:},             bg,FRC,fw,'Bold')
      uicontrol(      st,'Assign to',    pos,[8 + c 193 150  30],cb,cbf,ari{:},tg,'AddCal');
      uicontrol(      st,'Remove from',  pos,[8 + c 160 150  30],cb,cbf,ari{:},tg,'RemCal');
   end
   uicontrol(ty,tx,st,'Groups',             pos,[    8 458 ps2  25],       ari{:},             bg,WIC,fw,'Bold',fs,14);
   uicontrol(ty,tx,st,'Samples',            pos,[vbase 458 ps2  25],       ari{:},             bg,WIC,fw,'Bold',fs,14);
   SamHan.groupin = uicontrol(ty,lb,st,SG,  pos,[    8 262 ps1 194],       cu{:},              bg,wh);   
   SamHan.samplin = uicontrol(ty,lb,st,SL,  pos,[vbase  10 ps2 446],       cu{:},              bg,wh, va,[],    ms{:},ucm,cm);
   SamHan.ok      = uicontrol(      st,'Ok',pos,[8 + c  10 150  30],cb,cbf,ari{:},tg,'Ok');
   
   %Save Handlers and ModelIn in current figure
   setcvdata(fig,'ModelIn',ModelIn,'Handles',SamHan)      
   
   %Prevent other activities
   waitfor(fig,'waitstatus','close')
   
   %Return ModelIn: empty if "cancel" or the structure if "Ok"
   ModelIn = getcvdata(fig,'ModelIn');
   
   %Delete figure
   delete(fig)
   
else
   
   %Get parameters
   [ModelIn,SamHan] = getcvdata(gcbf,'ModelIn','Handles');
   san              = get(SamHan.samplin,va); %Row vector
   
   %Callbacks
   switch ctag                    
   case 'SelEvery'
      set(SamHan.samplin,va,san(1:get(gcbo,ud):end))
      
   case 'SelAll'
      set(SamHan.samplin,va,1:length(get(SamHan.samplin,st)))
      
   case 'RemAll'
      ModelIn.sam(1).sel.desel = [1,2];
            
   case 'AddCal'
      ModelIn.sam(1).sel(san).sel = 1;
      
   case 'AddTest'
      ModelIn.sam(1).sel(san).sel = 2;
      
   case 'RemCal'
      ModelIn.sam(1).sel(san).desel = 1;
      
   case 'RemTest'
      ModelIn.sam(1).sel(san).desel = 2;
      
   case 'Ok'
      set(gcbf,'waitstatus','close')
      return   
      
   case 'window'
      setcvdata(gcbf,'ModelIn',[])
      set(gcbf,'waitstatus','close')
      return
      
   case 'Help'
      a   = lower(which('CuBatch'));
      a   = strrep(a,'cubatch.m','Help\user\edit.html#SelectSamples');
      stat=web(['file:///' a],'-browser');
      
   end
   if ~strcmp(ctag,{'SelEvery','SelAll'})
      ps1 = max(length(ModelIn.sam(1).sel.name) * charwidth,180);
      ps2 = max(size(ModelIn.sam(1).sel.labels,2) * charwidth,160);
      set(SamHan.groupin,st,ModelIn.sam(1).sel.name,pos,[8 262 ps1 194])
      set(SamHan.samplin,st,ModelIn.sam(1).sel.labels,pos,[ps1 + 17  10 ps2 446])
      ScSize = ScreenSize;
      VPos   = fix((ScSize(3) - (ps1 + ps2 + 20)) / 2);
      HPos   = fix((ScSize(4) - 491) / 2);
      set(gcbf,pos,[VPos HPos (ps1 + ps2 + 20) 491]);
   end
   setcvdata(gcbf,'ModelIn',ModelIn)
   return
end
