function SaveReport(MainFig)
% function SaveReport (MainFig);
% 
% Description:
% Saves the informations (such as labels, description, author,...) relative to a loaded set of data or
% to the data employed for computing a model. It produces a window allowing to choose which information
% to save in the report and passes the generated ReportStruct to the model-specific *Report functions.
% It also passes the file handle relative to the generated txt files and handles its closure. The
% report structure has the following fields:
% 
%    'data', save data info, 
%    'file', txt file name, 
%    'maxlabels', maximum labels saved per mode, 
%    'm_info', save model information, 
%    'm_results', save model results, 
%    'p_info',save new data information, 
%    'p_results',save projection results, 
%    'v_info',save validation information, 
%    'v_results',save validation results. 
% 
% The *_info and *_results are logical vectors of length 10. Currently only the first element is set to
% 0/1, the others can be used in the future for finer tuning of the saved information.
% 
% NB, It must be remembered (when implementing a new method) that the txt file must not be closed in
% the model-specific function.
% 
% Inputs: 
% MainFig: handle to the main figure.
% 
% 
% Outputs:
% None
% 
% 
% Called by:
% CuBatch
% 
% Subroutines:
% Internal: None
% External: cenwindow, ctag, cval, setcvdata, *Report
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 02-Nov-2002 19:24:20
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
%

GuiNames
cbf = 'SaveReport(NaN)';
if ishandle(MainFig)
   ModelOut  = getappdata(MainFig,'Results');
   ReportStruct = struct('data',1,...             %data info
      'file','ModelReport',...                    %file name
      'maxlabels',10,...                          %maximum labels displayed
      'm_info',[1 zeros(1,9)],...                    %model info
      'm_results',[1 zeros(1,9)],...                  %model results
      'p_info',[1 zeros(1,9)],...                     %new data info
      'p_results',[1 zeros(1,9)],...                  %projection results the vector is for further
      'v_info',[1 zeros(1,9)],...
      'v_results',[1 zeros(1,9)]);
   h = CenWindow(WIC,'Report option',cbf,290,325);
   set(h,'toolbar','none');
   uicontrol(ty,fr,pos,[  7  47 276 291],                       bg,FRC);
   uicontrol(ty,ch,pos,[ 10 290 130  25],st,'Data',             bg,FRC,ari{:},cb,cbf,tg,'data',va,1);
   uicontrol(ty,tx,pos,[ 70 265 130  25],st,'Max labels',       bg,FRC,ari{:});
   uicontrol(ty,ed,pos,[200 263  50  30],st,'10',               bg,LBC,ari{:},cb,cbf,tg,'maxlabels');
   uicontrol(ty,ps,pos,[150  10 130  30],st,'Cancel',                  ari{:},cb,cbf,tg,'cancel');
   uicontrol(ty,ps,pos,[ 10  10 130  30],st,'OK',                      ari{:},cb,cbf,tg,'ok');
   m(1) = uicontrol(ty,ch,pos,[ 10 230 130  25],st,'Model',     bg,FRC,ari{:},cb,cbf,tg,'model',va,1);
   m(2) = uicontrol(ty,ch,pos,[ 70 205 130  25],st,'Info',      bg,FRC,ari{:},cb,cbf,tg,'m_info',va,1);
   m(3) = uicontrol(ty,ch,pos,[ 70 180 130  25],st,'Results',   bg,FRC,ari{:},cb,cbf,tg,'m_results',va,1);
   v(1) = uicontrol(ty,ch,pos,[ 10 145 130  25],st,'Validation',bg,FRC,ari{:},cb,cbf,tg,'validation',va,1);
   v(2) = uicontrol(ty,ch,pos,[ 70 120 130  25],st,'Info',      bg,FRC,ari{:},cb,cbf,tg,'v_info',va,1);
   v(3) = uicontrol(ty,ch,pos,[ 70  95 130  25],st,'Results',   bg,FRC,ari{:},cb,cbf,tg,'v_results',va,1);
   p    = uicontrol(ty,ch,pos,[ 10  60 130  25],st,'Proj',      bg,FRC,ari{:},cb,cbf,tg,'projection',va,1);
   if isempty(ModelOut(1).model) | isempty(ModelOut(1).modelname)
      set(m,dis{:},va,0)
      ReportStruct.m_info    = zeros(1,10);
      ReportStruct.m_results = zeros(1,10);
   end
   if isempty(ModelOut(1).validation) | isempty(ModelOut(1).modelname)
      set(v,dis{:},va,0)
      ReportStruct.v_info    = zeros(1,10);
      ReportStruct.v_results = zeros(1,10);
   end
   if isempty(ModelOut(1).prediction) | isempty(ModelOut(1).modelname) | isempty(ModelOut(1).prediction.data)
      set(p,dis{:},va,0)
      ReportStruct.p_results = zeros(1,10);
   end
   setcvdata(h,'ReportStruct',ReportStruct)
   waitfor(h,'waitstatus','close')
   ReportStruct = getappdata(h,'ReportStruct');
   delete(h)
   if isempty(ReportStruct)
      return
   end
   try
      CDir = cd;
      Dir  = which('Cubatch.m');
      Dir  = Dir(1:max(find(Dir == '\')));
      cd([Dir,'Reports'])             
      [Report_file{2:-1:1}] = uiputfile('*.txt','Save report file');
      cd(CDir)
   catch
      [Report_file{2:-1:1}] = uiputfile('*.txt','Save report file');
   end;         
   if Report_file{1} ~= 0
      ReportStruct.file = [Report_file{:}];
      if ~strcmp(ReportStruct.file(max(end-3,1):end),'.txt')
         ReportStruct.file = [ReportStruct.file,'.txt'];
      end
      Fil_id = fopen(ReportStruct.file,'w');
      if Fil_id == -1
         uiwait(errordlg('Unable to open file','Report'))
         return
      end
      fwrite(Fil_id,['CuBatch Report' 10 10],'char');
      fwrite(Fil_id,[vers 10 10],'char');
      fwrite(Fil_id,['Report saved on: ' date 10 10],'char');
      if any(ReportStruct.data)
         fwrite(Fil_id,['Data information' 10],'char');
         if isempty(ModelOut(1).data.dataset) | ...
               ~isa(ModelOut(1).data.dataset,'cbdataset') | ...
               (isa(ModelOut(1).data.dataset,'cell') & ~isa(ModelOut(1).data.dataset{1},'cbdataset'))
            X = getappdata(MainFig,'X');
            fwrite(Fil_id,['Loaded data' 10],'char');
         else
            fwrite(Fil_id,['Modelled data' 10],'char');
            X    = {};
            if isa(ModelOut(1).data.dataset,'cell')
               X{1} = ModelOut(1).data.dataset{1};
               if length(ModelOut(1).data.dataset) > 1
                  if length(ModelOut) > 1
                     for i = 1:length(ModelOut)
                        X{i + 1} = ModelOut(i).data.dataset{2};
                     end
                  else
                     X{2} = ModelOut(1).data.dataset{2};
                  end
               end
            else
               X{1} = ModelOut(1).data.dataset;
            end
         end
         %labels in the first mode
         fwrite(Fil_id,[10 'Mode 1',10],'char');
         fwrite(Fil_id,['Name      : ',X{1}.modenames{1},char(10)],'char');
         fwrite(Fil_id,['Meas. unit: ',X{1}.measunits{1},char(10)],'char');
         fwrite(Fil_id,['Def. label: ',X{1}.deflab{1},char(10)],'char');
         s = length(X{1}.axislabels{1});
         fwrite(Fil_id,[['Axis label: ';ones(s-1,12)*' '],char(X{1}.axislabels{1}),ones(s,1)*char(10)]','char');
         if ReportStruct.maxlabels
            n = min(ReportStruct.maxlabels,size(X{1},1));
            fwrite(Fil_id,[['Labels    : ';ones(n-1,12)*' '],char(X{1}(1:n).labels{1}),ones(n,1)*char(10)]','char');
         end
         fwrite(Fil_id,10,'char');
         
         for i = 1:length(X)
            if i == 1
               fwrite(Fil_id,['Data array: X' 10],'char');
            elseif i == 2
               fwrite(Fil_id,['Data array(s): Y' 10],'char');
            end
            %Data info for i-th element of X data structure
            fwrite(Fil_id,10,'char');
            fwrite(Fil_id,['Name          : ',X{i}.name,char(10)],'char');
            fwrite(Fil_id,['Type          : ',X{i}.type,char(10)],'char');
            if strcmp(X{1}.type,'Batch')
               Dims = [int2str(length(X{i}.userdata)) ' --- ',int2str(X{1}.dims(2:end))];
               Batch = 1;
            else
               Dims = int2str(X{i}.dims);
            end
            fwrite(Fil_id,['Dimensions    : [', Dims,']',char(10)],'char');
            NaNPerc = isnan(X{i});
            fwrite(Fil_id,['Missing %     : ' num2str(100 * sum(NaNPerc(:)) / prod(size(NaNPerc)),'%3.2f %%'),char(10)],'char');
            w = char(X{i}.author);
            fwrite(Fil_id,[['Author        : ';32*ones(size(w,1)-1,16)],w,10*ones(max(size(w,1),1),1)]','char');
            d = char(X{i}.description);
            fwrite(Fil_id,[['Description   : ';32*ones(size(d,1)-1,16)],d,10*ones(max(size(d,1),1),1)]','char');
            %Write batch lengths if cbdataset if of 'Batch' type
            if strcmp(X{i}.type,'Batch')
               Lengths = X{i}.userdata;
               fwrite(Fil_id,['Batch lengths : ' int2str(Lengths(:)'),'  ',char(10)],'char');
            end
            %Format last modified.
            w = '';
            Modified = X{i}.modified;
            for j = 3:-1:1
            
               a = num2str(Modified(j));
               if length(a) == 1,a = ['0',a];end
               if j < 3,a = ['/',a];end
               w = [w,a];
               
            end
            w=[w,', '];
            for j = 4:6
         
               a = num2str(Modified(j));
               if length(a) == 1,a=['0',a];end
               if j > 4,a = [':',a];end
               w = [w,a];
               
            end
            fwrite(Fil_id,['Last modified : ',w,char(10)],'char');
            fwrite(Fil_id,[32 32 10],'char');
            for j = 2:ndims(X{i})
            
               %labels in the second and further modes
               fwrite(Fil_id,[32 32 10],'char');
               fwrite(Fil_id,[sprintf('Mode %i',j),char(10)],'char');
               fwrite(Fil_id,['Name      : ',X{i}.modenames{j},char(10)],'char');
               fwrite(Fil_id,['Meas. unit: ',X{i}.measunits{j},char(10)],'char');
               fwrite(Fil_id,['Def. label: ',X{i}.deflab{j},char(10)],'char');
               s = length(X{1}.axislabels{j});
               fwrite(Fil_id,[['Axis label: ';ones(s-1,12)*' '],char(X{i}.axislabels{j}),ones(s,1)*char(10)]','char');
               if ReportStruct.maxlabels
                  n = min(ReportStruct.maxlabels,size(X{i},j));
                  fwrite(Fil_id,[['Labels    : ';ones(n - 1,12)*' '],char(X{i}.labels{j}(1:n)),ones(n,1)*char(10)]','char');
               end
               fwrite(Fil_id,[32 32 10],'char');
               
            end
            
         end
         
      end
      
      if any(ReportStruct.m_info) | any(ReportStruct.m_results) | any(ReportStruct.v_info) | any(ReportStruct.v_results) | any(ReportStruct.p_results)
         switch ModelOut(1).modelname
            case 'PARAFAC'
               PARAFACReport(MainFig,ReportStruct,Fil_id);
               
            case {'nPLS','PLS'}
               nPLS1Report(MainFig,ReportStruct,Fil_id);
               
            case 'OPA'
               OPAReport(MainFig,ReportStruct,Fil_id);
               
            case 'OPA3D'
               OPA3Report(MainFig,ReportStruct,Fil_id);
               
            case 'PARAFAC2'
               PARAFAC2Report(MainFig,ReportStruct,Fil_id);
               
            case 'Tucker',
               TuckerReport(MainFig,ReportStruct,Fil_id);
               
            case 'PCA',
               ACPReport(MainFig,ReportStruct,Fil_id);
               
            case 'IV-PCA',
               ACPVIReport(MainFig,ReportStruct,Fil_id);
               
            case 'IV-Tucker'
               TuckerVIReport(MainFig,ReportStruct,Fil_id);
               
         end
         
      end
      fclose(Fil_id);
      
   else
      return
   end
   
elseif isnan(MainFig)
   
   ReportStruct = getappdata(gcbf,'ReportStruct');
   switch ctag
      case 'ok'
         set(gcbf,'waitstatus','close')
         return
         
      case {'cancel','window'}
         setappdata(gcbf,'ReportStruct',[])
         set(gcbf,'waitstatus','close')
         return
         
      case 'data'
         if cval
            set(findobj(gcbf,ty,ed,tg,'maxlabels'),en{:})
            ReportStruct.data(1) = 1;
         else
            set(findobj(gcbf,ty,ed,tg,'maxlabels'),dis{:})
            ReportStruct.data(1) = 0;
         end
         
      case 'maxlabels'
         try

            x = str2num(get(gcbo,st));
            if isempty(x)
               uiwait(errordlg({[get(gcbo,st), ' is not valid.'];'Value must be an integer'},'Invalid input'));
               x = 10;
               set(gcbo,st,'10')
            elseif rem(x,1)
               uiwait(errordlg({[get(gcbo,st), ' is not valid.'];'Value must be an integer'},'Invalid input'));
               x = 10;
               set(gcbo,st,'10')
            end
            ReportStruct.maxlabels = x;
            
         catch
            uiwait(errordlg({[get(gcbo,st), 'is not valid.';'Value must be an integer']},'Invalid input'));
            return
         end
         
      case 'm_info'
         ReportStruct.m_info(1) = cval;
         
      case 'm_results'
         ReportStruct.m_results(1) = cval;
         
      case 'v_info'
         ReportStruct.v_info(1) = cval;
         
      case 'v_results'
         ReportStruct.m_results(1) = cval;
         
      case 'model'
         if cval
            set(findobj(gcbf,ty,ch,tg,'m_info'),en{:},va,1)
            set(findobj(gcbf,ty,ch,tg,'m_results'),en{:},va,1)
            ReportStruct.m_info(1)    = 1;
            ReportStruct.m_results(1) = 1;
         else
            set(findobj(gcbf,ty,ch,tg,'m_info'),dis{:},va,0)
            set(findobj(gcbf,ty,ch,tg,'m_results'),dis{:},va,0)
            ReportStruct.m_info(1)    = 0;
            ReportStruct.m_results(1) = 0;
         end
         
      case 'validation'
         if cval
            set(findobj(gcbf,ty,ch,tg,'v_info'),en{:},va,1)
            set(findobj(gcbf,ty,ch,tg,'v_results'),en{:},va,1)
            ReportStruct.v_info(1)    = 1;
            ReportStruct.v_results(1) = 1;
         else
            set(findobj(gcbf,ty,ch,tg,'v_info'),dis{:},va,0)
            set(findobj(gcbf,ty,ch,tg,'v_results'),dis{:},va,0)
            ReportStruct.v_info(1)    = 0;
            ReportStruct.v_results(1) = 0;
         end
         
      case 'projection'
         ReportStruct.p_results(1) = 0;
         
   end
   setappdata(gcbf,'ReportStruct',ReportStruct);
   
else
   error('Wrong value for MainFig')
end
