function [A,H,C,P,fit]=parafac2app(X,F,Constraints,Options,A,H);
%   Apply PARAFAC2 model to a new set of samples hold in X
%   [A,H,C,P,fit]=parafac2app(X,F,Constraints,Options,A,H);
%   Check PARAFAC2.m from the nway toolbox for syntax and use. Some options
%   (such as Options(5) have been deactivated for simplicity.
%   A and H instead of initial values are the model's corresponding
%   matrices and will not be updated along the process.
%
%   Type: PARAFAC2demo for an example.
%
%   Derived from the PARAFAC2.m of the nWAY toolbox. It requires version
%   2.1 (or compatible) of such toolbox to work adequately
%   Author: Giorgio Tomasi
%         Royal Agricolture University, MLI, LMT
%         1958 Frederiksberg C
%         Danmark
%         email: gt@kvl.dk


ShowFit  = 100; % Show fit every 'ShowFit' iteration
NumRep   = 10; %Number of repetead initial analyses
NumItInRep = 80; % Number of iterations in each initial fit
if ~(length(size(X))==3|iscell(X))
   error(' X must be a three-way array or a cell array')
end
%set random number generators
randn('state',sum(100*clock));
rand('state',sum(100*clock));

if nargin < 4
   Options = zeros(1,5);
end
if length(Options)<5
   Options = Options(:);
   Options = [Options;zeros(5-length(Options),1)];
end

% Convergence criterion
if Options(1)==0
   ConvCrit = 1e-7;
else
   ConvCrit = Options(1);
end
% Maximal number of iterations 
if Options(2)==0
   MaxIt = 2000;
else
   MaxIt = Options(2);
end
% Initialization method
initi = Options(3);

if nargin<3
   Constraints = [0 0];
end
if length(Constraints)~=2
   Constraints = [0 0];
   disp(' Length of Constraints must be two. It has been set to zeros')
end
% Modify to handle GPA (Constraints = [10 10]);
if Constraints(2)==10
   Constraints(1)=0;
   ConstB = 10;
else
   ConstB = 0;
end

% Make X a cell array if it isn't
if ~iscell(X)
   for k = 1:size(X,3)
      x{k} = X(:,:,k);
   end
   X = x;
   clear x
end
I = size(X{1},1);
K = max(size(X));

% Find missing and replace with average 
MissingElements = 0;
MissNum=0;AllNum=0;
for k = 1:K
   x=X{k};
   miss = sparse(isnan(x));
   MissingOnes{k} = miss;
   if any(miss(:))
      MissingElements = 1;
      % Replace missing with mean over slab (not optimal but what the heck)
      % Iteratively they'll be replaced with model estimates
      x(find(miss)) = mean(x(find(~miss)));
      X{k} = x;
      MissNum = MissNum + prod(size(find(miss)));
      AllNum = AllNum + prod(size(x));
   end
end
if MissingElements
   if Options(5)==0
      PercMiss = 100*MissNum/AllNum;
      RoundedOf = .1*round(PercMiss*10);
      disp([' Missing data handled by EM   : ',num2str(RoundedOf),'%'])
   end
end
clear x

% Initialize by ten small runs
bestfit = inf;
if initi==0
   AllFit = [];
   for i = 2:NumRep
      Opt(3) = 2;   % Init with random
      [Nil,Nil,c,p,fit]=parafac2app(X,F,Constraints,Opt,A,H);
      AllFit = [AllFit fit];
      if fit<bestfit
         C=c;
         P=p;
         bestfit = fit;
      end
   end
   AddiOutput.AllFit = AllFit;
   % Initialize by SVD
elseif initi==2
   C = rand(K,F);
else
   error(' Options(2) wrongly specified')
end

if initi~=1
   XtX=X{1}*X{1}'; % Calculate for evaluating fit (but if initi = 1 it has been calculated)
   for k = 2:K
      XtX = XtX + X{k}*X{k}';
   end
end  
fit    = sum(diag(XtX));
oldfit = fit*2;
fit0   = fit;
it     = 0;
Delta = 1;

% Iterative part
while abs(fit-oldfit)>oldfit*ConvCrit & it<MaxIt & fit>1000*eps
   oldfit = fit;
   it   = it + 1;
   
   % Update P
   for k = 1:K
      Qk       = X{k}'*(A*diag(C(k,:))*H');
      P{k}     = Qk*psqrt(Qk'*Qk);
      %  [u,s,v]  = svd(Qk.');P{k}  = v(:,1:F)*u(:,1:F)';
      Y(:,:,k) = X{k}*P{k};
   end
   
   % Update A,H,C using PARAFAC-ALS
   Fac     = parafac(Y,F,[1e-4 0 0 0 NaN 2500],[Constraints(1) ConstB Constraints(2)],{A,H,C},[1 1 0]);
   C       = Fac{3};
   [fit,X] = pf2fit(X,A,H,C,P,K,MissingElements,MissingOnes);
   
end



function [fit,X]=pf2fit(X,A,H,C,P,K,MissingElements,MissingOnes);

% Calculate fit and impute missing elements from model

fit = 0;
for k = 1:K
   M   = A*diag(C(k,:))*(P{k}*H)';
   % if missing values replace missing elements with model estimates
   if nargout == 2 
      if any(MissingOnes{k})
         x=X{k};
         x(find(MissingOnes{k})) = M(find(MissingOnes{k}));
         X{k} = x;
      end
   end
   fit = fit + sum(sum(abs (X{k} - M ).^2));
end


function X = psqrt(A,tol)

% Produces A^(-.5) even if rank-problems

[U,S,V] = svd(A,0);
if min(size(S)) == 1
   S = S(1);
else
   S = diag(S);
end
if (nargin == 1)
   tol = max(size(A)) * S(1) * eps;
end
r = sum(S > tol);
if (r == 0)
   X = zeros(size(A'));
else
   S = diag(ones(r,1)./sqrt(S(1:r)));
   X = V(:,1:r)*S*U(:,1:r)';
end

