function [Axes,MO] = Plot_XY_Loadings(PlotStruct,ModelOut,MainFig)
% function [Axes,MO] = Plot_XY_Loadings (PlotStruct,ModelOut,MainFig);
% 
% Description:
% Plots the scores/loadings for the mode specified in PlotStruct.nummode according to what selected in
% the control figure opened with nPLS1PlotControls. In the 1D plot the parameters are plot versus the
% scalars of the corresponding method. The choice of the LV(s) for the axes is taken from
% PlotStruct.factors. When the model has been applied to new data this plot automatically applies to
% the new data. For the axes specified by Axes the 'ModelOut' application-defined data is set, and it
% contains a copy of ModelOut.(employed by nPLS1DisplayInfo)
% 
% 
% Inputs: 
% PlotStruct: PlotStruct structure
% ModelOut: ModelOut structure for the model selected via the menus in the main window.
% MainFig: main figure handle
% 
% 
% Outputs:
% Axes: handle of the axes scores/loadings.
% MO: empty
% 
% 
% Called by:
% Model_nPLS1\nPLS1ActivatePlots
% NB. The actual call is made from nPLS1Plot via feval and not from Model_nPLS1\nPLS1ActivatePlots.
% 
% 
% Subroutines:
% Internal: None
% External: npls1plotcontrols, plotmodel, plotvalidation, resamplingextract, skcell
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 16-Oct-2002 14:37:08
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
%

MO   = [];
Axes = [];
if isempty(findobj(0,'type','figure','userdata','nPLS1 Plot Control'))
   %Called by the main window.
   PlotStruct.plotdims = get(gcbo,'userdata');
   PlotStruct.nummode  = get(get(gcbo,'parent'),'userdata');
   if ~PlotStruct.factors(1)
      PlotStruct.factors(1) = 1;
   end
   [Nil,Pos]           = max(find([ModelOut.model.nbfactors]));
   ModelOut.model      = ModelOut.model(Pos);
   if ~isempty(ModelOut.validation)
      ModelOut.validation = ModelOut.validation(Pos);
   end
   H = nPLS1PlotControls(PlotStruct,ModelOut,MainFig);
   waitfor(H,'waitstatus','Ok')
   if ~ishandle(H)
      Axes = [];
      return
   else 
      PlotStruct = getappdata(H,'PlotStr');
   end
end
[DSp,Fp]  = deal([]);
Pred_Flag = ~isempty(ModelOut.prediction) & PlotStruct.nummode == 1;
switch PlotStruct.additional.type2
   case 'X'
      DSm = ModelOut.data.dataset{1};
      Fm  = ModelOut.model.xfactors{PlotStruct.nummode};
      T   = ['\\bf',ModelOut.data.dataset{2}.labels{2}{1},' - X '];
      if ~isempty(ModelOut.validation)
         Fv  = ModelOut.validation.xfactors{PlotStruct.nummode};
      end
      if Pred_Flag
         Fp     = ModelOut.prediction.xfactors{1};
         LabObj = ModelOut.prediction.data.labobj{1};
      else
         LabObj  = DSm.labobj{PlotStruct.nummode};
      end
   case 'Y'
      DSm     = ModelOut.data.dataset{2};
      Fm      = ModelOut.model.yfactors{PlotStruct.nummode};
      LabObj  = DSm.labobj{PlotStruct.nummode};
      T   = ['\\bf',DSm.labels{1}{1},' - Y '];
      if PlotStruct.additional.validation == 1
         PlotStruct.additional.validation = 0;
      end
      if ~isempty(ModelOut.validation)
         Fv  = ModelOut.validation.yfactors{PlotStruct.nummode};
      end   
      if Pred_Flag
         Fp  = ModelOut.prediction.yfactors{1};
         LabObj  = ModelOut.prediction.data.labobj{1};  
      end
      if PlotStruct.nummode == 1
         T = [T,'Scores: '];
      else
         T = [T,'Loadings: '];
      end
end
figure(MainFig)
Axes = axes('tag','cbaxes','nextplot','replacechildren');
if PlotStruct.additional.validation & (~isempty(ModelOut.validation) | ~isempty(ModelOut.prediction))
   [x,y,z] = deal([]);
   if PlotStruct.nummode == 1
      
      if PlotStruct.additional.validation == 2 & ~isempty(ModelOut.validation)%Replicates
         
         if strcmp(ModelOut(1).validation.method,'rboo')
            if PlotStruct.plotdims >= 1,x = squeeze(Fv(:,PlotStruct.factors(1),:));end
            if PlotStruct.plotdims >= 2,y = squeeze(Fv(:,PlotStruct.factors(2),:));end
            if PlotStruct.plotdims >= 3,z = squeeze(Fv(:,PlotStruct.factors(3),:));end
         else
            if PlotStruct.plotdims >= 1,x = ResamplingExtract(Fv,PlotStruct.factors(1),ModelOut.validation.segments);end
            if PlotStruct.plotdims >= 2,y = ResamplingExtract(Fv,PlotStruct.factors(2),ModelOut.validation.segments);end
            if PlotStruct.plotdims >= 3,z = ResamplingExtract(Fv,PlotStruct.factors(3),ModelOut.validation.segments);end
         end
         
      elseif PlotStruct.additional.validation(1) == 1 & isempty(ModelOut.prediction) & any(strcmp({'loo','test'},ModelOut.validation.method))
         if PlotStruct.plotdims >= 1,x = Fv(:,PlotStruct.factors(1),end);end
         if PlotStruct.plotdims >= 2,y = Fv(:,PlotStruct.factors(2),end);end
         if PlotStruct.plotdims >= 3,z = Fv(:,PlotStruct.factors(3),end);end
      elseif Pred_Flag
         if PlotStruct.plotdims >= 1,x = Fp(:,PlotStruct.factors(1),end);end
         if PlotStruct.plotdims >= 2,y = Fp(:,PlotStruct.factors(2),end);end
         if PlotStruct.plotdims >= 3,z = Fp(:,PlotStruct.factors(3),end);end
      end
      
   else
      if PlotStruct.plotdims >= 1,x = squeeze(Fv(:,PlotStruct.factors(1),:));end
      if PlotStruct.plotdims >= 2,y = squeeze(Fv(:,PlotStruct.factors(2),:));end
      if PlotStruct.plotdims >= 3,z = squeeze(Fv(:,PlotStruct.factors(3),:));end
   end
   if isempty(y)
      y = x;
      x = LabObj.scalars;
      xlabel(LabObj.axislabels);
   else
      xlabel(sprintf('Component %i',PlotStruct.factors(1)));
   end
   if isempty(z)
      z = zeros(size(x));
   end
   ylabel(sprintf('Component %i',PlotStruct.factors(2)));
   zlabel(sprintf('Component %i',PlotStruct.factors(3)));
   PlotValidation(x,y,z,PlotStruct,LabObj,Axes)
   switch PlotStruct.plotdims
      case 1
         title(sprintf([T 'Component %i'],PlotStruct.factors(1)));
      case 2
         title(sprintf([T,'Component %i vs Component %i'],PlotStruct.factors(1:2)));
      case 3
         title(sprintf([T,'Comp. %i vs Comp. %i vs Comp. %i'],PlotStruct.factors(1:3)));   
   end
   
end
if any(strcmpi({'on','label','marker','number'},PlotStruct.visible))
   LabObj  = DSm.labobj{PlotStruct.nummode};
   switch PlotStruct.plotdims 
      case 1
         if PlotStruct.nummode == 1 & ~isempty(ModelOut.validation) & strcmp(ModelOut.validation.method,'test')
            x = LabObj.scalars(setdiff(1:size(DSm,1),ModelOut.validation.segments));
         else
            x = LabObj.scalars;
         end
         if PlotStruct.factors(1)
            y = Fm(:,PlotStruct.factors(1));
            title(sprintf([T,'Component %i'],PlotStruct.factors(1)),'fontname','arial','fontsize',10);
         else
            y = Fm;
            title(T(2:end),'fontname','arial','fontsize',10);
         end
         z = 0 * x;
         xlabel(LabObj.axislabels,'fontname','arial','fontsize',9);
         ylabel('A.U.','fontname','arial','fontsize',9)
         
      case 2
         title(sprintf([T,'Component %i vs Component %i'],PlotStruct.factors(1:2)),'fontname','arial','fontsize',10);
         x = Fm(:,PlotStruct.factors(1));
         y = Fm(:,PlotStruct.factors(2));
         z = 0 * x;
         xlabel(sprintf('Component %i (A.U.)',PlotStruct.factors(1)),'fontname','arial','fontsize',9);
         ylabel(sprintf('Component %i (A.U.)',PlotStruct.factors(2)),'fontname','arial','fontsize',9);
         
      otherwise
         title(sprintf([T,'Comp. %i vs Comp. %i vs Comp. %i'],PlotStruct.factors(1:3)),'fontname','arial','fontsize',10);
         x = Fm(:,PlotStruct.factors(1));
         y = Fm(:,PlotStruct.factors(2));
         z = Fm(:,PlotStruct.factors(3));
         xlabel(sprintf('Component %i (A.U.)',PlotStruct.factors(1)),'fontname','arial','fontsize',9);
         ylabel(sprintf('Component %i (A.U.)',PlotStruct.factors(2)),'fontname','arial','fontsize',9);
         zlabel(sprintf('Component %i (A.U.)',PlotStruct.factors(3)),'fontname','arial','fontsize',9);
         
   end
   if PlotStruct.nummode == 1 & ~isempty(ModelOut.validation) & strcmp(ModelOut.validation.method,'test') 
      Ind = setdiff(1:size(ModelOut.validation.xfactors{1},1),ModelOut.validation.segments);
      PlotModel(x,y,z,PlotStruct,LabObj(Ind),Axes)
   else
      PlotModel(x,y,z,PlotStruct,LabObj,Axes)
   end
   if ~PlotStruct.factors(1)
      Factors = skcell(1,size(Fm,2),'Factor')';
      legend(flipud(findobj(Axes,'tag','model','type','line')),Factors)
   end
end
set(Axes,'position',[0.2464,0.0377,0.7312,0.9127])
grid(PlotStruct.grid)
setappdata(Axes,'ModelOut',ModelOut)
return
