function [Axes,MO] = Plot_Partial_RSS(PlotStruct,ModelOut,MainFig)
% function [Axes,MO] = Plot_Partial_RSS (PlotStruct,ModelOut,MainFig);
% 
% Description:
% It plots the sum of squares of the residuals vs the scalars in the mode specified in PlotStruct.nummode.
% Additional selections can be made so that subsets of variables (or sample/batches) are used.
% If only one sample/batch is selected and the mode is not the first, confidence limits, computed with Residuals_CL, are automatically displayed.
% If 'leave one out' validation has been calculated it is asked which scores are to be used: those from the complete models or the "predicted" ones.
% When the model has been applied to new data this plot automatically applies to the new data; the confidence limits are computed on the original data set.
% This plot is currently available only for the X matrix.
% For the axes specified by Axes the 'ModelOut' application-defined data is set, and it contains a copy of ModelOut.(employed by nPLS1DisplayInfo)
%  
% 
% Inputs: 
% PlotStruct: PlotStruct structure
% ModelOut: ModelOut structure for the model selected via the menus in the main window.
% MainFig: main figure handle
% 
% 
% Outputs:
% Axes: handle of the axes where the Residuals are plot.
% MO: ModelOut structure with the updated statistic. Empty if no new statistic was computed.
% 
% 
% Called by:
% Model_nPLS1\nPLS1ActivatePlots
% NB. The actual call is made from nPLS1Plot via feval and not from Model_nPLS1\nPLS1ActivatePlots.
% 
% Subroutines:
% Internal: None
% External: cenwindow, nshape, residuals_cl, selectmodel, dispinfo
% NB DispInfo is set as the 'ButtonDownFcn' of the line plots when the residuals are displayed versus any mode but the first one.
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 15-Oct-2002 11:25:24
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
% 

Axes          = [];
MO            = [];
Rank          = 1;
Pos           = 1;
Dim           = 1;
[IndX,IndX2]  = deal(repmat({':'},ndims(ModelOut(1).info.content),1));
Various_Modes = 1:ndims(ModelOut.info.content);
T             = get(gcbo,'userdata');
[Mode,Type]   = deal(T{:});
YAF           = 0;
ModeNames     = ModelOut.info.content.modenames;
ystats        = [];
Ok            = 1;
Model_Val     = 'calibration';
if ~isempty(ModelOut.prediction)
   %If there are projections
   Model_Val = 'external set';
elseif isempty(ModelOut.prediction) & ~isempty(ModelOut.validation) & strcmp(ModelOut.validation(Pos).method,'test')
   
   %If there is a test set and it is mode 1
   if Mode == 1
      Model_Val = 'both';
   else
      %Use only calibration samples
      IndX2{1} = setdiff(1:size(ModelOut.data.dataset,1),ModelOut.validation(Pos).segments);
   end
   
end

%Choose model
if length(ModelOut.model) > 1
   Ranks      = [ModelOut.model.nbfactors]';
   [Rank,Pos] = SelectModel(Ranks);
   if ~Rank,return,end
end

%Labels etcf
L = ModelOut.info.content.labels;
if ~isempty(ModelOut.prediction)
   DS = ModelOut.prediction.data;
   L{1} = DS.labels{1};
else
   DS = ModelOut.data.dataset{1};
end

%Select subset
while Dim
   
   Dim = SelectSubSet(ModeNames);
   if isnan(Dim)
      return
   elseif Dim 
      
      YAF      = 1; %Yet another flag
      ListProp = {'initialvalue',1:length(L{Dim}),'selectionmode','multiple','name','Select interval','promptstring','Select interval'};
      ListStr  = L{Dim};
      if ~isempty(ModelOut.prediction) & Dim == 1
         %If there are projections
         Model_Val = 'external set';
         [Int,Ok] = listdlg('Liststring',ListStr,ListProp{:});
      elseif ~isempty(ModelOut.validation) & isempty(ModelOut.prediction) & Dim == 1 
         
         if strcmp('test',ModelOut.validation(1).method)
            
            if Mode ~= 1
               
               %If there are no projections, but there is test set validation and the RSS is not plot
               %against the first mode, but the subset is being selected in the first mode
               %Select calibration/test set
               Model_Val = questdlg('Choose predictions','Which samples shall be used?','Calibration set','Test set','Cancel','Calibration set');
               switch Model_Val
                  case 'Cancel'
                     return
                     
                  case 'Test set'
                     ListStr     = ModelOut.data.dataset{1}(ModelOut.validation(1).segments).labels{Dim};
                     ListProp{2} = 1:length(ListStr);
                     [Int,Ok]    = listdlg('Liststring',ListStr,ListProp{:});
                     Int         = ModelOut.validation(1).segments(Int);
                     Model_Val   = 'validation';
                     
                  case 'Calibration set'
                     Fic         = setdiff(1:size(ModelOut.data.dataset{1},1),ModelOut.validation(1).segments);
                     Model_Val   = 'calibration'
                     ListStr     = ModelOut.data.dataset{1}(Fic).labels{Dim};
                     ListProp{2} = 1:length(ListStr);
                     [Int,Ok]    = listdlg('Liststring',ListStr,ListProp{:});
                     IndX2{1}    = Fic(Int);
                     YAF         = 0;
               end
               
            else
               
               %Modify labels
               for i = 1:length(ModelOut.validation(1).segments)
                  ListStr{ModelOut.validation(1).segments(i)} = [ListStr{ModelOut.validation(1).segments(i)},' Test set'];
               end
               Model_Val = 'both';
               [Int,Ok]  = listdlg('Liststring',ListStr,ListProp{:});
               
            end
            
         elseif strcmp('loo',ModelOut.validation(1).method)
            
            %If leave one out validation and no projection 
            [Int,Ok]  = listdlg('Liststring',ListStr,ListProp{:});
            if ~Ok,
               Dim = 0;
               return,
            end
            %use model/predictions?
            Model_Val = questdlg('Choose predictions','Which scores shall be used?','calibration','validation','Cancel',Model_Val);
            
         else
            [Int,Ok] = listdlg('Liststring',ListStr,ListProp{:});
         end
         
      else
         [Int,Ok] = listdlg('Liststring',ListStr,ListProp{:});
      end
      if ~Ok | strcmp(Model_Val,'Cancel')
         Dim = 0;
         return,
      end
      IndX{Dim} = Int;
      if YAF
         IndX2{Dim} = Int;
      end
      
   end
   
end

%Compute residuals
%Size = size(ModelOut.info.content(IndX2{:}));
Size = size(ModelOut.info.content);
switch lower(Model_Val)
   case 'calibration'
      X       = reshape(ModelOut.data.dataset{1}.data,Size);
      X       = X(IndX2{:});
      Size(1) = size(ModelOut.model(Pos).xpred,1);
      M       = reshape(ModelOut.model(Pos).xpred,Size);
      M       = M(IndX{:});
      
   case 'both'
      %Final array
      M                  = repmat(NaN,size(ModelOut.info.content));
      %Get test set
      Fic                = IndX{1};
      IndX{1}            = intersect(IndX{1},ModelOut.validation(1).segments);
      Size_fic           = size(ModelOut.info.content(IndX{:}));
      Mt                 = reshape(ModelOut.validation(Pos).xpred,Size);
      M(IndX{:})         = Mt(IndX{:});
      %Get calibration test
      CalSet                = setdiff(1:size(ModelOut.info.content,1),ModelOut.validation(1).segments);
      [CalSet,IndX{1}]      = intersect(CalSet,Fic);
      Mc                    = reshape(ModelOut.model(Pos).xpred,Size);
      M(CalSet,IndX{2:end}) = Mc(IndX{:});
      IndX{1}               = Fic;
      %M = M(IndX{:}); %necessary?
      %Get original data
      X = reshape(ModelOut.data.dataset{1}.data,Size);
      X = X(IndX{:});
      
   case 'validation' 
      X = reshape(ModelOut.data.dataset{1}.data,Size);
      X = X(IndX{:});
      M = reshape(ModelOut.validation(Pos).xpred,Size);
      M = M(IndX{:});
      
   case 'external set'
      Size(1) = size(ModelOut.prediction.data,1);
      X       = reshape(ModelOut.prediction.data.data,Size);
      X       = X(IndX{:});
      M       = reshape(ModelOut.prediction.xpred,Size);
      M       = M(IndX{:});
      
end
M(isnan(X)) = 0;
X(isnan(X)) = 0;
switch Type
   case 'Residuals'
      R = X - M;
      y = R.^2;
      for i = 1:ndims(X)
         if i ~= [1 Mode]
            y = sum(y,i);
         end
      end
      y = squeeze(y);
      
   case '% Var'
      R = X - M;
      y = sum(nshape(R.^2,Mode),2);
      y = 100 - 100*y./sum(nshape(X.^2,Mode),2);
      
      
   case 'Congruence'
      y = sum(normit(nshape(X,Mode)').*normit(nshape(M,Mode)'));
      
end
if Mode == 1 & strcmp(lower(Model_Val),'external set')
   x = ModelOut.prediction.data(IndX{Mode}).scalars{Mode};
else
   x = ModelOut.info.content.scalars{Mode}(IndX{Mode});
end
Leg  = DS(IndX{1}).labels{1};

%Create axes
Axes = axes('tag','cbaxes','nextplot','replacechildren','yscale','log');

%Plot
if Mode == 1
   
   if strcmp(lower(Model_Val),'both') %Mode is necessarily 1
      
      Nil           = intersect(setdiff(1:size(ModelOut.data.dataset{1},1),ModelOut.validation(1).segments),IndX{1});
      [Nil,Index]   = intersect(IndX{1},Nil);
      Fic_Y         = NaN * y;
      Fic_Y(Index)  = y(Index);
      if length(x) < 20
         set(gca,'yscale','linear')
         Handle = bar(x,Fic_Y);
         set(Handle,'marker','none','facecolor',[0 0 0.5625])
      else
         Handle = plot(x,Fic_Y,'-x');
         set(Handle,'marker','none','color',[0 0 0.5625])
      end
      set(gca,'nextplot','add')
      Index2        = setdiff(1:size(y,1),Index);
      Fic_Y         = NaN * y;
      Fic_Y(Index2) = y(Index2);
      if length(x) < 20
         set(gca,'yscale','linear')
         h = bar(x,Fic_Y);
         set(h,'facecolor',[0.502 0 0])
      else
         h = plot(x,Fic_Y,'o-');
         set(h,'color',[0.502 0 0])
      end
      hold off
      Model_Val = 'cal. set and test set';
      legend('Cal. set','Test set')
      
   else
      
      if length(x) < 20
         set(gca,'yscale','linear')
         Handle = bar(x,y);
         set(Handle,'marker','none','facecolor',[0 0 0.5625])
      else
         Handle = plot(x,y');
         set(Handle,'marker','none','color',[0 0 0.5625])
      end
      
   end
   
else
   
   if length(x) < 20 & any(strcmp({'Congruence','% Var'},Type))
      set(gca,'yscale','linear')
      Handle = bar(x,y);
      set(Handle,'marker','none','facecolor',[0 0 0.5625])
   elseif isequal(Type,'Residuals')
      Handle = plot(x(:),y');
      set(Handle,'marker','none','buttondownfcn','DispInfo(gcbo)')
      set(Handle,{'tag'},Leg)
   else
      Handle = plot(x,y);
      set(Handle,'marker','none','color',[0 0 0.5625])
   end
   
end

switch Type
   case 'Residuals'
      
      %Compute and plot confidence limits
      if Mode ~= 1
         IndX3 = IndX; 
         %Don't use test set
         if ~isempty(ModelOut.validation) & strcmp(ModelOut.validation(1).method,'test')
            IndX3{1} = setdiff(1:size(ModelOut.data.dataset{1},1),ModelOut.validation(1).segments);
         else
            IndX3{1} = 1:size(ModelOut.data.dataset{1},1);
         end
         Size    = size(ModelOut.info.content);
         X       = reshape(ModelOut.data.dataset{1}.data,Size);
         Size(1) = size(ModelOut.model(Pos).xpred,1);
         M       = reshape(ModelOut.model(Pos).xpred,Size);
         for i = 1:size(R,Mode)
            IndX3{Mode} = i;
            R           = nshape(X(IndX3{:}) - M(:,IndX3{2:end}),1);
            ystats(:,i) = Residuals_CL(R,[0.95 0.99]);
         end
         hold on
         Handle2 = plot(x,ystats(1,:),'g',x,ystats(2,:),'r:');
         set(Handle2,'linewidth',2,'color',[0 0.5020 0.7529])
         if length(Handle) <= 10
            legend([Handle;Handle2],Leg{:},'0.95 Limit','0.99 Limit',0)
         else
            legend(Handle2,'0.95 Limit','0.99 Limit',0)
         end
         hold off
         
      end
      xlabel(ModelOut.info.content.modenames{Mode},'fontname','arial','fontsize',9);
      ylabel('Q','fontname','arial','fontsize',9)
      if length(IndX{1}) == 1 & ~ischar(IndX{1}) 
         if isempty(ModelOut.prediction)
            title(['\bfSum of squared residuals (' lower(Model_Val) ') - ',ModelOut.data.dataset{1}(IndX2{1}).labels{1}{1}],'fontname','arial','fontsize',10);
         else length(IndX{1}) == 1 & ~ischar(IndX{1}) & ~isempty(ModelOut.prediction)
            title(['\bfSum of squared residuals (' lower(Model_Val) ') - ',ModelOut.prediction.data(IndX2{1}).labels{1}{1}],'fontname','arial','fontsize',10);
         end
      else
         title(['\bfSum of squared residuals (' lower(Model_Val) ')'],'fontname','arial','fontsize',10);
      end
      
   case '% Var'
      xlabel(ModelOut.info.content.modenames{Mode},'fontname','arial','fontsize',9);
      ylabel('% Expl. var.','fontname','arial','fontsize',9)
      set(gca,'yscale','linear')
      if length(IndX{1}) == 1 & ~ischar(IndX{1})
         if isempty(ModelOut.prediction)
            title(['\bfExpl. var. (' lower(Model_Val) ') - ',ModelOut.data.dataset{1}(IndX2{1}).labels{1}{1}],'fontname','arial','fontsize',10);
         else length(IndX{1}) == 1 & ~ischar(IndX{1}) & ~isempty(ModelOut.prediction)
            title(['\bfExpl. var. (' lower(Model_Val) ') - ',ModelOut.prediction.data(IndX2{1}).labels{1}{1}],'fontname','arial','fontsize',10);
         end
      else
         title(['\bfExpl. var. (' lower(Model_Val) ')'],'fontname','arial','fontsize',10);
      end
      
   case 'Congruence'
      set(gca,'yscale','linear')
      xlabel(DS.axislabels{Mode},'fontname','arial','fontsize',9);
      ylabel('Slab-wise congruence','fontname','arial','fontsize',9)
      title(['\bfQuality (' lower(Model_Val) ')'],'fontname','arial','fontsize',10);
      
end

%Adjust axes limits
if min(y) < 0
   Ay(1) = 1.03*min([y(:);ystats(:)]);
   Ay(2) = 0.97*max([y(:);ystats(:)]);
else
   Ay(1) = 0.97*min([y(:);ystats(:)]);
   Ay(2) = 1.03*max([y(:);ystats(:)]);
end
if Ay(1) == Ay(2) 
   Ay(1) = Ay(1)-0.5;
   Ay(2) = Ay(2)+0.5;
end
if strcmp(Type,'Congruence')
   Ay(1) = max(-1,Ay(1));
   Ay(2) = min(1,Ay(2));
elseif strcmp(Type,'% Var')
   Ay(2) = min(100,Ay(2));
end
Ax = get(Axes,'xlim');
set(Axes,'ylim',Ay,'xlim',Ax)

%Copy in the current axes the ModelOut corresponding to the plots
ModelOut.model = ModelOut.model(Pos);
if ~isempty(ModelOut.validation)
   ModelOut.validation = ModelOut.validation(Pos);
end
setappdata(Axes,'ModelOut',ModelOut)
return

%------------------------------------------------------------------------------------------------------------------------------------------------------

function Dim = SelectSubSet(ModeNames)
h = cenwindow([0.5029 0.5814 0.6402],'Select subset','set(gcbf,''waitstatus'',''close'',''userdata'',0)',150,40*(length(ModeNames) + 1)+10);
set(h,'userdata',0,'toolbar','none','closerequestfcn','set(gcbf,''waitstatus'',''close'',''userdata'',NaN)')
for i = length(ModeNames):-1:1
   uicontrol('position',[10 50 + (i-1) * 40 130 30],'string',ModeNames{length(ModeNames)-i+1},'callback',sprintf('set(%i,''waitstatus'',''close'',''userdata'',%i)',h,length(ModeNames)-i+1))
end
uicontrol('position',[10 10 130 30],'string','Proceed','callback',sprintf('set(%i,''waitstatus'',''close'')',h))
waitfor(h,'waitstatus','close')
Dim = get(h,'userdata');
delete(h)

