function [Axes,MO] = Plot_D_Statistic(PlotStruct,ModelOut,MainFig)
% function [Axes,MO] = Plot_D_Statistic (PlotStruct,ModelOut,MainFig);
% 
% Description:
% Plots the D statistics for the selected sample/batches in the 'offline' mode and the D statistics vs
% time (i.e. the scalars in the last mode) for the selected sample/batch in the 'online' mode. The mode
% is based on the 'userdata' of the calling menu: 'on' for 'online' and 'off' for 'offline'. The values
% are computed by means of DLims if they where not originally present and saved in
% ModelOut.model.stats.d_on, and ModelOut.model.stats.d_off. When the model has been applied to new
% data this plot automatically applies to the new data. This plot is currently available only for the X
% matrix. For the axes specified by Axes the 'ModelOut' application-defined data is set, and it
% contains a copy of ModelOut.(employed by nPLS1DisplayInfo)
% 
% 
% Inputs: 
% PlotStruct: PlotStruct structure
% ModelOut: ModelOut structure for the model selected via the menus in the main window.
% MainFig: main figure handle
% 
% 
% Outputs:
% Axes: handle of the axes where the D-statistics are plot. 
% MO: ModelOut structure with the updated statistic. Empty if no new statistic was computed.
% 
% 
% Called by:
% Model_nPLS1\nPLS1ActivatePlots
% NB. The actual call is made from nPLS1Plot via feval and not from Model_nPLS1\nPLS1ActivatePlots.
% 
% 
% Subroutines:
% Internal: None
% External: dlims, nplspred, plspred, selectmodel
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 11-Oct-2002 18:47:18
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
% 

%Initial parameters
MO      = [];
On      = strcmp(get(gcbo,'userdata'),'on');
Rank    = 1;
Pos     = 1;
Axes    = [];
Leg     = cell(8,1);
Handle  = -1 * ones(8,1);
TF      = []; %Test set flag
TestPos = [];
IndX    = repmat({':'},ndims(ModelOut.data.dataset{1}),1);
QstOpt  = struct('Default','','Interpreter','tex');

%Select model
if length(ModelOut.model) > 1
   Ranks      = [ModelOut.model.nbfactors]';
   [Rank,Pos] = SelectModel(Ranks);
   if ~Rank,return,end
end

%Get labels etc
if ~isempty(ModelOut.prediction)
   DS = ModelOut.prediction.data;
else
   DS = ModelOut.data.dataset{1};
end
X_Lab  = DS.modenames{1};

%Select which samples' D-statistic should be plot
if size(DS,1) > 1

   ListStr  = DS.labels{1};
   if ~isempty(ModelOut.validation) & isequal('test',ModelOut.validation(1).method) & isempty(ModelOut.prediction)
      for i = 1:length(ModelOut.validation(1).segments)
         ListStr{ModelOut.validation(1).segments(i)} = [ListStr{ModelOut.validation(1).segments(i)},' (Test set)'];
      end
   else
      ListStr = DS.labels{1};
   end
   ListProp = {'selectionmode','multiple','name','Select sample','promptstring','Select sample/batch'};
   if On
      ListProp{2} = 'single';
   end
   [IndX{1},Ok] = listdlg('Liststring',ListStr,ListProp{:});
   if ~Ok,return,end

end
if On
   x  = ModelOut.info.content.scalars{ndims(ModelOut.info.content)};
else
   x  = DS(IndX{1}).scalars{1};
end
TF      = logical(zeros(length(IndX{1}),1));
CalPos  = 1:size(DS,1);
if isempty(ModelOut.prediction) & ~isempty(ModelOut.validation) & isequal(ModelOut.validation.method,'test')
   
   Fic                      = setdiff(1:size(ModelOut.data.dataset{1},1),ModelOut.validation(1).segments);
   [YAPC,CalPos]            = intersect(Fic,IndX{1}); %calibration set
   Fic                      = intersect(1:size(ModelOut.data.dataset{1},1),ModelOut.validation(1).segments);
   [YAPT,TestPos]           = intersect(Fic,IndX{1}); %test set
   TF(length(CalPos)+1:end) = 1;
   
end

%Define title
Ax_Tit ='\bf D statistic';
if strcmpi(ModelOut.info.content.modenames{ndims(ModelOut.info.content)},'Time')

   if ~On
      Ax_Tit = [Ax_Tit ' - offline'];
   else
      X_Lab  = DS.modenames{ndims(DS)};
      if any(TF)
         Ax_Tit = {[Ax_Tit ' - online'],[DS(IndX{1}).labels{1}{1},' - test set']};
      else
         Ax_Tit = [Ax_Tit ' - online - ',DS(IndX{1}).labels{1}{1}];
      end
   end

end

%Change fill-in method?
Ans = ModelOut.info.fillin;
if On & strcmpi(getappdata(MainFig,'Mode'),'advanced')
   QstOpt.Default = ModelOut.info.fillin;
   Prompt         = {['The fill-in method is currently set to \bf',ModelOut.info.fillin,'\rm'],['Choose fill-in method']};
   Ans            = questdlg(Prompt,'Choose fill-in method','Current deviation','Zero','Cancel',QstOpt);
   if strcmp(Ans,'Cancel')
      return
   end
end

%Compute statistics and limits if that is the case
Compute = isempty(ModelOut.model(Pos).stats.s) | ...        %No scores covariance has been computed
      (On & ndims(ModelOut.model(Pos).stats.s) ~= 3) |...   %Score covariance does not have the time (i.e online) dimension
      (~On & isempty(ModelOut.model(Pos).stats.d_off)) |... %D statistics offline has not been computed
      (On & ~strcmp(ModelOut.info.fillin,Ans));
if Compute
   
   %Update fill-in method
   ModelOut.info.fillin = Ans;
   %If validation is available use it for computing the D-contributions limits
   if ~On & ~isempty(ModelOut.validation)
      Val = ModelOut.validation(Pos);
      Seg = ModelOut.validation(Pos).segments;
   else
      Val = [];
      Seg = [];
   end
   %The limits and the covariance matrix are be computed on the NOC data
   if ~isempty(ModelOut.validation) & strcmp('test',ModelOut.validation(Pos).method)
      Data   = reshape(ModelOut(1).data.dataset{1}.data,size(ModelOut(1).info.content));
      Ind    = repmat({':'},ndims(Data),1);
      Ind{1} = setdiff(1:size(ModelOut.data.dataset{1},1),ModelOut.validation(Pos).segments)
      Data   = Data(Ind{:});
   else
      Data = ModelOut.data.dataset{1}.data;
      Data = reshape(Data,size(ModelOut.info.content));
   end
   ModelOut.model(Pos).stats = ...
      DLims([ModelOut.modelname,'Pred'],...%Score computing function
      get(gcbo,'userdata'),...             %Offline/OnLine
      1,...                                %Compute limits based on Data
      Data,...                             %Data
      ModelOut.model(Pos),...              %Model Parameters
      Val,...                              %Validation Parameters
      [.95 .99],...                        %Limits
      ModelOut.info.fillin,...             %Fill In
      Seg,...                              %Segments
      ModelOut.model(Pos).stats,...        %Stats structure
      ModelOut.model(Pos).xpreproc.cen,... %Centring
      ModelOut.model(Pos).xpreproc.scal,...%Scaling
      ModelOut.info.preprocess);
   
   if ~isempty(ModelOut.validation) & isequal(ModelOut.validation.method,'test')

      Data                                                 = reshape(ModelOut(1).data.dataset{1}.data,size(ModelOut(1).info.content));
      Ind                                                  = repmat({':'},ndims(Data),1);
      Ind{1}                                               = ModelOut.validation(Pos).segments;
      Data                                                 = Data(Ind{:});
      Val                                                  = ModelOut.validation(Pos);
      Val.xfactors{1}                                      = Val.xfactors{1}(ModelOut.validation(Pos).segments,:);
      [Val.core,Val.bcoeff]                                = deal(ModelOut.model(Pos).core,ModelOut.model(Pos).bcoeff);
      Val.yfactors(2:length(ModelOut.model(Pos).yfactors)) = ModelOut.model(Pos).yfactors(2:end);
      ModelOut.validation(Pos).stats = ...
         DLims([ModelOut.modelname,'Pred'],...%Score computing function
         get(gcbo,'userdata'),...             %Offline/OnLine
         0,...                                %Compute limits based on Data
         Data,...                             %Data
         Val,...                              %Model Parameters
         [],...                               %Validation Parameters
         [.95 .99],...                        %Limits
         ModelOut.info.fillin,...             %Fill In
         Seg,...                              %Segments
         ModelOut.model(Pos).stats,...        %Stats structure
         ModelOut.model(Pos).xpreproc.cen,... %Centring
         ModelOut.model(Pos).xpreproc.scal,...%Scaling
         ModelOut.info.preprocess);
      
   end

end
if isempty(ModelOut.prediction)

   %y is the D-Statistic if on-line a vector per batch is necessary: D-statistic over time
   if ~isempty(ModelOut.validation) & isequal(ModelOut.validation.method,'test')
   
      if On
         
         if isempty(TestPos)
            y  = ModelOut.model(Pos).stats.d_on(CalPos,:);
         else
            y  = ModelOut.validation(Pos).stats.d_on(TestPos,:);
         end
         
      else
        
         y = [ModelOut.model(Pos).stats.d_off(CalPos);ModelOut.validation(Pos).stats.d_off(TestPos)];
         if isempty(YAPT) & ~isempty(YAPC)
            x = DS(YAPC).scalars{1};
         elseif isempty(YAPC) & ~isempty(YAPT)
            x = DS(YAPT).scalars{1};
         else
            x = [DS(YAPC).scalars{1};DS(YAPT).scalars{1}];
         end
         
      end
      
   else

      %y is the D-Statistic if on-line a vector per batch is necessary: D-statistic over time
      if On
         y = ModelOut.model(Pos).stats.d_on(IndX{1},:);
      else
         y = ModelOut.model(Pos).stats.d_off(IndX{1});
      end
      
   end
      
else
   
   %Get the loadings from the model (they are not saved in the .prediction structure)
   ModelOut.prediction(Pos).xfactors(2:length(ModelOut.model(Pos).xfactors)) = ModelOut.model(Pos).xfactors(2:end);
   [ModelOut.prediction(Pos).core,ModelOut.prediction(Pos).bcoeff]           = deal(ModelOut.model(Pos).core,ModelOut.model(Pos).bcoeff);
   ModelOut.prediction(Pos).yfactors{1}                                      = zeros(size(ModelOut.prediction.data,1),ModelOut.prediction.nbfactors);
   ModelOut.prediction(Pos).yfactors(2:length(ModelOut.model(Pos).yfactors)) = ModelOut.model(Pos).yfactors(2:end);
   %Compute the statistic for the new sample/batches
   ModelOut.prediction(Pos).stats = ...
      DLims([ModelOut.modelname,'Pred'],...%Score computing function
      get(gcbo,'userdata'),...             %Offline/OnLine
      0,...                                %Do not compute limits based on Data
      ModelOut.prediction.data.data,...    %Data
      ModelOut.prediction(Pos),...         %Model Parameters
      [],...                               %Validation Parameters
      [.95 .99],...                        %Limits          
      ModelOut.info.fillin,...             %Fill In
      [],...                               %Segments
      ModelOut(Pos).model.stats,...        %Stats structure
      ModelOut.model(Pos).xpreproc.cen,... %Centring
      ModelOut.model(Pos).xpreproc.scal,...%Scaling
      ModelOut.info.preprocess);
   
   if On
      y = ModelOut.prediction(Pos).stats.d_on(IndX{1},:);
   else
      y = ModelOut.prediction(Pos).stats.d_off(IndX{1},:);
   end

end
%Create axes
Axes   = axes('tag','cbaxes','nextplot','replacechildren');

%Plot
%Blue  x: normal
%Green *: > 0.95
%Red   *: > 0.99
hold on
if On

   ystats = repmat(ModelOut.model(Pos).stats.dlim_on',length(y),1);
   l1     = y(:) > ystats(:,1);
   l2     = y(:) > ystats(:,2);
   if ~isempty(ModelOut.prediction)
      y(ModelOut.info.initialsize + 1:end)  = NaN;
   end
   if any(l2)
      yacc      = y;
      yacc(~l2) = NaN;
      Handle(1) = plot(x,yacc,'*r');
      Leg(1)    = {'p > 0.99'};
   end
   if any(l1)
      
      yacc            = y;
      yacc(~l1 | l2)  = NaN;
      Handle(2) = plot(x,yacc,'*g');
      set(Handle(2),'color',[0 0.5625 0])
      Leg(2)    = {'p > 0.95'};
   
   end
   if ~all(l1|l2)
   
      yacc            = y;
      yacc(l1)        = NaN;
      Handle(3) = plot(x,yacc,'xb');
      set(Handle(3),'color',[0 0 0.5625])
      Leg(3)    = {'Normal'};
   
   end

else

   ystats = repmat(ModelOut.model(Pos).stats.dlim_off',length(y),1);
   l1     = y(:) > ystats(:,1);
   l2     = y(:) > ystats(:,2);
   if any(l2)
      %Calibration set
      yacc             = y;
      yacc(~l2 | TF)   = NaN;
      Handle(1)  = bar(x,yacc);
      set(Handle(1),'facecolor',[0.5625 0 0])
      if any(TF & l2)
         Leg(1) = {'p > 0.99 (cal)'};
         %Test set
         yacc             = y;
         yacc(~l2 | ~TF)  = NaN;
         Handle(4)  = bar(x,yacc,'r');
         set(Handle(4),'facecolor',[1 .5625 .5625])
         Leg(4)     = {'p > 0.99 (test)'};
      else
         Leg(1) = {'p > 0.99'};
      end
      
   end
   if any(l1)
   
      yacc                 = y;
      yacc(~l1 | l2 | TF)  = NaN;
      Handle(2)            = bar(x,yacc);
      set(Handle(2),'facecolor',[0 0.5625 0])
      if any(TF & (~l1 | l2))
         Leg(2) = {'p > 0.95 (cal.)'};
         %Test set
         yacc                 = y;
         yacc(~l1 | l2 | ~TF) = NaN;
         Handle(5)            = bar(x,yacc,'g');
         set(Handle(5),'facecolor',[0.5625 1 0.5625])
         Leg(5) = {'p > 0.95 (test)'};
      else
         Leg(2) = {'p > 0.95'};
      end
   
   end
   if ~all(l1)
      
      yacc                = y;
      yacc(find(l1 | TF)) = NaN;
      Handle(3)     = bar(x,yacc);
      set(Handle(3),'facecolor',[0 0 0.5625])
      if any(TF & ~(l1|l2))
         Leg(3) = {'Normal (cal.)'};
         %Test set
         yacc                 = y;
         yacc(find(l1 | ~TF)) = NaN;
         Handle(6)            = bar(x,yacc,'b');
         set(Handle(6),'facecolor',[0.5625 0.5625 1])
         Leg(6) = {'Normal (test)'};
      else
         Leg(3) = {'Normal'};
      end
   
   end

end
if length(x) == 1
   set(Axes,'XLim',[x-.5 x+.5])
end
x           = get(Axes,'xlim');
ystats      = repmat(ystats,1,2);
Handle(7:8) = plot(x,ystats([1,end],1),'r',x,ystats([1,end],2),'r:');
xlabel(X_Lab)
ylabel('D Statistic')
title(Ax_Tit)
Leg(7:8) = {'0.95 Lim.';'0.99 Lim.'};
legend(Handle(ishandle(Handle)),Leg(ishandle(Handle)),0)
hold off


%Save the newly computed statistics
%Save everything in the "main" ModelOut
if Compute
   MO = ModelOut;
end

%Copy in the current axes the ModelOut corresponding to the plots
ModelOut.model = ModelOut.model(Pos);
if ~isempty(ModelOut.validation)
   ModelOut.validation = ModelOut.validation(Pos);
end
setappdata(Axes,'ModelOut',ModelOut)

return