function [A,B,C,errflag,iter] = swatld(X,A0,B0,F,tol,maxiter)
% function [A,B,C,errflag,iter] = swatld (X,A0,B0,F,tol,maxiter);
% 
% Description:
% Fits a PARAFAC model to a three way array via Self-Weighted Alternating Trilinear Decomposition
% 
% Inputs: 
% X: array of doubles
% A0,B0: initial estimations for the loading matrices A and B (here random matrices)
% F: number of components to extract
% tol: convergence criterion based on the relative decrease in fit
% maxiter: maximum number of iterations.F: Insert description
%  
% 
% Outputs:
% A,B,C: loading matrices
% errflag: indicates normal exit (0) or maximum number of iterations (1)
% iter: number of iterations
%  
% 
% Called by:
% Model_PARAFAC\InitPar
% 
% Subroutines:
% Internal: None
% External: nmodel
% 
% 
% Author: 
% Faber N.M.
% ...
% Modified by:
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% 
% Last modified: 09-May-2002 16:30:54
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
% 
% References
% [1] Chen et al. Chemometrics and Intelligent Laboratory Systems, 52 (2000) 75-86.
%
if ~exist('tol','var')
   tol = 1e-5;
end
if ~exist('maxiter','var')
   maxiter = 1000;
end

% Initialization:
[Jx,Jy,Jz] = size(X);
A          = A0 / diag(sqrt(sum(A0.^2)) .* sign(sum(A0.^3)));
B          = B0 / diag(sqrt(sum(B0.^2)) .* sign(sum(B0.^3)));
pinvB      = pinv(B); 
diagBtB    = diag(B'*B); 
C          = zeros(Jz,F);

% Iterative refinement:
sigmaold = -1;
cnv      = eps + 1;
iter     = 0;
while cnv > eps & iter < maxiter
   
   pinvA   = pinv(A);
   diagAtA = diag(A'*A);
   for jz = 1:Jz
      C(jz,:) = 0.5 * (diag(pinvB*X(:,:,jz)'*A)./diagAtA + diag(pinvA*X(:,:,jz) *B)./diagBtB)';
   end
   pinvC   = pinv(C);
   diagCtC = diag(C'*C);
   for jy = 1:Jy
      B(jy,:) = 0.5 * (diag(pinvA*squeeze(X(:,jy,:)) *C)./diagCtC + diag(pinvC*squeeze(X(:,jy,:))'*A)./diagAtA)';
   end
   B       = B / diag(sqrt(sum(B.^2)) .* sign(sum(B.^3)));
   pinvB   = pinv(B);
   diagBtB = diag(B'*B);
   for jx = 1:Jx
      A(jx,:) = 0.5 * (diag(pinvC*squeeze(X(jx,:,:))'*B)./diagBtB + diag(pinvB*squeeze(X(jx,:,:)) *C)./diagCtC)';
   end
   A     = A / diag(sqrt(sum(A.^2)) .* sign(sum(A.^3)));
   Res   = X - nmodel({A,B,C});
   sigma = sum(Res(:));
   cnv   = abs((sigma-sigmaold)/sigmaold);
   iter  = iter + 1; sigmaold = sigma;
   
end

if iter < maxiter
   errflag = 0;
else
   errflag = 1;
end
invAtABtB = pinv((A'*A).*(B'*B));
for j = 1:Jz
   C(j,:) = (invAtABtB * diag(A'* X(:,:,jz) * B))';
end