function [Axes,MO] = Plot_RIP(PlotStruct,ModelOut,MainFig)
% function [Axes,MO] = Plot_RIP (PlotStruct,ModelOut,MainFig);
% 
% Description:
% Plots the RIP (Resampling Influence Plot) for a the mode specified in PlotStruct.nummode. The MSEs
% for the selected mode are taken from ModelOut.validation.stats.mse. If they are not present they are
% computed by means of Compute_Risk. For the axes specified by Axes the 'ModelOut' application-defined
% data is set, and it contains a copy of ModelOut (employed by PARAFACDisplayInfo).
% 
% 
% Inputs: 
% PlotStruct: PlotStruct structure
% ModelOut  : ModelOut structure for the current model
% MainFig   : main figure handle
% 
% 
% Outputs:
% Axes: handle of the axes with the RIP.
% MO: ModelOut structure with the updated statistic. Empty if no new statistic was computed.
% 
% 
% Called by:
% Model_PARAFAC\PARAFACActivatePlots
% NB. The actual call is made from PARAFACPlot via feval and not from Model_PARAFAC\
% PARAFACActivatePlots.
% 
% 
% Subroutines:
% Internal: None
% External: compute_risk, nshape, selectmodel
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 11-Oct-2002 19:33:30
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 

%Set initial parameters
MO                 = [];
Axes               = [];
PlotStruct.nummode = get(gcbo,'userdata');

%Select the rank for the model
Rank  = 1;
Ranks = [ModelOut.model.nbfactors]';
if length(Ranks) > 1
   Rank = SelectModel(Ranks);
   if ~Rank,return,end
else
   Rank = Ranks;
end
Pos  = find([ModelOut.model.nbfactors] == Rank);

%Optimally scale the loading matrices
Compute = isempty(ModelOut.validation(1).stats.risk);
if Compute
   Ans = questdlg({'These values need to be calculated, it may take some time.'},'Proceed','Proceed','Cancel','Proceed');
   if ~strcmp(Ans,'Proceed')
      return
   end
   Hand(1) = max([ModelOut.model.nbfactors]) * size(ModelOut.validation(1).segments,2);
   Hand(2) = 0;
   for i = 1:length(ModelOut.validation)
      [ModelOut.validation(i),Hand] = Compute_Risk(Hand,ModelOut.model(i),ModelOut.validation(i));
   end
   delete(Hand(1))
end

%Create axes
figure(MainFig)
Axes        = axes('tag','cbaxes','nextplot','replacechildren');

%Plot
%Number of non-missing in each sample/batch
x           = nshape(ModelOut.validation(Pos).xpred,1).^2;
Ex          = sum(~isnan(x),2).^-1;
%Compute samples/batches RSS
x(isnan(x)) = 0;
x           = sum(x,2).*Ex;
%Compute the loadings/RSS in the selected mode
a           = repmat(ModelOut.model(Pos).xfactors{PlotStruct.nummode},[1,1,size(ModelOut.validation(Pos).xfactors{PlotStruct.nummode},3)]);
y           = squeeze(sum(sum((ModelOut.validation(Pos).xfactors{PlotStruct.nummode} - a).^2,2),1));
plot(x,y,'o')
text(x,y,[' '*ones(size(ModelOut.data.dataset,1),1),char(ModelOut.data.dataset.labels{1})])
ylabel('R_{loadings}');
xlabel('R_{-m}');
title(['\bfResample Influence Plot - Mode: ',ModelOut.data.dataset.modenames{PlotStruct.nummode}],'fontname','arial','fontsize',10);
set(Axes,'position',[0.2464,0.0377,0.7312,0.9127])
grid(PlotStruct.grid)

%Save everything in the "main" ModelOut
if Compute
   MO = ModelOut;
end

%Copy in the current axes the ModelOut corresponding to the plots
ModelOut.model = ModelOut.model(Pos);
if ~isempty(ModelOut.validation)
   ModelOut.validation = ModelOut.validation(Pos);
end
setappdata(Axes,'ModelOut',ModelOut)

return
