function [Axes,MO] = Plot_Partial_RSS_OnLine(PlotStruct,ModelOut,MainFig)
% function [Axes,MO] = Plot_Partial_RSS_OnLine (PlotStruct,ModelOut,MainFig);
% 
% Description:
% It plots the sum of squares of the residuals (Q statistics) computed in an online fashion or the
% Squared Prediction Error (SPE) vs time (scalars in the last mode, which has to be given name:
% 'time'). The distiction between Q and SPE case is made by the 'userdata' of the calling menu ('Q' for
% Q and 'SPE' otherwise). The confidence limits, computed by means of OnLineRes are automatically plot.
% If they where not originally present they are computed and saved in ModelOut.model.stats.qlim and in
% ModelOut.model.stats.spelim. When the model has been applied to new data this plot automatically
% applies to the new data; the confidence limits are computed on the original data set. For the axes
% specified by Axes the 'ModelOut' application-defined data is set, and it contains a copy of ModelOut
% (employed by PARAFACDisplayInfo).
%  
% 
% Inputs: 
% PlotStruct: PlotStruct structure
% ModelOut: ModelOut structure for the current model
% MainFig: main figure handle
% 
% 
% Outputs:
% Axes: handle of the axes where the Q or SPE are plot
% MO: ModelOut structure with the updated statistic. Empty if no new statistic was computed.
% 
% 
% Called by:
% Model_PARAFAC\PARAFACActivatePlots
% NB. The actual call is made from PARAFACPlot via feval and not from Model_PARAFAC\
% PARAFACActivatePlots.
% 
% 
% Subroutines:
% Internal: None
% External: onlineres, parafacpred, selectmodel
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 04-Nov-2002 13:25:48
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
% 

%Initial parameters
Axes    = [];
Rank    = 1;
Pos     = 1;
Mode    = strcmp(get(gcbo,'userdata'),'Q');
IndX    = repmat({':'},ndims(ModelOut.data.dataset),1);
QstOpt  = struct('Default','','Interpreter','tex');
Factors = {};
MO      = [];

%Choose model
if length(ModelOut.model) > 1
   Ranks      = [ModelOut.model.nbfactors]';
   [Rank,Pos] = SelectModel(Ranks);
   if ~Rank,return,end
end
if ~isempty(ModelOut.prediction)
   DS            = ModelOut.prediction.data;
   Model_Validat = 'external';
else
   DS            = ModelOut.data.dataset;
   Model_Validat = 'model';
end
x    = DS.scalars{ndims(DS)};
X_La = DS.axislabels{ndims(DS)};

%Choose sample/batch
ListProp = {'initialvalue',1,'selectionmode','single','name','Select interval','promptstring','Select batch'};
ListStr  = DS.labels{1};
if ~isempty(ModelOut.validation) & strcmp('test',ModelOut.validation(1).method) & isempty(ModelOut.prediction)
   for i = 1:length(ModelOut.validation(1).segments)
      ListStr{ModelOut.validation(1).segments(i)} = [ListStr{ModelOut.validation(1).segments(i)},' (Test set)'];
   end
end
[Int,Ok] = listdlg('Liststring',ListStr,ListProp{:});
if ~Ok,return,end
IndX{1} = Int;

TF      = logical(zeros(length(IndX{1}),1));
CalPos  = 1:size(DS,1);
if isempty(ModelOut.prediction) & ~isempty(ModelOut.validation) & isequal(ModelOut.validation.method,'test')
   
   Fic                      = setdiff(1:size(ModelOut.data.dataset,1),ModelOut.validation(1).segments);
   [Nil,CalPos]             = intersect(Fic,IndX{1}); %calibration set
   Fic                      = intersect(1:size(ModelOut.data.dataset,1),ModelOut.validation(1).segments);
   [Nil,TestPos]            = intersect(Fic,IndX{1}); %test set
   TF(length(CalPos)+1:end) = 1;
   
end

%Change fill-in method?
Ans = ModelOut.info.fillin;
if strcmpi(getappdata(MainFig,'Mode'),'advanced')
   QstOpt.Default = ModelOut.info.fillin;
   Prompt         = {['The fill-in method is currently set to \bf',ModelOut.info.fillin,'\rm'],['Choose fill-in method']};
   Ans            = questdlg(Prompt,'Choose fill-in method','Current deviation','Zero','Cancel',QstOpt);
   if strcmp(Ans,'Cancel')
      return
   end
end

%Compute confidence limits if not present or calculated with a different fill-in method
Compute = isempty(ModelOut.model(Pos).stats.qlim_on) | ~strcmp(ModelOut.info.fillin,Ans);
if Compute

   ModelOut.info.fillin = Ans;
   %The limits and the covariance matrix are be computed on the NOC data
   if ~isempty(ModelOut.validation) & strcmp('test',ModelOut.validation(Pos).method)
      Data = ModelOut.data.dataset(setdiff(1:size(ModelOut.data.dataset,1),ModelOut.validation(Pos).segments)).data;
   else
      Data = ModelOut.data.dataset.data;
   end   
   [ModelOut.model(Pos).stats.spe,ModelOut.model(Pos).stats.spelim,...
    ModelOut.model(Pos).stats.q_on,ModelOut.model(Pos).stats.qlim_on,...
    ModelOut.model(Pos).stats.scores] = ...
      OnLineRes('PARAFACPred',...               %Predicting function
      Data,...                                  %Data for online estimation (those used for the model)
      ModelOut.model(Pos).xfactors,...          %Model factors
      ModelOut.info.fillin,...                  %Fill In
      [.95 .99],...                             %Conf Limits
      1,...                                     %Compute limits
      ModelOut.model(Pos).xpreproc.cen,...      %Preprocessing
      ModelOut.model(Pos).xpreproc.scal);
   if ~isempty(ModelOut.validation) & isequal(ModelOut.validation.method,'test')

      Data            = ModelOut.data.dataset(ModelOut.validation(Pos).segments).data;
      Mod             = ModelOut.validation(Pos);
      Mod.xfactors{1} = Mod.xfactors{1}(ModelOut.validation(Pos).segments,:);
      [ModelOut.validation(Pos).stats.spe,Nil,...
          ModelOut.validation(Pos).stats.q_on,Nil,...
          ModelOut.validation(Pos).stats.scores] = ...
            OnLineRes('PARAFACPred',...               %Predicting function
            Data,...                                  %Data for online estimation (those used for the model)
            Mod,...                                   %Model factors
            ModelOut.info.fillin,...                  %Fill In
            [.95 .99],...                             %Conf Limits
            1,...                                     %Compute limits
            ModelOut.model(Pos).xpreproc.cen,...      %Preprocessing
            ModelOut.model(Pos).xpreproc.scal);
      
   end

end

%In the online case it is the RSS vs time (i.e. the last mode)
%For the selected samples
if isempty(ModelOut.prediction) 

   if ~isempty(ModelOut.validation) & isequal(ModelOut.validation.method,'test')
            
      if isempty(TestPos)

         if Mode %Either Q statistic on-line or SPE 
            y = ModelOut.model(Pos).stats.q_on(CalPos,:);
         else
            y = ModelOut.model(Pos).stats.spe(CalPos,:);
         end    

      else

         if Mode %Either Q statistic on-line or SPE 
            y = ModelOut.validation(Pos).stats.q_on(TestPos,:);
         else
            y = ModelOut.validation(Pos).stats.spe(TestPos,:);
         end

      end
                        
   else
   
      if Mode %Either Q statistic on-line or SPE 
         y = ModelOut.model(Pos).stats.q_on(IndX{1},:);
      else
         y = ModelOut.model(Pos).stats.spe(IndX{1},:);
      end    
   
   end

else      

   if isempty(ModelOut.prediction.stats.spe)

      %Compute only if it has not been computed before
      Factors{1} = ModelOut.prediction(Pos).xfactors{1}(:,:,end);
      for j = 2:ndims(ModelOut.data.dataset)
         Factors{j} = ModelOut.model(Pos).xfactors{j};
      end
      [ModelOut.prediction(Pos).stats.spe,Nil,...
      ModelOut.prediction(Pos).stats.q_on,...
       Nil,ModelOut.prediction(Pos).stats.scores] = ...
         OnLineRes('PARAFACPred',...               %Predicting function
         ModelOut.prediction.data.data,...         %Data for online estimation (those used for the model)
         Factors,...                               %Model factors
         ModelOut.info.fillin,...                  %Fill In
         [],...                                    %Conf Limits
         0,...                                     %Compute limits
         ModelOut.model(Pos).xpreproc.cen,...      %Preprocessing
         ModelOut.model(Pos).xpreproc.scal);

   end
   if Mode %Either Q statistic on-line or SPE
      y = ModelOut.prediction(Pos).stats.q_on(IndX{1},:);
   else
      y = ModelOut.prediction(Pos).stats.spe(IndX{1},:);
   end

end

%Choose the confidence limits depending on the statistic
if Mode 
   ystats = ModelOut.model(Pos).stats.qlim_on;
else
   ystats = ModelOut.model(Pos).stats.spelim;
end

%Create axes
Axes   = axes('tag','cbaxes','nextplot','replacechildren','yscale','log');

%Plot
%Blue  x: normal
%Green *: > 0.95
%Red   *: > 0.99  
l1     = y(:)>ystats(1,:)';
l2     = y(:)>ystats(2,:)';
if ~isempty(ModelOut.prediction)
   y(ModelOut.info.initialsize + 1:end)  = NaN;
end
Handle = -1 * ones(5,1);
if any(l2)
   Handle(1,1) = semilogy(x(l2),y(l2),'*r');
   hold on
end
if any(l1 & ~l2)
   Handle(2,1) = semilogy(x(l1 & ~l2),y(l1 & ~l2),'*');
   set(Handle(2,1),'color',[0 0.5625 0])
   hold on
end
if any(~l1)
   Handle(3,1) = semilogy(x(~l1),y(~l1),'x');
   set(Handle(3,1),'color',[0 0 0.5625])
   hold on
end
Handle(4:5) = semilogy(x,ystats(1,:),'r:',x,ystats(2,:),'r-');
hold off

%Legend and text
if any(TF)
   Model_Validat = 'test set';
end
if Mode
   L = {'Q > 0.99 lim','Q > 0.95 lim','Q','95% lim','99% lim'};
   ylabel('log(Q)','fontname','arial','fontsize',9)
   title({'\bfSum of Squared Residuals (Q)- on line - ',[DS(IndX{1}).labels{1}{1},' (' lower(Model_Validat) ')']},'fontname','arial','fontsize',10);
else
   L = {'SPE > 0.99 lim','SPE > 0.95 lim','SPE','95% lim','99% lim'};
   ylabel('log(SPE)','fontname','arial','fontsize',9)
   title({'\bfSquared Prediction Error (SPE)',[DS(IndX{1}).labels{1}{1},' (' lower(Model_Validat) ')']},'fontname','arial','fontsize',10);
end
legend(Handle(ishandle(Handle)),L(ishandle(Handle)))
xlabel(X_La,'fontname','arial','fontsize',9);

%Save the newly computed statistics
%Save everything in the "main" ModelOut
if Compute
   MO = ModelOut;
end

%Copy in the current axes the ModelOut corresponding to the plots
ModelOut.model = ModelOut.model(Pos);
if ~isempty(ModelOut.validation)
   ModelOut.validation = ModelOut.validation(Pos);
end
setappdata(Axes,'ModelOut',ModelOut)
return
