function [Axes,MO] = Plot_Partial_RSS(PlotStruct,ModelOut,MainFig)
% function [Axes,MO] = Plot_Partial_RSS (PlotStruct,ModelOut,MainFig);
% 
% Description:
% It plots the sum of squares of the residuals vs the scalars in the mode specified in
% PlotStruct.nummode. Additional selections can be made so that subsets of variables (or
% sample/batches) are used. If only one sample/batch is selected and the mode is not the first,
% confidence limits, computed with Residuals_CL, are automatically displayed. If 'leave one out'
% validation has been calculated it is asked which scores are to be used: those from the complete
% models or the "predicted" ones. When the model has been applied to new data this plot automatically
% applies to the new data; the confidence limits are computed on the original data set. For the axes
% specified by Axes the 'ModelOut' application-defined data is set, and it contains a copy of ModelOut.
% (employed by PARAFACDisplayInfo)
%  
% 
% Inputs: 
% PlotStruct: PlotStruct structure
% ModelOut: ModelOut structure for the current model
% MainFig: main figure handle
% 
% 
% Outputs:
% Axes: handle of the axes where the Residuals are plot
% MO: ModelOut structure with the updated statistic. Empty if no new statistic was computed.
% 
% 
% Called by:
% Model_PARAFAC\PARAFACActivatePlots
% NB. The actual call is made from PARAFACPlot via feval and not from Model_PARAFAC\
% PARAFACActivatePlots.
% 
% 
% Subroutines:
% Internal: None
% External: cenwindow, nshape, residuals_cl, selectmodel, dispinfo
% NB DispInfo is set as the 'ButtonDownFcn' of the line plots when the residuals are displayed versus any mode but the first one.
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 15-Oct-2002 11:28:08
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
%

Axes          = [];
MO            = [];
Rank          = 1;
Pos           = 1;
Dim           = 1;
[IndX,IndX2]  = deal(repmat({':'},ndims(ModelOut.data.dataset),1));
Various_Modes = 1:ndims(ModelOut.data.dataset);
T             = get(gcbo,'userdata');
[Mode,Type]   = deal(T{:});
YAF           = 0;
Ok            = 1;
ModeNames     = ModelOut.data.dataset.modenames;
ystats        = [];
Model_Val     = 'calibration';
if ~isempty(ModelOut.prediction)
   %If there are projections
   Model_Val = 'external set';
elseif isempty(ModelOut.prediction) & ~isempty(ModelOut.validation) & strcmp(ModelOut.validation(1).method,'test')
   
   %If there is a test set and it is mode 1
   if Mode == 1
      Model_Val = 'both';
   else
      %Use only calibration samples
      IndX2{1} = setdiff(1:size(ModelOut.data.dataset,1),ModelOut.validation(1).segments);
   end
   
end

%Choose model
if length(ModelOut.model) > 1
   Ranks      = [ModelOut.model.nbfactors]';
   [Rank,Pos] = SelectModel(Ranks);
   if ~Rank,return,end
end

%Labels etc
if ~isempty(ModelOut.prediction)
   DS = ModelOut.prediction.data;
else
   DS = ModelOut.data.dataset;
end
L = DS.labels;
%Select subset
while Dim
   
   Dim = SelectSubSet(ModeNames);
   if isnan(Dim)
      return
   elseif Dim 
      
      YAF      = 1; %Yet another flag
      ListProp = {'initialvalue',1:length(L{Dim}),'selectionmode','multiple','name','Select interval','promptstring','Select interval'};
      ListStr  = L{Dim};
      if ~isempty(ModelOut.prediction) & Dim == 1
         %If there are projections
         Model_Val = 'external set';
         [Int,Ok] = listdlg('Liststring',ListStr,ListProp{:});
      elseif ~isempty(ModelOut.validation) & isempty(ModelOut.prediction) & Dim == 1 
         
         if strcmp('test',ModelOut.validation(1).method)
            
            if Mode ~= 1
               
               %If there are no projections, but there is test set validation and the RSS is not plot
               %against the first mode, but the subset is being selected in the first mode
               %Select calibration/test set
               Model_Val = questdlg('Choose predictions','Which samples shall be used?','Calibration set','Test set','Cancel','Calibration set');
               switch Model_Val
                  case 'Cancel'
                     return
                     
                  case 'Test set'
                     ListStr     = ModelOut.data.dataset(ModelOut.validation(1).segments).labels{Dim};
                     ListProp{2} = 1:length(ListStr);
                     [Int,Ok]    = listdlg('Liststring',ListStr,ListProp{:});
                     Int         = ModelOut.validation(1).segments(Int);
                     Model_Val   = 'validation';
                     
                  case 'Calibration set'
                     Fic         = setdiff(1:size(ModelOut.data.dataset,1),ModelOut.validation(1).segments);
                     Model_Val   = 'calibration'
                     ListStr     = ModelOut.data.dataset(Fic).labels{Dim};
                     ListProp{2} = 1:length(ListStr);
                     [Int,Ok]    = listdlg('Liststring',ListStr,ListProp{:});
                     IndX2{1}    = Fic(IndX{1});
                     YAF         = 0;
               end
               
            else
               
               %Modify labels
               for i = 1:length(ModelOut.validation(1).segments)
                  ListStr{ModelOut.validation(1).segments(i)} = [ListStr{ModelOut.validation(1).segments(i)},' Test set'];
               end
               Model_Val = 'both';
               [Int,Ok]  = listdlg('Liststring',ListStr,ListProp{:});
               
            end
            
         elseif strcmp('loo',ModelOut.validation(1).method)
            
            %If leave one out validation and no projection 
            [Int,Ok]  = listdlg('Liststring',ListStr,ListProp{:});
            if ~Ok,
               Dim = 0;
               return,
            end
            %use model/predictions?
            Model_Val = questdlg('Choose predictions','Which scores shall be used?','calibration','validation','Cancel',Model_Val);
            
         else
            [Int,Ok] = listdlg('Liststring',ListStr,ListProp{:});
         end
         
      else
         [Int,Ok] = listdlg('Liststring',ListStr,ListProp{:});
      end
      if ~Ok | strcmp(Model_Val,'Cancel')
         Dim = 0;
         return,
      end
      IndX{Dim} = Int;
      if YAF
         IndX2{Dim} = Int;
      end
      
   end
   
end

%Compute residuals
switch lower(Model_Val)
   case 'calibration'
      X = ModelOut.data.dataset(IndX2{:}).data;
      M = ModelOut.model(Pos).xpred(IndX{:});
      
   case 'both'
      M                  = repmat(NaN,size(ModelOut.data.dataset));
      Fic                = IndX{1};
      IndX{1}            = intersect(IndX{1},ModelOut.validation(1).segments);
      M(IndX{:})         = ModelOut.validation(Pos).xpred(IndX{:});
      Nil                = setdiff(1:size(ModelOut.data.dataset,1),ModelOut.validation(1).segments);
      [Nil,IndX{1}]      = intersect(Nil,Fic);
      M(Nil,IndX{2:end}) = ModelOut.model(Pos).xpred(IndX{:});
      IndX{1}            = Fic;
      X                  = ModelOut.data.dataset(IndX{:}).data;
      M                  = M(IndX{:});
      
   case 'validation' 
      X = ModelOut.data.dataset(IndX{:}).data;
      M = ModelOut.validation(Pos).xpred(IndX{:});
      
   case 'external set'
      X = ModelOut.prediction.data(IndX{:}).data;
      M = ModelOut.prediction.xpred(IndX{:});
      
end
M(isnan(X)) = 0;
X(isnan(X)) = 0;
switch Type
   case 'Residuals'
      R = X - M;
		y = R.^2;
		for i = 1:ndims(X)
			if i ~= [1 Mode]
				y = sum(y,i);
			end
		end
		y = squeeze(y);
         
   case '% Var'
      R = X - M;
		y = sum(nshape(R.^2,Mode),2);
		y = 100 - 100*y./sum(nshape(X.^2,Mode),2);
      
   case 'Congruence'
      y = sum(normit(nshape(X,Mode)').*normit(nshape(M,Mode)'));
      
end
x    = DS.scalars{Mode}(IndX{Mode});
Leg  = DS(IndX{1}).labels{1};

%Create axes
Axes = axes('tag','cbaxes','nextplot','replacechildren','yscale','log');

%Plot
if Mode == 1
   
   if strcmp(lower(Model_Val),'both') %Mode is necessarily 1
      
      Nil           = intersect(setdiff(1:size(ModelOut.data.dataset,1),ModelOut.validation(1).segments),IndX{1});
      [Nil,Index]   = intersect(IndX{1},Nil);
      Fic_Y         = NaN * y;
      Fic_Y(Index)  = y(Index);
      if length(x) < 20
         set(gca,'yscale','linear')
         Handle = bar(x,Fic_Y);
         set(Handle,'marker','none','facecolor',[0 0 0.5625])
      else
         Handle = plot(x,Fic_Y,'-x');
         set(Handle,'marker','none','color',[0 0 0.5625])
      end
      set(gca,'nextplot','add')
      Index2        = setdiff(1:size(y,1),Index);
      Fic_Y         = NaN * y;
      Fic_Y(Index2) = y(Index2);
      if length(x) < 20
         set(gca,'yscale','linear')
         h = bar(x,Fic_Y);
         set(h,'facecolor',[0.502 0 0])
      else
         h = plot(x,Fic_Y,'o-');
         set(h,'color',[0.502 0 0])
      end
      hold off
      Model_Val = 'cal. set and test set';
      legend('Cal. set','Test set')
      
   else
      
      if length(x) < 20
         set(gca,'yscale','linear')
         Handle = bar(x,y);
         set(Handle,'marker','none','facecolor',[0 0 0.5625])
      else
         Handle = plot(x,y');
         set(Handle,'marker','none','color',[0 0 0.5625])
      end
      
   end
   
else
   
   if length(x) < 20 & any(strcmp({'Congruence','% Var'},Type))
      set(gca,'yscale','linear')
      Handle = bar(x,y);
      set(Handle,'marker','none','facecolor',[0 0 0.5625])
   elseif isequal(Type,'Residuals')
      Handle = plot(x(:),y');
      set(Handle,'marker','none','buttondownfcn','DispInfo(gcbo)')
      set(Handle,{'tag'},Leg)
   else
      Handle = plot(x,y);
      set(Handle,'marker','none','color',[0 0 0.5625])
   end
   
end

switch Type
   case 'Residuals'
      
      %Compute and plot confidence limits
      if Mode ~= 1
         
         IndX3 = IndX; 
         %Don't use test set
         if ~isempty(ModelOut.validation) & strcmp(ModelOut.validation(1).method,'test')
            IndX3{1} = setdiff(1:size(ModelOut.data.dataset,1),ModelOut.validation(1).segments);
         else
            IndX3{1} = 1:size(ModelOut.data.dataset,1);
         end
         
         for i = 1:size(R,Mode)
            IndX3{Mode} = i;
            R           = nshape(ModelOut.data.dataset(IndX3{:}).data - ModelOut.model(Pos).xpred(:,IndX3{2:end}),1);
            ystats(:,i) = Residuals_CL(R,[0.95 0.99]);
         end
         hold on
         Handle2 = plot(x,ystats(1,:),'g',x,ystats(2,:),'r:');
         set(Handle2,'linewidth',2,'color',[0 0.5020 0.7529])
         if length(Handle) <= 10
            legend([Handle;Handle2],Leg{:},'0.95 Limit','0.99 Limit',0)
         else
            legend(Handle2,'0.95 Limit','0.99 Limit',0)
         end
         hold off
         
      end
      xlabel(ModelOut.data.dataset.modenames{Mode},'fontname','arial','fontsize',9);
      ylabel('Q','fontname','arial','fontsize',9)
      if length(IndX{1}) == 1 & ~ischar(IndX{1})
         if isempty(ModelOut.prediction)
            title(['\bfSum of squared residuals (' lower(Model_Val) ') - ',ModelOut.data.dataset(IndX2{1}).labels{1}{1}],'fontname','arial','fontsize',10);
         else length(IndX{1}) == 1 & ~ischar(IndX{1}) & ~isempty(ModelOut.prediction)
            title(['\bfSum of squared residuals (' lower(Model_Val) ') - ',ModelOut.prediction.data(IndX2{1}).labels{1}{1}],'fontname','arial','fontsize',10);
         end
      else
         title(['\bfSum of squared residuals (' lower(Model_Val) ')'],'fontname','arial','fontsize',10);
      end
      
   case '% Var'
      xlabel(ModelOut.data.dataset.modenames{Mode},'fontname','arial','fontsize',9);
		ylabel('% Expl. var.','fontname','arial','fontsize',9)
		set(gca,'yscale','linear')
      if length(IndX{1}) == 1 & ~ischar(IndX{1})
         if isempty(ModelOut.prediction)
            title(['\bfExpl. var. (' lower(Model_Val) ') - ',ModelOut.data.dataset(IndX2{1}).labels{1}{1}],'fontname','arial','fontsize',10);
         else length(IndX{1}) == 1 & ~ischar(IndX{1}) & ~isempty(ModelOut.prediction)
            title(['\bfExpl. var. (' lower(Model_Val) ') - ',ModelOut.prediction.data(IndX2{1}).labels{1}{1}],'fontname','arial','fontsize',10);
         end
      else
         title(['\bfExpl. var. (' lower(Model_Val) ')'],'fontname','arial','fontsize',10);
      end
   
   case 'Congruence'
      set(gca,'yscale','linear')
      xlabel(DS.axislabels{Mode},'fontname','arial','fontsize',9);
      ylabel('Slab-wise congruence','fontname','arial','fontsize',9)
      title(['\bfQuality (' lower(Model_Val) ')'],'fontname','arial','fontsize',10);
      
end

%Adjust axes limits
if min(y) < 0
   Ay(1) = 1.03*min([y(:);ystats(:)]);
   Ay(2) = 0.97*max([y(:);ystats(:)]);
else
   Ay(1) = 0.97*min([y(:);ystats(:)]);
   Ay(2) = 1.03*max([y(:);ystats(:)]);
end
if Ay(1) == Ay(2) 
   Ay(1) = Ay(1)-0.5;
   Ay(2)=  Ay(2)+0.5;
end
if strcmp(Type,'Congruence')
   Ay(1) = max(-1,Ay(1));
   Ay(2) = min(1,Ay(2));
elseif strcmp(Type,'% Var')
   Ay(2) = min(100,Ay(2));
end

Ax = get(Axes,'xlim');
set(Axes,'ylim',Ay,'xlim',Ax)

%Copy in the current axes the ModelOut corresponding to the plots
ModelOut.model = ModelOut.model(Pos);
if ~isempty(ModelOut.validation)
   ModelOut.validation = ModelOut.validation(Pos);
end
setappdata(Axes,'ModelOut',ModelOut)
return

%------------------------------------------------------------------------------------------------------------------------------------------------------

function Dim = SelectSubSet(ModeNames)
h = cenwindow([0.5029 0.5814 0.6402],'Select subset','set(gcbf,''waitstatus'',''close'',''userdata'',0)',150,40*(length(ModeNames) + 1)+10);
set(h,'userdata',0,'toolbar','none','closerequestfcn','set(gcbf,''waitstatus'',''close'',''userdata'',NaN)')
for i = length(ModeNames):-1:1
   uicontrol('position',[10 50 + (i-1) * 40 130 30],'string',ModeNames{length(ModeNames)-i+1},'callback',sprintf('set(%i,''waitstatus'',''close'',''userdata'',%i)',h,length(ModeNames)-i+1))
end
uicontrol('position',[10 10 130 30],'string','Proceed','callback',sprintf('set(%i,''waitstatus'',''close'')',h))
waitfor(h,'waitstatus','close')
Dim = get(h,'userdata');
delete(h)
