function [Axes,MO] = Plot_Loadings(PlotStruct,ModelOut,MainFig)
% function [Axes,MO] = Plot_Loadings (PlotStruct,ModelOut,MainFig);
% 
% Description:
% Plots the scores/loadings for the mode specified in PlotStruct.nummode according to what selected in
% the control figure opened with PARAFACPlotControls. In the 1D plot the parameters are plot versus the
% scalars of the corresponding method. The choice of the component(s) for the axes is taken from
% PlotStruct.factors. When the model has been applied to new data this plot automatically applies to
% the new data. For the axes specified by Axes the 'ModelOut' application-defined data is set, and it
% contains a copy of ModelOut.(employed by PARAFACDisplayInfo)
%  
% 
% Inputs: 
% PlotStruct: PlotStruct structure
% ModelOut: ModelOut structure for the current model
% MainFig: main figure handle
% 
% 
% Outputs:
% Axes: handle of the axes with the scores/loadings.
% MO: empty
% 
% 
% Called by:
% Model_PARAFAC\PARAFACActivatePlots
% NB. The actual call is made from PARAFACPlot via feval and not from Model_PARAFAC\
% PARAFACActivatePlots.
% 
% 
% Subroutines:
% Internal: None
% External: parafacplotcontrols, plotmodel, plotvalidation, resamplingextract, selectmodel, skcell
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 14-Oct-2002 18:59:24
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
%

MO   = [];
Axes = [];
if isempty(findobj(0,'type','figure','userdata','PARAFAC Plot Control'))
   %Called by the main window.
   PlotStruct.plotdims                = get(gcbo,'userdata');
   PlotStruct.nummode                 = get(get(gcbo,'parent'),'userdata');
   if ~PlotStruct.factors(1)
      PlotStruct.factors(1) = 1;
   end
   Rank                               = 1;
   Ranks                              = [ModelOut.model.nbfactors]';
   Ranks(Ranks < PlotStruct.plotdims) = [];
   if length(Ranks) > 1
      Rank = SelectModel(Ranks);
      if ~Rank,return,end
   else
      Rank = Ranks;
   end
   Pos  = find([ModelOut.model.nbfactors] == Rank);
   ModelOut.model = ModelOut.model(Pos);
   if ~isempty(ModelOut.validation)
      ModelOut.validation = ModelOut.validation(Pos);
   end
   H = PARAFACPlotControls(PlotStruct,ModelOut,MainFig);
   waitfor(H,'waitstatus','Ok')
   if ~ishandle(H)
      Axes = [];
      return
   else 
      PlotStruct = getappdata(H,'PlotStr');
   end
end
if PlotStruct.nummode == 1
   T = '\\bfScores: ';
else
   T = '\\bfLoadings: ';
end
figure(MainFig)
Axes = axes('tag','cbaxes','nextplot','replacechildren');
if PlotStruct.additional.validation(1) & (~isempty(ModelOut.validation) | (~isempty(ModelOut.prediction) & PlotStruct.nummode == 1))
   
   [x,y,z] = deal([]);
   if isempty(ModelOut.prediction) | PlotStruct.nummode ~= 1
      LabObj = ModelOut.data.dataset.labobj{PlotStruct.nummode};
   else
      LabObj = ModelOut.prediction.data.labobj{1};
   end
   if PlotStruct.nummode == 1
      
      if PlotStruct.additional.validation(1) == 2 %Replicates

         if strcmp(ModelOut.validation.method,'rboo') %Residual bootstrap
            if PlotStruct.plotdims >= 1,x = squeeze(ModelOut.validation.xfactors{1}(:,PlotStruct.factors(1),:));end
            if PlotStruct.plotdims >= 2,y = squeeze(ModelOut.validation.xfactors{1}(:,PlotStruct.factors(2),:));end
            if PlotStruct.plotdims >= 3,z = squeeze(ModelOut.validation.xfactors{1}(:,PlotStruct.factors(3),:));end
         else
            if PlotStruct.plotdims >= 1,x = ResamplingExtract(ModelOut.validation.xfactors{1},PlotStruct.factors(1),ModelOut.validation.segments);end
            if PlotStruct.plotdims >= 2,y = ResamplingExtract(ModelOut.validation.xfactors{1},PlotStruct.factors(2),ModelOut.validation.segments);end
            if PlotStruct.plotdims >= 3,z = ResamplingExtract(ModelOut.validation.xfactors{1},PlotStruct.factors(3),ModelOut.validation.segments);end
         end
         
      elseif PlotStruct.additional.validation(1) == 1 & isempty(ModelOut.prediction) & any(strcmp({'loo','test'},ModelOut.validation.method))
         if PlotStruct.plotdims >= 1,x = ModelOut.validation.xfactors{1}(:,PlotStruct.factors(1),end);end
         if PlotStruct.plotdims >= 2,y = ModelOut.validation.xfactors{1}(:,PlotStruct.factors(2),end);end
         if PlotStruct.plotdims >= 3,z = ModelOut.validation.xfactors{1}(:,PlotStruct.factors(3),end);end
      elseif ~isempty(ModelOut.prediction)
         if PlotStruct.plotdims >= 1,x = ModelOut.prediction.xfactors{1}(:,PlotStruct.factors(1),end);end
         if PlotStruct.plotdims >= 2,y = ModelOut.prediction.xfactors{1}(:,PlotStruct.factors(2),end);end
         if PlotStruct.plotdims >= 3,z = ModelOut.prediction.xfactors{1}(:,PlotStruct.factors(3),end);end
      end
      
   else
      if PlotStruct.plotdims >= 1,x = squeeze(ModelOut.validation.xfactors{PlotStruct.nummode}(:,PlotStruct.factors(1),:));end
      if PlotStruct.plotdims >= 2,y = squeeze(ModelOut.validation.xfactors{PlotStruct.nummode}(:,PlotStruct.factors(2),:));end
      if PlotStruct.plotdims >= 3,z = squeeze(ModelOut.validation.xfactors{PlotStruct.nummode}(:,PlotStruct.factors(3),:));end
   end
   if isempty(y)
      y = x;
      x = LabObj.scalars;
      xlabel(LabObj.axislabels);
   else
      xlabel(sprintf('Component %i',PlotStruct.factors(1)));
   end
   if isempty(z)
      z = zeros(size(x));
   end
   ylabel(sprintf('Component %i',PlotStruct.factors(2)));
   zlabel(sprintf('Component %i',PlotStruct.factors(3)));
   PlotValidation(x,y,z,PlotStruct,LabObj,Axes)
   switch PlotStruct.plotdims
      case 1
         title(sprintf([T 'Component %i'],PlotStruct.factors(1)));
      case 2
         title(sprintf([T,'Component %i vs Component %i'],PlotStruct.factors(1:2)));
      case 3
         title(sprintf([T,'Comp. %i vs Comp. %i vs Comp. %i'],PlotStruct.factors(1:3)));   
   end
   
end
[Handle_Plot,Handle_Labels] = deal([]);
if any(strcmpi({'on','label','marker','number'},PlotStruct.visible))
   
   F = ModelOut.model.xfactors{PlotStruct.nummode};
   switch PlotStruct.plotdims 
      case 1
         if PlotStruct.nummode == 1 & ~isempty(ModelOut.validation) & strcmp(ModelOut.validation.method,'test')
            x = ModelOut.data.dataset.scalars{PlotStruct.nummode}(setdiff(1:size(ModelOut.data.dataset,1),ModelOut.validation.segments));
         else
            x = ModelOut.data.dataset.scalars{PlotStruct.nummode};
         end
         if PlotStruct.factors(1)
            y = F(:,PlotStruct.factors(1));
            title(sprintf([T,'Component %i'],PlotStruct.factors(1)),'fontname','arial','fontsize',10);
         else
            y = F;
            title(T(2:end),'fontname','arial','fontsize',10);
         end
         z = 0 * x;
         xlabel(ModelOut.data.dataset.axislabels{PlotStruct.nummode},'fontname','arial','fontsize',9);
         ylabel('A.U.','fontname','arial','fontsize',9)
         
      case 2
         title(sprintf([T,'Component %i vs Component %i'],PlotStruct.factors(1:2)),'fontname','arial','fontsize',10);
         x = F(:,PlotStruct.factors(1));
         y = F(:,PlotStruct.factors(2));
         z = 0 * x;
         xlabel(sprintf('Component %i (A.U.)',PlotStruct.factors(1)),'fontname','arial','fontsize',9);
         ylabel(sprintf('Component %i (A.U.)',PlotStruct.factors(2)),'fontname','arial','fontsize',9);
         
      otherwise
         title(sprintf([T,'Comp. %i vs Comp. %i vs Comp. %i'],PlotStruct.factors(1:3)),'fontname','arial','fontsize',10);
         x = F(:,PlotStruct.factors(1));
         y = F(:,PlotStruct.factors(2));
         z = F(:,PlotStruct.factors(3));
         xlabel(sprintf('Component %i (A.U.)',PlotStruct.factors(1)),'fontname','arial','fontsize',9);
         ylabel(sprintf('Component %i (A.U.)',PlotStruct.factors(2)),'fontname','arial','fontsize',9);
         zlabel(sprintf('Component %i (A.U.)',PlotStruct.factors(3)),'fontname','arial','fontsize',9);      
         
   end
   if PlotStruct.nummode == 1 & ~isempty(ModelOut.validation) & strcmp(ModelOut.validation.method,'test')
      Ind = setdiff(1:size(ModelOut.validation.xfactors{1},1),ModelOut.validation.segments);
      PlotModel(x,y,z,PlotStruct,ModelOut.data.dataset.labobj{PlotStruct.nummode}(Ind),Axes)
   else
      PlotModel(x,y,z,PlotStruct,ModelOut.data.dataset.labobj{PlotStruct.nummode},Axes)
   end
   if ~PlotStruct.factors(1)
      Factors = skcell(1,size(F,2),'Factor')';
      legend(flipud(findobj(Axes,'tag','model','type','line')),Factors)
   end
   
end
grid(PlotStruct.grid)
setappdata(Axes,'ModelOut',ModelOut)
return
