function Options = ParOptions(varargin);
% function Options = ParOptions (varargin);
% 
% Description:
% Defines the Options for GenPARAFAC and GNPARAFAC
% 
% Inputs: 
% varargin: it can be one structure (in which case the function controls only if the values in the
% structure are available) or a list of option names and obtions values
% Options = ParOptions (optionname1,optionvalue1,optionname2,....);
% In the latter case the non specified options will have default value.
% 
% 
% Outputs:
% Options: Options structure, fields:
% cles          : on(default)/off - activate/deactivate Component Loadings Equal Scaling 
% compmaxiter   : Max number of iterations in Compression                  (default: 3)
% compmode      : Mode to compress to dimension two in DTLD if the smallest slab has
%                 dimension larger than two. Default is 0 (the smallest will be compressed)
% compression   : on/off(default) - activate/deactivate compression
% diagnostics   : on(default)/off - show/do not show diagnostics
% display       : n(default 5)    - show fit every n iterations
%                 none            - no fit is shown
% fitconvcrit   : converge if loss function/sum(X(:).^2) < value           (default: eps)
% frin          : on(default)/off - activate/deactivate FRIN (Factors Reparametrisation by Infinte Norm)
% gradconvcrit  : converge if max(abs(gradient)) < value                   (default: 1e-9)
% initialisation: random(default) - random values
%                 orth            - orthogonalised random loadings
%                 svd             - nipals
%                 dtld            - DTLD/GRAM
%                 swatld          - SWATLD
% inititer      : Max number of iterations for initialisation              (default: 5)
% inittol       : Tolerance for initialisation (when iterative methods are employed)
% jacsv         : on/off(default) - save/don't save jacobian singular value
% lambdaupdate  : levmar(default) - standard Levenberg-Marquadt update
%                 alpha           - update strategy based on line search
%                 hbn             - ref. to Hans Bruun Nielsen paper
% lambdaudpar   : lambda update parameters                                 (default: 2,3)
% large         : on/off(default) - use/do not use large models routine to calculate the jacobian (See ParJacS)
% linesearch    : iter            - do line search at every iteration
%                 diverg          - do line search when divergence occurs
%                 none(default)   - no line search
% maxiter       : Max number of iterations in fitting                      (default: 500)
% normeqsolver  : ldivide(default)- use matlab's backslash operator
%                 pcg             - use preconditioned conjugate gradients
%                 cholesky        - use Cholesky factorisation and backsubstitution
% parconvcrit   : converge if relative change in step's 2-norm < value     (default: 1e-8)
% relfitconvcrit: converge if relative fit < value                         (default: 1e-6)
% refmaxiter    : Max number of iteration in the refining step             (default: 10000)
% tuckextra     : Number of extra components in Tucker3 compression model. (default: 2)
%
% Called by:
% Model_PARAFAC\PARAFACGNFit, Model_PARAFAC\GNParafac, Model_PARAFAC\GenParafac
% 
% Subroutines:
% Internal: None
% External: None
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 07-Jun-2002 19:16:08
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
%


if nargin == 1
   Op = fieldnames(varargin{1});
   for i = 1:length(Op)
      V = getfield(varargin{1},Op{i});
      switch lower(Op{i})
      case 'cles'
         if ~any(strcmp({'on','off'},lower(V)))
            error('Invalid assignment for ''CLES''. It must be ''on'' or ''off''')
         end
         
      case 'compression'   
         if ~any(strcmp({'on','off'},lower(V)))
            error('Invalid assignment for ''Compression''. It must be ''on'' or ''off''')
         end
         
      case 'compmaxiter'
         if ~isa(V,'double')
            error('Invalid property assignment for ''CompMaxIter''. Value must be an integer')
         elseif rem(V,1)
            error('Invalid property assignment for ''CompMaxIter''. Value must be an integer')
         end         
         
      case 'compmode'
         if ~isa(V,'double')
            error('Invalid property assignment for ''CompMode''. Value must be an integer')
         end
         
      case 'diagnostics'
         if ~any(strcmp({'on','off'},lower(V)))
            error('Invalid assignment for ''Diagnostics''. It must be ''on'' or ''off''')
         end
         
      case 'display'
         if ~strcmp('none',lower(V)) & isa(V,'double')
            if rem(V,1)
               error('Invalid property assignment for ''Display''. Value must be an integer or ''None''')
            end
         end
         
      case 'fitconvcrit'
         if ~isa(V,'double')
            error('Invalid property assignment for ''FitConvCrit''. Value must be a number')
         end
         
      case 'frin'
         if ~any(strcmp({'on','off'},lower(V)))
            error('Invalid assignment for ''FRIN''. It must be ''on'' or ''off''')
         end
         
      case 'gradconvcrit'
         if ~isa(V,'double')
            error('Invalid property assignment for ''GradConvCrit''. Value must be a number')
         end
         
      case 'initialisation'
         if ~any(strcmp({'random','orth','nipals','dtld','swatld','best'},lower(V)))
            error('Invalid property assignment for ''Initialisation''. Value must be either ''random'',''orth'',''nipals'',''dtld'' or ''swatld''')
         end
         
      case 'inititer'
         if ~isa(V,'double')
            error('Invalid property assignment for ''InitIter''. Value must be an integer')
         elseif rem(V,1)
            error('Invalid property assignment for ''InitIter''. Value must be an integer')
         end         
         
      case 'inittol'
         if ~isa(V,'double')
            error('Invalid property assignment for ''InitTol''. Value must be a number')
         end
         
      case 'jacsv'
         if ~any(strcmp({'on','off'},lower(V)))
            error('Invalid assignment for ''JacSV''. It must be ''on'' or ''off''')
         end
         
      case 'lambdaudpar'
         if ~isa(V,'double') | length(V) ~= 2
            error('Invalid property assignment for ''LambdaUDPar''. Value must be a vector of 2 numbers')
         elseif V(2) >= 1
            error('Invalid property assignment for ''LabdaUDPar'', Value(2) must be less than 1')
         end         
         
      case 'lambdaupdate'
         if ~any(strcmp({'levmar','alpha','hbn'},lower(V)))
            error('Invalid property assignment for ''LambdaUpdate''. Value must be either ''levmar'',''alpha'',''hbn''')
         end
         
      case 'large'
         if ~any(strcmp({'on','off'},lower(V)))
            error('Invalid assignment for ''Large''. It must be ''on'' or ''off''')
         end
         
      case 'linesearch'
         if ~any(strcmp({'diverg','iter','none'},lower(V)))
            error('Invalid property assignment for ''LineSearch''. Value must be either ''diverg'',''iter'',''none''')
         end
         
      case 'maxiter'
         if ~isa(V,'double')
            error('Invalid property assignment for ''MaxIter''. Value must be an integer')
         elseif rem(V,1)
            error('Invalid property assignment for ''MaxIter''. Value must be an integer')
         end         
         
      case 'newton'
         if ~any(strcmp({'on','off'},lower(V)))
            error('Invalid assignment for ''Newton''. It must be ''on'' or ''off''')
         end
         
      case 'normeqsolver'
         if ~any(strcmp({'ldivide','pcg','cholesky'},lower(V)))
            error('Invalid property assignment for ''NormEqSolver''. Value must be either ''ldivide'',''pcg'',''cholesky''')
         end
         
      case 'parconvcrit'
         if ~isa(V,'double')
            error('Invalid property assignment for ''ParConvCrit''. Value must be a number')
         end
         
      case 'pcgmaxiter'
         if ~isa(V,'double')
            error('Invalid property assignment for ''PCGMaxIter''. Value must be an integer')
         elseif rem(V,1)
            error('Invalid property assignment for ''PCGMaxIter''. Value must be an integer')
         end         
         
      case 'pcgprecond'
         if ~isa(V,'double') & length(V) < 2
            error('Invalid property assignment for ''PCGPreCond''. Value must be a matrix')
         end
         
      case 'pcgtol'
         if ~isa(V,'double')
            error('Invalid property assignment for ''PCGTol''. Value must be a number')
         end
         
      case 'relfitconvcrit'
         if ~isa(V,'double')
            error('Invalid property assignment for ''RelFitConvCrit''. Value must be a number')
         end
         
      case 'refmaxiter'
         if ~isa(V,'double')
            error('Invalid property assignment for ''RefMaxIter''. Value must be an integer')
         elseif rem(V,1)
            error('Invalid property assignment for ''RefMaxIter''. Value must be an integer')
         end         

      case 'separate'
         if ~any(strcmp({'on','off'},lower(V)))
            error('Invalid assignment for ''Separate''. It must be ''on'' or ''off''')
         end
         
      case 'tuckextra'    
         if ~isa(V,'double')
            error('Invalid property assignment for ''TuckExtra''. Value must be an integer')
         elseif rem(V,1)
            error('Invalid property assignment for ''TuckExtra''. Value must be an integer')
         end         
         
      otherwise
         error('Invalid property name')
         
      end
   end
   Options = varargin{1};
   if strcmp(Options.newton,'on') & strcmp(Options.separate,'on')
      error('Use of variable separation cannot be combined with second derivatives')
   end
   if strcmp(Options.lambdaupdate,'alpha')
      Options.linesearch = 'iter';
   end
   
else
   
   Options = struct('cles','on',...
      'compmaxiter',3,...
      'compmode',0,...
      'compression','on',...
      'diagnostics','on',...
      'display',1,...
      'fitconvcrit',eps,...
      'frin','on',...
      'gradconvcrit',1e-9,...
      'initialisation','random',...
      'inititer',5,...
      'inittol',1e-5,...
      'jacsv','off',...
      'lambdaudpar',[2,1/3],...
      'lambdaupdate','levmar',...
      'large','off',...
      'linesearch','none',...
      'maxiter',500,...
      'newton','off',...
      'normeqsolver','ldivide',...
      'parconvcrit',1e-8,...
      'pcgmaxiter',20,...
      'pcgprecond',[],...
      'pcgtol',1e-5,...
      'refmaxiter',10000,...
      'relfitconvcrit',1e-6,...
      'separate','off',...
      'tuckextra',2);

   if rem(nargin,2)
      if isa(varargin{1},'struct')
         if ~all(strcmp(sort(fieldnames(varargin{1})),sort(fieldnames(Options))))
            error('Incompatible structures')
         else 
            Options     = varargin{1};
            varargin(1) = [];
         end
      else
         error('Properties and values must come in pairs')
      end
   end
   for i = 1:2:length(varargin)
      Options = setfield(Options,lower(varargin{i}),varargin{i+1});
   end
end
