function [conv,varargout] = InitPar(X,F,Mode,Iter,Tol,Const)
% function [conv,A,B,C,...] = InitPar (X,F,Mode,Iter,Tol,Const);
% 
% Description:
% Find initial estimates for a PARAFAC model with F components fit on X
% 
% Inputs: 
% X: array of double
% F: number of factors to extract
% Mode: six type of initialisation are available
% 
%    'orth'  : ten starts with orthogonal random matrices. The best is picked after Iter iterations 
%    'random': ten starts with random matrices. The best is picked after Iter iterations 
%    'nipals': the matrices are obtained via SVD  
%    'swatld': SWATLD algorithm, available only up to 3 way and no missing values 
%    'dltd'  : DTLD algorithm, available only up to 3 way and no missing values 
%    'best'  : uses all the possible initalisations and picks the best one 
% 
% 
% Iter : max number of iterations for the iterative methods
% Tol  : tolerance (i.e. relative fit decrease) for convergence in the iterative methods
% Const: constraints (if required the initial values are found according to constraints); the format is
%        identical to the one used for PARAFAC.m
% 
% 
% Outputs:
% conv: returns 0 if the initialisation does not converge within the max number of iter.
% A,B,C,...: initial estimations for the loadings
% 
% Called by:
% Model_PARAFAC\Calculate, Model_PARAFAC\GNParafac, Model_PARAFAC\GenParafac
% 
% Subroutines:
% Internal: swatld
% External: dtld, kr, ini, missmult, nmodel, normit, nshape, parafac
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 05-Nov-2002 18:22:24
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
%


conv = 1;
if ~exist('Iter','var')
   Iter = 5;
end
if ~exist('Tol','var')
   Tol  = 1e-5;
end
Fit0 = inf;
DimX = size(X);
if any(isnan(X(:))) & any(strcmp({'swatld','dtld'},Mode))
   warning(['Due to missing values ', Mode, ' cannot be employed, random starts will be used instead'])
   Mode = 'random';
end   
if ndims(X) ~= 3 & any(strcmp({'swatld','dtld'},Mode))
   warning([Mode,' cannot be employed as initialisation for ' int2str(ndims(X)) '-way arrays, random starts will be used instead'])
   Mode = 'random';
end
switch Mode
case {'random','orth'}
   for i=1:10
      Z     = ones(1,F);
      Ztemp = ones(F);
      for j = ndims(X):-1:2
         Fac{j} = normit(rand(DimX(j),F)); %set non orthogonal basis
         if strcmp(Mode,'orth') == 2
            Fac{j} = orth(Fac{j});  %set orthogonal basis for factors
         end
         Z     = kr(Z,Fac{j});
         Ztemp = Ztemp .* (Fac{j}'*Fac{j});
      end
      Fac{1} = nshape(X,1) * Z * pinv(Ztemp);
      if any(Iter)
         [Fac,nil,fit] = parafac(X,F,[Tol,0,0,0,NaN,Iter],Const,Fac); %run It iterations with ALS
      else
         fit = nmodel(Fac);
         fit = nansum((X(:)-fit(:)).^2);
      end
      if fit < Fit0
         [varargout{1:ndims(X)}] = deal(Fac{:});
         Fit0 = fit;
      end
   end
   
case 'nipals'
   Fact                    = ini(X,F,2);
   [varargout{1:ndims(X)}] = deal(Fact{:});
   
case 'dtld'
   [varargout{1:3}] = dtld(X,F,0);
   
case 'swatld'
   [varargout{1:3},conv] = swatld(X,rand(DimX(1),F),rand(DimX(2),F),eps,Iter);
   
case 'best'
   for i=1:10
      Z     = ones(1,F);
      Ztemp = ones(F);
      for i = ndims(X):-1:2
         Fac{i} = orth(normit(rand(DimX(i),F))); %set non orthogonal basis
         Z     = kr(Z,Fac{i});
         Ztemp = Ztemp .* (Fac{i}'*Fac{i});
      end
      Fac{1} = nshape(X,1) * Z * pinv(Ztemp);
      [Fac,nil,fit] = parafac(X,F,[Tol,0,0,0,NaN,Iter],Const,Fac); %run It iterations with ALS
      if fit < Fit0
         [varargout{1:ndims(X)}] = deal(Fac{:});
         Fit0 = fit;
      end
   end
   X1 = X;
   svdfac = ini(X,F,2);
   R = X - nmodel(svdfac);
   fit = sum(R(:).^2);
   if fit < Fit0
      [varargout{1:ndims(X)}] = deal(svdfac{:});
      Fit0 = fit;
   end
   if ~(any(isnan(X(:))) | ndims(X) == 3)
      [swafac{1:3},conv] = swatld(X,rand(DimX(1),F),rand(DimX(2),F),eps,Iter);
      R = X - nmodel(swafac);
      fit = sum(R(:).^2);
      if fit < Fit0
         [varargout{1:ndims(X)}] = deal(swafac{:});
         Fit0 = fit;
      end
      [dtldfac{1:3}] = dtld(X,F,0);
      R = X - nmodel(dtldfac);
      fit = sum(R(:).^2);
      if fit < Fit0
         [varargout{1:ndims(X)}] = deal(dtldfac{:});
         Fit0 = fit;
      end
      %taken from nway toolbox
      varargout{1} = real(varargout{1});
      varargout{2} = real(varargout{2});
      varargout{3} = real(varargout{3});
      % Check for signs and reflect if appropriate
      for f=1:F
         if sign(sum(varargout{1}(:,f)))<0
            if sign(sum(varargout{2}(:,f)))<0
               varargout{2}(:,f)=-varargout{2}(:,f);
               varargout{1}(:,f)=-varargout{1}(:,f);
            elseif sign(sum(varargout{3}(:,f)))<0
               varargout{3}(:,f)=-varargout{3}(:,f);
               varargout{1}(:,f)=-varargout{1}(:,f);
            end
         end
         if sign(sum(varargout{2}(:,f)))<0
            if sign(sum(varargout{3}(:,f)))<0
               varargout{3}(:,f)=-varargout{3}(:,f);
               varargout{2}(:,f)=-varargout{2}(:,f);
            end
         end
      end
   end
   
otherwise 
   error('There is not such an initialisation')
end

%----------------------------------------------------------------------------------------------------------------------------------

function [A,B,C,errflag,iter] = swatld(X,A0,B0,tol,maxiter)
% [A,B,C,errflag,iter,cnv] = swatld(X,A0,B0,F,eps,maxiter)
%
% Purpose: self-weighted alternating trilinear decomposition.
%
% Source: Chen et al. Chemometrics and Intelligent Laboratory Systems, 52 (2000) 75-86.
%
% Requires: trilin_RSS, sort_factors and xy2z.
%
% Input:
% X is the three-way array,
% A0 and B0 are starting values for the profiles in the x and y orders,
% Jx, Jy and Jz are the number of variables in the x, y and z orders,
% F is an estimate of the number of components,
% eps is the threshold for the convergence criterion,
% maxiter is the maximum number of iterations.
%
% Output:
% A, B, C are the resolved profiles in x, y and z orders,
% errflag indicates normal exit (0) or maximum number of iterations (1),
% iter is the number of iterations,
% cnv is the convergence criterion.
%
% Author  : N.M. Faber.
% Modified: G. Tomasi
% Update  : May 9, 2002.

if ~exist('tol','var')
   tol = 1e-5;
end
if ~exist('maxiter','var')
   maxiter = 1000;
end
F          = size(A0,2);
% Initialization:
[Jx,Jy,Jz] = size(X);
A          = A0 / diag(sqrt(sum(A0.^2)) .* sign(sum(A0.^3)));
B          = B0 / diag(sqrt(sum(B0.^2)) .* sign(sum(B0.^3)));
pinvB      = pinv(B); 
diagBtB    = diag(B'*B); 
C          = zeros(Jz,F);

% Iterative refinement:
sigmaold = -1;
cnv      = eps + 1;
iter     = 0;
while cnv > eps & iter < maxiter
   
   pinvA   = pinv(A);
   diagAtA = diag(A'*A);
   for jz = 1:Jz
      C(jz,:) = 0.5 * (diag(pinvB*X(:,:,jz)'*A)./diagAtA + diag(pinvA*X(:,:,jz) *B)./diagBtB)';
   end
   pinvC   = pinv(C);
   diagCtC = diag(C'*C);
   for jy = 1:Jy
      B(jy,:) = 0.5 * (diag(pinvA*squeeze(X(:,jy,:)) *C)./diagCtC + diag(pinvC*squeeze(X(:,jy,:))'*A)./diagAtA)';
   end
   B       = B / diag(sqrt(sum(B.^2)) .* sign(sum(B.^3)));
   pinvB   = pinv(B);
   diagBtB = diag(B'*B);
   for jx = 1:Jx
      A(jx,:) = 0.5 * (diag(pinvC*squeeze(X(jx,:,:))'*B)./diagBtB + diag(pinvB*squeeze(X(jx,:,:)) *C)./diagCtC)';
   end
   A     = A / diag(sqrt(sum(A.^2)) .* sign(sum(A.^3)));
   Res   = X - nmodel({A,B,C});
   sigma = sum(Res(:));
   cnv   = abs((sigma-sigmaold)/sigmaold);
   iter  = iter + 1; sigmaold = sigma;
   
end

if iter < maxiter
   errflag = 0;
else
   errflag = 1;
end
invAtABtB = pinv((A'*A).*(B'*B));
for j = 1:Jz
   C(j,:) = (invAtABtB * diag(A'* X(:,:,jz) * B))';
end

%----------------------------------------------------------------------------------------------------------------------------------


function y = nanmean(x)
%NANMEAN Average or mean ignoring NaNs.
%   NANMEAN(X) returns the average treating NaNs as missing values.  
%   For vectors, NANMEAN(X) is the mean value of the non-NaN
%   elements in X.  For matrices, NANMEAN(X) is a row vector
%   containing the mean value of each column, ignoring NaNs.
%
%   See also NANMEDIAN, NANSTD, NANMIN, NANMAX, NANSUM.

%   Copyright (c) 1993-98 by The MathWorks, Inc.
%   $Revision: 2.8 $  $Date: 1997/11/29 01:45:53 $

if isempty(x) % Check for empty input.
    y = NaN;
    return
end

% Replace NaNs with zeros.
nans = isnan(x);
i = find(nans);
x(i) = zeros(size(i));

if min(size(x))==1,
  count = length(x)-sum(nans);
else
  count = size(x,1)-sum(nans);
end

% Protect against a column of all NaNs
i = find(count==0);
count(i) = ones(size(i));
y = sum(x)./count;
y(i) = i + NaN;
