function varargout = FacOptScale(varargin)
% function [As,Bs,Cs,...,as,bs,cs] = FacOptScale (A,B,C,...,a,b,c,...);
% 
% Description:
% Scales the loading vectors of each factors in the two sets so that the euclidean distance between the
% two of them is minimised.
% It uses a constrained optimisation routine (via the function parafacscaerr) and the scaling factor
% has to be between 0 and 2. The norm of each factor is equally spread over the vectors defining it.
% 
% 
% Inputs: 
% A,B,C,...: loadings set to be scaled
% a,b,c,...: reference set
% 
% Outputs:
% As,Bs,Cs,...: optimally scaled loadings set 
% as,bs,cs,...: scaled reference set
%  
% 
% Called by:
% Model_PARAFAC\Compute_Risk
% 
% Subroutines:
% Internal: isodd (returns 1 if argument is odd), normvec (normalises vector to length 1)
% External: parafacscaerr, scale_factors
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 11-Oct-2002 19:43:28
%


if isodd(nargin)
   error('The number of parameters must be even')
end
if nargout ~= nargin
   error('The number of in and out parameters is not consistent')
end
Dims = nargout/2;
Rk1  = size(varargin{1},2); 
Rk   = size(varargin{Dims + 1},2); 
if Rk1 > Rk
   for i = 1:Dims
      varargin{i}(:,Rk + 1:end) = [];
   end
else
   Fl = [];
end
[varargin{1:Dims}]        = scale_factors(0,varargin{1:Dims});
[varargin{Dims+1:Dims*2}] = scale_factors(0,varargin{Dims+1:Dims*2});
for i=1:Rk
   Op = optimset('display','off','maxiter',400,'tolx',1e-8,'largescale','off');
   [t,fic,flag] = fmincon('PARAFACScaErr',ones(Dims-1,1),[],[],[],[],zeros(Dims-1,1),2*ones(Dims-1,1),[],Op,i,varargin{:});
   if flag > 0
      Scal(:,i) = t(:);
   else
      Scal(:,i) = ones(Dims-1,1);
   end
end
for i = 1:Dims
   if i < Dims
      Scal(i,:) = Scal(i,:) .* diag(sign(corrcoef([varargin{i},varargin{Dims + i}])),Rk)';
      varargout{i} = varargin{i} * diag(Scal(i,:));
   else
      varargout{nargin/2} = varargin{nargin/2} * diag(prod(Scal))^-1;
   end
end
[varargout{Dims+1:2*Dims}] = deal(varargin{Dims+1:Dims*2});
   

function b = isodd(a)
%Check for odd numbers
b = logical(rem(a,2));

function A = normvec(B)
% A = normvec(B)
%
% A contains the columnwise normalised (to length 1) B

A = B*diag(sqrt(sum(abs(B).^2)))^-1;