function ModelOut = Calculate(ModelIn)
% function ModelOut = Calculate (ModelIn);
% 
% Description:
% Fills in the ModelOut structure and calls the adequate functions (depending on the algorithm chosen
% to fit it) to fit a PARAFAC model to the selected part of X{1}.
% 
% Inputs: 
% ModelIn: ModelIn structure
% 
% 
% Outputs:
% ModelOut: ModelOut structure
% 
% 
% Called by:
% Model_PARAFAC\PARAFACModel
% 
% Subroutines:
% Internal: None
% External: definearray, definemodelout, fitmodel, initpar, nprocess, parafacfit, parafacgnfit,
%           parafacgnpred, parafacpred, validatemodel
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 04-Nov-2002 10:40:54
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
%
ModelOut            = DefineModelOut;
ModelOut.modelname  = 'PARAFAC';
ModelOut.prediction = [];
ModelOut.info       = [];
%Define array
[X,Nil,Sams,Vars] = DefineArray(ModelIn,1,1);

%Handle constraints
Const    = {'None','Non-negativity','Orthogonality','Unimodality'};
[a,Ex_Const] = intersect({ModelIn.constraints.name},lower(Const));
Const = [2 1 3];
Constraints = zeros(1,ndims(X{1}));
for i = 1:length(a)
   if any(ModelIn.constraints(Ex_Const(i)).modes)
      Constraints(find(ModelIn.constraints(Ex_Const(i)).modes)) = Const(i);
   end
end
Options = [find(strcmp({'ALS','dGN'},ModelIn.options.algorithm.name))-1;ModelIn.options.algorithm.constants(:);ModelIn.options.algorithm.maxiter(:)];

%Handle preprocessing
a            = struct('cen',[ModelIn.sam(1).cen(1) ModelIn.var(1).cen],'scal',[ModelIn.sam(1).scal(1) ModelIn.var(1).scal]);
PreProcess   = struct('modx',a,'mody',struct('cen',[],'scal',[]));
h            = waitbar(0,'Please wait...');
Tot          = length(ModelIn.nbfactors.min:max([ModelIn.nbfactors.min,ModelIn.nbfactors.max]));
Count        = 0;
if ~Options(1) & ~Options(8)
   FitFun  = 'parafacfit';      %Implemented this way so that other algorithm and m-files can be employed in the future
   PredFun = 'parafacpred';
   Options([1 2 6]) = Options([2 5 11]);
   Options([3 4 5]) = [0 0 NaN];
   Options = Options(1:6);
else
   FitFun      = 'parafacgnfit';
   PredFun     = 'parafacgnpred';
   Constraints = zeros(1,ndims(X{1}));
end

%Remove slabs
[Xb,IndX{1:ndims(X{1})}] = cleanx(X{1});
Cleaned                  = false;
if ~isequal(size(Xb),size(X))
   uiwait(warndlg({'Some slabs contain only missing values.','These slabs will be automatically skipped'},'Skip slabs'))
   X{1} = Xb;
   Sams = Sams(IndX{1});
   for m = 2:length(IndX)
      Vars{1}{m - 1} = Vars{1}{m - 1}(IndX{m});
   end
   Cleaned = true;
else
   clear Xb
end

%Compute model
for f = ModelIn.nbfactors.min:max([ModelIn.nbfactors.min,ModelIn.nbfactors.max])
   if any(isnan(X{1}(:))) & any([4 5] == ModelIn.options.algorithm.constants(4))
      warndlg('Due to missing values nipals is used for initialisation')
      ModelIn.options.algorithm.constants(4) = 3;
   end
   InitOpt = {'random','orth','nipals','dtld','swatld','best'};
   [conv,InitValues{1:ndims(X{1})}] = ...
      InitPar(nprocess(X{1},[ModelIn.sam(1).cen(1) ModelIn.var(1).cen],[ModelIn.sam(1).scal(1) ModelIn.var(1).scal],[],[],1,-1),...
      f,InitOpt{find(1:6 == ModelIn.options.algorithm.constants(4))},ModelIn.options.algorithm.maxiter(3),1e-5,Constraints);
   ModelOut.model(f - ModelIn.nbfactors.min + 1) = FitModel(FitFun,PredFun,X{1},[],f,PreProcess,Options,Constraints,InitValues);
   if ~isempty(ModelIn.valmethod.name) 
      InitialValues = ModelOut.model(f - ModelIn.nbfactors.min + 1).xfactors;
      switch ModelIn.valmethod.name
      case 'loo'
         InitialValues{1} = InitialValues{1}(1:end-1,:);
         ModelOut.validation(f - ModelIn.nbfactors.min + 1) = ValidateModel(FitFun,PredFun,X{1},[],f,ModelIn.valmethod.name,PreProcess,1,Options,Constraints,InitialValues);
      case {'nboo','rboo'}
         ModelOut.validation(f - ModelIn.nbfactors.min + 1) = ValidateModel(FitFun,PredFun,X{1},[],f,ModelIn.valmethod.name,PreProcess,1,ModelIn.valmethod.replicates,Options,Constraints,InitialValues);
      case {'test'}
         Xt                        = DefineArray(ModelIn,1,0);
         [nil,Samt]                = intersect(ModelIn.sam(1).sel.sel{1:2},ModelIn.sam(1).sel.sel{2});
         IndXt                     = IndX;
         [Xb,IndXt{1}]             = cleanx(Xt{1});
         Cleaned                   = false;
         if ~isequal(size(Xb),size(Xt))
            uiwait(warndlg({'Some slabs in the test set contain only missing values.','These slabs will be automatically skipped'},'Skip slabs'))
            Xt{1}   = Xb;
            Samt    = intersect(Samt,IndXt{1});
            Cleaned = true;
         else
            clear Xb
         end
         ModelOut.validation(f - ModelIn.nbfactors.min + 1) = ValidateModel(FitFun,PredFun,Xt{1},[],f,ModelIn.valmethod.name,PreProcess,1,Samt,ModelOut.model(f - ModelIn.nbfactors.min + 1),Options,Constraints,InitialValues);
      end
   else
      ModelOut.validation = [];
   end
   Count = Count + 1;
   waitbar(Count/Tot,h)
end
close(h)
if strcmp(ModelIn.valmethod.name,'test')
   
   D    = DefineArray(ModelIn,1,0,1);
   D{1} = D{1}(IndXt{:}); 
   if Cleaned
      S    = ModelIn.sam(1).sel.sel{1:2};
      S    = S(IndXt{1});
   end
   Data = struct('batches',{{S}},'dataset',D,'variables',Vars(1));
   
else
   D    = DefineArray(ModelIn,1,1,1);
   D{1} = D{1}(IndX{:});
   S    = ModelIn.sam(1).sel.sel{1};
   Data = struct('batches',{{S(IndX{1})}},'dataset',D,'variables',Vars(1));
end
ModelOut(1).data = Data;
%By default the last mode is the time mode -> time dir
Info = struct('algorithm',ModelIn.options.algorithm.name,...
              'constants',ModelIn.options.algorithm.constants,...
              'constraints',Constraints,...
              'content',dscontent(ModelOut(1).data.dataset),...
              'fillin','Current deviation',...
              'maxiter',ModelIn.options.algorithm.maxiter,...
              'preprocess',PreProcess);
ModelOut(1).info = Info;
return