function DisplayInfo(MainFig,ModelOut)
% function PARAFACDisplayInfo (MainFig,ModelOut);
% 
% Description:
% It displays the informations relative to the current PARAFAC model(s). If there is a plot it displays
% the informations saved in the application-defined data 'ModelOut' of the current plot. The data
% passed to displayinfo are those utilised to compute the PARAFAC model and stored in
% ModelOut.data.dataset It also handles the opening of a window (associated with a
% 'click here for more' text) with additional information on the PARAFAC model currently loaded/
% displayed
% 
% 
% Inputs: 
% MainFig: handle to the main figure
% ModelOut: ModelOut structure or handle (in the second case it handles the additional information window.
% 
% 
% Outputs:
% None
% 
% 
% Called by:
% Model_PARAFAC\PARAFACPlot, DisplayModel
% 
% Subroutines:
% Internal: None
% External: displayinfo
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 04-Nov-2002 11:09:50
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
%

GuiNames
%Some initial values
fnor  = {'fontunits','normalized'};
vert  = {'verticalalignment','cap'};
Alg   = {'ALS','dGN'};
Names = {': Leave One Out',[': Na' char(239) 've Bootstrap'],': Residuals Bootstrap',': Test set'};
S30   = [];
S2a   = {'\fontsize{10}Algorithm';' ';'N. of factors';'Expl. Variation';'Centring';'Scaling'};
 
if ~ishandle(ModelOut)

   %Cleaning up the window
   delete(findobj(tg,'textfig'))
   
   %Get the model data
   X = ModelOut.data.dataset;
   if ~isa(X,'cell')
      X = {X};
   end
   
   %Display data info
   DisplayInfo(MainFig,X);

   a    = findobj(tg,'textfig');
   apos = get(a,'extent');
   %Either X and Y or additional information on the data set
   if isa(apos,'cell')
       apos = min(cat(1,apos{:}));
   end
   
   %Informations on the model
   %Label
   
   %Algorithm
   AlgSt = [Alg{find(strcmp(Alg,ModelOut.info.algorithm))}];
   if length(ModelOut.info.constants) > 3
       if ModelOut.info.constants(7) 
           AlgSt = [AlgSt,' with'];
           Nil   = 'compression';
       else
           Nil   = '  ';
       end
   end
   %Number of factors
   if length(ModelOut.model) ~= 1
       %if more than one was calculated
       NbFacStr = sprintf('%i - %i',ModelOut.model(1).nbfactors,ModelOut.model(end).nbfactors);
       EVStr    = sprintf('%3.2f%% - %3.2f%%',ModelOut(1).model(1).xev,ModelOut(1).model(end).xev);
   else
       NbFacStr = sprintf('%i',ModelOut.model.nbfactors);
       EVStr    = num2str(ModelOut.model.xev,'%3.2f%%');
   end
   %Actual info
   S2b = { [': ',AlgSt];...
           ['  ',Nil];...
           [': ',NbFacStr];...
           [': ',EVStr];...
           [': [ ',sprintf('%i ',ModelOut.info.preprocess.modx.cen(:)'),']'];...
           [': [ ',sprintf('%i ',ModelOut.info.preprocess.modx.scal(:)'),']']};
   
   %Display in the info box
   a    = text(.5,apos(2) - 0.05,'\bf\fontsize{12}Model: PARAFAC',ha,'center',vert{:},fnor{:},tg,'textfig','units','normalized');
   apos = get(a,'extent');
   Nil  = text(.05,apos(2)-0.02,S2a,ha,'left','verticalalignment','cap',tg,'textfig','units','normalized');
   a    = text(.5,apos(2)-0.02,S2b,ha,'left','verticalalignment','cap',tg,'textfig','units','normalized');
   apos = get(a,'extent');
   %Text: click for more, the callback is in the second part of this file
   a    =  text(.5,apos(2)-0.015,'\itClick here for more','verticalalignment','top',tg,'textfig',ha,'center','buttondownfcn','PARAFACDisplayInfo(gcbf,gcbo);',ud,{ModelOut.info,[ModelOut.model.iterations]});
   apos = get(a,'extent');

   %Display information of validation, if available
   if ~isempty(ModelOut.validation)
       Val   = find(strcmp({'loo','nboo','rboo','test'},ModelOut.validation(1).method));
       S30   = 'Validation';
       %Labels and info
       switch ModelOut.validation(1).method
       case {'nboo','rboo'}
           S3a = {'Method';'Replicates'};
           S3b = {Names{Val};sprintf(': %i',size(ModelOut.validation(1).segments,2))};

       case 'loo'
           S3a = {'Method','Expl. Variation'};
           if length(ModelOut.validation) > 1
               S3b = {Names{Val};...
               sprintf(': %3.2f%% - %3.2f%%',ModelOut(1).validation(1).xev,ModelOut(1).validation(end).xev)};
           else
               S3b   = {Names{Val};sprintf(': %3.2f%%',ModelOut.validation(1).xev)};
           end

       case 'test'
         Size    = size(ModelOut(1).info(1).content);
         Size(1) = length(ModelOut(1).validation(1).segments);
         Size    = [': [',sprintf(' %i',Size),']'];
         S3a     = {'Method';'Size';'Expl. Variation'};
         S3b     = {': Test set';...
              Size;...
              sprintf(': %3.2f%%',ModelOut.validation(1).xev)};
           if length(ModelOut.validation) > 1
              S3b(end) = {sprintf(': %3.2f%% - %3.2f%%',ModelOut(1).validation(1).xev,ModelOut(1).validation(end).xev)};
           end
       otherwise
           S3 = [];
       end
       %Display information right beneath the Model's
       a     = text(.5,apos(2) - 0.05,['\bf\fontsize{12}',S30],ha,'center',vert{:},fnor{:},tg,'textfig','units','normalized');
       apos  = get(a,'extent');
       nil   = text(.05,apos(2)-0.02,S3a,ha,'left','verticalalignment','cap',tg,'textfig','units','normalized');
       a     = text(.5,apos(2)-0.02,S3b,ha,'left','verticalalignment','cap',tg,'textfig','units','normalized');
       apos  = get(a,'extent');
   end
   if ~isempty(ModelOut.prediction)
     
      S30 = 'Projection';
      Siz = size(ModelOut.prediction(1).data);
      if ~isempty(ModelOut.info.initialsize) & ModelOut.info.initialsize ~= size(ModelOut.info.content,ndims(ModelOut.info.content));
         
         Ind      = repmat({':'},ndims(ModelOut.prediction(1).data),1);
         Ind{end} = 1:ModelOut.info.initialsize;
         Mis      = isnan(ModelOut(1).prediction(1).data(Ind{:}));
         Mis      = sum(Mis(:))/length(Mis(:)) * 100;
         S3a      = {'Size','Missing data','Expl. variation','Fill-in method'};
         Siz(end) = ModelOut.info.initialsize;
         S3b = {[': [',int2str(Siz),']'];...
               sprintf(': %3.2f%%',Mis);...
               sprintf(': %3.2f%%',ModelOut.prediction(1).xev);...
               [': ',ModelOut.info.fillin]};
         
      else

         Mis = isnan(ModelOut(1).prediction(1).data);
         Mis = sum(Mis(:))/length(Mis(:)) * 100;
         S3a = {'Size';'Missing data';'Expl. variation'};
         S3b = {[': [',int2str(Siz),']'];...
               sprintf(': %3.2f%%',Mis);...
               sprintf(': %3.2f%%',ModelOut.prediction(1).xev)};
         
      end
      %Display information right beneath the Validation's
      a     = text(.5,apos(2) - 0.05,['\bf\fontsize{12}',S30],ha,'center',vert{:},fnor{:},tg,'textfig','units','normalized');
      apos  = get(a,'extent');
      nil   = text(.05,apos(2)-0.02,S3a,ha,'left','verticalalignment','cap',tg,'textfig','units','normalized');
      a     = text(.5,apos(2)-0.02,S3b,ha,'left','verticalalignment','cap',tg,'textfig','units','normalized');

   end
   set(findobj(gcf,tg,'textfig'),'clipping','on')

else
   
   s         = get(gcbo,ud);
   [Info,It] = deal(s{:});
   h         = figure('menubar','none','name','PARAFAC Model Info','units','normalized','numbertitle','off');
   uimenu(h,la,'Close',cb,'delete(gcbf)')
   axes(pos,[0 0 1 1],'box','on','xtick',[],'ytick',[],'color',WIC,'units','normalized');
   Conv  = {' \fontsize{12}\bf Convergence criteria\rm\fontsize{9}','\fontsize{12}\bf \rm\fontsize{9}';... %1
            ' \circ Relative fit decrease: ','---';...                     %2
            ' \circ Gradient: ','---';...                                     %3
            ' \circ Relative Parameters Change: ','---';...                   %4
            ' ','';...                                                     %5
            ' \fontsize{12}\bf #/Max Iterations\rm\fontsize{9}','\fontsize{12}\bf \rm\fontsize{9}';...           %6
            ' \circ Fitting: ','---';...                                      %7
            ' \circ Refining: ','---';...                                     %8
            ' ','';...                                                     %9
            ' \fontsize{12}\bfInitialisation \rm\fontsize{9}','\fontsize{12}\bf \rm\fontsize{9}';...       %10                              
            ' \circ Method: ','---';...                                       %11
            ' \circ Iterations: ','---';...                                   %12
            ' ','';...                                                     %13
            ' \fontsize{12}\bfdGN Options \rm\fontsize{9}','\fontsize{12}\bf \rm\fontsize{9}';...          %14
            ' \circ FRIN: ','---';...                                   %15
            ' \circ Line-search: ','---';...                                   %16
            ' \circ Large data routines: ','---';...                          %17
            ' ','';...                                                     %18
            ' \fontsize{12}\bfCompression\rm\fontsize{9}','\fontsize{12}\bf \rm\fontsize{9}';...           %19                                                               
            ' \circ Extra components: ','---';...                                       %20
            ' ','';...                                                     %21
            ' \fontsize{12}\bfConstraints \rm\fontsize{9}','\fontsize{12}\bf \rm\fontsize{9}'};          %22                         

   Conv{2,2}  = sprintf('%1.0e',Info.constants(1));
   if size(It,2) > 1
      Conv{7,2}  = sprintf('%i-%i/%i',It(1,1),It(1,end),Info.maxiter(1));
   else
      Conv{7,2}  = sprintf('%i/%i',It(1),Info.maxiter(1));
   end
   %Conv{7,2}  = int2str(Info.maxiter(1));
   %Conv{8,2}  = int2str(Info.maxiter(2));
   if Info.constants(7)
      if size(It,2) > 1
         Conv{8,2}  = sprintf('%i-%i/%i',It(2,1),It(2,end),Info.maxiter(2));
      else
         Conv{8,2}  = sprintf('%i/%i',It(2),Info.maxiter(2));
      end
   end
   Init     = {'Random','Orth','NIPALS','DTLD','SWATLD','Best'};
   Conv{11,2} = Init{Info.constants(4)};
   if Info.constants(4) ~= 4
      Conv{12,2} = int2str(Info.maxiter(3));
   end
   if strcmp('dGN',Info.algorithm) & Info.constants(5)
      Conv{15,2} = 'Applied';
      Conv{3,2}  = sprintf('%1.0e',Info.constants(3));
      Conv{4,2}  = sprintf('%1.0e',Info.constants(2));
      if Info.constants(6) == 0
         Conv{16,2} = 'No';
      elseif Info.constants(6) == 1
         Conv{16,2} = 'Iteration';
      elseif Info.constants(6) == 2
         Conv{16,2} = 'Divergence';
      end
      if Info.constants(8)
         Conv(17,2) = 'Yes';
      else
         Conv(17,2) = 'No';
      end
   end
   if Info.constants(7)
      Conv{20,2} = int2str(Info.constants(9));
   end
   for i = 1:length(Info.constraints)
      Conv(end+1,1) = {sprintf(' \\circ Mode [%i]: ',i)};
      if Info.constraints(i) == 1
         Conv(end,2) = {'Orthogonality'};
      elseif Info.constraints(i) == 2
         Conv(end,2) = {'Non-negativity'};
      elseif Info.constraints(i) == 3
         Conv(end,2) = {'Unimodality'};
      else
         Conv(end,2) = {'No constraints'};
      end
      
   end
   a    = text(0.05,.96,Conv(:,1),'verticalalignment','cap',ha,'left','units','normalized',tg,'textfig');
   apos = get(a,'extent');
   b    = text(apos(3)+0.10,.96,Conv(:,2),'verticalalignment','cap',ha,'left','units','normalized',tg,'textfig');
   set([a,b],'units','pixels');
   txpos = get([a,b],'extent');
   set([a,b],'units','normalized');
   txsize     = cat(1,txpos{:});
   figdims    = sum(txsize) + 60;
   figdims(4) = txsize(1,4) + 20;
   t          = screensize;
   figdims([1:2]) = fix(.5*(t(3:4)-figdims(3:4)));
   if any(figdims(1:2)<0)
      figdims(1:2) = [0 t(4)];
   end
   set(h,'units','pixels',pos,figdims,'resize','off')
   apos = get(a,'extent');
   set(b,pos,[apos(3)+0.10,0.96,0])

end
