%#												
%# [U,S,V] = svd2(X)										
%#												
%# AIM:		Faster singular value decomposition.						
%#												
%# PRINCIPLE: 	Produces a diagonal matrix S, of the same dimension				
%# 		as the minor dimension of X and with nonnegative diagonal elements in decreasing
%# 		order, and unitary matrices U and V so that X = U*S*V'.				
%#												
%# 		When X is m-by-n with m < n, then only the first m columns of V are computed 	
%#		and S is m-by-m [kernel version, see W. WU et al,				
%#		Chemom. Intell. Lab. Sys. (in press)].						
%#												
%# 		By itself, SVD2(X) returns a vector containing diag(S).				
%#												
%# 		[U,S,V] = SVD2(X,0) (econonomy size) has been kept for similarity with SVD,	
%#		but this routine always produces the most economic size.			
%#												
%# INPUT: 	X (m x n) : matrix with m objects and p variables				
%#												
%# OUTPUT: 	U (score matrix), S (diagonal matrix with square root of eigenvalues) 		
%# 		and V (loadings), so that X = U*S*V'.						
%#												
%# AUTHOR:	Jorge Verd-Andrs								
%#		Copyright (c) 1997 for ChemoAC							
%#		Dienst FABI, Vrije Uiversiteit Brussel						
%#		Laarbeeklaan 103, 1090-Jette, Belgium						
%#												
%# VERESION: 1.1 (28/02/1998)									
%#												
%# TEST:	Roy De Maesschalck								
%#												

function [U,S,V]=svd2(X,k)

[m,n]=size(X);				% Matrix size dimension

if m<n					% If more variables than objects, kernel version
	[U,S]=eig(X*X');
	S=diag(S);
	[a,b]=sort(S);
	b=flipud(b);
	S=S(b);
	S=abs(S);
	U=U(:,b);
	V=X'*U*diag(sqrt(1./S));
	S=diag(sqrt(S));

else					% Else, normal SVD
	[U,S,V]=svd(X,0);

end

if nargout==0				% By itself, is returned a vector containing diag(S).
	U=diag(S);
end
