function CuBatch(Mode);
% function CuBatch (Mode);
% 
% Description:
% Cubatch main file. It handles the events occurring within the main window
% The main CuBatch window is given the 'userdata': 'CuBatch'. If this property changes error messages
% will appear. If it is necessary to save further data use SetCVData
% The following application-defined data are created within Cubatch.m for the main window
%    
% Handles        : structure with the handles to the menus 
% Content        : Content data structure referring to the data in X
% Model          : ModelIn structure 
% X              : data 
% Results        : ModelOut structure 
% ChildrenFigures: Handles to the child-windows created via the "Copy to window" menu 
% PlotStr        : PlotStr structure 
% Mode           : either 'advanced' or 'plant'
% 
% The following is application-defined data is created for 'root'
% CVCharWidth: character width (see CharWidth)
% 
% 
% Inputs: 
% Mode (optional): can either be 'advanced' or 'plant' (case insensitive), or NaN. 
%                  If it is not defined a requester is opened so that the user can choose the desired
%                  option (in this case only between 'advanced' or 'plant'). 
%                  The value is saved in the application-defined data 'Mode' of the main CuBatch window.
%                  This parameter assumes value NaN when the function is called as a callback function 
%                  of one of the objects within the main window or by any of the children figures. 
%                  The part of the code that has to be executed in the callback mode is determined on
%                  the basis of the callback-object's tag.
% 
% 
% Outputs:
% None 
%  
% Called by:
% CuBatch itself, in the callback mode.
% 
% Subroutines:
% Internal: DefMenus, SaveModel, CopyrightWarn
% External: activateplots, avmodels, cenwindow, clearusemodel, ctag, cubatch, decomp_menu,
% defaultplot, definemodelout, dispatch_apply, dispatch_compute, displayinfo, displaymodel, 
% editlabels, exportdata, getcvdata, guinames, importdata, initialise, permutewindow, 
% regress_menu, regression, rempath, reshapewindow, savereport, selectsamples, selectvariables,
% setcvdata, setpath, viewdata
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 05-Nov-2002 16:55:48
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
%

%warning off

if nargin == 0  
   if isempty(findobj(0,'type','figure','userdata','Cubatch'))
      Mode = questdlg('Choose operational mode','Choose mode','Advanced','Plant','Advanced');
   else
      Mode = 'advanced';
   end
end
Results = [];
global ModelOut ModelIn X

if ~any(isnan(Mode))
   SetPath
   GuiNames
   if ~any(strcmpi({'advanced','plant'},Mode))
      uiwait(errordlg('Only ''advanced'' and ''plant'' modes are allowed'))
      Mode = 'advanced';
   end
   if ~isempty(findobj(0,'type','figure','userdata','Cubatch'))
      uiwait(warndlg('Cubatch is already running!','Warning'));
      figure(findobj(0,'type','figure','userdata','Cubatch'))
      return
   end
   %Create window
   h = cenwindow([0.5029 0.5814 0.6402],[vers,' - ',lastrev],'CuBatch(NaN);',804,504);
   s = get(0,'screensize');
   set(h,ud,'Cubatch','resize','on','resizefcn','ResizeCuBatch')
   
   %Get default char width for listboxes and save it
   tw   = text(1,1,sprintf('%d',ones(5,1)),cu{:},'units','pixel',off{:});
   tref = get(tw,'extent')/4;
   setappdata(0,'CVCharWidth',tref(3))
   clf;
   [ModelOut,PlotStr] = DefineModelOut;
   %Define menues
   Han      = DefMenus(h,Mode);
   DisplayInfo(h,[]);
   
   %------------Removed for the moment
   %CopyrightWarn
   %------------------------------
   
   %Initial values for Handles, Content, Model, X and Results
   setcvdata(h,'Handles',Han,'Content',[],'Model',[],'X',[],'Results',ModelOut,'ChildrenFigures',[],'PlotStr',PlotStr,'Mode',lower(Mode));   
else
   
   H = gcbf;
   GuiNames
   %Available models
   AvModels
   %Callback mode
   GlHan                         = getcvdata(gcbf,'Handles');
   %[Content,ModelIn,X,ModelOut] = getcvdata(gcbf,'Content','Model','X','Results');
   Content                       = getcvdata(gcbf,'Content');
   Flag                          = [0 0 0];
   switch ctag
      case 'New'
         Ans = questdlg({'The current session will be lost';'Do you wish to continue?'},'New session','OK','Cancel','Cancel');
         if strcmp(Ans,'OK')
            [ModelIn,X]        = deal([]);
            [ModelOut,PlotStr] = DefineModelOut;
            setcvdata(gcbf,'Handles',GlHan,'Content',[],'Model',[],'X',[],'Results',[],'ChildrenFigures',[],'PlotStr',PlotStr)
            delete(findobj(gcbf,'type','axes'))
            delete(findobj(gcbf,tg,'textfig'))
            delete(findobj(gcbf,ty,fr))
            DisplayInfo(gcbf,[]);
            delete(findobj(0,'userdata','PARAFAC Plot Control'))
            delete(findobj(0,'userdata','nPLS1 Plot Control'))
         end
         
      case 'Open'
         try
            
            CDir = cd;
            Dir  = which('Cubatch.m');
            Dir  = Dir(1:max(find(Dir == '\')));
            cd([Dir,'Sessions'])             
            [Xfil{2:-1:1}] = uigetfile('*.mat','Open');
            cd(CDir)
            
         catch
            [Xfil{2:-1:1}] = uigetfile('*.mat','Open');
         end;
         FileContent    = who('-file',[Xfil{:}]);
         SessionContent = {'Content','ModelIn','ModelOut','X'};
         if Xfil{1} ~= 0
            
            Vars = intersect(FileContent,SessionContent);
            Vars = load([Xfil{:}],Vars{:});
            %at least ModelOut shall be present
            if ~any(strcmp(fieldnames(Vars),'ModelOut')) | ~any(strcmp(fieldnames(Vars),'ModelIn'))
               uiwait(errordlg('Wrong file content'))
               return
            end
            %Remove children figures
            h = getcvdata(gcbf,'ChildrenFigures');
            if ~isempty(h)
               set(h,crf,'delete(gcbf)');
            end
            delete(findobj(GlHan.win,tg,'WinMenuItem'))
            
            %Remove any other information
            [Nil,PlotStr] = DefineModelOut;
            setcvdata(gcbf,'Content',[],'Model',[],'X',[],'Results',[],'ChildrenFigures',[],'PlotStr',PlotStr)
            
            %Remove info axes and info
            delete(findobj(gcbf,'type','axes'))
            delete(findobj(gcbf,tg,'textfig'))
            delete(findobj(gcbf,ty,fr)) %to be removed as not necessary
            
            ModelOut = Vars.ModelOut;
            ModelIn  = Vars.ModelIn;
            Content  = Vars.Content;
            setcvdata(gcbf,'Model',ModelIn,'Results',ModelOut,'ChildrenFigures',[])
            
            %If it contains also some data it is a whole session file
            if any(strcmp(fieldnames(Vars),'X'))
               if any(strcmp(fieldnames(Vars),'Content'))
                  Content  = Vars.Content ;
               end
               X = Vars.X;
               if ~isa(X,'cell')
                  uiwait(errordlg('Wrong file format'))
                  return
               end
               if ~isequal(size(Content),size(X))
                  for m = 1:length(X)
                     Content{m} = DSContent(X{m});
                  end
               end
               ModelIn.xpointer = gcbf;
               setcvdata(gcbf,'Content',Content,'X',X)
            else
               X       = [];
            end
            DisplayInfo(gcbf,X);
            
         end
         if isa(ModelOut,'struct') & ~isempty(ModelOut(1).modelname)
            DefaultPlot(ModelIn,ModelOut,Content)
         end
         
      case 'Save'
         try
            CDir = cd;
            Dir  = which('Cubatch.m');
            Dir  = Dir(1:max(find(Dir == '\')));
            cd([Dir,'Sessions'])             
            [Xfil{2:-1:1}] = uiputfile('*.mat','Save');
            cd(CDir)
         catch
            [Xfil{2:-1:1}] = uiputfile('*.mat','Save');
         end;         
         if Xfil{1} ~= 0
            
            Ans            = 'No';
            if strcmp(getappdata(gcbf,'Mode'),'advanced')
               QstOp  = struct('Default','Yes','Interpreter','tex');
               Prompt = {'Save X as well?'
                  '\bfNo\rm keeps the file dimensions to a minimum, but it does not allow some operations in further sessions'
                  '\bfYes\rm allows sample/variable selection, data editing and raw data display in following sessions'
                  '\bfCancel\rm aborts the operation'};
               Ans    = questdlg(Prompt,'Export Options','Yes','No','Cancel',QstOp);
            end
            switch Ans
               case 'Cancel'
                  return
                  
               case 'No'
                  SaveModel([Xfil{:}],Content,ModelIn,ModelOut);
                  
               otherwise
                  SaveModel([Xfil{:}],Content,ModelIn,ModelOut,X);
                  
            end
            
         end
         
      case 'Import'
         ImportData;
         if OK
            %Flags = [1 1 1];
            delete(findobj(gcbf,tg,'cbaxes'))
            %Display the initial figure (something related to the data would be better)
            %AxH = axes(pos,[0.2464,0.0377,0.7312,0.9127]);
            %imagesc(imread('cubedali.jpg'))
            %set(AxH,'xtick',[],'ytick',[],tg,'cbaxes');
            %Display data info
            DisplayInfo(gcbf,X);
            ActivatePlots(gcbf)
         end
          
      case 'Export'
         ExportData(ModelIn)
         
      case 'EditLabels'
         t = findobj(gcbf,'type','uimenu');
         stat = get(t,'enable');
         set(t,dis{:})
         Cont1 = EditLabels(Content{get(gcbo,ud)});
         if ~isempty(Cont1)
            TempUD                    = X{get(gcbo,ud)}.userdata;
            X{get(gcbo,ud)}           = cbdataset(X{get(gcbo,ud)}.data,Cont1);
            X{get(gcbo,ud)}.userdata  = TempUD;
            [Mod,Con]                 = Initialise([],[],[],X{get(gcbo,ud)},{ModelIn.file(get(gcbo,ud)).file,X{get(gcbo,ud)}.name},1);
            ModelIn.var(get(gcbo,ud)) = Mod.var;
            Content(get(gcbo,ud))     = Con;
            Labs                      = Con{1}.labobj;
            ModelIn.samlabs           = Labs{1};
            for i = 1:length(X)
               Content{i}.labels{1} = Labs{1};
            end
            ModelIn.sam(1).sel.sel               = 1;
            ModelIn.var(get(gcbo,ud)).sel{1}.sel = get(gcbo,ud);
            setcvdata(gcbf,'Content',Content,'Model',ModelIn,'X',X)
            delete(allchild(GlHan.labels));
            for i = 1:length(ModelIn.var)
               uimenu(GlHan.labels,la,ModelIn.var(i).name,cb,'CuBatch(NaN)',tg,'EditLabels',ud,i)
            end
            DisplayInfo(gcbf,X);
         end
         set(t(ishandle(t)),{'enable'},stat(ishandle(t)))
         
      case 'Permute'
         PermuteWindow;
         Flag = [1 0 1];
         DisplayInfo(H,getappdata(H,'X'));
         
      case 'Reshape'
         ReshapeWindow;
         Flag = [1 0 1];
         DisplayInfo(H,getappdata(H,'X'));
         
      case 'SaveModel'
         try
            CDir = cd;
            Dir  = which('Cubatch.m');
            Dir  = Dir(1:max(find(Dir == '\')));
            cd([Dir,'Sessions'])             
            cd sessions
            [Xfil{2:-1:1}] = uiputfile('*.mat','Save Model');
            cd(CDir)
         catch
            [Xfil{2:-1:1}] = uiputfile('*.mat','Save Model');
         end
         if Xfil{1} ~= 0
            SaveModel([Xfil{:}],ModelOut);
         end
         
      case 'SelBat'
         Model1 = SelectSamples(ModelIn);
         if ~isempty(Model1);
            ModelIn.sam = Model1.sam;
            setcvdata(gcbf,'Model',Model1);
         end
         
      case 'SelXVar'
         Model1     = ModelIn;
         Model1.var = ModelIn.var(1);
         Model1     = SelectVariables(Model1,'In');
         if ~isempty(Model1);
            ModelIn.var(1) = Model1.var;
            setcvdata(gcbf,'Model',ModelIn)
         end
         
      case 'SelYVar'
         Model1     = ModelIn;
         Model1.var = ModelIn.var(2);
         Model1 = SelectVariables(Model1,'Out');
         if ~isempty(Model1);
            ModelIn.var(2) = Model1.var;
            setcvdata(gcbf,'Model',ModelIn)
         end
         
      case 'ViewData'
         viewdata; 
          
      case Available_Models 
         set(gcbf,'waitstatus','')
         delete(findobj(ModelIn.xpointer,'type','axes','tag','cbaxes'))
         Dispatch_Compute
         if ~isempty(ModelIn)
            setcvdata(gcbf,'Model',ModelIn);
            Flag(1) = 1;
         else 
            %Flag(2) = 1;
            Flag(1:2) = 1;
         end
         if ~isempty(ModelOut)
            setcvdata(gcbf,'Results',ModelOut);
         end
         Flag(1:2) = 1;
         
      case 'AppModel'
         ActExtraPar{1} = 1; %Used as a flag for *ActivatePlot to signal that is an applied model and not only a model
         Dispatch_Apply
         if ~isempty(ModelIn)
            setcvdata(gcbf,'Model',ModelIn);
            %Flag(1) = 1;
         else
            Flag(1:2) = 1;
         end
         if ~isempty(ModelOut)
            setcvdata(gcbf,'Results',ModelOut);
            %Flag(2) = 1;
         end
         Flag(1:2) = 1;
         
      case 'Report'
         SaveReport(gcbf)
         
      case 'clearusemodel',
         ClearUseModel(ModelIn,ModelOut);
         Flag(2) = 1;
         
      case {'window','End'}
         h = getcvdata(gcbf,'ChildrenFigures');
         if ~isempty(h)
            set(h,crf,'delete(gcbf)');
         end
         delete(gcbf)
         delete(findobj(get(0,'children'),'type','figure',tg,'plot control'))
         RemPath
         clear global X ModelOut ModelIn
         return
         
      case 'Export plot'
         if ~isempty(findobj(gcbf,'type','axes','tag','cbaxes'))
            h   = figure('visible','off');
            AxH = findobj(gcbf,'type','axes','tag','cbaxes');
            AxH = copyobj(AxH,h);
            if length(AxH) == 1
               set(AxH,'position',[0.1300 0.1100 0.7750 0.8150])
            end
            filemenufcn(h,'FileExport')
            delete(h)
         end
         return
         
      case 'Print'
         if ~isempty(findobj(gcbf,'type','axes','tag','cbaxes'))
            h   = figure('PaperOrientation','Landscape','visible','off','papertype','A4','paperunits','normalized','renderer','painters');
            set(h,'colormap',get(gcbf,'colormap'))
            AxH = findobj(gcbf,'type','axes','tag','cbaxes');
            AxH = copyobj(AxH,h);
            set(AxH,'units','normalized')
            load AxPos
            if length(AxH) > 1
               set(AxH(:),{'position'},AxPos{length(AxH)})
            else
               set(AxH,'position',AxPos{length(AxH)})
            end
            printdlg(h)
         end
         return
         
      case 'Copy'
         MainFig                                                               = gcbf;
         MainFigPos                                                            = get(MainFig,pos);
         cbf                                                                   = 'CuBatch(NaN)';
         AxH                                                                   = findobj(MainFig,'type','axes');
         AxH(strcmp(get(AxH,tg),'legend') | strcmp(get(AxH,tg),'CBInfoFrame')) = [];
         T          = char(get(get(AxH(1),'title'),'string'))';
         Def        = strrep(strrep(T(:)','\it',''),'\bf','');
         Answer     = inputdlg({'Insert the name for the new window'},'New window name',1,{Def(:)'});
         if ~isempty(Answer)
            
            h          = getappdata(MainFig,'ChildrenFigures');
            h(end + 1) = figure('name',Answer{1},pos,MainFigPos,crf,cbf,tg,'CBChildrenWin','color',WIC,'numbertitle','off');
            d          = sort(copyobj(get(GlHan.win,'children'),h(end)));
            delete(get(GlHan.win,'children'))
            a = uimenu(GlHan.win,la,Answer{1},tg,'WinMenuItem',ud,h(end),cb,cbf);
            set(h,ud,MainFig);
            if length(d) > 2
               copyobj(d(3:end),GlHan.win)
            end
            copyobj(d([2 1]),GlHan.win)
            delete(d)
            setcvdata(h(end),'Handles',GlHan,'ParentFigure',MainFig);
            setcvdata(MainFig,'ChildrenFigures',h);
            AxH  = copyobj(AxH,h(end));
            load AxPos
            set(AxH,'units','normalized')
            if length(AxH) > 1
               set(AxH(:),{'position'},AxPos{length(AxH)})
            else
               set(AxH,'position',AxPos{length(AxH)})
            end
            for i = 1:length(AxH)
               if isappdata(AxH(i),'ModelOut')
                  rmappdata(AxH(i),'ModelOut')
               end
            end
            set(AxH,'tag','')
         end
         return
         
      case 'WinMenuItem'
         figure(get(gcbo,ud))
         return
         
      case 'CBChildrenWin'
         Parent = getcvdata(gcbf,'ParentFigure');
         h      = getcvdata(Parent,'ChildrenFigures');
         h(find(h == gcbf)) = [];
         delete(findobj(allchild(get(gcbf,'userdata')),tg,'WinMenuItem','userdata',gcbf))
         delete(gcbf)
         setcvdata(Parent,'ChildrenFigures',h)
         return
         
      case 'Close Win'
         h = getcvdata(gcbf,'ChildrenFigures');
         delete(findobj(allchild(gcbf),tg,'WinMenuItem'))
         setcvdata(gcbf,'ChildrenFigures',[]);
         delete(h)
         return
         
   end
   if any(Flag) %Suboptimal but necessary to work with global variables, to be removed in the future
      if Flag(1)
         ModelIn = getappdata(H,'Model');
      end
      if Flag(2)
         ModelOut = getappdata(H,'Results');
      end
      if Flag(3)
         X = getappdata(H,'X');
      end
   end 
   if ~isempty(ModelOut) & ~isempty(ModelOut(1).modelname)
      DisplayModel(ModelOut)
   end
   ActivatePlots(H)
   return
end


%----------------------------------------------------------------------------------------------

function GlHan = DefMenus(h,Mode)
if nargin < 2
   Mode = 'Advanced';
elseif isempty(Mode)
   Mode = 'Advanced';
end
GuiNames
cbf = 'CuBatch(NaN);';

%Window's menus
Files = uimenu(h,la,'&Files');
GlHan.edit    = uimenu(h,la,'&Edit',dis{:});
GlHan.expl    = uimenu(h,la,'F&actor analysis',dis{:});
GlHan.regress = uimenu(h,la,'&Regression',dis{:});
GlHan.predict = uimenu(h,la,'&Use model',dis{:});
GlHan.plot    = uimenu(h,la,'Re&sults',dis{:},'tag','res');
GlHan.win     = uimenu(h,la,'&Window',dis{:});
Help          = uimenu(h,la,'&Help',on{:});
if ~strcmpi(Mode,'Advanced')
   set([GlHan.expl,GlHan.regress],off{:})
end 

%Files submenus
uimenu(Files,la,'&Clear session',                         cb,cbf,tg,'New',                      acc,'N');
uimenu(Files,la,'&Open',                                  cb,cbf,tg,'Open',                     acc,'O');
uimenu(Files,la,'&Save',                                  cb,cbf,tg,'Save',                     acc,'S');
i           = uimenu(Files,la,'&Import data from .mat file',sep{:});
uimenu(i,la,'&Single set',cb,cbf,tg,'Import',ud,0,acc,'I');
uimenu(i,la,'&Multiple sets',   cb,cbf,tg,'Import',ud,1,acc,'B')
e           = uimenu(Files,la,'&Export',sep{:});
GlHan.exp   = uimenu(e,la,'&Data to .mat file',           cb,cbf,tg,'Export',            dis{:},acc,'E');
GlHan.exppl = uimenu(e,la,'&Plot',                        cb,cbf,tg,'Export plot',       dis{:},acc,'W');
GlHan.rep   = uimenu(e,la,'&Report to .txt file',         cb,cbf,tg,'Report',     sep{:},dis{:},acc,'R');
GlHan.print = uimenu(Files,la,'&Print',                   cb,cbf,tg,'Print',      sep{:});
uimenu(Files,la,'&Exit',                                  cb,cbf,tg,'End',        sep{:},       acc,'Q');

%Edit submenu
GlHan.view   = uimenu(GlHan.edit,la,'&View data');
GlHan.permute= uimenu(GlHan.edit,la,'&Permute',tg,'Permute',cb,cbf);
GlHan.reshape= uimenu(GlHan.edit,la,'&Reshape',tg,'Reshape',cb,cbf);
GlHan.labels = uimenu(GlHan.edit,la,'&Labels && Info');
GlHan.sel    = uimenu(GlHan.edit,la,'&Select',dis{:});
uimenu(GlHan.sel,la,'&Select Batches',  cb,cbf,tg,'SelBat');
uimenu(GlHan.sel,la,'&Select X Variables',cb,cbf,tg,'SelXVar');
GlHan.yvars  = uimenu(GlHan.sel,la,'&Select Y Variables',cb,cbf,tg,'SelYVar',off{:});

%View submenus
GlHan.Xview     = uimenu(GlHan.view,la,'&X');
GlHan.landXview = uimenu(GlHan.Xview,la,'&Landscape',tg,'landXview');
GlHan.trajXview = uimenu(GlHan.Xview,la,'&2D plots',tg,'trajXview');
GlHan.Yview     = uimenu(GlHan.view,la,'&Y');
GlHan.landYview = uimenu(GlHan.Yview,la,'&Landscape',tg,'landYview');
GlHan.trajYview = uimenu(GlHan.Yview,la,'&2D plots',tg,'trajYview');
% orig: 
% remove 'T' shortcut for -- tucker
% GlHan.viewtb    = uimenu(GlHan.view,la,'&View Toolbar',cb,cbf,tg,'ViewToolbar',sep{:},con{:},off{:},acc,'T');
GlHan.viewtb    = uimenu(GlHan.view,la,'&View Toolbar',cb,cbf,tg,'ViewToolbar',sep{:},con{:},off{:});
s               = uimenu(GlHan.view,la,'&Statistics',sep{:},  dis{:});

%Tasks submenus
Decomp_Menu
Regress_Menu

%Plot submenus
%r = uimenu(GlHan.plot,la,'&Results',tg,'res');
%d = uimenu(GlHan.plot,la,'&Data',dis{:});

%Use model submenus
GlHan.useproj = uimenu(GlHan.predict,la,'Projection',tg,'AppModel',cb,cbf,       acc,'P');
GlHan.usepred = uimenu(GlHan.predict,la,'Prediction',tg,'AppModel',cb,cbf,off{:});
GlHan.useclea = uimenu(GlHan.predict,la,'Clear projected points',tg,'clearusemodel',cb,cbf);

%Window submenus
uimenu(GlHan.win,la,'&Copy to window',cb,cbf,tg,'Copy',     sep{:})
uimenu(GlHan.win,la,'&Close all',     cb,cbf,tg,'Close Win',sep{:})

%Help submenus
Dir = which('Cubatch.m');
Dir = [Dir(1:max(find(Dir == '\'))),'Help'];
uimenu(Help,la,'&User Guide',          cb,['stat=web(''file:///' which('userguide.html') ''',''-browser'');'],en{:},acc,'U');
uimenu(Help,la,'&Programmer Guide',    cb,['stat=web(''file:///' which('progguide.html') ''',''-browser'');'],en{:},acc,'H');
uimenu(Help,la,'&About CuBatch ...',   cb,'AboutCubatch',sep{:})

%Modify default toolbar
set(0,'ShowHiddenHandles','on')
ToolBarHandle = findobj(h,'type','uitoolbar');
Buttons       = get(ToolBarHandle,'children');
New = findobj(Buttons,'TooltipString','New Figure');
set(New,'ClickedCallback','Cubatch(NaN);','Tag','New','TooltipString','Clear session')
Open = findobj(Buttons,'TooltipString','Open File');
set(Open,'ClickedCallback','Cubatch(NaN);','Tag','Open','TooltipString','Open')
Save = findobj(Buttons,'TooltipString','Save Figure');
set(Save,'ClickedCallback','Cubatch(NaN);','Tag','Save','TooltipString','Save')
Print = findobj(Buttons,'TooltipString','Print Figure');
set(Print,'ClickedCallback','Cubatch(NaN);','Tag','Print','TooltipString','Print axes')
set(0,'ShowHiddenHandles','off')

%--------------------------------------------------------------------------------------------------------
 
function SaveModel(File,Content,ModelIn,ModelOut,X)
%It avoids that ModelOut is saved as global
if ~exist('X','var') 
   save(File,'Content','ModelOut','ModelIn') 
else 
   save(File,'ModelOut','Content','ModelIn','X')
end 

%--------------------------------------------------------------------------------------------------------

function CopyrightWarn
war={''
   'This software is the property of TOTALFINAELF-FRANCE',
   'of the VRIJE UNIVERSITEIT BRUSSEL (Belgium),'
   'and of ROYAL AGRICULTURAL AND VETERINARY UNIVERSITY (Danmark). '
   ''
   'It is protected by copyright law and international treaties.'
   'Unauthorized reproduction or distribution of this program', 
   'or any portion of it,'
   'may result in severe civil and criminal penalties and will be prosecuted'
   'to the maximum extent possible under the law.'
   ''};
pause(.2)
uiwait(warndlg(war,'WARNING !'));  
