function Validation = ValidateModel(FitFun,PredFun,X,Y,Factors,Method,PreProcess,Options,varargin)
% function Validation = ValidateModel(FitFun,PredFun,X,Y,Factors,Method,PreProcess,Options,varargin);
% 
% Description:
% Validate a model (either in regression or decomposition) and saves the results in ModelOut.validation.
% With respect to FitModel some further parameters are necessary and their number depend on the type
%(Method) of validation chosen.
% 
% 
% Inputs: 
% FitFun    : function fitting the desired model to the data. 
%             It can be a string (for MatLab 5.3 or superior) or  a function handle 
%             (for MatLab 6.x or superior). See FitModel.
% PredFun   : function computing the predictions for the desired model. It can be a string 
%             (for MatLab 5.3 or superior) or  a function handle (for MatLab 6.x or superior).
%             FitFun and  have specific requirements with respect to input and output
% X         : double array relative to the calibration set
% Y         : double array relative to the calibration set (it must be empty if it is a decomposition 
%             method)  
% Factors   : number of factors to extract (it can be a single number or a vector depending on the 
%             desired model)
% Method    : 'loo' => Leave One Out 
%             'nboo' => Nave Bootstrap 
%             'rboo' => Residuals Bootstrap 
%             'test' => Test Set Validation 
% PreProcess: structure with two fields: 'modx' and 'mody'; each of them is also a structure 
%             with two fields: 'cen' and 'scal'. 'modx' defines the preprocessing relative to X,
%             while 'mody' refers to Y. 
%             The two fields 'cen' and 'scal' are vectors with the same number of elements as the modes 
%             of the relative array (e.g. if X is 3-way PreProcess.modx.cen and PreProcess.modx.scal 
%             must be 3 x 1 (or 1 x 3) vectors. For more information on the values that these two vectors 
%             can be given check the help of nprocess (n-way toolbox).
% Options   : if it is set to 1 RecFac and PermFac are applied to remove the permutational indeterminacy
%             between a model and all the replicates
% varargin  : additional inputs (e.g. Options, Constraints, etc) handled directly by FitFun and PredFun.
%             Depending on the Method the first one or two elements must be used for additional validation
%             parameters:
%             - Leave One Out      : no specific requirements are made on varargin 
%             - BootStrap          : the first element of varargin must be the number of replicates. 
%             - Test Set Validation: varargin{1} is the index, within X, of the samples/batches 
%                                    (horizontal slabs) that compose the test set. varargin{2} are 
%                                    the model parameters, i.e. the structure ModelOut.model. 
% 
% 
% Outputs: 
% Validation_Results: the results of the computations are stored in a structure equal 
%                     to ModelOut.validation. Depending on the chosen method some of the parameters 
%                     (such as Validation.xfactors{1:n} or Validation.core) are concatenated along 
%                     an additional mode. 
%
%                     E.g. Leave One Out Validation of a PARAFAC model for a 3-way array leads to 
%                     Validation.xfactors{1:n} elements that are 3-way arrays.
% 
%                     In particular, in the Leave One Out validation, for some of the parameters a 
%                     further slab along this additional mode is added, holding the predictions for
%                     the left out samples/batches. 
%                     This means that Validation.xfactors{1} for a PARAFAC model with F factors, 
%                     validated via 'loo' on an I x J x K array, has dimensions I x F x (I + 1); 
%                     i.e. I segments with only one sample excluded plus one slab with the predictions.
%                     Validation.xfactors{2} and {3} have sizes of respectively J x F x I and K x F x I 
%                     The Explained Variance and the RMSE in validation are not computed when 
%                     BootStrap is applied.
% 
% 
% Called by:
% Model_nPLS1\Calculate, Model_PARAFAC\Calculate
% 
% Subroutines:
% Internal: Validate_Decomposition, Validate_Regression
% External: definemodelout, facperm, preprocess, recfac
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 02-Nov-2002 18:59:46
% 
% Contact: Giorgio Tomasi, gt@kvl.dk
% 

Available_Methods = {'loo','nboo','rboo','test'};
if ~any([2:6] == exist(FitFun))
   error('The function calculating the model is not in the path')
elseif ~any([2:6] == exist(PredFun))
   error('The function to calculate the predictions is not in the path')
end

if isempty(X)
   error('X array is missing')
end

if ~exist('Y','var')
   Y = [];
end
if ~isempty(Y)
   if size(X,1) ~= size(Y,1)
      error('X and Y must have the same dimension in mode 1')
   end
end

if ~exist('Factors','var')
   Factors = 1;
end
if isempty(Factors)
   Factors = 1;
end

if ~exist('Method','var')
   Method = 'loo';
end
if isempty(Method)
   Method = 'loo';
end
if ~any(strcmp(Available_Methods,Method))
   error('Validation method is not available')
end

Centre_ScaleX_Modes = struct('cen',zeros(1,ndims(X)),'scal',zeros(1,ndims(X)));
Centre_ScaleY_Modes = struct('cen',zeros(1,ndims(Y)),'scal',zeros(1,ndims(Y)));
if ~exist('PreProcess','var')
   PreProcess   = struct('modx',Centre_ScaleX_Modes,'mody',Centre_ScaleY_Modes);
end
if isempty(PreProcess)
   PreProcess   = struct('modx',Centre_ScaleX_Modes,'mody',Centre_ScaleY_Modes);
end

if ~exist('Options','var')
   Options =[];
end
if isempty(Options)
   Options = zeros(10,1);
end

if isempty(Y)
   Validation = Validate_Decomposition(FitFun,PredFun,X,Factors,Method,PreProcess,Options,varargin{:});
else
   Validation = Validate_Regression(FitFun,PredFun,X,Y,Factors,Method,PreProcess,Options,varargin{:});
end

%------------------------------------------------------------------------------------------------------------------------

function Validation_Results = Validate_Decomposition(Validated_Function,Predict_Function,X,Factors,Method,PreProcess,Options,varargin)
M                            = DefineModelOut;
Validation_Results           = M.validation;
Validation_Results.method    = Method;
Validation_Results.nbfactors = Factors;

SizeXor                 = size(X);
[Flag,IndX{1:ndims(X)}] = CleanX(X);
Xp = nprocess(X,PreProcess.modx.cen,PreProcess.modx.scal,[],[],1,-1);
switch Method
case {'nboo','rboo'}
   NRep        = varargin{1};
   varargin(1) = [];
case 'test'
   Validation_Results.segments = varargin{1};
   Model_Parameters            = varargin{2};
   varargin(1:2)               = [];
end
if Options(1) 
   if ~strcmp(Method,'test')
      Model_Parameters = feval(Validated_Function,Xp,Factors,varargin{:});
      if isa(Model_Parameters,'cell')
         Model_Parameters.xfactors = Model_Parameters;
      end
   end
   Parameters_Reference = Model_Parameters.xfactors;
end
SSX = sum(Xp(~isnan(Xp)).^2);

if Flag
   for i = 1:ndims(X)
      if length(IndX{i})~=SizeXor(i)
         warning([num2str(SizeXor(i)-length(IndX{i})) ' slabs in mode ', num2str(i),'are being ignored because containing only missing values'])
      end
   end
end
switch Method
case 'loo'
   Validation_Results.xpred    = NaN*ones(size(X));
   R                           = [1:size(X,1)]';
   R                           = R(:,ones(size(X,1),1));
   R(1:size(X,1)+1:end)        = [];
   Validation_Results.segments = IndX{1}(reshape(R,size(X,1)-1,size(X,1)));
   NRep = length(IndX{1});
   for i=1:NRep;
      [u{i},v{i}]=unique(Validation_Results.segments(:,i));
   end
   for i = 1:NRep
      [Xcal,Centre_Par,Scale_Par]                      = nprocess(X(Validation_Results.segments(:,i),IndX{2:end}),PreProcess.modx.cen,PreProcess.modx.scal,[],[],1,-1);
      [Xtest,Centre_Par,Scale_Par]                     = nprocess(X(IndX{1}(i),IndX{2:end}),PreProcess.modx.cen,PreProcess.modx.scal,Centre_Par,Scale_Par,1,-1);
      Model_Parameters                                 = feval(Validated_Function,Xcal,Factors,varargin{:}); %The validated function shall return a structure where each field 
      Validation_Results                               = MergeResults(Validation_Results,Model_Parameters,NRep,u,v,i,size(Xp,1),Options,[],[],[],Parameters_Reference);
      Predictions                                      = feval(Predict_Function,Xtest,Model_Parameters,varargin{:});
      Validation_Results.xpred(IndX{1}(i),IndX{2:end}) = nprocess(Predictions.xpred,PreProcess.modx.cen,PreProcess.modx.scal,Centre_Par,Scale_Par,-1,-1);   
      Res_X                                            = Xp(IndX{1}(i),IndX{2:end}) - Predictions.xpred;
      Num_Elements                                     = sum(~isnan(Res_X(:)));
      Validation_Results.xpress(1,i)                   = sum(Res_X(~isnan(Res_X)).^2);
      if ~isa(Predictions.xfactors,'cell')
         Predictions.xfactors = {Predictions.xfactors};
      end
      D                                                        = repmat({':'},ndims(Validation_Results.xfactors{1})-2,1);
      Validation_Results.xfactors{1}(IndX{1}(i),D{:},NRep + 1) = Predictions.xfactors{1};
   end
   Validation_Results.xcumpress = sum(Validation_Results.xpress);
   Validation_Results.xrmse     = sqrt(Validation_Results.xcumpress / Num_Elements);
   Validation_Results.xev       = 100 * (1 - Validation_Results.xcumpress / SSX);
   
case 'nboo'
   Validation_Results.segments = round((length(IndX{1})-1)*rand(length(IndX{1}),NRep))+1; %segments: each column contain the samples to include in a realisation of the model
   for i=1:NRep;
      [u{i},v{i}]=unique(Validation_Results.segments(:,i));
   end
   for i = 1:NRep
      [Xcal,Centre_Par,Scale_Par] = nprocess(X(Validation_Results.segments(:,i),IndX{2:end}),PreProcess.modx.cen,PreProcess.modx.scal,[],[],1,-1);
      Model_Parameters            = feval(Validated_Function,Xcal,Factors,varargin{:}); %The validated function shall return a structure where each field 
                                                                                        %correspond to the necessary parameter for the validation function
      Validation_Results          = MergeResults(Validation_Results,Model_Parameters,NRep,u,v,i,size(Xp,1),Options,[],[],[],Parameters_Reference);
   end
   Validation_Results.xpred = [];
   
case 'rboo'
   Model_Parameters            = feval(Validated_Function,Xp,Factors,varargin{:});
   Predictions                 = feval(Predict_Function,Xp,Model_Parameters,varargin{:});
   Xp                          = Xp - Predictions.xpred;
   Model                       = Predictions.xpred;
   Validation_Results.segments = round((length(IndX{1})-1)*rand(length(IndX{1}),NRep))+1;
   [u{1:NRep},v{1:NRep}]       = deal(IndX{1});
   %for i=1:NRep;
   %   [u{i},v{i}]=unique(Validation_Results.segments(:,i));
   %end
   for i = 1:NRep
      Xcal               = Model + Xp(Validation_Results.segments(:,i),IndX{2:end});
      Model_Parameters   = feval(Validated_Function,Xcal,Factors,varargin{:}); 
      Validation_Results = MergeResults(Validation_Results,Model_Parameters,NRep,u,v,i,size(Xp,1),Options,[],[],[],Parameters_Reference);
   end
   Validation_Results.xpred = [];
   
case 'test'
   Validation_Results.xpred                                           = NaN*ones(size(X));
   Validation_Results.xfactors{1}                                     = NaN*ones(size(X,1),Factors);
   [Validation_Results.xfactors{2:length(Model_Parameters.xfactors)}] = deal(Model_Parameters.xfactors{2:end});
   IndX{1}                                                            = intersect(Validation_Results.segments,IndX{1});
   NRep                                                               = 1;
   [Xp,Centre_Par,Scale_Par]                                          = nprocess(X,PreProcess.modx.cen,PreProcess.modx.scal,Model_Parameters.xpreproc.cen,Model_Parameters.xpreproc.scal,1,-1);
   Xtest                                                              = Xp(IndX{:});
   SSX                                                                = sum(Xtest(~isnan(Xtest(:))).^2);
   Predictions                                                        = feval(Predict_Function,Xtest,Model_Parameters,varargin{:});
   Validation_Results.xpred(IndX{:})                                  = nprocess(Predictions.xpred,PreProcess.modx.cen,PreProcess.modx.scal,Centre_Par,Scale_Par,-1,-1);   
   Res_X                                                              = Xp(IndX{:}) - Predictions.xpred;
   Num_Elements_X                                                     = sum(~isnan(Res_X(:)));
   Validation_Results.xpress(1,1)                                     = sum(Res_X(~isnan(Res_X)).^2);
   if ~isa(Predictions.xfactors,'cell')
      Predictions.xfactors = {Predictions.xfactors};
   end
   D                                              = repmat({':'},ndims(Predictions.xfactors{1})-1,1);
   Validation_Results.xfactors{1}(IndX{1},D{:},1) = Predictions.xfactors{1};
   Validation_Results.xcumpress                   = sum(Validation_Results.xpress);
   Validation_Results.xrmse                       = sqrt(Validation_Results.xcumpress / Num_Elements_X);
   Validation_Results.xev                         = 100 * (1 - Validation_Results.xcumpress / SSX);
   
end

%------------------------------------------------------------------------------------------------------------------------

function Validation_Results = Validate_Regression(Validated_Function,Predict_Function,X,Y,Factors,Method,PreProcess,Options,varargin)
M                            = DefineModelOut;
Validation_Results           = M.validation;
Validation_Results.method    = Method;
Validation_Results.nbfactors = Factors;
switch Method
case {'nboo','rboo'}
   NRep        = varargin{1};
   varargin(1) = [];
case 'test'
   Validation_Results.segments = varargin{1};
   Model_Parameters            = varargin{2};
   varargin(1:2)               = [];
end
SizeXor                                  = size(X);
[Flag,IndX{1:ndims(X)},IndY{1:ndims(Y)}] = CleanX(X,Y);
Xp                                       = nprocess(X,PreProcess.modx.cen,PreProcess.modx.scal,[],[],1,-1);
SSX                                      = sum(Xp(~isnan(Xp)).^2);
[Yp,Centre_Par_Y,Scale_Par_Y]            = nprocess(Y,PreProcess.mody.cen,PreProcess.mody.scal,[],[],1,-1);
SSY                                      = sum(Yp(~isnan(Yp)).^2);
for i = 1:ndims(X)
   if length(IndX{i})~=SizeXor(i)
      warning([num2str(SizeXor(i)-length(IndX{i})) ' slabs in mode ', num2str(i),'are being ignored because containing only missing values'])
   end
end
switch Method
case 'loo'
   Validation_Results.xpred    = NaN*ones(size(X));
   Validation_Results.ypred    = NaN*ones([size(Y),size(X,1)+1]);
   R                           = [1:size(X,1)]';
   R                           = R(:,ones(size(X,1),1));
   R(1:size(X,1)+1:end)        = [];
   Validation_Results.segments = IndX{1}(reshape(R,size(X,1)-1,size(X,1)));
   NRep                        = length(IndX{1});
   for i = 1:NRep
      [u{i},v{i}]=unique(Validation_Results.segments(:,i));
      [Xcal,Centre_Par_X,Scale_Par_X]                               = nprocess(X(Validation_Results.segments(:,i),IndX{2:end}),PreProcess.modx.cen,PreProcess.modx.scal,[],[],1,-1);
      [Xtest,Centre_Par_X,Scale_Par_X]                              = nprocess(X(IndX{1}(i),IndX{2:end}),PreProcess.modx.cen,PreProcess.modx.scal,Centre_Par_X,Scale_Par_X,1,-1);
      [Ycal,Centre_Par_Y,Scale_Par_Y]                               = nprocess(Y(Validation_Results.segments(:,i),IndY{2:end}),PreProcess.mody.cen,PreProcess.mody.scal,[],[],1,-1);
      [Ytest,Centre_Par_Y,Scale_Par_Y]                              = nprocess(Y(IndX{1}(i),IndY{2:end}),PreProcess.mody.cen,PreProcess.mody.scal,Centre_Par_Y,Scale_Par_Y,1,-1);
      Model_Parameters                                              = feval(Validated_Function,Xcal,Ycal,Factors,varargin{:}); %The validated function shall return a structure where each field 
                                                                                                                               %correspond to the necessary parameter for the validation function
      Validation_Results                                            = MergeResults(Validation_Results,Model_Parameters,NRep,u,v,i,size(Xp,1),Options,PreProcess,Centre_Par_Y,Scale_Par_Y);
      Predictions                                                   = feval(Predict_Function,Xtest,Model_Parameters,varargin{:});
      if ~isa(Predictions.xfactors,'cell')
         Predictions.xfactors = {Predictions.xfactors};
      end
      IndFac                                                        = repmat({':'},ndims(Predictions.xfactors{1})-1,1);
      Validation_Results.xfactors{1}(IndX{1}(i),IndFac{:},NRep + 1) = Predictions.xfactors{1};
      %IndFac                                                        = repmat({':'},ndims(Predictions.yfactors{1})-1,1);
      %Validation_Results.yfactors{1}(IndY{1}(i),IndFac{:},NRep + 1) = Predictions.yfactors{1};
      Validation_Results.xpred(IndX{1}(i),IndX{2:end})              = nprocess(Predictions.xpred,PreProcess.modx.cen,PreProcess.modx.scal,Centre_Par_X,Scale_Par_X,-1,-1);   
      Validation_Results.ypred(IndY{1}(i),IndY{2:end},NRep + 1)     = nprocess(Predictions.ypred,PreProcess.mody.cen,PreProcess.mody.scal,Centre_Par_Y,Scale_Par_Y,-1,-1);   
      Res_X                                                         = Xp(IndX{1}(i),IndX{2:end}) - Predictions.xpred;
      Num_Elements_X                                                = sum(~isnan(Res_X(:)));
      Res_Y                                                         = Yp(IndY{1}(i),IndY{2:end}) - Predictions.ypred;
      Num_Elements_Y                                                = sum(~isnan(Res_Y(:)));
      Validation_Results.xpress(1,i)                                = sum(Res_X(~isnan(Res_X)).^2);
      Validation_Results.ypress(1,i)                                = sum(Res_Y(~isnan(Res_Y)).^2);
   end
   Validation_Results.xcumpress = sum(Validation_Results.xpress);
   Validation_Results.xrmse     = sqrt(Validation_Results.xcumpress / Num_Elements_X);
   Validation_Results.xev       = 100 * (1 - Validation_Results.xcumpress / SSX);
   Validation_Results.ycumpress = sum(Validation_Results.ypress);
   Validation_Results.yrmse     = sqrt(Validation_Results.ycumpress / Num_Elements_Y);
   Validation_Results.yev       = 100 * (1 - Validation_Results.ycumpress / SSY);
   
case 'nboo'
   Validation_Results.segments = round((length(IndX{1})-1)*rand(length(IndX{1}),NRep))+1; %segments: each column contain the samples to include in a realisation of the model
   for i=1:NRep;
      [u{i},v{i}]=unique(Validation_Results.segments(:,i));
   end
   for i = 1:NRep
      [Xcal,Centre_Par,Scale_Par]     = nprocess(X(Validation_Results.segments(:,i),IndX{2:end}),PreProcess.modx.cen,PreProcess.modx.scal,[],[],1,-1);
      [Ycal,Centre_Par_Y,Scale_Par_Y] = nprocess(Y(Validation_Results.segments(:,i),IndY{2:end}),PreProcess.mody.cen,PreProcess.mody.scal,[],[],1,-1);
      Model_Parameters                = feval(Validated_Function,Xcal,Ycal,Factors,varargin{:}); %The validated function shall return a structure where each field 
      %correspond to the necessary parameter for the validation function
      Validation_Results              = MergeResults(Validation_Results,Model_Parameters,NRep,u,v,i,size(Xp,1),Options,PreProcess,Centre_Par_Y,Scale_Par_Y);
   end
   
case 'rboo'
   Model_Parameters            = feval(Validated_Function,Xp,Yp,Factors,varargin{:});
   Predictions                 = feval(Predict_Function,Xp,Model_Parameters,varargin{:});
   Xp                          = Xp - Predictions.xpred;
   Yp                          = Yp - Predictions.ypred;
   ModelX                      = Predictions.xpred;
   ModelY                      = Predictions.ypred;
   %varargin{1}                 = [];
   Validation_Results.segments = round((length(IndX{1})-1)*rand(length(IndX{1}),NRep))+1;
   [u{1:NRep},v{1:NRep}]       = deal(IndX{1});
   for i = 1:NRep
      Xcal               = ModelX + Xp(Validation_Results.segments(:,i),IndX{2:end});
      Ycal               = ModelY + Yp(Validation_Results.segments(:,i),IndY{2:end});
      Model_Parameters   = feval(Validated_Function,Xcal,Ycal,Factors,varargin{:}); 
      Validation_Results = MergeResults(Validation_Results,Model_Parameters,NRep,u,v,i,size(Xp,1),Options,PreProcess,Centre_Par_Y,Scale_Par_Y);
   end
   
case 'test'
   Validation_Results.xpred                                           = NaN*ones(size(X));
   Validation_Results.ypred                                           = NaN*ones(size(Y));
   Validation_Results.xfactors{1}                                     = NaN*ones(size(X,1),Factors);
   Validation_Results.yfactors{1}                                     = NaN*ones(size(X,1),Factors);
   [Validation_Results.xfactors{2:length(Model_Parameters.xfactors)}] = deal(Model_Parameters.xfactors{2:end});
   NRep                                                               = 1;
   IndX{1}                                                            = intersect(Validation_Results.segments,IndX{1});
   IndY{1}                                                            = intersect(Validation_Results.segments,IndY{1});
   Xp                                                                 = nprocess(X,PreProcess.modx.cen,PreProcess.modx.scal,Model_Parameters.xpreproc.cen,Model_Parameters.xpreproc.scal,1,-1);
   Yp                                                                 = nprocess(Y,PreProcess.mody.cen,PreProcess.mody.scal,Model_Parameters.ypreproc.cen,Model_Parameters.ypreproc.scal,1,-1);
   Xtest                                                              = Xp(IndX{:});
   SSX                                                                = sum(Xp(~isnan(Xp)).^2);
   Ytest                                                              = Yp(IndY{:});
   SSY                                                                = sum(Yp(~isnan(Yp)).^2);
   Predictions                                                        = feval(Predict_Function,Xtest,Model_Parameters,varargin{:});
   if ~isa(Predictions.xfactors,'cell')
      Predictions.xfactors = {Predictions.xfactors};
   end
   IndFac                                              = repmat({':'},ndims(Predictions.xfactors{1})-1,1);
   Validation_Results.xfactors{1}(IndX{1},IndFac{:},1) = Predictions.xfactors{1};
   %IndFac                                              = repmat({':'},ndims(Predictions.yfactors{1})-1,1);
   %Validation_Results.yfactors{1}(IndY{1},IndFac{:},1) = Predictions.yfactors{1};
   Validation_Results.xpred(IndX{:})                   = nprocess(Predictions.xpred,PreProcess.modx.cen,PreProcess.modx.scal,Model_Parameters.xpreproc.cen,Model_Parameters.xpreproc.scal,-1,-1);   
   Validation_Results.ypred(IndY{:})                   = nprocess(Predictions.ypred,PreProcess.mody.cen,PreProcess.mody.scal,Model_Parameters.ypreproc.cen,Model_Parameters.ypreproc.scal,-1,-1);   
   Res_X                                               = Xp(IndX{:}) - Predictions.xpred;
   Num_Elements_X                                      = sum(~isnan(Res_X(:)));
   Res_Y                                               = Yp(IndY{:}) - Predictions.ypred;
   Num_Elements_Y                                      = sum(~isnan(Res_Y(:)));
   Validation_Results.xpress(1)                        = sum(Res_X(~isnan(Res_X)).^2);
   Validation_Results.ypress(1)                        = sum(Res_Y(~isnan(Res_Y)).^2);
   Validation_Results.xcumpress                        = sum(Validation_Results.xpress);
   Validation_Results.xrmse                            = sqrt(Validation_Results.xcumpress / Num_Elements_X);
   Validation_Results.xev                              = 100 * (1 - Validation_Results.xcumpress / SSX);
   Validation_Results.ycumpress                        = sum(Validation_Results.ypress);
   Validation_Results.yrmse                            = sqrt(Validation_Results.ycumpress / Num_Elements_Y);
   Validation_Results.yev                              = 100 * (1 - Validation_Results.ycumpress / SSY);
   
end

%------------------------------------------------------------------------------------------------------------------------

function [Flag,varargout] = CleanX(X,Y);
% 
% function [Flag,IndX1,IndX2,...IndXn,...,IndY1,IndY2,...IndYm] = CleanX(X,Y);
% 
% Description:
% Extracts the indexes on X corresponding to slabs that are not constituted only by missing values.
% In case a Y is provided the horizontal slabs (i.e. the samples/batches) that have only missing values
% either in X or in Y are absent from these indexes.  
% Flag is 1 in case any of the slabs had to be removed. Otherwise it is 0.
% 
% Inputs: 
% X: n-way array (where n >= 2)
% Y: m-way array (where m >= 2)
% The first mode of X and Y must have the same dimensions in the first mode.
% 
% Outputs: 
% Flag: is one in presence of at least one slab in either X or Y containing only missing values
% IndX1,IndX2,...IndXn: are the indexes in the n-dimensions of X.
% IndY1,IndY2,...IndYm: are the indexes in the m-dimensions of Y.
% 
% Subroutines: 
% Internal:    none
% External:    none
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 21-Oct-02 
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
%
Flag    = 0;
NDX     = ndims(X);
Xnan    = isnan(X);
SizeXor = size(X);
for i=1:NDX
   Xv           = nshape(Xnan,i);
   Canc         = find(all(Xv,2));
   varargout{i} = setdiff(1:size(Xv,1),Canc);
end
if nargin == 2
   SizeYor = size(Y);
   NDY  = ndims(Y);
   Ynan = isnan(Y);
   for i=1:NDY
      Yv                 = nshape(Ynan,i);
      Canc               = find(all(Yv,2));
      varargout{NDX + i} = setdiff(1:size(Yv,1),Canc);
   end
   [varargout{[1,NDX+1]}] = deal(union(varargout{NDX + 1},varargout{1}));
   if any(SizeYor ~= size(Y(varargout{NDX + 1:end})))
      Flag = 1;
   end
end
if any(SizeXor ~= size(X(varargout{1:NDX})))
   Flag = 1;
end

%------------------------------------------------------------------------------------------------------------------------------

function Validation_Results = MergeResults(Validation_Results,Model_Parameters,NRep,u,v,i,N_HorSlabs,Options,PreProcess,Centre_Par_Y,Scale_Par_Y,Parameters_Reference)
if isa(Model_Parameters,'cell');
   Model_Parameters.xfactors = Model_Parameters;
end
if isa(Model_Parameters,'struct')
   if Options(1) %Fix permutational and sign indeterminacy to be extended by means of feval
      [fic,Perm] = RecFac('MitBur',Model_Parameters.xfactors{2:end},Parameters_Reference{2:end});
      [Model_Parameters.xfactors{1:end}] = FacPerm(Perm,Model_Parameters.xfactors{1:end});
   end
   Model_Fields = intersect(fieldnames(Validation_Results),fieldnames(Model_Parameters));
   for j = 1:length(Model_Fields)
      a = getfield(Model_Parameters,Model_Fields{j});
      b = getfield(Validation_Results,Model_Fields{j});
      if isa(a,'cell')
         for k = 1:length(a)
            if isempty(b)
               b = cell(1,length(a));
            end
            Inda = repmat({':'},ndims(a{k}),1);
            if k == 1 & any(strcmp({'xfactors','yfactors'},Model_Fields{j}))
               if isempty(b{k})
                  D    = size(a{k});
                  D(1) = N_HorSlabs;
                  b{k} = repmat(NaN,[D,NRep]);
               end
               Inda{1}         = u{i};
               b{k}(Inda{:},i) = a{k}(v{i},Inda{2:end});
            else
               if isempty(b{k})
                  b{k} = repmat(NaN,[size(a{k}),NRep]);
               end
               Inda{1} = 1:size(a{k},1);
               b{k}(Inda{:},i) = a{k};
            end
         end
      else
         if isempty(b)
            b = repmat(NaN,[size(a),NRep]);
         end
         Inda         = repmat({':'},ndims(a),1);
         if strcmp(Model_Fields{j},'ypred')
            a = nprocess(a,PreProcess.mody.cen,PreProcess.mody.scal,Centre_Par_Y,Scale_Par_Y,-1,-1);            
            Inda{1} = u{i};
            b(Inda{:},i) = a(v{i},Inda{2:end});
         else
            b(Inda{:},i) = a;
         end
      end
      Validation_Results = setfield(Validation_Results,Model_Fields{j},b);
   end
else
   error('Invalid format for output')
end