function Qlim = Residuals_CL(Res,ConfLim)
% function Qlim = Residuals_CL (Res,ConfLim);
% 
% Description:
% Computes the residuals confidence limits relative to the matrix Res according to Jackson and Mudholkar.
% 
% Inputs: 
% Res: matrix of the residuals
% ConfLim (optional): vector with the desired "alpha"s (default [.95 .99]) 
% 
% 
% Outputs:
% Qlim: vector with the confidence limits the number of elements correspond to the number of thresholds specified in ConfLim
% 
% 
% Called by:
% OnLineRes
% 
% Subroutines:
% Internal: None
% External: norminv (Statistical Toolbox)
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 11-Oct-2002 14:35:28
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
% 
% References 
% [1] Mudholkar, Technometrics, 21, (1979), 341
% [2] Nomikos, Technometrics, 37, (1995), 41
% 
Res(isnan(Res)) = 0;
if ~exist('ConfLim','var') | isempty(ConfLim)
   ConfLim = [.95 .99]';
end
if size(Res,2) > size(Res,1)
   C = (Res * Res')/(size(Res,1)-1); 
else
   C = (Res' * Res)/(size(Res,1)-1); 
end
EigValues = eig(C);
Theta1    = sum(EigValues);
Theta2    = sum(EigValues.^2);
Theta3    = sum(EigValues.^3);
h0        = 1 - 2 * Theta1 * Theta3 / (3 * Theta2^2);
Qlim      = Theta1 * (1 - Theta2 * h0 * (1 - h0) / Theta1^2 + sign(h0) * norminv(ConfLim(:)) * sqrt(2 * Theta2 * h0^2) / Theta1).^(1/h0);
