function [Recovery,Perm] = RecFac(Meas,varargin)
% function [Recovery,Perm] = RecFac(Meas,a,b,c,...,A,B,C,...) 
% 
% Description:
% If Meas is 'MSE' the factors have to be previously ordered and gives the mean square error between
% the first set of vectors and the second: the m-th element of the n-th row is the error for the m-th
% vector in the n-th dimension (or loading matrix). 
% Perm is in this case an identity matrix as this method does not seek the correct order of the factors.
% 
% If Meas is 'MitBur' the function checks all the possible permutations of factors in the first set 
% opting for the one having the maximum average congruence with the second set. Perm, is the 
% corresponding permutation matrix. The recovery is given per factor and is the cosine (congruence)
% between the component of the first set and the corresponding ones in the second. 
% 
% The number of factors in the two sets can be different, in the latter case the set with the highest
% number of factors must come first. In such a case the permutation matrix will be F x F, where F is 
% the number of factors in the first set. The resulting permutation will cause the exceeding factors 
% in the "permuted" matrices to be columns of zeros.
% 
% 
% Inputs: 
% Meas: type of quality of the recovery requested it can either be 'MitBur' or 'MSE' 
% A,B,C,...: set of factors 1
% a,b,c,...: set of factors 2
% 
% Outputs:
% Recovery: quality of the recovery
% Perm: permutation matrix
% 
% Called by:
% ValidateModel, Model_PARAFAC\Compute_Risk
% 
% Subroutines:
% Internal: None
% External: NWay\NormIt
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 21-Oct-2002 08:22:16
% 
% Contact: Giorgio Tomasi, gt@kvl.dk
% 


if logical(rem(nargin-1,2))
   error('Odd number of loading matrices')
end
Dims = (nargin - 1) / 2;
switch Meas
case 'MitBur'
   Rk   = size(varargin{Dims + 1},2);
   Rk1  = size(varargin{1},2);
   R    = perms(1:Rk1);
   Perm = zeros(Rk1);
   Per  = [];
   Co   = 0;
   for o = 1:length(R)
      Fac = R(o,1:Rk);
      CCL = ones(Rk);
      for q = 1:Dims
         CCL = CCL .* (normit(varargin{q}(:,Fac))' * normit(varargin{Dims + q}));
      end
      Rec = mean(diag(CCL));
      if Rec > Co
         Co       = Rec;
         Recovery = diag(CCL);
         Per      = o;
      end
   end   
   if ~isempty(Per)
      Perm(sub2ind([Rk1 Rk1],R(Per,1:Rk),1:Rk)) = 1;
   else 
      Perm            = zeros(Rk1);
      Perm(1:Rk,1:Rk) = eye(Rk);
      Recovery        = NaN*zeros(1,Rk);
   end
   Recovery = Recovery(:);

case 'MSE'
   Rk       = size(varargin{Dims + 1},2);
   Recovery = NaN * ones(Dims,Rk);
   for p = 1:Rk
      M = 1;
      X = 1;
      for q = Dims:-1:1
         Recovery(q,p) = sum((varargin{q}(:,p) - varargin{Dims + q}(:,p)).^2) / size(varargin{q},1);
      end
   end
   Perm = eye(Rk);
end