function [SPE,SPElim,Q,Qlim,Scores] = OnLineRes(Pred_Fun,Data,ModelPars,Mode,ConfLim,Limit,Cen,Scal,varargin)
% function [SPE,SPElim,Q,Qlim,Scores] = ...
%             OnLineRes(Pred_Fun,Data,ModelPars,Mode,ConfLim,Limit,Cen,Scal,varargin);
% 
% Description:
% Compute the Squared Prediction Error and its confidence limits (if requested), 
% the Q statistics (i.e. the sum of squared residuals) and its confidence limits (if required) 
% and the scores for the model associated to Pred_Fun in an on-line fashion. 
% The "Time" mode is by default the last.
% NB This function will likely undergo some simplification of the syntax as many of the input parameters
%    are contained in the same data structure.
% 
% Inputs: 
% Pred_Fun : function computing the prediction for the desired model. 
%            It can be a string (for MatLab 5.3 or superior) or  a function handle 
%            (for MatLab 6.x or superior). See FitModel
% Data     : array of doubles with the data used to compute the statistics
% ModelPars: model's parameters as required by Pred_Fun (i.e. as a structure ModelOut.model)
% Mode     : fill in mode method: 'Current deviation'/'Zero'
% ConfLim  : "alpha"s for the confidence intervals (if empty or not given: 95% and 99%).
% Limit    : compute limits.
% Cen,Scal : preprocessing parameters as contained in Model.xpreproc.
% varargin : additional inputs if required by Pred_Fun
% 
% 
% Outputs:
% SPE   : Squared Prediction Error. It has the same dimensions of Data 
% SPElim: confidence intervals for SPE
% Q     : Sum of Squares of the Residuals computed in an online fashion Data
% Qlim  : confidence intervals for Q 
% Scores: scores computed in an online fashion. The time mode is added as the third (i.e. the Scores is a tridimensional array where the first mode are the batches, the second hold the various factors/LV and the last is time).
% 
% Called by:
% Model_nPLS1\DLims,Model_PARAFAC\DLims, Model_nPLS1\Plot_PartialRSS_Online, 
% Model_PARAFAC\Plot_PartialRSS_Online
% 
% Subroutines:
% Internal: None
% External: residuals_cl, NWay\nprocess, NWay\nshapes, Pred_Fun
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 04-Nov-2002 08:11:30
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
% 
% References
%
if ~exist('ConfLim','var') | isempty(ConfLim)
   ConfLim = [0.95 0.99];
end
if ~exist('Cen','var') | isempty(Cen)
   Cen = cell(ndims(Data),1);
end
if ~exist('Scal','var') | isempty(Scal)
   Scal = cell(ndims(Data),1);
end
[SPE,Q,Qlim,Scores,SPElim] = deal([]);
TimeDir                    = ndims(Data);
switch lower(Mode)
case 'zero'
   IndDataAll  = repmat({':'},ndims(Data),size(Data,TimeDir));
   IndDataAll2 = repmat({':'},ndims(Data),size(Data,TimeDir));
   for i =  1:size(Data,TimeDir)
      IndDataAll{TimeDir,i} = i + 1:size(Data,TimeDir);
   end
   IndDataAll2(TimeDir,:) = num2cell(1:size(Data,TimeDir));
   for i = 1:size(Data,TimeDir)
      Data2                   = nprocess(Data,~cellfun('isempty',Cen),~cellfun('isempty',Scal),Cen,Scal,1,-1);
      Data2(IndDataAll{:,i})  = 0;
      Predictions             = feval(Pred_Fun,Data2,ModelPars,varargin{:});
      Scores(:,:,i)           = Predictions.xfactors;
      TempRes                 = Data2 - Predictions.xpred;
      TempRes(isnan(TempRes)) = 0;
      SPE(:,i)                = sum(nshape(TempRes(IndDataAll2{:,i}),1).^2,2);
      Q(:,i)                  = sum(nshape(TempRes.^2,1),2);
      if Limit
         SPElim(:,i) = Residuals_CL(nshape(TempRes(IndDataAll2{:,i}),1),ConfLim);
         Qlim(:,i)   = Residuals_CL(nshape(TempRes,1),ConfLim);
      end
   end
   
case 'current deviation'
   IndDataAll  = repmat({':'},ndims(Data),size(Data,TimeDir));
   IndDataAll2 = repmat({':'},ndims(Data),size(Data,TimeDir));
   for i =  1:size(Data,TimeDir)
      a(i,1:size(Data,TimeDir)) = [1:i i * ones(1,size(Data,TimeDir)- i)];
   end
   IndDataAll(TimeDir,:) = num2cell(a,2)'; 
   IndDataAll2(TimeDir,:) = num2cell(1:size(Data,TimeDir));
   for i = 1:size(Data,TimeDir)
      Data2                   = nprocess(Data(IndDataAll{:,i}),~cellfun('isempty',Cen),~cellfun('isempty',Scal),Cen,Scal,1,-1);
      Predictions             = feval(Pred_Fun,Data2,ModelPars,varargin{:});
      Scores(:,:,i)           = Predictions.xfactors;
      TempRes                 = Data2 - Predictions.xpred;
      TempRes(isnan(TempRes)) = 0;
      SPE(:,i)                = sum(nshape(TempRes(IndDataAll2{:,i}),1).^2,2);
      Q(:,i)                  = sum(nshape(TempRes.^2,1),2);
      if Limit
         SPElim(:,i) = Residuals_CL(nshape(TempRes(IndDataAll2{:,i}),1),ConfLim);
         Qlim(:,i)   = Residuals_CL(nshape(TempRes,1),ConfLim);
      end
   end
   
end