function [Stat,ModelOut] = Load_ExtData(MO,MainFig)
% function [Stat,ModelOut] = Load_ExtData (MO,MainFig);
% 
% Description:
% It loads new data for a model to be applied on. It checks that the dimensions are compatible 
% with the model: if any of the dimensions apart from the first is not compatible, the procedure 
% is not completed. An exception is made when the last mode has name 'Time' (case insensitive);
% in this case, the array is filled in (i.e. when the model is applied, not upon loading, because of the 
% preprocessing that is sometimes required) with NaN. The original dimensions are then stored in
% ModelOut.info.initialsize (possibly this will be moved to ModelOut.prediction). 
% In case the ModelOut.info.fillin field is empty or non-existing and the operational mode is 'advanced',
% a requester appears to establish the desired method.
% When the data is loaded as a double array (i.e. not a cbdataset object), the labels are set by
% default to: 'Proj. #')
% 
% 
% Inputs: 
% MO     : ModelOut structure (necessary as the function access ModelOut.info.content 
%          (to check the dimensions of the original array, in case they are not compatible) and 
%          ModelOut.info.fillin (in case it is necessary or it is empty the suitable requester appears).
% MainFig: main figure handle.
% 
% Outputs:
% Stat    : 0 if an error occurred when attempting to load new data, 
%           1 if the operation is successfully completed
% ModelOut: ModelOut data structure with suitable values in the 'prediction' field
% 
% Called by:
% nPLS1Apply, PARAFACApply
% 
% Subroutines:
% Internal: CleanX
% External: definemodelout
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 19-Nov-2002 16:50:14
% 
% Contact: Giorgio Tomasi, gt@kvl.dk
%
% Some initial values
Stat              = 0;
ModelOut          = [];
[Xt,Yt,Xfil,Yfil] = deal({});
ListProp          = {'selectionmode','single','name','Load array','promptstring','Select the array to load'};
RankPos           = 1;
DimM              = size(MO.info.content);
Time              = strcmpi(MO.info.content.modenames{end},'time');

%Load array   
try
   
   CDir = cd;
   Dir  = which('Cubatch.m');
   Dir  = Dir(1:max(find(Dir == '\')));
   %cd([Dir,'Data'])             
   cd([Dir,'Session'])             
   [Xfil{2:-1:1}] = uigetfile('*.mat','Load data');
   cd(CDir)
   
catch
   [Xfil{2:-1:1}] = uigetfile('*.mat','Load data');
end;
if Xfil{1} ~= 0
   Cont = who('-file',[Xfil{:}]);
   Ok   = 1;
   Sel  = 1;
   if length(Cont) > 1
      [Sel,Ok] = listdlg('Liststring',Cont,ListProp{:});
   end
   if Ok
      Xt   = load([Xfil{:}],Cont{Sel});
      Xt   = getfield(Xt,Cont{Sel});
   else
      return
   end
else 
   return
end
  
%Define labels in case the loaded array is an array of doubles
if ~isa(Xt,'cbdataset') 
   Lat = label(1:size(Xt,1),'Proj.');
else
   Lat = Xt.labobj{1};
end

%Check if the dimensions are compatible
Xt    = CleanX(Xt);
DimXt = size(Xt);
IndX  = repmat({':'},ndims(Xt),1);
if ~isequal(DimXt(2:end),DimM(2:end))
   
   if isequal(find(DimXt(2:end) ~= DimM(2:end)) + 1,length(DimM)) & Time      
      IndX{end}   = DimXt(end) + 1 : DimM(end);
      Xt          = Xt.data;
      Xt(IndX{:}) = NaN;
      %Change fill-in method?
      QstOpt  = struct('Default',MO.info.fillin,'Interpreter','tex');
      if strcmpi(getappdata(MainFig,'Mode'),'advanced')
         Prompt         = {['The fill-in method is currently set to \bf',MO.info.fillin,'\rm'],['Choose fill-in method']};
         Ans            = questdlg(Prompt,'Choose fill-in method','Current deviation','Zero','Cancel',QstOpt);
         if strcmp(Ans,'Cancel')
            return
         else
            MO.info.fillin = Ans;
         end
      end      
   else
      error(['The current model cannot be applied to this data set:',char(10),'the dimensions are not compatible'])
   end
   
end

%Xt                       = cbdataset(Xt(:,MO.data.variables{:}));
Xt                       = cbdataset(Xt);
Lat                      = [{Lat},MO.info.content.labobj(2:end)];
Xt.labels                = Lat;
Nil                      = DefineModelOut;
Pred                     = Nil.prediction;
MO.info.initialsize      = DimXt(end);
ModelOut                 = MO;
ModelOut.prediction      = Pred;
ModelOut.prediction.data = Xt;
Stat                     = 1;

function [X,varargout] = CleanX(X);
% function [CleanX,IndX1,IndX2,...IndXn,...,IndY1,IndY2,...IndYm] = CleanX(X);
% 
% Description:
% Extracts the indexes on X corresponding to slabs that are not constituted only by missing values.
% 
% Inputs: 
% X: n-way array (where n >= 2)
% 
% Outputs:
% CleanX: "cleaned" n-way array (that is without slabs of sole missing values)
% IndX1,IndX2,...IndXn: indexes in the n-dimensions of X.
%  
% Subroutines:
% Internal: None
% External: nshape
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 21-Oct-2002 02:38:54
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 


ND   = ndims(X);
Xnan = isnan(X);
for i=1:ND
   Xv           = nshape(Xnan,i);
   Canc         = find(all(Xv,2));
   varargout{i} = setdiff(1:size(Xv,1),Canc);
end
X = X(varargout{:});


         
         
   