function Model = FitModel(FitFun,PredFun,X,Y,Factors,PreProcess,varargin)
% function Model = FitModel(FitFun,PredFun,X,Y,Factors,PreProcess,varargin);
% 
% Description:
% Fits a model (either in regression or simply to decompose an array in factors) and saves the results in ModelOut.model. 
% 
% Inputs: 
% FitFun and PredFun have specific requirements with respect to input and output. 
% 
% FitFun(X,[Y,]F,varargin)
%    It is a function fitting the desired model to the data. 
%    It can be a string (for MatLab 5.3 or superior) or  a function handle (for MatLab 6.x or superior)
%    The first input parameter must be X, followed (only in regression) by Y and then by the 
%    dimensionality of the model F. 
%    The additional inputs come next. 
%    The output of this function should be only one: either a cell vector with the loading matrices 
%    in each mode as elements or a structure. 
%    In the second case the fields that the returned structure has in common with ModelOut.model 
%    are filled with the results. 
%    
%    E.g. function Model = Tucker3Fit(X,F,varargin)
%         [Model.xfactors,Model.core] = tucker(X,F,varargin{:});
% 
%         can be used to compute a simple Tucker3 model on a generic n-way array and the resulting 
%         loading matrices and scores will be stored in ModelOut.model.xfactors and ModelOut.model.core.
%         varargin{1} can be the Options, varargin{2} the constraints on the Factors and so forth  
% 
% PredFun(X,Model,varargin)
%    Function computing the predictions for the desired model. 
%    It can be a string (for MatLab 5.3 or superior) or  a function handle (for MatLab 6.x or superior).
%    The first input parameter must be X, followed by the model parameters (i.e. by ModelOut.model 
%    or a reduced version, so long as the function is capable of fitting the model on the data).
%    If a Y is necessary it shall be passed within the varargin parameter.    
%    
%    E.g. function Model = Tucker3Pred(X,Model,varargin)
% 
%       W = Model.xfactors{ndims(X)};
%       for m = ndims(X)-1:-1:2
%          W = kr(W,Model.xfactors{m});
%       end
%       Z = nshape(Model.core,1)*W';
%       Model.xfactors{1} = nshape(X,1)*pinv(Z);
%       Model.xpred = reshape(Model.xfactors{1}*Z,size(X));
% 
% X         : double array relative to the calibration set
% Y         : double array relative to the calibration set (it must be empty if it is a decomposition method)  
% Factors   : number of factors to extract (it can be a single number or a vector depending on the desired model)
% PreProcess: structure with two fields: 'modx' and 'mody'; 
%             each of them is also a structure with two fields: 'cen' and 'scal'.
%             'modx' defines the preprocessing relative to X, while 'mody' refers to Y. 
%             The two fields 'cen' and 'scal' are vectors with the same number of elements 
%             as the modes of the relative array (e.g. if X is 3-way PreProcess.modx.cen and 
%             PreProcess.modx.scal must be 3 x 1 (or 1 x 3) vectors. 
%             For more information on the values that these two vectors can be given check 
%             the help of nprocess (n-way toolbox).
% varargin  : additional inputs (e.g. Options, Constraints, etc) handled directly by FitFun and PredFun
% 
% 
% Outputs: 
% The results of the computations as well as the new model's parameters are stored in ModelOut.model 
% 
% 
% Called by:
% Model_nPLS1\Calculate, Model_PARAFAC\Calculate
% 
% Subroutines:
% Internal: Fit_Decomposition, Fit_Regression, CleanX
% External: definemodelout, nprocess (N-Way toolbox), nshape (N-Way toolbox),
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 21-Oct-2002 10:32:04
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 


Available_Methods = {'loo','nboo','rboo'};
if ~any([2:6] == exist(FitFun))
   error('The function calculating the model is not in the path')
elseif ~any([2:6] == exist(PredFun))
   error('The function to calculate the predictions is not in the path')
end

if isempty(X)
   error('X array is missing')
end

if ~exist('Y','var')
   Y = [];
end
if ~isempty(Y)
   if size(X,1) ~= size(Y,1)
      error('X and Y must have the same dimension in mode 1')
   end
end

if ~exist('Factors','var')
   Factors = 1;
end
if isempty(Factors)
   Factors = 1;
end

Centre_ScaleX_Modes = struct('cen',zeros(1,ndims(X)),'scal',zeros(1,ndims(X)));
Centre_ScaleY_Modes = struct('cen',zeros(1,ndims(Y)),'scal',zeros(1,ndims(Y)));
if ~exist('PreProcess','var')
   PreProcess   = struct('modx',Centre_ScaleX_Modes,'mody',Centre_ScaleY_Modes);
end
if isempty(PreProcess)
   PreProcess   = struct('modx',Centre_ScaleX_Modes,'mody',Centre_ScaleY_Modes);
end

if isempty(Y)
   Model = Fit_Decomposition(FitFun,PredFun,X,Factors,PreProcess,varargin{:});
else
   Model = Fit_Regression(FitFun,PredFun,X,Y,Factors,PreProcess,varargin{:});
end

%------------------------------------------------------------------------------------------------------------------------

function Model = Fit_Decomposition(Fit_Function,Predict_Function,X,Factors,PreProcess,varargin)
% 
% function Model = Fit_Decomposition(FitFun,PredFun,X,Factors,PreProcess,varargin) 
% 
% Description:
% Fits a model which decomposes the X in absence of a Y of predicted variables. 
% 
% Inputs: 
% The same as FitModel 
% 
% Outputs: 
% The same as FitModel. Only the fields referring to the X array are filled in. 
% 
% Subroutines: 
% Internal:    CleanX
% External:    DefineModelOut, NProcess (n-way toolbox)
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 21-Oct-02 
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
% 
M                       = DefineModelOut;
Model                   = M.model;
Model.nbfactors         = Factors;
Model.xpred             = NaN*ones(size(X));
SizeXor                 = size(X);
[Flag,IndX{1:ndims(X)}] = CleanX(X);
[Xp,Model.xpreproc.cen,Model.xpreproc.scal] = nprocess(X(IndX{:}),PreProcess.modx.cen,PreProcess.modx.scal,[],[],1,-1);
SSX = sum(Xp(~isnan(Xp)).^2);
if Flag
   for i = 1:ndims(X)
      if length(IndX{i})~=SizeXor(i)
         warning([num2str(SizeXor(i)-length(IndX{i})) ' slabs in mode ', num2str(i),'are being ignored because containing only missing values'])
      end
   end
end
Model_Parameters  = feval(Fit_Function,Xp,Factors,varargin{:}); %The validated function shall return a structure where each field 
Parameter_as_Cell = isa(Model_Parameters,'cell');                                                                                                       %correspond to the necessary parameter for the validation function
if Parameter_as_Cell
   Model_Parameters.xfactors = Model_Parameters;%A;
end
if isa(Model_Parameters,'struct')
   Model_Fields = intersect(fieldnames(Model),fieldnames(Model_Parameters));
   for j = 1:length(Model_Fields)
      Model = setfield(Model,Model_Fields{j},getfield(Model_Parameters,Model_Fields{j}));
   end
else
   error('Invalid format for output')
end
if Parameter_as_Cell
   Model_Parameters = Model_Parameters.xfactors;
end
Predictions                     = feval(Predict_Function,Xp,Model_Parameters,varargin{:});
Model.xpred(IndX{:})            = nprocess(Predictions.xpred,PreProcess.modx.cen,PreProcess.modx.scal,Model.xpreproc.cen,Model.xpreproc.scal,-1,-1);   
Res_X                           = Xp(IndX{1},:) - Predictions.xpred(1:length(IndX{1}),:);
Num_Elements_X                  = sum(~isnan(Res_X(:)));
Missing_X                       = isnan(Res_X);
Res_X(Missing_X)                = 0;
Model.xpress(1:length(IndX{1})) = sum(Res_X.^2,2);
Model.xcumpress                 = sum(Model.xpress);
Model.xrmse                     = sqrt(Model.xcumpress / Num_Elements_X);
Model.xev                       = 100 * (1 - Model.xcumpress / SSX);

%------------------------------------------------------------------------------------------------------------------------

function Model = Fit_Regression(Fit_Function,Predict_Function,X,Y,Factors,PreProcess,varargin)
% 
% function Model = Fit_Regression(FitFun,PredFun,X,Y,Factors,PreProcess,varargin) 
% 
% Description:
% Fits a regression model having X as the predictor matrix and Y as predicted variables. 
% 
% Inputs: 
% The same as FitModel 
% 
% Outputs: 
% The same as FitModel. 
% 
% Subroutines: 
% Internal:    CleanX
% External:    DefineModelOut, NProcess (n-way toolbox)
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 21-Oct-02 
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
% 

M                                           = DefineModelOut;
Model                                       = M.model;
Model.nbfactors                             = Factors;
Model.xpred                                 = NaN*ones(size(X));
Model.ypred                                 = NaN*ones(size(Y));
SizeXor                                     = size(X);
[Flag,IndX{1:ndims(X)},IndY{1:ndims(Y)}]    = CleanX(X,Y);
[Xp,Model.xpreproc.cen,Model.xpreproc.scal] = nprocess(X,PreProcess.modx.cen,PreProcess.modx.scal,[],[],1,-1);
SSX = sum(Xp(~isnan(Xp)).^2);
[Yp,Model.ypreproc.cen,Model.ypreproc.scal] = nprocess(Y,PreProcess.mody.cen,PreProcess.mody.scal,[],[],1,-1);
SSY = sum(Yp(~isnan(Yp)).^2);
for i = 1:ndims(X)
   if length(IndX{i})~=SizeXor(i)
      warning([num2str(SizeXor(i)-length(IndX{i})) ' slabs in mode ', num2str(i),'are being ignored because containing only missing values'])
   end
end
Model_Parameters  = feval(Fit_Function,Xp,Yp,Factors,varargin{:});
Parameter_as_Cell = isa(Model_Parameters,'cell');                                                                                                       %correspond to the necessary parameter for the validation function
if Parameter_as_Cell
   Model_Parameters.xfactors = Model_Parameters;
end
if isa(Model_Parameters,'struct')
   Model_Fields = intersect(fieldnames(Model),fieldnames(Model_Parameters));
   for j = 1:length(Model_Fields)
      Model = setfield(Model,Model_Fields{j},getfield(Model_Parameters,Model_Fields{j}));
   end
else
   error('Invalid format for output')
end
if Parameter_as_Cell
   Model_Parameters = Model_Parameters.xfactors;
end
Predictions                    = feval(Predict_Function,Xp,Model_Parameters,varargin{:});

Model.xpred(IndX{:})            = nprocess(Predictions.xpred,PreProcess.modx.cen,PreProcess.modx.scal,Model.xpreproc.cen,Model.xpreproc.scal,-1,-1);   
Res_X                           = Xp(IndX{1},:) - Predictions.xpred(1:length(IndX{1}),:);
Missing_X                       = isnan(Res_X);
Num_Elements_X                  = sum(~Missing_X(:));
Res_X(Missing_X)                = 0;
Model.xpress(1:length(IndX{1})) = sum(Res_X.^2,2);
Model.xcumpress                 = sum(Model.xpress);
Model.xrmse                     = sqrt(Model.xcumpress / Num_Elements_X);
Model.xev                       = 100 * (1 - Model.xcumpress / SSX);

Model.ypred(IndY{:})            = nprocess(Predictions.ypred,PreProcess.mody.cen,PreProcess.mody.scal,Model.ypreproc.cen,Model.ypreproc.scal,-1,-1);   
Res_Y                           = Yp(IndY{1},:) - Predictions.ypred(1:length(IndY{1}),:);
Missing_Y                       = isnan(Res_Y);
Num_Elements_Y                  = sum(~Missing_Y(:));
Res_Y(Missing_Y)                = 0;
Model.ypress(1:length(IndY{1})) = sum(Res_Y.^2,2);
Model.ycumpress                 = sum(Model.ypress);
Model.yrmse                     = sqrt(Model.ycumpress / Num_Elements_Y);
Model.yev                       = 100 * (1 - Model.ycumpress / SSY);

%------------------------------------------------------------------------------------------------------------------------

function [Flag,varargout] = CleanX(X,Y);
% 
% function [Flag,IndX1,IndX2,...IndXn,...,IndY1,IndY2,...IndYm] = CleanX(X,Y);
% 
% Description:
% Extracts the indexes on X corresponding to slabs that are not constituted only by missing values.
% In case a Y is provided the horizontal slabs (i.e. the samples/batches) that have only missing values
% either in X or in Y are absent from these indexes.  
% Flag is 1 in case any of the slabs had to be removed. Otherwise it is 0.
% 
% Inputs: 
% X: n-way array (where n >= 2)
% Y: m-way array (where m >= 2)
% The first mode of X and Y must have the same dimensions in the first mode.
% 
% Outputs: 
% Flag: is one in presence of at least one slab in either X or Y containing only missing values
% IndX1,IndX2,...IndXn: are the indexes in the n-dimensions of X.
% IndY1,IndY2,...IndYm: are the indexes in the m-dimensions of Y.
% 
% Subroutines: 
% Internal:    none
% External:    nshape (n-way toolbox)
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 21-Oct-02 
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
% 
Flag    = 0;
NDX     = ndims(X);
Xnan    = isnan(X);
SizeXor = size(X);
for i=1:NDX
   Xv           = nshape(Xnan,i);
   Canc         = find(all(Xv,2));
   varargout{i} = setdiff(1:size(Xv,1),Canc);
end
if nargin == 2
   SizeYor = size(Y);
   NDY  = ndims(Y);
   Ynan = isnan(Y);
   for i=1:NDY
      Yv                 = nshape(Ynan,i);
      Canc               = find(all(Yv,2));
      varargout{NDX + i} = setdiff(1:size(Yv,1),Canc);
   end
   [varargout{[1,NDX+1]}] = deal(union(varargout{NDX + 1},varargout{1}));
   if any(SizeYor ~= size(Y(varargout{NDX + 1:end})))
      Flag = 1;
   end
end
if any(SizeXor ~= size(X(varargout{1:NDX})))
   Flag = 1;
end