function Pred = ApplyModel(PredFun,X,Y,Model_Parameters,Pred,InitialSize,FillInMode,varargin)
% function Pred = ApplyModel (PredFun,X,Y,Model_Parameters,Pred,InitialSize,FillInMode,varargin);
% 
% Description:
% Applies a model, as defined within Model_Parameters, to a new set of data, represented by X 
% (and Y when necessary).
% In order to allow "on-line" applications, 'filling in' is also supported, according to Nomikos 
% and Mc. Gregor
% 
% Inputs: 
% PredFun         : function computing the prediction for the desired model. 
%                   It can be a string (for MatLab 5.3 or superior) or  a function handle 
%                   (for MatLab 6.x or superior). See FitModel
% X               : double array with the new data
% Y               : double array with the Y data (in case an additional array is needed - 
%                   see Instrumental Variables Models)
% Model_Parameters: model's parameters as required by PredFun (i.e. as a structure ModelOut.model)
% PredIn          : structure where to save the results (i.e.  structure ModelOut.prediction)
% InitialSize     : size of the batch before the filling-in (stored in ModelOut.info.initialsize)
% FillInMode      : fill-in mode in case the model can be applied "on-line" 
%                   (i.e. the last mode has been given name 'time' - case insensitive) and the length 
%                   of the new "batch" is not identical to the NOC batches. 
%                   It can either be 'Current deviation' or 'Zero'
% varargin        : additional inputs as (and if) required by PredFun
% 
% Outputs:
% PredOut: structure (ModelOut.prediction) holding the results for the new set of data.
% 
% Called by:
% PARAFACApply, nPLS1Apply
% 
% Subroutines:
% Internal: Names and hyperlinks
% External: Nway\nprocess
% 
% 
% Author: 
% Giorgio Tomasi 
% Royal Agricultural and Veterinary University 
% MLI, LMT, Chemometrics group 
% Rolighedsvej 30 
% DK-1958 Frederiksberg C 
% Danmark 
% 
% Last modified: 26-Oct-2002 23:07:10
% 
% Contact: Giorgio Tomasi, gt@kvl.dk 
% 
% References
% 
if ~any([2:6] == exist(PredFun))
   error('The function to calculate the predictions is not in the path')
end
if isempty(X)
   error('X array is missing')
end
%Some initial values
IndX           = deal(repmat({':'},ndims(X),1));
Pred.nbfactors = Model_Parameters.nbfactors;
Pred.xpred     = NaN * ones(size(X));
Centre_X       = ~cellfun('isempty',Model_Parameters.xpreproc.cen);
Scale_X        = ~cellfun('isempty',Model_Parameters.xpreproc.scal);

%Preprocess
Xp             = nprocess(X,Centre_X,Scale_X,Model_Parameters.xpreproc.cen,Model_Parameters.xpreproc.scal,1,-1);
SSX            = sum(Xp(~isnan(X)).^2);
%Current deviation
IndX{end}     = [1:InitialSize,InitialSize * ones(1,size(X,ndims(X))-InitialSize)];
Xp             = Xp(IndX{:});
IndX{end}     = InitialSize + 1 : size(X,ndims(X));
if strcmp(FillInMode,'zero')
   Xp(IndX{:}) = 0;
end
%Compute the new scores and preictions
Predictions = feval(PredFun,Xp,Model_Parameters,varargin{:});
if any(strcmp(fieldnames(Predictions),'ypred'))
   Pred2.ypred = Predictions.ypred;
end
%Save factors
Pred.xfactors = Predictions.xfactors;
if ~isa(Pred.xfactors,'cell')
   Pred.xfactors = {Pred.xfactors};
end

%Remove from prediction the filled-in values
Pred.xpred(IndX{:})         = NaN;
Predictions.xpred(IndX{:})  = NaN;
if isempty(IndX{end})
   IndX{end} = ':';
end
%Residuals for the unprocessed array
Res_X                       = Xp(1:size(Xp,1),:) - Predictions.xpred(1:size(Xp,1),:);
Missing_X                   = isnan(Res_X);
Res_X(Missing_X)            = 0;
Pred.xpress(1:size(Xp,1),1) = sum(Res_X.^2,2)';                       %Horizontal slab-wise sum of squares
Pred.xcumpress              = sum(Pred.xpress);                       %Sum of squared errors
Pred.xrmse                  = sqrt(Pred.xcumpress / sum(~Missing_X(:))); 
Pred.xev                    = 100 * (1 - Pred.xcumpress / SSX);
%Back scale
Pred.xpred                  = nprocess(Predictions.xpred,Centre_X,Scale_X,Model_Parameters.xpreproc.cen,Model_Parameters.xpreproc.scal,-1,-1);   
if any(strcmp(fieldnames(Predictions),'ypred'))
   Centre_Y         = ~cellfun('isempty',Model_Parameters.ypreproc.cen);
   Scale_Y          = ~cellfun('isempty',Model_Parameters.ypreproc.scal);
   Pred.ypred       = nprocess(Pred2.ypred,Centre_Y,Scale_Y,Model_Parameters.ypreproc.cen,Model_Parameters.ypreproc.scal,-1,-1);   
end