function value=subsref(x,index)
%DATASET/SUBSREF Subindex for DataSet objects using structure and index notation
%I/O: value = x.field;
%  when (x) is a dataset object (See DATASET) this returns
%  the value (value) of the dataset object field ('field').
%  This syntax is used for the following fields:
%    name         : char vector.
%    author       : char vector.
%    date         : 6 element vector (see CLOCK).
%    moddate      : 6 element vector (last modified date).
%    type         : either 'data', 'batch' or 'image'.
%    data         : double,single, logical, or [u]int8/16/32 array.
%    label        : cell of char arrays.
%    labelname    : cell of char row vectors.
%    axisscale    : cell of row vectors.
%    axisscalename: cell of char row vectors.
%    title        : cell of char row vectors.
%    titlename    : cell of char row vectors.
%    class        : cell of row vectors.
%    classname    : cell of char row vectors.
%    userdata     : user defined.
%    description  : char or cell array.
%    history      : cell array of char (e.g. char(x.history))
%    datasetversion: dataset object version.
%
%I/O: value = x.field{vdim};
%    includ       : value is row vector of indices for mode vdim
%
%I/O: value = x.field{vdim,vset};
%  when (x) is a dataset object this returns the value (value)
%  for the field ('field') for the specified dimension/mode
%  (vdim) and optional set (vset) {default: vset=1}. E.g. (vset) is
%  used when multiple sets of labels are present in (x).
%  This syntax is used for the following fields:
%    label        : value is a char array with size(x.data,vdim) rows.
%    labelname    : value is a char row vector.
%    axisscale    : value is a row vector with size(x.data,vdim) real elements.
%    axisscalename: value is a char row vector.
%    title        : value is a char row vector.
%    titlename    : value is a char row vector.
%    class        : value is a row vector with size(x.data,vdim) integer elements.
%    classname    : value is a char row vector.
%
%Examples:
%    plot(mydataset.axisscale{2})            %second dim axis scale
%    xlabel(mydataset.title{1,2})            %Second title for first dim
%    disp(['Made by: ' mydataset.author])
%
%See also: DATASET, DATASETDEMO, DATASET/SUBSASGN, DATASET/GET, ISA, DATASET/EXPLODE

%Copyright Eigenvector Research, Inc. 2000-2002
%nbg 8/3/00, 8/16/00, 8/17/00, 8/30/00, 9/1/00, 10/10/00
%jms 4/11/01  -converted from dataset/get
%   -added indexing into history (vs. get which doesn't allow that)
%jms 5/3/01 -added indexing into data using ()s (e.g. x.data(1:5,:) )
%jms 5/9/01 -adapted for use with data being other classes (single, uint8, etc).
%jms 5/31/01 -added subscripting into ALL fields
%jms 8/02 -allow extract of all labels, axisscales, etc at once by calling
%           without vdim
%  -revised error messages
%jms 9/02 -added object-level subscripting

%Fields 'label', 'axisscale', 'title', and 'class' can not be set if 'data'
%is empty. Once field 'data' has been filled it can not be "set" to a 
%different size.

nnargin = nargin;
error(nargchk(2,2,nnargin));   %give error if nargin is not appropriate

if isempty(index); error('  Invalid subscripting'); end;

if ~strcmp(index(1).type,'.')
  switch index(1).type
    case '()'
      for dim = 1:length(index(1).subs)
        %convert logical to double so we can test for out of range
        if isa(index(1).subs{dim},'logical')  
          index(1).subs{dim} = find(index(1).subs{dim});
        end
        %interpret doubles as indexing into dataset object (do delsamps on UNspecified indices)
        if isa(index(1).subs{dim},'double')
          if max(index(1).subs{dim}) > size(x.data,dim);
            error('  Invalid subscript')
          end
          if dim <= ndims(x.data)   %don't try this if we don't have the dim
            x = delsamps(x, index(1).subs{dim}, dim, 3);    %hard keep/reorder delete
          end
        end
      end
      index(1) = [];
      if isempty(index);    %nothing other than indexing into main object? just return reduced object
        value = x;
        return
      end
      
    otherwise
      error('  Invalid subscripting for DataSet object');
      %       error('  Subscripting not permitted on main dataset object (yet!)');
  end
end
feyld = lower(index(1).subs);

if size(feyld,1)>1
  error('  Input ''field'' must be char row vector or 1 element cell.')
elseif isa(feyld,'cell')&size(feyld,2)>1
  error('  Input ''field'' must be char row vector or 1 element cell.')
end
if isa(feyld,'cell')&logical(prod(size(feyld)))
  feyld  = char(feyld{:});
end
indx     = '';
if strncmp(feyld,'data',4) & length(feyld>4)
  indx   = feyld(1,5:end);
  feyld  = 'data';
end

if length(index)<2 | ~strcmp(index(2).type,'{}') | ...
    (strcmp(x.type,'batch') & strcmp(feyld,'data'));
  vdim  = 1;
  vset  = 1;
  nnargin = 2;     %fake out logic later on
  index(1) = [];  %clear field name from index structure
else
  vdim=index(2).subs{1};
  if length(index(2).subs)>1;
    vset=index(2).subs{2};
    nnargin = 4;     %fake out logic later on
  else
    vset=1;
    nnargin = 3;     %fake out logic later on
  end;
  index(1:2) = [];  %clear field name & vset/vdim from index structure
  %Remaining parts of index will be used to subscript into results after getting results
end;

if nnargin==2
  switch lower(feyld)
  case 'name'
    value  = x.name;
  case 'author'
    value  = x.author;
  case 'date'
    value  = x.date;
  case 'moddate'
    value  = x.moddate;
  case 'type'
    value  = x.type;
  case 'data'
    if isempty(indx)
      value  = x.data;
    else
      %value = getfield(x,['data' indx]);    
      eval(['value = x.data',indx,';'])
    end
  case 'description'
    value  = x.description;
  case 'datasetversion'
    value  = x.datasetversion;
  case 'userdata'
    value  = x.userdata;
  case 'history'
    value  = x.history;
  case 'label'
    value  = squeeze(x.label(:,1,:));
  case 'labelname'
    value  = squeeze(x.label(:,2,:));
  case 'axisscale'
    value  = squeeze(x.axisscale(:,1,:));
  case 'axisscalename'
    value  = squeeze(x.axisscale(:,2,:));
  case 'title'
    value  = squeeze(x.title(:,1,:));
  case 'titlename'
    value  = squeeze(x.title(:,2,:));
  case 'class'
    value  = squeeze(x.class(:,1,:));
  case 'classname'
    value  = squeeze(x.class(:,2,:));
  case 'includ'
    value  = x.includ(:,1)';
  otherwise
    error('  Not valid dataset object field.')
  end
elseif nnargin==3|nnargin==4
  if vdim(1)<1
    error(['vdim(1) must be integer >0.'])
  elseif any(strcmp(feyld,{'history'}));    %allow : expansion on these fields
    if isa(vdim,'char');
      if ~strcmp(vdim,':');
        error(['expecting integer index or '':'' operator']);
      else
        vdim=1:length(getfield(x,feyld));
      end;
    elseif((max(vdim)>length(x.history))|min(vdim)<1)
      error(['vdim must be integer in [1:',int2str(length(getfield(x,feyld))),'].'])
    end;
  else
    if ~isa(x.data,'cell')&((vdim>ndims(x.data))|length(vdim)>1)
      error(['vdim must be integer in [1:',int2str(ndims(x.data)),'].'])
    elseif isa(x.data,'cell')&(vdim(1)>ndims(x.data{1}))
      error(['vdim(1) must be integer in [1:',int2str(ndims(x.data{1})),'].'])
    elseif isa(x.data,'cell')&(length(vdim)>2)
      error(['vdim must be 2 element vector.'])
    elseif isa(x.data,'cell')&(length(vdim)==2)& ...
        ((vdim(2)>length(x.data))|(vdim(2)<1))
      error(['vdim(2) must be integer in [1:',int2str(length(x.data)),'].'])
    elseif vset<1
      error('Input vset can not be less than zero.')
    end
  end;
  
  nlabel     = size(x.label,3);     %get number of sets for each
  naxisscale = size(x.axisscale,3); %field
  ntitle     = size(x.title,3);
  nclass     = size(x.class,3);
  switch lower(feyld)
  case 'name'
    error('  Indicies not allowed for field ''name''.')
  case 'author'
    error('  Indicies not allowed for field ''author''.')
  case 'date'
    error('  Indicies not allowed for field ''date''.')
  case 'moddate'
    error('  Indicies not allowed for field ''moddate''.')
  case 'type'
    error('  Indicies not allowed for field ''type''.')
  case 'data'
    error('  Indicies not allowed for field ''data''.')
  case 'description'
    error('  Indicies not allowed for field ''description''.')
  case 'datasetversion'
    error('  Indicies not allowed for field ''datasetversion''.')
  case 'userdata'
    error('  Indicies not allowed for field ''userdata''.')
  case 'history'
    if vset>1;
      error('   Field ''history'' only allows one set.');
    else
      if length(vdim)==1;
        value  = x.history{vdim};
      else
        value  = char(x.history(vdim));
      end;
    end;
    %error('  Indicies not allowed for field ''history''.')
  case 'label'
    if vset>nlabel
      error(['  ''' feyld ''' set ' num2str(vset) ' does not exist.'])
    else
      value  = x.label{vdim,1,vset};
    end
  case 'labelname'
    if vset>nlabel
      error(['  ''' feyld ''' set ' num2str(vset) ' does not exist.'])
    else
      value  = x.label{vdim,2,vset};
    end
  case 'axisscale'
    if vset>naxisscale
      error(['  ''' feyld ''' set ' num2str(vset) ' does not exist.'])
    else
      if isa(x.data,'cell')
        if (length(vdim)==1)
          value  = x.axisscale{vdim,1,vset};
        elseif (length(vdim)==2)&(vdim(1)==1)
          value  = x.axisscale{1,1,vset}{vdim(2)};
        else
          error('vdim must be a 2 element vector with vdim(1)== 1');
        end;
      else
        value    = x.axisscale{vdim,1,vset};
      end
    end
  case 'axisscalename'
    if vset>naxisscale
      error(['  ''' feyld ''' set ' num2str(vset) ' does not exist.'])
    else
      value  = x.axisscale{vdim,2,vset};
    end
  case 'title'
    if vset>ntitle
      error(['  ''' feyld ''' set ' num2str(vset) ' does not exist.'])
    else
      value  = x.title{vdim,1,vset};
    end
  case 'titlename'
    if vset>ntitle
      error(['  ''' feyld ''' set ' num2str(vset) ' does not exist.'])
    else
      value  = x.title{vdim,2,vset};
    end
  case 'class'
    if vset>nclass
      error(['  ''' feyld ''' set ' num2str(vset) ' does not exist.'])
    else
      value  = x.class{vdim,1,vset};
    end
  case 'classname'
    if vset>nclass
      error(['  ''' feyld ''' set ' num2str(vset) ' does not exist.'])
    else
      value  = x.class{vdim,2,vset};
    end
  case 'includ'
    value    = x.includ{vdim};
  otherwise
    error('  Not valid dataset object field.')
  end
end

if ~isempty(index);
  %some indexing instructions left in index
  try;
    value = subsref(value,index);
  catch
    error(lasterr)
  end;
end;
  