function x=subsasgn(x,index,val)
%DATASET/SUBSASGN Assign for DataSet objects using Structure and index notation
%I/O: x.field = value;
%  sets the field for a dataset object to (value).
%  This syntax is used for the following fields:
%    name         : char vector or cell of size 1 with char contents.
%    author       : char vector or cell of size 1 with char contents.
%    date         : 6 element vector (see CLOCK).
%    type         : field 'data' must not be empty to set 'type',
%                   can have value 'data' or 'image'.
%                   See documentation for type 'batch'.
%    data         : double, single, int or uint array, or cell array (type 'batch' see documentation).
%    description  : char array or cell with char contents.
%    userdata     : filled with user defined data.
%
%I/O: x.field{vdim} = value;
%    includ       : row vector of max length size(x.data,vdim) e.g. 1:size(x.data,vdim)
%
%I/O: x.field{vdim,vset} = value;
%  sets the field for a dataset object to (value) for the specified
%  dimension (vdim) and optional set (vset) {default: vset=1}, e.g.
%  the (vset) input allows for multiple sets of labels to be used.
%  The field 'data' must not be empty to set the values.
%  This syntax is used for the following fields:
%    label        : character or cell array with size(x.data,vdim)
%                   rows containing labels for the vdim dimension/mode.
%    labelname    : row vector of class char or 1 element cell containing
%                   a name for the label set.
%    axisscale    : vector of class double (real elements) with
%                   size(x.data,vdim) elements. Each contains an axis
%                   scale to plot against for the vdim dimension/mode
%                   (e.g. x.axisscale{2,1} is a numerical scale for columns,
%                   and x.axisscale{1,1} is a numerical scale for rows).
%                   See documentation for type 'batch'.
%    axisscalename: row vector of class char or 1 element cell containing 
%                   a name for the axis scale.
%    title        : row vector of class char or 1 element cell containing
%                   a title for the vdim dimension/mode.
%    titlename    : row vector of class char or 1 element cell containing
%                   a name for the title.
%    class        : vector of class double (integer elements) and size(x.data,vdim) 
%                   elements. Each element is a class identifier that can be used
%                   in plotting, classification, or crossvalidation.
%    classname    : row vector of class char or 1 element cell containing
%                   a name for the class set.
%
%Examples:
%    mydataset.author = 'James Joyce'
%    mydataset.axisscale{2} = [1:.5:25]        %second dim axis scale
%    mydataset.title{1,2} = 'Second title for first dim'
%
%See also: DATASET, DATASETDEMO, DATASET/SUBSREF, DATASET/GET, ISA, DATASET/EXPLODE

%Copyright Eigenvector Research, Inc. 2000-2002
%nbg 8/3/00, 8/16/00, 8/17/00, 8/30/00, 9/1/00, 10/10/00
%jms 4/11/01  
%   -this started off as nearly the same as SET for datasets except
%     that we added the logic to take input like a subsasgn call and
%     translate it into the correct info for the old set routine
%   -fixed error in errors (missing [ ]s)
%   -modified error statements to make more sense with this kind of call
%   -changed history logging function (renamed old one to keep it)
%nbg/jms 4/12/01 -fixed includ error if adding data to empty dataset
% jms 5/9/01 -added handling of other classes for data field
% jms 5/31/01 -added subscripting into fields for assignment
%   -fixed bug which kept batch data axisscales from be addressed correctly
%   -fixed bugs in data assign checking batch data size
% jms 6/5/01 -fixed includ size bug (adding data to empty dataset created
%    includ field with only 1 column)
% jms 10/19/01 -added error if vset used on includ field
%   -added error if subscripting used on author field
%   -added calling function identification in sethistory
% jms 11/05/01 -axisscale and label can now be set to empty []
% jms 4/16/02  -removed extraneous code in sethistory
% jms 5/20/02 -changed calls to sethistory and sethistory output format
% jms 8/02 -removed extraneous tests of data class
%    -revised error messages
%    -fixed bugs associated with batch support
% jms 9/02 -changed subscripting error message

%Fields 'label', 'axisscale', 'title', and 'class' can not be set if 'data'
%is empty. Once field 'data' has been filled it can not be "set" to a 
%different size.

nnargin = nargin;
error(nargchk(3,3,nnargin));   %give error if nargin is not appropriate

if isempty(index); error('  Invalid subscripting'); end;

if ~strcmp(index(1).type,'.');
  error('  Assignment subscripting not permitted on main dataset object');
else
  feyld=lower(index(1).subs);
end;
 
if length(index)<2 | ~strcmp(index(2).type,'{}') | ...
    (strcmp(x.type,'batch') & strcmp(feyld,'data'));
  %Calls without vdim or vset (e.g. on data, date, author, etc)
  vdim  = 1;
  vset  = 1;
  nnargin = 3;     %fake out logic later on
  index(1) = [];    %clear field name from index structure
else
  vdim=index(2).subs{1};
  if length(index(2).subs)>1;
    vset=index(2).subs{2};
    nnargin = 5;     %fake out logic later on
  else
    vset=1;
    nnargin = 4;     %fake out logic later on
  end;
  index(1:2) = [];  %clear field name & vset/vdim from index structure
end;

if (length(vdim)>1 & (~isa(x.data,'cell') | ~strcmp(feyld,'axisscale'))) | ...
    (length(vdim)>2 & isa(x.data,'cell') & strcmp(feyld,'axisscale')) | length(vset)>2;
  error('  Indicies must be single integer values');
end;

if nnargin==3
  if ~strcmp(class(feyld),'char')
    error('  Not valid dataset object field.')
  end
  switch feyld
  case 'name'
    if ~isempty(index);
      error(['  Field ''' feyld ''' does not allow assignment subscripting']);
    end;
    if ~(strcmp(class(val),'char')|strcmp(class(val),'cell'))
      error('  Value must be class char or cell for field ''name''.')
    elseif size(val,1)>1|ndims(val)>2
      error('  Size(value,1) & ndims(value) must = 1 for field ''name''.')
    elseif strcmp(class(val),'cell')&size(val,2)>1
      error('  Cell input must be size 1x1 for field ''name''.')
    else
      if strcmp(class(val),'cell')
        val    = char(val{:});
      end
      x.name    = val;
      x.history = sethistory(x.history,inputname(1),'name',['''' val '''']);
    end
  case 'author'
    if ~isempty(index);
      error(['  Field ''' feyld ''' does not allow assignment subscripting']);
    end;
    if ~(strcmp(class(val),'char')|strcmp(class(val),'cell'))
      error('  Value must be class char or cell for field ''author''.')
    elseif (size(val,1)>1)|ndims(val)>2
      error('  Size(value,1) & ndims(value) must = 1 for field ''author''.')
    elseif strcmp(class(val),'cell')&size(val,2)>1
      error('  Cell input must be size 1x1 for field ''author''.')
    else
      if strcmp(class(val),'cell')
        val    = char(val{:});
      end
      x.author = val;
      x.history = sethistory(x.history,inputname(1),'author',['''' val '''']);
    end
  case 'date'
    if isempty(x.date)
      if ~isempty(index);
        error(['  Field ''' feyld ''' does not allow assignment subscripting']);
      end;
      if ~strcmp(class(val),'double')
        error('  Value must be class double for field ''date''.')
      elseif (size(val,1)>1)|(size(val,2)~=6)|(ndims(val)>2)
        error('  Value must be a 6 element row vector for field ''date''.')
      else
        x.date    = val;
        x.moddate = val;
      end
    else
      error('  Field ''date'' can''t be reset.')
    end
  case 'moddate'
    error('  Field ''moddate'' can''t be reset.')
  case 'type'
    if ~isempty(index);
      error(['  Field ''' feyld ''' does not allow assignment subscripting']);
    end;
    if ~(strcmp(class(val),'char')|strcmp(class(val),'cell'))
      error('  Invalid data object field value.')
    elseif ~(strcmp(val,'data')| ...
        strcmp(val,'image')| ...
        strcmp(val,'batch'))
      error('  Invalid data structure field value.')
    elseif isempty(x.data)
      error(['  ''' feyld ''' can not be set when ''data'' is empty.'])
    elseif strcmp(val,'image')
      if ~(ndims(x.data)==3|ndims(x.data)==2)
        error('  Number of dimensions not correct for ''image''.')
      else
        x.type = val;
        x.history = sethistory(x.history,inputname(1),'type',['''' val '''']);
      end
    elseif strcmp(val,'batch')
      if ~isa(x.data,'cell')
        error('  class(x.data) must be cell for type batch.')
      else
        x.type = batch;
      end
    else
      x.type   = val;
      x.history = sethistory(x.history,inputname(1),'type',['''' val '''']);
    end
  case 'data'
    if ~any(strcmp(class(val),{'double','single','logical','int8','int16','int32','uint8','uint16','uint32','cell'}))
      error('  Field ''data'' must be of class double, single, logical, [u]int8/16/32, or cell.')
    end;
    if ~isempty(x.data)
      if ~isempty(index);
        try; val = subsasgn(x.data,index,val);
        catch; error(lasterr); end;
      end;
      if ~isa(val,'cell')
        if strcmp(x.type,'batch') | isa(x.data,'cell')
          error('  Value must be class cell for dataset of type ''batch''.')
        end;
        if ~all(size(x.data)==size(val))
          error('  Size of value must equal size of previously loaded data (see dataset/cat and dataset/delsamps).')
        elseif ndims(x.data)~=ndims(val)
          error('  Number of dimensions of value must equal previously loaded data.')
        else
          x.data = val;
          if ~iscell(x.includ);
            x.includ = cell(ndims(val),2);
          end;
          for ii=1:ndims(val)
            if isempty(x.includ{ii});  %any empty cell resets all cells
              for iii=1:ndims(val)
                x.includ{iii} = 1:size(val,iii);
              end;
              break;    %no point in looking further, we just reset them all
            end;
          end;
          notes  = int2str(size(val,1));
          for ii=2:ndims(val)
            notes = [notes,'x',int2str(size(val,ii))];
          end
          if isempty(index);
            notes   = ['class ' class(val) ' ',notes,' entire data array replaced.'];
          else
            notes   = ['class ' class(val) ' ',notes,' partial data array replaced.'];
          end;
          x.history = sethistory(x.history,inputname(1),'data',inputname(3),[],[],notes);
        end
      else    %batch mode
        if ~strcmp(x.type,'batch') | ~isa(x.data,'cell')
          error('  Value cannot be class cell for dataset of type ''data'' or ''image''.')
        end
        if (size(val,1)>1) & (size(val,2)>1)
          error('Multidimensional cells not valid for dataset of type ''batch''.')
        else
          csize   = size(val{1});
          csize   = csize(2:end);    %size of dimensions~=1
          for ii=2:length(val)       %make certain that contents of all cells are same size except dim 1
            csize2 = size(val{ii});
            csize2 = csize2(2:end);
            if any(csize2~=csize)
              error('All modes except 1 must be same size.')
            end
          end      
          x.data  = val;
          notes   = int2str(size(val,1));
          for ii=2:ndims(val{1})
            notes = [notes,'x',int2str(size(val,ii))];
          end
          if isempty(index);
            notes   = ['class cell ',notes,' entire cell array replaced.'];
          else
            notes   = ['class cell ',notes,' partial cell array replaced.'];
          end;
          x.history = sethistory(x.history,inputname(1),'data',inputname(3),[],[],notes);
        end
      end
    else
      if ~isempty(index);
        error(['  Field ''Data'' is empty and cannot allow assignment subscripting']);
      end;
      temp             = dataset(val);  %use basic dataset function to create the appropriate dataset
      temp.name        = x.name;
      temp.author      = x.author;
      temp.description = x.description;
      temp.userdata    = x.userdata;
      temp.history     = x.history;
      x = temp;
      
      if ~isa(val,'cell')
        nmodes = ndims(val);
      else
        nmodes = ndims(val{1});
      end
      notes    = int2str(size(val,1));
      for ii=2:nmodes
        notes  = [notes,'x',int2str(size(val,ii))];
      end
      notes    = ['class ',class(val),' ',notes];
      x.history = sethistory(x.history,inputname(1),'data',inputname(3),[],[],notes);
    end
  case 'label'
    error('  Index must be specified for field ''label''.')
  case 'labelname'
    error('  Index must be specified for field ''labelname''.')
  case 'axisscale'
    error('  Index must be specified for field ''axisscale''.')
  case 'axisscalename'
    error('  Index must be specified for field ''axisscalename''.')
  case 'title'
    error('  Index must be specified for field ''title''.')
  case 'titlename'
    error('  Index must be specified for field ''titlename''.')
  case 'class'
    error('  Index must be specified for field ''class''.')
  case 'classname'
    error('  Index must be specified for field ''classname''.')
  case 'description'
    if ~isempty(index);
      error(['  Field ''' feyld ''' does not allow assignment subscripting']);
    end;
    if ~(strcmp(class(val),'char')|strcmp(class(val),'cell'))
      error('  Field ''description'' must be of class char or cell.')
    else
      if strcmp(class(val),'cell')
        val         = char(val{:});
      end
      x.description = val;
      x.history = sethistory(x.history,inputname(1),'description',inputname(3));      
    end
  case 'includ'
    error('  Index must be specified for field ''includ''.')
  case 'datasetversion'
    error('  Field ''datasetversion'' can''t be reset.')
  case 'userdata'
    if isempty(index);
      x.userdata = val;
    else
      try
        x.userdata = subsasgn(x.userdata,index,val);
        val = x.userdata;
      catch; error(lasterr); end;
    end;
    notes     = [int2str(size(val,1))];
    for ii=2:ndims(val)
      notes   = [notes,'x',int2str(size(val,ii))];
    end
    notes     = ['class ',class(val),' ',notes];
    if ~isempty(index);
      notes = [notes ' (subscript replacement)'];
    end;
    x.history = sethistory(x.history,inputname(1),'userdata',inputname(3),[],[],notes);      
  otherwise
    error('  Not valid dataset object field.')
  end
  if isempty(x.date)
    x.date          = clock;
    x.moddate       = x.date;
  else
    x.moddate       = clock;
  end
elseif nnargin==4|nnargin==5
  if ~strcmp(class(feyld),'char')
    error('  Not valid DataSet field.')
  elseif (~isa(x.data,'cell')&(length(vdim)>1))
    error('  vdim must be scalar.')
  elseif ~isa(x.data,'cell') & vdim>ndims(x.data)
    error('  vdim cannot be > ndims(x.data).')
  elseif isa(x.data,'cell') & vdim>ndims(x.data{1})
    error('  vdim cannot be > ndims(x.data{1}).')
  end
  switch feyld
  case 'name'
    error('  Indicies not allowed for field ''name''.')
  case 'type'
    error('  Indicies not allowed for field ''type''.')
  case 'author'
    error('  Indicies not allowed for field ''author''.')
  case 'data'
    error('  Indicies not allowed for field ''data''.')
  case 'label'
    if ~isempty(index);
      if vset>size(x.label,3);
        error(['  ''label'' set ' num2str(vset) ' does not exist.'])
      end
      try
        val = subsasgn(x.label{vdim,1,vset},index,val);
      catch; error(lasterr); end;
    end;
    if isempty(x.data)
      error('  ''label'' can not be set when ''data'' is empty.')
    elseif ~(strcmp(class(val),'char')|strcmp(class(val),'cell'))
      error('  ''label'' Value must be array of class char or cell.')
    elseif strcmp(class(val),'cell')
      notes     = ['class cell ', int2str(size(val,1))];
      for ii=2:ndims(val)
        notes     = [notes,'x',int2str(size(val,ii))];
      end
      val   = char(val{:});
    else
      notes = ['class char ', int2str(size(val,1))];
      for ii=2:ndims(val)
        notes     = [notes,'x',int2str(size(val,ii))];
      end
    end
    if ~isa(x.data,'cell') | vdim == 1
      if size(x.data,vdim)~=size(val,1) & ~isempty(val);
        error(['  Number of rows (or cell elements) in input ''label'' must =', ...
            'size(x.data,vdim) = ',int2str(size(x.data,vdim)),'.'])
      end
    else
      if size(x.data{1},vdim)~=size(val,1) & ~isempty(val);
        error(['  Number of rows (or cell elements) in input ''label'' must =', ...
            'size(x.data{1},vdim) = ',int2str(size(x.data{1},vdim)),' (for vdim > 1).'])
      end
    end    

    nsets = size(x.label,3);
    if nsets<vset;
      [x.label{:,:,nsets+1:vset}] = deal('');   %expand to new set(s) with empty STRINGS
    end
    x.label{vdim,1,vset} = val;
    if ~isempty(index);
      notes = [notes ' (subscript replacement)'];
    end;
    x.history = sethistory(x.history,inputname(1),'label',inputname(3),vdim,vset,notes);
    
  case 'labelname'
    if ~isempty(index);
      error(['  Field ''' feyld ''' does not allow assignment subscripting']);
    end;
    if isempty(x.data)
      error(['  ''' feyld ''' can not be set when ''data'' is empty.'])
    elseif ~(strcmp(class(val),'char')|strcmp(class(val),'cell'))
      error('  Value must be class char or cell for field ''labelname''.')
    elseif strcmp(class(val),'cell')
      val    = char(val{:});
    end
    if size(val,1)>1
      error(['  Size(value,1) (or number of cell elements) must ', ...
          '= 1 for field ''labelname''.'])
    end
    nsets = size(x.label,3);
    if nsets<vset;
      [x.label{:,:,nsets+1:vset}] = deal('');   %expand to new set(s) with empty STRINGS
    end
    x.label{vdim,2,vset} = val;      
    x.history = sethistory(x.history,inputname(1),'labelname',['''' val ''''],vdim,vset);
    
  case 'axisscale'
    if isempty(x.data)
      error(['  ''' feyld ''' can not be set when ''data'' is empty.'])
    end;
    if isa(x.data,'cell') & ~isempty(vdim) & vdim(1) == 1 | length(vdim)>1;  %type batch, first mode assignment
      if isa(val,'cell') & length(vdim)==1 & vdim==1;
        %assignment of all first-mode axisscales at once through a cell
        if ~isempty(index);
          if vset>size(x.axisscale,3);
            error(['  ''axisscale'' set ' num2str(vset) ' does not exist.'])
          end
          try
            val = subsasgn(x.axisscale{1,1,vset},index,val);
          catch; error(lasterr); end;
        end;
        if ~all(size(x.data)==size(val))  & ~isempty(val);% val must be cell of length(x.data)
          error(['  Cell size must = length(x.data) = ' int2str(length(x.data))])
        end
        for ii=1:length(val)
          if all(size(val{ii})>1) | ndims(val{ii})>2
            error(['Cell contents (value{',int2str(ii),'}) must be vector.'])
          end
          if length(val{ii})~=size(x.data{ii},1)
            error(['length(value{',int2str(ii),'}) must = ', ...
                'size(x.data{',int2str(ii),'},1).'])
          end
          if size(val{ii},1)>size(val{ii},2)
            val{ii}  = val{ii}';
          end
        end

        nsets = size(x.axisscale,3);
        if nsets<vset;
          [x.axisscale{:,2,nsets+1:vset}] = deal('');   %expand to new set(s) with empty STRINGS
        end
        x.axisscale{1,1,vset} = val;
        
        notes = ['class cell ', int2str(size(val,1))];
        for ii=2:ndims(val)
          notes   = [notes,'x',int2str(size(val,ii))];
        end
        if ~isempty(index);
          notes = [notes ' (subscript replacement)'];
        end;
        x.history = sethistory(x.history,inputname(1),'axisscale',inputname(3),vdim,vset,notes);
      else  %vdim must be 2 element vector w/ vdim(1)==1
        if ~isempty(index);
          error(['  Multi-indexed batch axisscale does not allow assignment subscripting. Use scalar vdim.']);
        end;
        if (prod(size(vdim))>2)|(prod(size(vdim))<2)
          error('vdim must be 2 element vector with vdim(1)==1.')
        elseif vdim(2)>length(x.data)
          error(['vdim(2)>number of cells ',int2str(length(x.data))])
        end
        if ~isa(val,'double')
          error('value must be vector of class double.')
        elseif (size(val,1)>1&size(val,2)>1)|ndims(val)>2
          error('value must be vector of class double.')
        elseif length(val)~=size(x.data{vdim(2)},1)
          error('length(value)~=size(x.data{vdim(2)},1)')
        elseif size(val,1)>size(val,2)
          notes   = ['class double ',int2str(size(val,1)),'x1'];
          val     = val';
        else
          notes   = ['class double 1x',int2str(size(val,1))];
        end
    
        nsets = size(x.axisscale,3);
        if nsets < vset;
          [x.axisscale{1,1,nsets+1:vset}] = deal(cell(length(x.data),1));  %create new empty set if set didn't exist yet
          [x.axisscale{:,2,nsets+1:vset}] = deal('');   %expand to new set(s) with empty STRINGS
        end

        x.axisscale{1,1,vset}{vdim(2)} = val;
        if ~isempty(index);
          notes = [notes ' (subscript replacement)'];
        end;
        x.history = sethistory(x.history,inputname(1),'axisscale',inputname(3),vdim,vset,notes);
      end
      
    else %x.data is class 'double' or it's 'cell' w/ vdim>1
      if ~isempty(index);
        if vset>size(x.axisscale,3);
          error(['  ''axisscale'' set ' num2str(vset) ' does not exist.'])
        end
        try
          val = subsasgn(x.axisscale{vdim,1,vset},index,val);
        catch; error(lasterr); end;
      end;
      if (size(val,1)>1&size(val,2)>1)|(ndims(val)>2)|(~isa(val,'double'))
        error('  Value must be vector of class double.')
      elseif ~isa(x.data,'cell') & size(x.data,vdim)~=length(val) & ~isempty(val);
        error(['  Length of input ''axisscale'' must = ', ...
            'size(x.data,vdim) = ',int2str(size(x.data,vdim)),'.'])
      elseif isa(x.data,'cell') & size(x.data{1},vdim)~=length(val) & ~isempty(val);
        error(['  Length of input ''axisscale'' must = ', ...
            'size(x.data{1},vdim) = ',int2str(size(x.data{1},vdim)),'.'])
      end

      if size(val,1)>size(val,2)
        val  = val';
      end
      notes = ['class double ', int2str(size(val,1))];
      for ii=2:ndims(val)
        notes   = [notes,'x',int2str(size(val,ii))];
      end
      
      nsets = size(x.axisscale,3);
      if nsets<vset;
        [x.axisscale{:,2,nsets+1:vset}] = deal('');   %expand to new set(s) with empty STRINGS
      end
      
      x.axisscale{vdim,1,vset} = val;
      if ~isempty(index);
        notes = [notes ' (subscript replacement)'];
      end;
      x.history = sethistory(x.history,inputname(1),'axisscale',inputname(3),vdim,vset,notes);
      
    end
    
  case 'axisscalename'
    if ~isempty(index);
      error(['  Field ''' feyld ''' does not allow assignment subscripting']);
    end;
    if isempty(x.data)
      error(['  ''' feyld ''' can not be set when ''data'' is empty.'])
    elseif ~(strcmp(class(val),'char')|strcmp(class(val),'cell'))
      error('  Value must be class char or cell for field ''axisscalename''.')
    elseif strcmp(class(val),'cell')
      val    = char(val{:});
    end
    if size(val,1)>1
      error(['  Size(value,1) (or number of cell elements) must ', ...
          '= 1 for field ''axisscalename''.'])
    end
    
    nsets = size(x.axisscale,3);
    if nsets<vset;
      [x.axisscale{:,2,nsets+1:vset}] = deal('');   %expand to new set(s) with empty STRINGS
    end
    x.axisscale{vdim,2,vset}= val;
    x.history = sethistory(x.history,inputname(1),'axisscalename',['''' val ''''],vdim,vset);
    
  case 'title'
    if ~isempty(index);
      error(['  Field ''' feyld ''' does not allow assignment subscripting']);
    end;
    if isempty(x.data)
      error(['  ''' feyld ''' can not be set when ''data'' is empty.'])
    elseif ~(strcmp(class(val),'char')|strcmp(class(val),'cell'))
      error('  Value must be class char or cell for field ''title''.')
    elseif strcmp(class(val),'cell')
      val    = char(val{:});
    end
    if size(val,1)>1
      error(['  Size(value,1) (or number of cell elements) must ', ...
          '= 1 for field ''title''.'])
    end
    
    nsets = size(x.title,3);
    if nsets<vset;
      [x.title{:,:,nsets+1:vset}] = deal('');   %expand to new set(s) with empty STRINGS
    end
    x.title{vdim,1,vset}= val;

    x.history = sethistory(x.history,inputname(1),'title',['''' val ''''],vdim,vset);
    
  case 'titlename'
    if ~isempty(index);
      error(['  Field ''' feyld ''' does not allow assignment subscripting']);
    end;
    if isempty(x.data)
      error(['  ''' feyld ''' can not be set when ''data'' is empty.'])
    elseif ~(strcmp(class(val),'char')|strcmp(class(val),'cell'))
      error('  Value must be class char or cell for field ''titlename''.')
    elseif strcmp(class(val),'cell')
      val    = char(val{:});
    end
    if size(val,1)>1
      error(['  Size(value,1) (or number of cell elements) must ', ...
          '= 1 for field ''titlename''.'])
    end
    
    nsets = size(x.title,3);
    if nsets<vset;
      [x.title{:,:,nsets+1:vset}] = deal('');   %expand to new set(s) with empty STRINGS
    end
    x.title{vdim,2,vset}= val;
    
    x.history = sethistory(x.history,inputname(1),'titlename',['''' val ''''],vdim,vset);
    
  case 'class'
    if ~isempty(index);
      if vset>size(x.class,3);
        error(['  ''class'' set ' num2str(vset) ' does not exist.'])
      end
      try
        val = subsasgn(x.class{vdim,1,vset},index,val);
      catch; error(lasterr); end;
    end;
    if isempty(x.data)
      error(['  ''' feyld ''' can not be set when ''data'' is empty.'])
    end
    if (size(val,1)>1&size(val,2)>1)|(ndims(val)>2)|(~strcmp(class(val),'double'))
      error('  Value must be vector of class double.')
    end
    if ~isa(x.data,'cell') | vdim == 1;
      if size(x.data,vdim)~=length(val) & ~isempty(val);
        error(['  Length of input ''class'' must =', ...
            'size(x.data,vdim) = ',int2str(size(x.data,vdim)),'.'])
      end
    else
      if size(x.data{1},vdim)~=length(val) & ~isempty(val);
        error(['  Length of input ''class'' must =', ...
            'size(x.data{1},vdim) = ',int2str(size(x.data{1},vdim)),' (for vdim > 1).'])
      end
    end    
  if size(val,1)>size(val,2)
      val  = val';
    end
    
    nsets = size(x.class,3);
    if nsets<vset;
      [x.class{:,2,nsets+1:vset}] = deal('');   %expand to new set(s) with empty STRINGS
    end
    x.class{vdim,1,vset} = val;
    
    notes = ['class double ', int2str(size(val,1))];
    for ii=2:ndims(val)
      notes   = [notes,'x',int2str(size(val,ii))];
    end
    if ~isempty(index);
      notes = [notes ' (subscript replacement)'];
    end;
    x.history = sethistory(x.history,inputname(1),'class',inputname(3),vdim,vset,notes);
    
  case 'classname'
    if ~isempty(index);
      error(['  Field ''' feyld ''' does not allow assignment subscripting']);
    end;
    if isempty(x.data)
      error(['  ''' feyld ''' can not be set when ''data'' is empty.'])
    elseif ~(strcmp(class(val),'char')|strcmp(class(val),'cell'))
      error('  Value must be class char or cell for field ''classname''.')
    elseif strcmp(class(val),'cell')
      val    = char(val{:});
    end
    if size(val,1)>1
      error(['  Size(value,1) (or number of cell elements) must ', ...
          '= 1 for field ''classname''.'])
    end
    
    nsets = size(x.class,3);
    if nsets<vset;
      [x.class{:,2,nsets+1:vset}] = deal('');   %expand to new set(s) with empty STRINGS
    end
    x.class{vdim,2,vset}= val;
    
    x.history = sethistory(x.history,inputname(1),'classname',['''' val ''''],vdim,vset);
    
  case 'description'
    error('  Indicies not allowed for field ''description''.')
    
  case 'includ'
    if vset>1;
      error(['  Field ''' feyld ''' does not allow multiple sets']);
    end;
    if ~isempty(index);
      try
        val = subsasgn(x.includ{vdim},index,val);
      catch; error(lasterr); end;
    end;
    if ~isa(x.data,'cell')  | vdim == 1
      nmodes  = ndims(x.data);
      nsize   = size(x.data);
    else
      nmodes  = ndims(x.data{1});
      nsize   = size(x.data{1});
      nsize(1)= length(x.data);
    end
    if any(val<1)|(max(val)>nsize(vdim))
      error(['  Value must be row vector of integers and a', ...
          ' subset of [1:',int2str(nsize(vdim)),'].'])
    elseif size(val,1)>1&size(val,2)>1
      error('  Value must be row vector.')
    else
      if size(val,1)>size(val,2)
        val   = val';
      end
      
      x.includ{vdim} = val;
      
      notes = ['class double ', int2str(size(val,1))];
      for ii=2:ndims(val)
        notes   = [notes,'x',int2str(size(val,ii))];
      end
      if ~isempty(index);
        notes = [notes ' (subscript replacement)'];
      end;
      x.history = sethistory(x.history,inputname(1),'includ',inputname(3),vdim,vset,notes);
    end
  case 'userdata'
    error('  Indicies not allowed for field ''userdata''.')
  otherwise
    error('  Not valid dataset object field.')
  end
  if isempty(x.date)
    x.date          = clock;
    x.moddate       = x.date;
  else
    x.moddate       = clock;
  end
elseif nnargin>5
  error('Too many inputs.')
end

%--------------------------------------------------------------------------
function history = sethistory(history,objname,feyld,val,vdim,vset,notes)

if isempty(history{1})
  ihis   = 1;
else
  ihis   = length(history)+1;
end

caller = '';
try
  [ST,I] = dbstack;
  if length(ST)>2;
    [a,b,c]=fileparts(ST(3).name); 
    caller = [' [' b c ']'];
  end
catch
end

if isempty(val);
  val = '?FunctionCall';
end  

if nargin>6;
  notes  = ['   %' notes caller]; 
else
  notes  = ['   %' caller ];
end;

switch feyld
case {'name', 'author', 'type', 'description', 'data', 'userdata'}
  history{ihis} = [objname '.' feyld '=' val ';' notes];
otherwise
  history{ihis} = [objname '.' feyld '{' int2str(vdim) ',' int2str(vset) '}=' val ';' notes];
end

