function data = dataset(a)
%DATASET DataSet object class constructor.
% Creates a DataSet object which can contain data along with related
%  informational fields including: 
%   
%    name         : name of data set.
%    author       : authors name.
%    date         : date of creation.
%    moddate      : date of last modification.
%    type         : either 'data', 'batch' or 'image'.
%    data         : actual data consisting of any Matlab array of class
%                     double, single, logical, or [u]int8/16/32.
%    label        : text labels for each row, column, etc of data.
%    labelname    : descriptive name for each set of labels.
%    axisscale    : axes scales for each row, column, etc of data.
%    axisscalename: descriptive name for each set of axis scales.
%    title        : axis titles for each row, column, etc of data.
%    titlename    : descriptive name for each axis title.
%    class        : class indentifiers for each row, column, etc of data.
%    classname    : descriptive name for each set of class identifiers.
%    includ       : indices of rows, columns, etc to use from data (allows
%                    "exclusion" of data without hard-deletion)
%    userdata     : user defined content.
%    description  : text description of DataSet content.
%    history      : text description of modification history.
%    datasetversion : dataset object version.
%
% For more information on working with DataSet objects, see the methods: 
%    DATASET/SUBSREF and DATASET/SUBSASGN
% For more detail on DataSet functionality, see the DataObject documentation. 
%
%I/O: data = dataset(a);
%
%See also: DATASET/EXPLODE, DATASET/SUBSASGN, DATASET/SUBSREF

%Copyright Eigenvector Research, Inc. 2000-2001

%nbg 8/3/00, 8/16/00, 8/17/00, 8/30/00, 10/05/00, 10/09/00
%nbg added 5/11/01  b.includ = cell(nmodes,2); (this is different from
%    the previous version which used b.includ = cell(ndims) which didn't
%    follow the convention of different modes on different rows
%jms 5/30/01 added transposition of row-vector batch cell to column-vector
%nbg 10/07/01 changed version from 2.01 to 3.01
%jms 8/30/02 added validclasses string
%    added empty dataset construction
%jms 11/06/02 change version to 3.02
%    updated help

%Construct a dataset object template
b.name      = '';            %variable name   char
b.type      = '';            %data type       char
                               %   data  {default}
                               %   image
                               %   batch
b.author    = '';            %dataset author  char
b.date      = [];            %creation date
b.moddate   = [];            %last modified date
b.data      = [];            %double array

%define valid classes for dataset of type "data" 
validclasses = {'double','single','logical','int8','int16','int32','uint8','uint16','uint32'};

if nargin==0
  nmodes    = 2;
elseif nargin>0
  if isempty(a)
    nmodes  = 2;
    b.type  = 'data';        %default
  elseif any(strcmp(class(a),validclasses))
    nmodes  = ndims(a);
    b.type  = 'data';
    b.data  = a;
  elseif isa(a,'cell')
    if (size(a,1)>1)&(size(a,2)>1)
      error('Not set up for multidimensional cells.')
    else
      if size(a,2)>1; a=a'; end; %flip to be COLUMN vector
      nmodes  = ndims(a{1});     %number of modes for each cell
      csize   = size(a{1});
      csize   = csize(2:end);    %size of dimensions~=1
      if length(a)>1             %make certain that contents of all
        for ii=2:length(a)       % cells are same size except dim 1
          csize2 = size(a{ii});
          csize2 = csize2(2:end);
          if any(csize2~=csize)
            error('All modes except 1 must be same size.')
          end
        end      
      end
      b.type  = 'batch';
      b.data  = a;
    end
  else
    error(['Unable to create a dataset to contain variables of class ' class(a)])
  end
end

b.label     = cell(2,2,1);   %empty cell
b.axisscale = cell(2,2,1);   %empty cell
b.title     = cell(2,2,1);   %empty cell
b.class     = cell(2,2,1);   %empty cell
b.includ    = cell(2,2,1);   %empty cell
for ii      = 1:nmodes
  b.label{ii,1,1}     = ''; b.label{ii,2,1}     = ''; %'Set 1';
  b.axisscale{ii,1,1} = []; b.axisscale{ii,2,1} = ''; %'Set 1';
  b.title{ii,1,1}     = ''; b.title{ii,2,1}     = ''; %'Set 1';
  b.class{ii,1,1}     = []; b.class{ii,2,1}     = ''; %'Set 1';
end
if nargin==0
  b.includ       = cell(2,2);  %empty cell with size = ndims x 1  %nbg changed 5/11/01
elseif ~isa(a,'cell')
  b.includ       = cell(nmodes,2);  %nbg added 5/11/01
  for ii=1:nmodes
    b.includ{ii} = [1:size(b.data,ii)]; 
  end
else
  b.includ       = cell(nmodes,2);  %nbg added 5/11/01
  b.includ{1}    = [1:length(b.data)];
  for ii=2:nmodes
    b.includ{ii} = [1:size(b.data{1},ii)]; 
  end
  b.axisscale{1,1,1} = cell(length(a),1);
end
b.description    = '';       %character string
b.userdata       = [];       %userdata

b.datasetversion = '3.02';   %version number of the dataset object
                             %this is hidden (can't be set) but
                             %can be obtained using GET.
b.history      = cell(1,1);  %changed in SET
b.history{1}   = '';

if nargin==0
  data         = class(b,'dataset');
elseif isa(a,'dataset')
  data         = a;
elseif any(strcmp(class(a),validclasses)) | isa(a,'cell');
  data         = b; clear b
  data.name    = inputname(1);
  data.date    = clock;
  data.moddate = data.date;
  data         = class(data,'dataset');
else
  error(['Datasets can not be made from class ' class(a) ' variables'])
end

