function y = cat(dim,varargin);
%DATASET/CAT Concatenation of DataSet objects.
%  Generalized concatenation of DataSet objects
%  [a; b] is the vertical concatenation of DataSet objects
%  (a) and (b). Any number of DataSet objects can be
%  concatenated within the brackets. For this operation
%  to be defined the following must be true:
%    1) All inputs must be valid dataset objects.
%    2) The Dataset 'type' fields must all be the same.
%    3) Concatenation is along the dimension dim. The
%       remaining dimension sizes must match.
%
%  This is similar to matrix concatenation, but each field
%  is treated differently. In structure notation this is:
%    z.name      = a.name;
%    z.type      = a.type;
%    z.author    = a.author
%    z.date      date of concatenation
%    z.moddate   date of concatenation
%    z.data      = cat(dim,a.data,b.data,c.data,...);
%    z.label     mode dim label sets are concatenated, and new
%                label sets are created for all other modes
%    z.axisscale mode dim axisscale sets are concatenated, and new
%                axisscale sets are created for all other modes
%    z.title     new label sets are created
%    z.class     mode dim class sets are concatenated with
%                empty classes set to zeros, and new
%                class sets are created for all other modes
%    z.description concatenates all descriptions
%    z.includ    mode dim include sets are concatenated all
%                others are taken from first dataset
%    z.userdata  if more than one input has userdata it is
%                filled into a cell array, if only one input
%                has userdata it is returned, else it is empty
%
%I/O: z = cat(dim,a,b,c);
%
%See also: DATASET, DATASET/HORZCAT, DATASET/VERTCAT

%Copyright Eigenvector Research, Inc. 2001
%jms 11/06/01 generalized from vertcat and horzcat
%jms 11/06/02 (funny, one year later!) fix bug related to having different
%    numbers of sets of labels, classes, or axisscales in each datatset
%  -allowed cat in dim > ndim (e.g. cat of slabs)
%  -history field note of cat operation
%  -attempt cat of datasets and non-datasets by making datasets from nons
%jms 11/07/02
%  -name and author taken from ALL datasets

iv     = length(varargin);

if nargin<2;
  error('Must supply dim and at least one dataset');
end

for ii=1:iv
  if ~isa(varargin{ii},'dataset')
    varargin{ii} = dataset(varargin{ii});
%     error('All inputs must be dataset objects.')
  end
  if strcmp('batch',lower(varargin{ii}.type))
    error('Not defined for dataset objects of type ''batch''.')
  end
end

if iv==1
  y    = varargin{1};
  return
end

for ii=2:iv
  if ~strcmp(varargin{ii}.type,varargin{1}.type)
    error('All input dataset objects must be of the same type.')
  end
end

y         = dataset;
y.name    = varargin{1}.name;
for ii=2:iv;
  if ~isempty(varargin{ii}.name) & ~strcmp(y.name,varargin{ii}.name);
    y.name = [y.name ',' varargin{ii}.name];
  end
end
y.author  = varargin{1}.author;
for ii=2:iv;
  if ~isempty(varargin{ii}.author) & ~strcmp(y.author,varargin{ii}.author);
    y.author = [y.author ',' varargin{ii}.author];
  end
end
y.date    = clock;
y.moddate = clock;

otherdims        = setdiff(1:ndims(varargin{1}.data),dim);

dimsize  = zeros(iv,1);
for ii=1:iv
  dimsize(ii) = size(varargin{ii}.data,dim);
end

%concatenate the data
for ii=1:iv;
  s1{ii} = varargin{ii}.data;
  if ii > 1;
    if ndims(s1{ii}) ~= ndims(s1{1});
      error(['Number of dimensions in each data sets must match']);
    end
    for ij = otherdims;
      if size(s1{ii},ij) ~= size(s1{1},ij);
        error(['All data sets must match in size except dim ' num2str(dim)]);
      end
    end
  end
end
y.data = cat(dim,s1{:});

y.type = varargin{1}.type;

%label
y.label = varargin{1}.label;
% all modes except dim
for ii=2:iv   %loop over number of input arguments
  ia = size(y.label,3); %number of label sets
  for ik=1:size(varargin{ii}.label,3)
    i1 = logical(0);  
    for ij=otherdims   %loop over number of dimensions
      if ~isempty(varargin{ii}.label{ij,1,ik})
        i1 = logical(1);
        break
      end
    end
    if i1
      for ij=otherdims
        y.label{ij,1,ia+1} = varargin{ii}.label{ij,1,ik};
        y.label{ij,2,ia+1} = varargin{ii}.label{ij,2,ik};
      end
      ia = ia+1;
    end
  end
end
% mode dim
if dim<=length(dimsize);
  ia   = size(varargin{1}.label,3);
  for ii=2:iv   %loop over the number of input arguments
    ia = max([ia size(varargin{ii}.label,3)]);
  end
  for ii=1:ia   %loop over sets
    i1 = logical(0);
    for ij=1:iv
      if size(varargin{ij}.label,3)>=ii & ~isempty(varargin{ij}.label{dim,1,ii})
        i1 = logical(1);
        break
      end
    end
    if i1
      s1 = '';
      s2 = ' ';
      name = [];
      for ij=1:iv
        if size(varargin{ij}.label,3)<ii | isempty(varargin{ij}.label{dim,1,ii})
          s1 = char(s1,s2(ones(dimsize(ij),1),:));
        else
          s1 = char(s1,varargin{ij}.label{dim,1,ii});
          thisname = varargin{ij}.label{dim,2,ii};
          if ~isempty(thisname) & ~strcmp(thisname,name);
            name = [name ',' thisname];
          end
        end
      end
      y.label{dim,1,ii} = deblank(s1(2:end,:));
      y.label{dim,2,ii} = deblank(name(2:end));
    end
  end
end

%axisscale
y.axisscale = varargin{1}.axisscale;
% all modes except dim
for ii=2:iv   %loop over number of input arguments
  ia = size(y.axisscale,3); %number of axisscale sets
  for ik=1:size(varargin{ii}.axisscale,3)
    i1 = logical(0);  
    for ij=otherdims   %loop over number of dimensions
      if ~isempty(varargin{ii}.axisscale{ij,1,ik})
        i1 = logical(1);
        break
      end
    end
    if i1
      for ij=otherdims
        y.axisscale{ij,1,ia+1} = varargin{ii}.axisscale{ij,1,ik};
        y.axisscale{ij,2,ia+1} = varargin{ii}.axisscale{ij,2,ik};
      end
      ia = ia+1;
    end
  end
end
% mode dim
if dim<=length(dimsize);
  ia   = size(varargin{1}.axisscale,3);
  for ii=2:iv   %loop over the number of input arguments
    ia = max([ia size(varargin{ii}.axisscale,3)]);
  end
  for ii=1:ia   %loop over sets
    i1 = logical(1);    %we WILL concatenate unless any input argument has an empty entry for this set
    for ij=1:iv
      if size(varargin{ij}.axisscale,3)<ii | isempty(varargin{ij}.axisscale{dim,1,ii})
        i1 = logical(0);
        break
      end
    end
    if i1
      s1 = [];
      name = [];
      for ij=1:iv
        s1 = [s1 varargin{ij}.axisscale{dim,1,ii}];
        thisname = varargin{ij}.axisscale{dim,2,ii};
        if ~isempty(thisname) & ~strcmp(thisname,name);
          name = [name ',' thisname];
        end
      end
      y.axisscale{dim,1,ii} = s1;
      y.axisscale{dim,2,ii} = deblank(name(2:end));
    else
      y.axisscale{dim,1,ii} = [];
    end
  end
end

%title
y.title = varargin{1}.title;
% all modes
for ii=2:iv   %loop over number of input arguments
  ia = size(y.title,3); %number of title sets
  for ik=1:size(varargin{ii}.title,3)
    i1 = logical(0);  
    for ij=1:ndims(varargin{ii}.data)   %loop over number of dimensions
      if ~isempty(varargin{ii}.title{ij,1,ik})
        i1 = logical(1);
        break
      end
    end
    if i1
      for ij=1:ndims(varargin{ii}.data)
        y.title{ij,1,ia+1} = varargin{ii}.title{ij,1,ik};
        y.title{ij,2,ia+1} = varargin{ii}.title{ij,2,ik};
      end
      ia = ia+1;
    end
  end
end

%class
y.class = varargin{1}.class;
% all modes except dim
for ii=2:iv   %loop over number of input arguments
  ia = size(y.class,3); %number of class sets
  for ik=1:size(varargin{ii}.class,3)
    i1 = logical(0);  
    for ij=otherdims   %loop over number of dimensions
      if ~isempty(varargin{ii}.class{ij,1,ik})
        i1 = logical(1);
        break
      end
    end
    if i1
      for ij=otherdims
        y.class{ij,1,ia+1} = varargin{ii}.class{ij,1,ik};
        y.class{ij,2,ia+1} = varargin{ii}.class{ij,2,ik};
      end
      ia = ia+1;
    end
  end
end
% mode dim
if dim<=length(dimsize);
  ia   = size(varargin{1}.class,3);
  for ii=2:iv   %loop over the number of input arguments
    ia = max([ia size(varargin{ii}.class,3)]);
  end
  for ii=1:ia   %loop over sets
    i1 = logical(0);
    for ij=1:iv
      if size(varargin{ij}.class,3)>=ii & ~isempty(varargin{ij}.class{dim,1,ii})
        i1 = logical(1);
        break
      end
    end
    if i1
      s1 = [];
      name = [];
      for ij=1:iv
        if size(varargin{ij}.class,3)<ii | isempty(varargin{ij}.class{dim,1,ii})
          s1 = [s1, zeros(1,dimsize(ij))];
        else
          s1 = [s1, varargin{ij}.class{dim,1,ii}];
          thisname = varargin{ij}.class{dim,2,ii};
          if ~isempty(thisname) & ~strcmp(thisname,name);
            name = [name ',' thisname];
          end
        end
      end
      y.class{dim,1,ii} = s1;
      y.class{dim,2,ii} = deblank(name(2:end));
    else
      y.class{dim,1,ii} = [];
    end
  end
end

%includ
y.includ = varargin{1}.includ;
if 0;     %the next part of the code is disabled
  % all modes except dim
  for ii=2:iv   %loop over number of input arguments
    ia = size(y.includ,3); %number of sets
    for ik=1:size(varargin{ii}.includ,3)
      i1 = logical(0);  
      for ij=otherdims   %loop over number of dimensions
        if ~isempty(varargin{ii}.includ{ij,1,ik})
          i1 = logical(1);
          break
        end
      end
      if i1
        for ij=otherdims
          y.includ{ij,1,ia+1} = varargin{ii}.includ{ij,1,ik};
          y.includ{ij,2,ia+1} = varargin{ii}.includ{ij,2,ik};
        end
        ia = ia+1;
      end
    end
  end
end
%mode dim
if dim<=length(dimsize);
  ia   = size(varargin{1}.includ,3);
  for ii=2:iv   %loop over the number of input arguments
    ia = max([ia size(varargin{ii}.includ,3)]);
  end
  for ii=1:ia   %loop over sets
    s1 = varargin{1}.includ{dim,1,ii};
    for ij=2:iv
      if size(varargin{ij}.includ,3) < ii 
        s1 = [s1, sum(dimsize(1:(ij-1)))+1:sum(dimsize(1:ij))];  %include ALL if no includ set
      elseif ~isempty(varargin{ij}.includ{dim,1,ii});
        s1 = [s1, sum(dimsize(1:(ij-1)))+varargin{ij}.includ{dim,1,ii}];
      end
    end
    y.includ{dim,1,ii} = s1;
  end
end

%Note: on 8/30/00 the following code was used, but failed
%to provide an inputname (should be checked with R12 to
%see if the same problem exists). The Code following the %---
%is used until this is fixed by the MathWorks
%description
%for ii=1:iv
%  if isempty(varargin{ii}.description)
%    y.description = strvcat(y.description,[inputname(ii),':']);
%  else
%    y.description = strvcat(y.description,[inputname(ii),':'], ...
%                            varargin{ii}.description);
%  end
%end
%---
for ii=1:iv
  if ~isempty(varargin{ii}.description)
    y.description = strvcat(y.description,varargin{ii}.description);
  end
end

%userdata
i1    = 0;
i2    = [];
for ii=1:iv
  if ~isempty(varargin{ii}.userdata)
    i1 = i1+1;
    i2 = [i2 ii];
  end
end
if i1>1
  y.userdata = cell(i1,1);
  for ii=1:i1
    y.userdata{ii,1} = varargin{i2(ii)}.userdata;
  end
elseif i1==1
  y.userdata = varargin{i2}.userdata;
end

%history
name = [];
for ii=1:iv;
  thisname = inputname(ii+1);
  if isempty(thisname);
    thisname = ['"' varargin{ii}.name '"'];
  end
  if isempty(thisname);
    thisname = 'unknown_dataset';
  end
  name = [name ',' thisname];
end
y.history = {['x=cat(' num2str(dim) name ')']};

