function Label = label(varargin)
%   Label object
% 
%   SYNTAX
%   
%   Labels = label(Lab,DefLab,Scalars,AxisLabels,Name,MeasUn)
%   
%    Lab:         can be char array or cell vector of chars or vector of doubles. In the latter case 
%                 the labels are in the form [DefLab number]. If scalars is defined this field is not necessary
%    DefLab:      default: 'Var.'
%    Scalars:     scalars to plot against. If Lab is empty the labels are in the form [scalar MeasUn], if not defined
%                 Scalars = 1:length(Lab)
%    MeasUn:      default: ''
%    AxisLabels:  char array or cell vector of arrays
%    Name:        name of the label object
%   
%   
%   A label object can refer to any dimension in a multidimensional array. 
%   Only a few functions are defined on the label object:
%   1. char.m       transforms the label object into a char array    
%   2. display.m    this function is used to display the content of a label object 
%   3. end.m        it is used to allow the use of "end" in subsref and subsasgn
%   4. length.m     returns the number of labels of the labels object
%   5. size.m       returns the number of labels in the first mode and 1 in the second mode, multidimensional label objects are
%   	        	not allowed
%   6. subsasgn.m   allows to assign values to certain parts of the label object (such as labels, scalars...)
%   7. subsindex.m  since the label object cannot be used as an index for other matrices it simply returns an error message
%   8. subsref.m    allows to refer to subsets of labels. Its output is a new label object or variables belonging to a suitable
%   		class (such as cell arrays, char arrays, etc.)
%   
%   Examples:
%   
%   Assuming that one has a data set of dimensions 20 x 4 x 10, where the first dimension represents the samples, 
%   the second the concentration of 4 different compounds and the third the measurement times (from 1 to 28 seconds 
%   every third second).
%   One can than create three label objects respectively having length 20, 4 and 10.
%   
%   Lab1 = label(1:20,'Sam',[],'Samples','Samples');
%   Lab2 = label({'Tryptophan','DOPA','Tyrosine','Phenilalanine'},'Comp.',[],'Compounds','Compounds');
%   Lab3 = label([],'',1:3:28,'Time (s)','Time','s');
%   Lab1,Lab2,Lab3 
%   
%   Name      : Samples
%   Meas. unit: 
%   Def. label: Sam 
%   Axis label: Samples
%   Labels    : Sam 1 
%               Sam 2 
%               Sam 3 
%               Sam 4 
%               Sam 5 
%               Sam 6 
%               Sam 7 
%               Sam 8 
%               Sam 9 
%               Sam 10
%               Sam 11
%               Sam 12
%               Sam 13
%               Sam 14
%               Sam 15
%               Sam 16
%               Sam 17
%               Sam 18
%               Sam 19
%               Sam 20
%   
%   Name      : Compounds
%   Meas. unit: 
%   Def. label: Comp. 
%   Axis label: Compounds
%   Labels    : Tryptophan   
%               DOPA         
%               Tyrosine     
%               Phenilalanine
%               
%   Name      : Time
%   Meas. unit: s
%   Def. label:  
%   Axis label: Time (s)
%   Labels    : 1 s 
%               4 s 
%               7 s 
%               10 s
%               13 s
%               16 s
%               19 s
%               22 s
%               25 s
%               28 s
%   
%   The label object can be treated as a normal vector: 
%   
%    t = Lab1(1:2:end) 
%   
%   Name      : Samples
%   Meas. unit: 
%   Def. label: Sam  
%   Axis label: Samples
%   Labels    : Sam 1 
%               Sam 3 
%               Sam 5 
%               Sam 7 
%               Sam 9 
%               Sam 11
%               Sam 13
%               Sam 15
%               Sam 17
%               Sam 19
%   
%   and t is a 10 x 1 label object:
%    whos t
%     Name      Size         Bytes  Class
%   
%     t        10x1           2116  label object
%   
%   Grand total is 103 elements using 2116 bytes
%   
%   To access labels, scalars and the other properties of the label object one can use a dot as for structures:
%   
%    Lab1.labels
%   
%   ans = 
%   
%       'Sam 1'
%       'Sam 2'
%       'Sam 3'
%       'Sam 4'
%       'Sam 5'
%       'Sam 6'
%       'Sam 7'
%       'Sam 8'
%       'Sam 9'
%       'Sam 10'
%       'Sam 11'
%       'Sam 12'
%       'Sam 13'
%       'Sam 14'
%       'Sam 15'
%       'Sam 16'
%       'Sam 17'
%       'Sam 18'
%       'Sam 19'
%       'Sam 20'
%       
%    Lab3.scalars
%   
%   ans =
%   
%        1
%        4
%        7
%       10
%       13
%       16
%       19
%       22
%       25
%       28
%   
%    Lab2.axislabels
%   
%   ans = 
%   
%       'Compounds'
%   
%    Lab2.deflab
%   
%   ans =
%   
%   Comp. 
%   
%    Lab2.name
%   
%   ans =
%   
%   Compounds
%   
%   Complex constructions are also allowed for reference, e.g.
%   
%    Lab1(1:2:end).labels
%   
%   ans = 
%   
%       'Sam 1'
%       'Sam 3'
%       'Sam 5'
%       'Sam 7'
%       'Sam 9'
%       'Sam 11'
%       'Sam 13'
%       'Sam 15'
%       'Sam 17'
%       'Sam 19'
%   
%   The scalars and the labels are kept together:
%   
%    p = Lab3(1:5)
%   Name      : Time
%   Meas. unit: s
%   Def. label:   
%   Axis label: Time (s)
%   Labels    : 1 s 
%               4 s 
%               7 s 
%               10 s
%               13 s
%   
%    p.scalars
%   
%   ans =
%   
%        1
%        4
%        7
%       10
%       13
%   
%   The "Def. label" (default label) is used to reconstruct undefined labels:
%    Lab(11:15)=Lab1(1:5)
%   yields
%   Name      : 
%   Meas. unit: 
%   Def. label: Var. 
%   Axis label: 
%   Labels    : Var. 1 
%               Var. 2 
%               Var. 3 
%               Var. 4 
%               Var. 5 
%               Var. 6 
%               Var. 7 
%               Var. 8 
%               Var. 9 
%               Var. 10
%               Sam 1  
%               Sam 2  
%               Sam 3  
%               Sam 4  
%               Sam 5  
%   
%   because the standard value for "Def. label" is 'Var.' and Lab is an empty undefined matrix
%   
%    Lab1(25)='Sample 30'
%    
%   Name      : Samples
%   Meas. unit: 
%   Def. label: Sam 
%   Axis label: Samples
%   Labels    : Sam 1    
%               Sam 2    
%               Sam 3    
%               Sam 4    
%               Sam 5    
%               ...
%               ...
%               Sam 20   
%               Sam 21   
%               Sam 22   
%               Sam 23   
%               Sam 24   
%               Sample 30
%   
%   Because "Def. label" for Lab1 is 'Sam' the labels between 21 and 24 are created automatically.
%   The assigned value can either be a char array, a cell vector containing only strings or another label object. 
%   E.g.:
%    t={'Fluoranthene','Nonylphenol'}'
%    Lab2(6:7)=t
%    
%   Name      : Compounds
%   Meas. unit: 
%   Def. label: Comp. 
%   Axis label: Compounds
%   Labels    : Tryptophan   
%               DOPA         
%               Tyrosine     
%               Phenilalanine
%               Comp. 5      
%               Fluoranthene 
%               Nonylphenol  
%   Since "Def. label" for Lab2 is 'Comp.'. Note that t must be a column cell vector otherwise the operation will return
%   the original label object
%   There is no filling in for the scalars themselves, which are instead turned automatically to 1:length(label) in
%   case the filling for the labels by means of "Def. label" is applied. Hence if one defines the labels by scalars 
%   it is basically not possible to update the label object without updating all the scalars at the same time, 
%   otherwise all the scalars are lost:
%    Lab3.scalars = [2:3:29]
%    
%   Name      : Time
%   Meas. unit: s
%   Def. label:   
%   Axis label: Time (s)
%   Labels    : 2 s 
%               5 s 
%               8 s 
%               11 s
%               14 s
%               17 s
%               20 s
%               23 s
%               26 s
%               29 s
%    
%   There is an automatic check for identical labels and scalars, if any the operation is interrupted and an error message
%   is generated:
%    Lab1(2)='Sam 1'
%   ??? Error using ==> label/subsasgn
%   No label or scalar can be repeated
%   
%   The length on the two sides of an assignment must be the same:
%    Lab3.scalars = [2:3:22]
%   ??? Error using ==> label/subsasgn
%   Unequal lengths
%   
%   If one redefines one or more of the actual labels of a label object defined by scalars, the automatic label update based 
%   on the automatic scheme [number measurement-unit] stops working hence upon modification the labels remain the same:
%   
%    Lab3(1)='Null'
%    
%   Name      : Time
%   Meas. unit: s
%   Def. label:   
%   Axis label: Time (s)
%   Labels    : Null
%               5 s 
%               8 s 
%               11 s
%               14 s
%               17 s
%               20 s
%               23 s
%               26 s
%               29 s
%   
%    Lab3.scalars = 1:3:28
%    
%   Name      : Time
%   Meas. unit: s
%   Def. label:    
%   Axis label: Time (s)
%   Labels    : Null
%               5 s 
%               8 s 
%               11 s
%               14 s
%               17 s
%               20 s
%               23 s
%               26 s
%               29 s
%               
%   Used in: Content object, CBDataSet object
%   
%
%   Author: Giorgio Tomasi
%       Royal Agricolture University, MLI, LMT
%       1958 Frederiksberg C
%       Danmark
%       email: gt@kvl.dk
%   


Labs = struct('name','','labels',[],'scalars',[],'axislabels',[],'measun','','deflab','Var. ','flag',0);
if nargin > 1
   varargin(nargin + 1:6) = {[]};
end
switch nargin
case 0
   Label = class(Labs,'label');
   
case 1
   if isa(varargin{1},'double')
      Labs.labels  = cellstr([ones(length(varargin{1}(:)),1)*'Var. ',char(strrep(cellstr(num2str(varargin{1}(:))),' ',''))]);
      Labs.scalars = [1:length(varargin{1}(:))]'; 
   elseif isa(varargin{1},'cell')
      
      if ~all(cellfun('isclass',varargin{1}(:),'char'))
         error
      else
         Labs.labels   = varargin{1}(:);
         Labs.scalars = [1:length(Labs.labels)]';
      end
      
   elseif ischar(varargin{1})
      Labs.labels  = cellstr(varargin{1});
      Labs.scalars = [1:length(Labs.labels)]';
   elseif isa(varargin{1},'label')
      Label = varargin{1};
      return
   else
      error('Error')
   end
   Label = class(Labs,'label');
   
case {2,3,4,5,6}
   if isempty(varargin{1}) & isempty(varargin{3})
      error('Error')
   end
   if ischar(varargin{6})
      Labs.measun = varargin{6};
   end
   if ischar(varargin{2})
      Labs.deflab = [varargin{2},' '];
   end
   if isa(varargin{1},'double') & ~isempty(varargin{1})
      Labs.labels  = cellstr([ones(length(varargin{1}(:)),1) * Labs.deflab,char(strrep(cellstr(num2str(varargin{1}(:))),' ',''))]);
   elseif isa(varargin{1},'cell')
      if ~all(cellfun('isclass',varargin{1}(:),'char'))
         error('Type mismatch error')
      else
         Labs.labels = varargin{1}(:);
      end
   elseif ischar(varargin{1})
      Labs.labels = cellstr(varargin{1});
   elseif isempty(varargin{1})
      a = [num2str([varargin{3}(:)]),ones(length(varargin{3}(:)),1)*[' ',Labs.measun]];
      i = 1;
      while i
         if ~all(a(:,i) ~=  ' ');
            a(a(:,i) == ' ') = '';
            i = i + 1;
         else
            i = 0;
         end
      end
      Labs.labels = cellstr([char(strrep(cellstr(a),'',''))]);
      Labs.flag   = 1;
   else
      error('Type mismatch error')
   end
   if isempty(varargin{3})
      Labs.scalars = [1:length(Labs.labels)]';
   elseif isa(varargin{3},'double')
      if length(varargin{1}) == length(varargin{3}) | isempty(varargin{1})
         Labs.scalars = varargin{3}(:);
      else
         error('Inconsistent lengths for labels and scalars')
      end
   else
      error('Type mismatch error')
   end
   if ~isempty(varargin{4})
      if ischar(varargin{4})
         varargin{4} = cellstr(varargin{4});
      end
      if iscell(varargin{4})
         if all(cellfun('isclass',varargin{4}(:),'char'))
            Labs.axislabels = varargin{4}(:);
         end
      else
         error('Type mismatch error')
      end
   end
   if ~isempty(varargin{5})
      if ischar(varargin{5})
         Labs.name = varargin{5}(:)';
      else
         error('Type mismatch error')
      end
   end
   %if length(Labs.scalars) ~= length(unique(Labs.scalars)) | length(Labs.labels) ~= length(unique(Labs.labels))
   %   error('No label or scalar can be repeated');
   %end
   Label = class(Labs,'label');
otherwise
   error('Error')
end
