function DS = cbdataset(varargin)
%   CBDataSet object
%
%   Syntax: DataSet = cbdataset(data,['field1',value1[,'field2',value2,[...]]]
%
%   Fields       Allowed values
%
%   Name       : string of char, if it not inserted the name of the matrix containing the data will be used.
%                 Default: Matrix
%   Type       : can either be 'data' (only one implemented so far), 'batch' or 'image'
%             Default: Data
%   Author     : either matrix of chars or cell of strings (the cell is vectorised when it is not a vector)
%   Description: either matrix of chars or cell of strings (the cell is vectorised when it is not a vector)
%   Labels     : cell vector - length must be equal to ndims(data))
%                containing label objects or cell vectors of strings or char arrays or vectors of doubles
%
%   The cbdataset object is extensively used in the GUI but can also be used separately from it
%   The simplest syntax is the following (X being here a 20 x 30 x 20 array):
%   >> Data = cbdataset(X);
%
%   Name          : X
%   Type          : Data
%   Dimensions    : [20 x 30 x 25]
%   Author        : 
%   Description   : 
%   Last modified : 30/01/2002, 22:00:10
%
%   
%   The labels will be generated automatically as 'Sam. n' for the first mode and 'Var. p' for the 
%   following. The different modes will also be named as 'Mode 1','Mode 2' and 'Mode 3'
%   
%   If one wants to use different labels it is possible to provide them at the moment of the creation
%   of the cbdataset object:
%   
%   >> Labels = {[1:2:40],[1:30],[]};
%   >> Data = cbdataset(X,'labels',Labels)
%    
%    Name          : X
%    Type          : Data
%    Dimensions    : [20 x 30 x 25]
%    Author        : 
%    Description   : 
%    Last modified : 30/01/2002, 22:20:55
%    
%   The Labels must be a cell vector with as many elements as modes in the array otherwise an error
%   message will be generated. At the same time the length of each element of this cell array must be
%   0 (i.e. an empty array) or equal to the corresponding dimension otherwise an error message is
%   produced.
%   
%   >> Labels = {[1:2:60],[1:30],[]};
%   >> Data = cbdataset(X,'labels',Labels)
%   ??? Error using ==> d:\matlab\cubatch\@cbdataset\private\checklabels
%   Inconsistent labels length for mode 1
%   
%   When the cell element is empty the default values will be used.
%   The labels can be provided as a vector of doubles (generating 'Var. value' labels), as a char array
%   holding one label per line, as a cell vector of chars with one label per cell-element or as a label
%   object. In the latter case the scalars, the mode names and the other feature of the label object will
%   be saved in the cbdataset object without being automatically generated.
%   
%   e.g. the cell Labs generated as
%   
%   >> Labs{1}=label(1:30,'Sam.',[],'Samples','Samples');
%   >> Labs{2}=label([],'Var.',273.15:20:373.15,'T (K)','Temperature','K');
%   >> Labs{3}=label([],'Var.',100:150:700,'P (mB)','Pressure','mB');
%   >> Labs{4}=label([],'Var.',2:0.4:5,'pH','pH');
%   
%   can be used as labels for an X array of dimensions 30 x 6 x 5 x 8.
%   If the labels are not provided as label objects the default values for the label object will be
%   used as the labels are stored in the cbdataset object as label objects
%   
%   The labels can be extracted using the command:
%   
%   >> Data.labels
%   
%   yielding
%   
%   ans = 
%   
%       {20x1 cell}    {30x1 cell}    {25x1 cell}
%   
%   that is a cell row vector with the labels expressed for the n-th mode as cell arrays of chars
%   in the n-th element. Complex subreference is also allowed, hence to access the labels for the 2nd
%   mode one just has to type:
%   
%   >> Data.labels{2}
%   
%   ans = 
%   
%       'Var. 1'
%       'Var. 2'
%       'Var. 3'
%       ...
%       'Var. 29'
%       'Var. 30'
%       
%   (the ... is only for shortening the whole list of labels is actually displayed)
%   
%   All the features available for the label object (see @label\label.m) can also be directly accessed.
%   In particular the scalars are extracted by adding .scalars to the cbdataset object name:
%   
%   >> Data.scalars
%   
%   ans = 
%   
%       [20x1 double]    [30x1 double]    [25x1 double]
%       
%   As for the other features:
%   
%   .modenames : produce the axis names
%   .axislabels: produce the axis labels
%   .measunits : produce the measurement units for the different axis
%   
%   Unless further reference is provided this construction will yield a cell row vector with as many 
%   elements as the number of modes (dimension) of the data set.
%   
%   The labels of an existing cbdataset object can be modified using an analogous syntax:
%   
%   >> Data.labels{1} = label(1:3:60,'Sam.');
%   
%   >> Data.labels{1}
%   
%   ans = 
%   
%       'Sam. 1'
%       'Sam. 4'
%          ...
%       'Sam. 55'
%       'Sam. 58'
%       
%   Beside the labels the cbdataset object support the following fields: author, description, name
%   which can be modified and referred to using the same syntax used for the labels:
%   
%   Data.author     : content of the 'author' field
%   Data.dims       : dimensions of the cbdataset (it is mostly for internal use, it is possible to 
%                     use the standard size and length commands)
%   Data.description: content of the 'description' field
%   Data.data       : actual data array
%   Data.labobj     : cell row vector containing the label object for the different modes
%   Data.modified   : last modification of the array
%   Data.name       : name assigned to the data set (it can of course be different from the data set name and
%                     more complex, with the only limitation of not containing more than one line or CR or Tab 
%                     characters
%   
%   Data.dims, Data.labobj and Data.modified can be used only to read the values and cannot be set.
%   
%   A very important feature of the cbdataset object is that it can essentially be treated as a standard array
%   apart from a significant exception that will be explained below.
%   
%   Hence, defined a cbdataset object Data of dimensions 20 x 30 x 25 as above, one can write:
%   
%   >> Data2 = Data(1:2:end,:,:)
%   
%   yielding a data set with one every second horizontal slab:
%    
%    Name          : X
%    Type          : Data
%    Dimensions    : [10 x 30 x 25]
%    Author        : 
%    Description   : 
%    Last modified : 30/01/2002, 22:45:54
%   
%   The labels, scalars and all the other features are preserved and modified accordingly with the
%   advantage of not requiring separate commands.
%   A significant difference is that if an incomplete reference is given, e.g. if an index is
%   given only for the first mode, it will be completed so that the whole slabs are included:
%   the previous line can then be simplified as:
%   
%   >> Data2 = Data(1:2:end);
%   
%   This means that Data(:) does not vectorise the cbdataset as for normal arrays but simply yields
%   the whole data set. 
%   This syntax can be used in combination with the previous ones referring labels. 
%   F.i.
%   
%   >> Data(1:2:end).labels{1}
%   
%   ans = 
%   
%       'Sam. 1'
%       'Sam. 3'
%       'Sam. 5'
%       'Sam. 7'
%       'Sam. 9'
%       'Sam. 11'
%       'Sam. 13'
%       'Sam. 15'
%       'Sam. 17'
%       'Sam. 19'
%
%
%
%   Author: Giorgio Tomasi
%         Royal Agricolture University, MLI, LMT
%         1958 Frederiksberg C
%         Danmark
%         email: gt@kvl.dk


DS = [];
switch nargin
case 0
   DS.lastmod  = fix(clock);
   DS.data     = [];
   DS.userdata = [];
   C           = content;
   DS          = class(DS,'cbdataset',C);
   
case 1
   switch class(varargin{1})
   case 'cbdataset'
      DS = varargin{1};
   case 'double'
      if isempty(varargin{1})
         DS.lastmod  = fix(clock);
         DS.data     = [];
         DS.userdata = [];
         Cont        = content;
         Cont.name   = 'Empty';
      else
         DS.lastmod = fix(clock);
         DS.data    = varargin{1};
         DS.userdata = [];
         if ~isempty(inputname(1))
            Name = inputname(1);
         else
            Name = 'Matrix';
         end
         Cont = Content(size(varargin{1}),'name',Name);
      end
      DS   = class(DS,'cbdataset',Cont);
   end
   
otherwise
   if isa(varargin{1},'double')
      if ~any(strcmp(lower(varargin(cellfun('isclass',varargin,'char'))),'name'))
         if ~isempty(inputname(1))
            Name = inputname(1);
         else
            Name = 'Matrix';
         end
         varargin = [varargin(:)',{'name'},{Name}];
      end
      DS.lastmod  = fix(clock);
      DS.data     = varargin{1};
      DS.userdata = [];
      varargin(1) = [];
      W           = varargin(1:2:end);
      if all(cellfun('isclass',W,'char'))
         if rem(length(varargin),2)
            error('Properties'' names and their values must be inserted in pairs')
         end
         Prop = varargin(1:2:end);
         if length(unique(Prop)) ~= length(Prop)
            error('Properties can be defined only once')
         end
         if ~all(cellfun('isclass',Prop,'char'))
            error('Property names must be char strings')
         end
         if any(strcmp(lower(W),'userdata'))
            UD                = find(strcmp(lower(W),'userdata')) * 2;
            DS.userdata       = varargin{UD};
            varargin(UD-1:UD) = [];
            W = varargin(1:2:end);
         end
         LabPos = find(strcmp(lower(W),'labels')) * 2;
      else
         LabPos = {};
      end
      if ~isempty(LabPos)
         T = 1:length(varargin);
         T(LabPos - 1:LabPos) = [];
         t = find(~cellfun('length',varargin{LabPos}));
         for i = 1:length(t)
            if t(i) == 1
               varargin{LabPos}{t(i)} = label(1:size(DS.data,t(i)),'Sam.',[],'','Samples');
            else
               varargin{LabPos}{t(i)} = label(1:size(DS.data,t(i)),'Var.','',sprintf('Mode %i',t(i)));
            end
         end    
         Cont = Content(varargin{LabPos},varargin{T});   
      else
         if nargin == 2
            if isa(varargin{1},'content') %Simplified version
               DS = class(DS,'cbdataset',varargin{1});
               return
            end
         end
         Cont = Content(size(DS.data),varargin{1:end});
      end
   else
      error('The first parameter must be the data set')
   end
   DS = class(DS,'cbdataset',Cont);
end
