/******************************************************************POLYS
** funcmin.c                                                          **
**                                                                    **
** Function minimizations                                             **
** ------------------                                                 **
** This module contains the code for function minimizations           **
** It is used in the following situations:                            **
** Energy minimization in dihedral space (dihedmin)                   **
** Helix optimization (helix)                                         **
**                                                                    **
** Will be replaced by a simpler line search method such as linsch in **
** <linesearch.c> which is used for minimization in cartesian space.  **
**                                                                    **
** Compiled by Soren Balling Engelsen, INRA-93                        **
** Revisions:                                                         **
***********************************************************************/
#include <math.h>
/* TOLE = 0.02 used for phimin */

int    ncom=0;    /* defining declarations */
double *pcom=0, *xicom=0, (*nrfunc)();

void linmin(double p[], double xi[], int n, double tole, 
             double *fret, double (*func)())

{  int    j;
   double xx,xmin,fx,fb,fa,bx,ax;
   double brent(),f1dim(),*dvector();
   void   mnbrak(),free_dvector();

   ncom=n;
   pcom=dvector(1,n);
   xicom=dvector(1,n);
   nrfunc=func;
   for (j=1;j<=n;j++) 
   {  pcom[j]=p[j];
      xicom[j]=xi[j];
   }
   ax=0.0;
   xx=1.0;
   bx=2.0;
   mnbrak(&ax,&xx,&bx,&fa,&fx,&fb,f1dim);
   *fret=brent(ax,xx,bx,f1dim,tole,&xmin);
   for (j=1;j<=n;j++) 
   {  xi[j] *= xmin;
      p[j] += xi[j];
   }
   free_dvector(xicom,1,n);
   free_dvector(pcom,1,n);

} /* End of linmin */ 


extern int    ncom;         /* defined in LINMIN */
extern double *pcom, *xicom, (*nrfunc)();

double f1dim(double x)
 
{  int j;
   double f,*xt,*dvector();
   void free_dvector();

   xt=dvector(1,ncom);
   for (j=1;j<=ncom;j++) 
      xt[j]=pcom[j]+x*xicom[j];
   f=(*nrfunc)(xt);
   free_dvector(xt,1,ncom);
   return f;

} /* End of f1dim */


#define GOLD 1.618034
#define GLIMIT 100.0
#define TINY 1.0e-20
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#define SIGN(a,b) ((b) > 0.0 ? fabs(a) : -fabs(a))
#define SHFT(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);

void mnbrak(double *ax, double *bx, double *cx,
            double *fa, double *fb, double *fc, double (*func)(double))

{  double ulim,u,r,q,fu,dum;

   *fa=(*func)(*ax);
   *fb=(*func)(*bx);
   if (*fb > *fa) 
   {  SHFT(dum,*ax,*bx,dum)
      SHFT(dum,*fb,*fa,dum)
   }
   *cx=(*bx)+GOLD*(*bx-*ax);
   *fc=(*func)(*cx);
   while (*fb > *fc) 
   {  r=(*bx-*ax)*(*fb-*fc);
      q=(*bx-*cx)*(*fb-*fa);
      u=(*bx)-((*bx-*cx)*q-(*bx-*ax)*r)/
        (2.0*SIGN(MAX(fabs(q-r),TINY),q-r));
      ulim=(*bx)+GLIMIT*(*cx-*bx);
      if ((*bx-u)*(u-*cx) > 0.0) 
      {  fu=(*func)(u);
         if (fu < *fc) 
         {  *ax=(*bx);
            *bx=u;
            *fa=(*fb);
            *fb=fu;
            return;
         } 
         else 
         if (fu > *fb) 
         {  *cx=u;
            *fc=fu;
            return;
         }
         u=(*cx)+GOLD*(*cx-*bx);
         fu=(*func)(u);
      } 
      else 
      if ((*cx-u)*(u-ulim) > 0.0) 
      {  fu=(*func)(u);
         if (fu < *fc) 
         {  SHFT(*bx,*cx,u,*cx+GOLD*(*cx-*bx))
            SHFT(*fb,*fc,fu,(*func)(u))
         }
      } 
      else 
      if ((u-ulim)*(ulim-*cx) >= 0.0) 
      {  u=ulim;
         fu=(*func)(u);
      } 
      else 
      {  u=(*cx)+GOLD*(*cx-*bx);
         fu=(*func)(u);
      }
      SHFT(*ax,*bx,*cx,u)
      SHFT(*fa,*fb,*fc,fu)
   }
}

#undef GOLD
#undef GLIMIT
#undef TINY
#undef MAX
#undef SIGN
#undef SHFT


#define ITMAX 100
#define CGOLD 0.3819660
#define ZEPS 1.0e-10
#define SIGN(a,b) ((b) > 0.0 ? fabs(a) : -fabs(a))
#define SHFT(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);


double brent(double ax, double bx, double cx, double (*f)(double),
             double tol, double *xmin)

{  int iter;
   double a,b,d,etemp,fu,fv,fw,fx,p,q,r,tol1,tol2,u,v,w,x,xm;
   double e=0.0;

   a=((ax < cx) ? ax : cx);
   b=((ax > cx) ? ax : cx);
   x=w=v=bx;
   fw=fv=fx=(*f)(x);
   for (iter=1;iter<=ITMAX;iter++) 
   {  xm=0.5*(a+b);
      tol2=2.0*(tol1=tol*fabs(x)+ZEPS);
      if (fabs(x-xm) <= (tol2-0.5*(b-a))) 
      {  *xmin=x;
         return fx;
      }
      if (fabs(e) > tol1) 
      {  r=(x-w)*(fx-fv);
         q=(x-v)*(fx-fw);
         p=(x-v)*q-(x-w)*r;
         q=2.0*(q-r);
         if (q > 0.0) 
            p = -p;
         q=fabs(q);
         etemp=e;
         e=d;
         if (fabs(p) >= fabs(0.5*q*etemp) || p <= q*(a-x) || p >= q*(b-x))
            d=CGOLD*(e=(x >= xm ? a-x : b-x));
         else 
         {  d=p/q;
            u=x+d;
            if (u-a < tol2 || b-u < tol2)
               d=SIGN(tol1,xm-x);
         }
      } 
      else 
      {  d=CGOLD*(e=(x >= xm ? a-x : b-x));
      }
      u=(fabs(d) >= tol1 ? x+d : x+SIGN(tol1,d));
      fu=(*f)(u);
      if (fu <= fx) {
			if (u >= x) a=x; else b=x;
			SHFT(v,w,x,u)
			SHFT(fv,fw,fx,fu)
		} else {
			if (u < x) a=u; else b=u;
			if (fu <= fw || w == x) {
				v=w;
				w=u;
				fv=fw;
				fw=fu;
			} else if (fu <= fv || v == x || v == w) {
				v=u;
				fv=fu;
			}
		}
	}
	printf("Number of iterations in BRENT exceeded\n");
	*xmin=x;
	return fx;
}

#undef ITMAX
#undef CGOLD
#undef ZEPS
#undef SIGN
