function [Factors,Diagnostics] = dGN(varargin);
% Fit a PARAFAC model to a three-way array using a damped Gauss-Newton (Levenberg-Marquadt) algorithm
%
% SYNTAX
% [Factors,Diag] = dGN(X,F,Options,A,B,C);
% 
% INPUTS
% X      : data array
% F      : model's rank
% Options: see ParOptions
% A,B,C  : initial estimates for the loading matrices
%
% OUTPUTS
% Factors: cell vectors with the loading matrices (NB Sign permutation is
%          not fixed according to any convention)
% Diag   : structure with some diagnostics
%          Diag.fit(1)        : value of the loss function at convergence
%          Diag.fit(2)        : not used
%          Diag.fit(3)        : not used
%          Diag.it(4)         : not used
%          Diag.it(1)         : total number of iterations
%          Diag.it(2)         : number of iterations where the dGN step was accepted
%          Diag.it(3)         : not used
%          Diag.it(4)         : not used
%          Diag.convergence(1): Relative fit decrease
%          Diag.convergence(2): Relative change in the parameter's norm
%          Diag.convergence(3): Relative loss function value lower than machine precision
%          Diag.convergence(4): Gradient's infinite norm
%          Diag.convergence(5): Not used
%          Diag.convergence(6): Max number of iterations reached
%          Diag.convergence(7): Not used
%
% Author: Giorgio Tomasi 
%         Royal Agricultural and Veterinary University 
%         Rolighedsvej 30 
%         DK-1958 Frederiksberg C 
%         Denmark 
% 
% Last modified: 10-Jan-2005 15:48
% 
% Contact: Giorgio Tomasi, gt@kvl.dk; Rasmus Bro, rb@kvl.dk 
% 
% Reference: "A comparison of algorithms for fitting the PARAFAC model"
%            G. Tomasi, R. Bro, Computational Statistics and Data Analysis, in press.
%

% Check for minimal input
if ~nargin
    help gnparafac_lm
    return
end

%Check input values
[X,F,Options,Factors] = Check_GenParafac_Input(varargin{:});
if isempty(F)
   [Factors,Diagnostics] = deal([]);
   return
end

% Compute initial estimates if not given
if isempty(Factors)                                         
    [Factors{1:ndims(X)}] = InitPar(X,F,Options);
end

%Some initial values
ConvCrit   = false(7,1);                                    % Convergence criteria
it         = zeros(4,1);                                    % Number of iterations [global hessian_computations not_used not_used]
Ni         = 2;                                             % Parameter in the damping parameter's updating scheme
FactorsNew = Factors;                                       % Model parameters new estimates 
dimX       = size(X);                                       % X dimensions vector
SSX        = tss(X,false);                                  % Total sum of squares
ConvGrad   = Options.convcrit.grad;                         % Convergence criteria: gradient infinite norm
ConvRelFit = Options.convcrit.relfit;                       %                       relative fit decrease
ConvPar    = Options.convcrit.par;                          %                       relative parameter change
ConvFit    = Options.convcrit.fit;                          %                       loss function/total sum of squares
ConvMaxIt  = Options.convcrit.maxiter;                      %                       max n of iterations
LamHistory = zeros(ConvMaxIt,1);                            % History of the damping parameter, used as a diagnostic
FitNew     = tss(X - nmodel(Factors),false);                % Initial fit

if strcmpi(Options.diagnostics,'on')
    FitHistory  = zeros(ConvMaxIt,1);                       % Initialise history of loss function values
end

if ~isequal(Options.display,'none')                         % Display titles for iteration's information     
    fprintf('\n\n            Fit           It      EV%%     Rho            Lambda             Max Gr\n')
end

%Start fitting
while ~any(ConvCrit)                                        % Start the outer loop
    
    % Outer loop: iterations that require a new computation of H and g
    
    it(2)                 = it(2) + 1;                      % Update the # iterations for the outer loop 
    [Factors{1:ndims(X)}] = scale_factors(0,Factors{:});    % Scale loadings vectors to equal norm
    FactorsOld            = Factors;                        % Save old factors
    FitOld                = FitNew;                         % Save old fit
    p                     = vec(cat(1,Factors{:})');        % Vector of parameter estimates
    NormPar               = norm(p);                        % Norm of the vector of parameters
    if ~it(1)
        [g,JtJ,Lambda] = ParafacDer(Factors,X);             % Calculate Hessian and Gradient
        Lambda         = Lambda * Options.lambdainit;       % initialise the damping parameter
    else
        [g,JtJ]        = ParafacDer(Factors,X);             % Calculate Hessian and Gradient
    end
    LamHistory(it(2)) = Lambda;                             % Store the values of the damping parameter to display diagnostic
    Norm_g            = max(abs(g));                        % Gradient infinite norm
    if strcmpi(Options.diagnostics,'on')                    % Save some diagnostics
        FitHistory(it(2))   = FitOld;                       % Store fit values for accepted steps
    end
    ConvCrit(4,1) = ConvGrad >= Norm_g;                     % Stop if gradient's infinite norm is smaller than a predefined criterion
    Do_It         = true;                                   % Do inner loop
    
    while Do_It && ~any(ConvCrit)                           % Start the inner loop
        
        % Inner loop: if the step is rejected, only the damping parameter is
        %             updated. There is no need to recompute Hessian and gradient
        
        it(1)  = it(1) + 1;                                 % Update # iterations             
        Htilde = JtJ + Lambda * eye(size(JtJ,1));           % Compute left hand side of the modified normal equations               
        
        % Solve the system of modified normal equations using Cholesky factorisation
        [Htilde,CFlag] = chol(Htilde);                      % Compute the Cholesky factor of Htilde
        if ~CFlag                                           % Htilde is positive definite -> a descent direction can be computed
            
            warning off                                     % Avoid displaying the warning on bad scaling
            lastwarn('')
            deltap = Htilde\(Htilde'\g);                    % Compute update calculated by back substitution
            if isempty(lastwarn)                            % The matrix is nicely scaled
                
                Norm_deltap  = norm(deltap);                % Norm of the update
                p_new        = p + deltap;                  % Compute updated parameters
                Count        = 0;                           % Change format to cell
                for m = 1:length(dimX)
                    FactorsNew{m} = reshape(p_new(Count + 1:Count + dimX(m) * F),F,dimX(m))';
                    Count         = Count + dimX(m) * F;
                end      
                FitNew = tss(X - nmodel(FactorsNew),false); % Compute fit with updated parameters
                
                % Update damping parameter according to 
                % K. Madsen et al, Methods for non-linear least squares problems (Dept. Mathematical Modelling, Technical University of Denmark, Lyngby, Denmark, ed. 2nd, 2004).)
                LinDec = deltap' * (Lambda * deltap + g);   % Linear decrease
                Do_It  = false;                             % Accept update and exit inner loop (default)
                warning off                                 % In case LinDec is exactly zero
                Rho    = (FitOld - FitNew) / LinDec;        % Compute Gain ratio
                warning on
                if Rho <= 0                                 % Reject update and repeat inner loop
                    Do_It  = true;
                    Lambda = Lambda * Ni;
                    Ni     = Ni * 2;
                else
                    Lambda = Lambda * max(Options.lambdaudpar(2),1 - (2 * Rho - 1)^3);
                    Ni     = 2;
                end
                
                %Check convergence
                if ~FitNew
                    ConvCrit(1,1) = true ;                                      % In case model fit the data exactly
                else
                    ConvCrit(1,1) = ConvRelFit >= abs(FitOld - FitNew)/FitOld;  % Relative fit decrease
                end
                ConvCrit(2,1) = ConvPar >= Norm_deltap/NormPar;                 % Relative change in the parameters
                
            else
                Lambda = Lambda * Options.lambdaudpar(1);   % Increase damping parameter if Htilde is nearly singular
            end
            
        else
            Lambda = Lambda * Options.lambdaudpar(1);       % Increase the Hessian approximation because Htilde is singular to 
                                                            % machine precision 
        end
        ConvCrit(6,1) = it(1) == ConvMaxIt;                 % Max. number of iterations
        
    end % End of the inner loop
    Factors       = FactorsNew;                             % Accept new estimates
    ConvCrit(3,1) = ConvFit  >= FitNew / SSX;               % Check that loss function is not too small compared to Frobenius norm of X
    
    % Display some information on iterations 
    if ~isequal(Options.display,'none') && ~rem(it(1),Options.display) && ~any(ConvCrit)
        DisplayIt_dGN(it(1),FitNew,SSX,Rho,LamHistory(it(2)),Norm_g,Options)
    end
    
end % End of the outer loop

% Scale the factors according to the common convention
[Factors{end:-1:1}] = scale_factors(1,Factors{end:-1:1});

% Sort the factors according to their norm (in decresing order)
[nil,Seq]             = sort(-sum(Factors{1}.^2));
[Factors{1:ndims(X)}] = FacPerm(Seq,Factors{:});

if nargout > 1
    Diagnostics             = struct('fit',FitNew,'it',it,'convergence',ConvCrit);    
end
if strcmpi(Options.diagnostics,'on')  
    
    % Display some details about convergence
    ConvMsg = {'Relative fit decrease'
        'Parameters'' update'
        'Loss function value of less than machine precision'
        'Gradient equal to zero'
        ''
        'Max number of iterations reached'};
    ConvMsg = char(ConvMsg(ConvCrit));
    fprintf('\n The algorithm has converged after %i iterations (%i Hessian computations)',it(1:2))
    fprintf('\n Met convergence criteria: %s',ConvMsg(1,:))
    if size(ConvMsg,1) > 1
        fprintf('\n                           %s',ConvMsg(1,:))
    end   
    fprintf('\n')
    
    figure('name','Diagnostics','number','off')
    %Show the damping parameter and the fit history
    ax = plotyy(1:it(2),LamHistory(1:it(2)),1:it(2),FitHistory(1:it(2)),@semilogy);
    axes(ax(1))
    ylabel('\lambda')
    xlabel('# it')
    axes(ax(2))
    ylabel('L(\bfA\rm,\bfB\rm,\bfC\rm)')
    title('Lambda and loss function history')
    drawnow
    
end

%----------------------------------------------------------------------------------------------------------------------

function DisplayIt_dGN(It,Fit,SSX,Rho,LambdaOld,Gr,Options)
FitStr = sprintf('%12.10f',Fit);
FitStr = [char(32*ones(1,22-length(FitStr))),FitStr];
ItStr  = num2str(It);
ItStr  = [char(32*ones(1,length(num2str(Options.convcrit.maxiter))-length(ItStr))),ItStr];
VarStr = sprintf('%2.4f',100*(1-Fit/SSX));
VarStr = [char(32*ones(1,7-length(VarStr))),VarStr];
RatStr = sprintf('%2.4f',Rho);
RatStr = [char(32*ones(1,7-length(RatStr))),RatStr];
LamStr = sprintf('%12.4f',LambdaOld);
LamStr = [char(32*ones(1,17-length(LamStr))),LamStr];
GrStr  = sprintf('%8.4f',Gr);
GrStr  = [char(32*ones(1,15-length(GrStr))),GrStr];
fprintf([' ',FitStr,'  ',ItStr,'  ',VarStr,'  ',RatStr,'  ',LamStr,'   ',GrStr]);
fprintf('\n');

