function [g,H,L] = ParafacDer(Factors,X)
% Compute Gradient, JTJ and largest diagonal element of JTJ for a three-way
% PARAFAC model
%
% SYNTAX
% [Gr,H,L] = ParafacDer(Factors,X)
%
% INPUT
% Factors: cell vector with the loading matrices as elements
% X      : three-way array
%
% OUTPUT
% Gr     : gradient
% H      : J' * J where J is the Jacobian
% L      : largest diagonal element of H
%
% Author: Giorgio Tomasi 
%         Royal Agricultural and Veterinary University 
%         Rolighedsvej 30 
%         DK-1958 Frederiksberg C 
%         Denmark 
% 
% Last modified: 10-Jan-2005 15:23
% 
% Contact: Giorgio Tomasi, gt@kvl.dk; Rasmus Bro, rb@kvl.dk 
% 
% References: "A comparison of algorithms for fitting the PARAFAC model"
%             G. Tomasi, R. Bro, Computational Statistics and Data Analysis, in press.
%             "Cramer-Rao lower bounds for low-rank decomposition of multidimensional arrays"
%             X. Q. Liu, N. D. Sidiropoulos, Ieee Transactions on Signal Processing 49, 2074 (2001).

[I,J,K] = size(X);
[A,B,C] = deal(Factors{:});
F       = size(A,2);
cdim    = F * [cumsum([I,J,K])];
N       = ndims(X);
g       = zeros(cdim(end),1);
H       = [];

if nargout >= 2
    
    H   = zeros(cdim(end));
    % Accessorial matrices (cf. reference)
    GA  = A' * A;
    GB  = B' * B;
    GC  = C' * C;
    Ind = repmat([1:I],F,1);FAT = A'; FAT = FAT(:,Ind);
    Ind = repmat([1:J],F,1);FBT = B'; FBT = FBT(:,Ind);
    Ind = repmat([1:K],F,1);FCT = C'; FCT = FCT(:,Ind);
    
    % Diagonal blocks
    T                                          = GB .* GC;
    L                                          = max(diag(T));
    H(1:cdim(1),1:cdim(1))                     = full(kron(speye(I),T));
    T                                          = GA .* GC;
    L                                          = max([L;diag(T)]);
    H(cdim(1) + 1:cdim(2),cdim(1) + 1:cdim(2)) = full(kron(speye(J),T));
    T                                          = GA .* GB;
    L                                          = max([L;diag(T)]);
    H(cdim(2) + 1:cdim(3),cdim(2) + 1:cdim(3)) = full(kron(speye(K),T));
    
    %AB off-diagonal block
    H(cdim(1) + 1:cdim(2),1:cdim(1)) = repmat(FAT,J,1) .* repmat(FBT',1,I) .* repmat(GC,J,I);
    H(1:cdim(1),cdim(1) + 1:cdim(2)) = H(cdim(1) + 1:cdim(2),1:cdim(1))';
    
    %AC off-diagonal block
    H(cdim(2) + 1:cdim(3),1:cdim(1)) = repmat(FAT,K,1) .* repmat(FCT',1,I) .* repmat(GB,K,I);
    H(1:cdim(1),cdim(2) + 1:cdim(3)) = H(cdim(2) + 1:cdim(3),1:cdim(1))';
    
    %BC off-diagonal block
    H(cdim(1) + 1:cdim(2),cdim(2) + 1:cdim(3)) = repmat(FCT,J,1) .* repmat(FBT',1,K) .* repmat(GA,J,K);
    H(cdim(2) + 1:cdim(3),cdim(1) + 1:cdim(2)) = H(cdim(1) + 1:cdim(2),cdim(2) + 1:cdim(3))';
    
end

% Compute gradient (cf. reference)
X  = reshape(X,I,J * K) - A * kr(C,B)';
g(1:cdim(1)) = vec((X * kr(C,B))'); 
X  = reshape(X',J,K * I);
g(cdim(1) + 1:cdim(2)) = vec((X * kr(A,C))');
X  = reshape(X',K,I * J);
g(cdim(2) + 1:cdim(3)) = vec((X * kr(B,A))');